/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <gmf.h>
#include <libgmf/gnome-genericfactory.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>

#include <audiofile.h>
#include "gmf-digital-audio.h"

#define NNOTES 2

typedef struct {
  CORBA_ORB orb;
  CORBA_Environment *ev;
  GMFTimeReference *timeref;
  GnomeGenericFactory *factory;
  GList *filters;
} AppInfo;

typedef struct {
  AppInfo *ai;
  GMFFilter *filter;
  char *filename;
  AFfilehandle fh;
  GMF_Media_AudioDigitalFormat streaminfo;
  GMF_Callback_CallbackID cbid;
  GMF_TimeVal timeBase;
  GMFTimeReference *timeref;
} FiltInfo;

static void wav_create_filter(GnomeGenericFactory *factory,
			      const char *goad_id, GNOME_stringlist *params,
			      CORBA_Object *new_object, AppInfo *ai);

static void wav_out_get_pipe(GMFFilter *filter,
			     GMF_Direction pDirection,
			     GMFPipe **pipe,
			     FiltInfo *fi);
static void wav_run(GMFFilter *filter, GMF_TimeVal *timeBase, FiltInfo *fi);
static void wav_stop(GMFFilter *filter, FiltInfo *fi);
static void wav_pause(GMFFilter *filter, FiltInfo *fi);

int main(int argc, char *argv[])
{
  CORBA_Environment ev;
  poptContext ctx;
  AppInfo appinfo;

  CORBA_exception_init(&ev);
  appinfo.ev = &ev;
  appinfo.orb = gnome_CORBA_init_with_popt_table("wav-file-filter", VERSION,
						 &argc, argv, NULL, 0,
						 &ctx, GNORBA_INIT_SERVER_FUNC,
						 &ev);

  appinfo.factory = GNOME_GENERIC_FACTORY(gnome_generic_factory_new("gmf-filter-wav-file-factory"));
  appinfo.timeref = GMF_TIME_REFERENCE(gmf_time_reference_new());

  gtk_signal_connect(GTK_OBJECT(appinfo.factory), "create_object",
		     GTK_SIGNAL_FUNC(wav_create_filter), &appinfo);

  gtk_main();

  return 0;
}

static void
wav_destroy_filter(GMFFilter *filter, FiltInfo *fi)
{
  AppInfo *ai;

  ai = fi->ai;
  ai->filters = g_list_remove(ai->filters, fi);
  if(fi->filename) {
    afCloseFile(fi->fh);
    g_free(fi->filename);
  }
  g_free(fi);

  if(!g_list_length(ai->filters))
    gtk_main_quit();
}

static void
wav_create_filter(GnomeGenericFactory *factory,
		  const char *goad_id, GNOME_stringlist *params,
		  CORBA_Object *new_object, AppInfo *ai)
{
  GMFFilter *new_filter;
  GMF_Filter_Type ftype;
  FiltInfo *fi;

  if(!strcmp(goad_id, "gmf-filter-wav-file")) {
    ftype = GMF_Filter_SOURCE;
  } else {
    return; /* Can't activate anything else */
  }

  new_filter = GMF_FILTER(gmf_filter_new(ftype, goad_id));
  g_assert(new_filter);

  fi = g_new0(FiltInfo, 1);
  fi->ai = ai;
  gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		     GTK_SIGNAL_FUNC(wav_out_get_pipe), fi);

  fi->filter = new_filter;

  ai->filters = g_list_prepend(ai->filters, fi);

  gtk_signal_connect(GTK_OBJECT(fi->filter), "destroy",
		     GTK_SIGNAL_FUNC(wav_destroy_filter), fi);
  gtk_signal_connect(GTK_OBJECT(fi->filter), "run",
		     GTK_SIGNAL_FUNC(wav_run), fi);
  gtk_signal_connect(GTK_OBJECT(fi->filter), "stop",
		     GTK_SIGNAL_FUNC(wav_stop), fi);
  gtk_signal_connect(GTK_OBJECT(fi->filter), "pause",
		     GTK_SIGNAL_FUNC(wav_pause), fi);

  *new_object = CORBA_Object_duplicate(new_filter->corba_object, ai->ev);
}

static void wav_can_process(GMFPipe *pipe, GMF_MediaTypeInfo *type_info,
			    gboolean *retval)
{
  if(type_info->majorType != GMF_MEDIA_AUDIO_DIGITAL)
    return;

  if(strcmp(type_info->minorType, "audio/raw"))
    return;

  *retval = TRUE;
}

static void wav_processable_types(GMFPipe *pipe,
				  GMF_MediaTypeInfoList **out_typelist,
				  FiltInfo *fi)
{
  GMF_MediaTypeInfoList *list;
  GMF_MediaTypeInfo *ti;

  list = GMF_MediaTypeInfoList__alloc();
  list->_length = 1;
  list->_buffer = CORBA_sequence_GMF_MediaTypeInfo_allocbuf(list->_length);

  ti = &list->_buffer[0];
  {
    GMF_Media_AudioDigitalFormat *val;

    memset(ti, 0, sizeof(*ti));
    ti->majorType = GMF_MEDIA_AUDIO_DIGITAL;
    ti->minorType = CORBA_string_dup("audio/raw");
    ti->typeData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
    ti->formatData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_GMF_Media_AudioDigitalFormat, fi->ai->ev);
    ti->formatData._value = val = GMF_Media_AudioDigitalFormat__alloc();
    CORBA_any_set_release(&ti->formatData, CORBA_TRUE);

    *val = fi->streaminfo;
  }

  *out_typelist = list;
}

static void wav_out_get_pipe(GMFFilter *filter,
			     GMF_Direction pDirection,
			     GMFPipe **pipe,
			     FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_OUT); /* We only allow data to be fed in */

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(wav_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(wav_processable_types), fi);

  *pipe = newpipe;
}

static gboolean
wav_send_sample(FiltInfo *fi)
{
  return TRUE;
}

static void
wav_run(GMFFilter *filter, GMF_TimeVal *timeBase, FiltInfo *fi)
{
  GMF_TimeVal freq;
  fi->timeBase = *timeBase;
  fi->cbid =
    gmf_time_reference_request_metronome(fi->timeref,
					 timeBase, &freq,
					 (GSourceFunc)wav_send_sample,
					 fi);
}

static void
wav_stop(GMFFilter *filter, FiltInfo *fi)
{
  gmf_time_reference_remove_request(fi->timeref, fi->cbid);
}

static void
wav_pause(GMFFilter *filter, FiltInfo *fi)
{
  gmf_time_reference_remove_request(fi->timeref, fi->cbid);
}
