/*  Glimmer - insert.c
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "declarations.h"
#include "insert.h"
#include "dialogs.h"
#include "file-io.h"
#include "main.h"
#include "misc.h"
#include "settings.h"

static void insert_ascii_char(GtkWidget *widget, GnomeAsciiList *ascii);
static void insert_file_cb_continue(GtkWidget *widget, GnomeFileList *file_list);
static void insert_file_real(gchar *filename);
static void insert_color_cb_continue(GtkWidget *widget, GtkColorSelectionDialog *colorsel);
static void runcmd_cb_continue(GtkWidget *widget, GnomeRunCmd *grc);
static void indent_cb_continue(GtkWidget *widget, GnomeRunCmd *grc);

void insert_date_time(GtkWidget *widget, gpointer data)
{
   char buffer[128];
   time_t cur_time;
   struct tm *time_ptr;
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   time(&cur_time);
   time_ptr = localtime(&cur_time);
   strftime(buffer, sizeof(buffer), "%a %b %d %X %Z %Y", time_ptr);
   gds_editor_insert_text(GDS_EDITOR(extext), buffer, strlen(buffer), (gint *)&extext->current_pos);
}

void insert_username(GtkWidget *widget, gpointer data)
{
   gchar *text;
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   text = g_new(char, strlen(getenv("USER"))+1);
   strcpy(text, getenv("USER"));
   gds_editor_insert_text(GDS_EDITOR(extext), text, strlen(text), (gint *)&extext->current_pos);
   g_free(text);
}

void insert_name(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), general_preferences.name, strlen(general_preferences.name), (gint *)&extext->current_pos);
}

void insert_email(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), general_preferences.email, strlen(general_preferences.email), (gint *)&extext->current_pos);
}

// C/C++ related tags

void insert_gpl_notice(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;

   gchar *gpl_notice =
   {
      "/*\n"
      " *  This program is free software; you can redistribute it and/or modify\n"
      " *  it under the terms of the GNU General Public License as published by\n"
      " *  the Free Software Foundation; either version 2 of the License, or\n"
      " *  (at your option) any later version.\n"
      " *\n"
      " *  This program is distributed in the hope that it will be useful,\n"
      " *  but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
      " *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
      " *  GNU Library General Public License for more details.\n"
      " *\n"
      " *  You should have received a copy of the GNU General Public License\n"
      " *  along with this program; if not, write to the Free Software\n"
      " *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
      " */\n"
   };

   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), gpl_notice, strlen(gpl_notice), (gint *)&extext->current_pos);
}

void insert_gpl_noticepp(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;

   gchar gpl_notice[] =
   {
      "// This program is free software; you can redistribute it and/or modify\n"
      "// it under the terms of the GNU General Public License as published by\n"
      "// the Free Software Foundation; either version 2 of the License, or\n"
      "// (at your option) any later version.\n"
      "//\n"
      "// This program is distributed in the hope that it will be useful,\n"
      "// but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
      "// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
      "// GNU Library General Public License for more details.\n"
      "//\n"
      "// You should have received a copy of the GNU General Public License\n"
      "// along with this program; if not, write to the Free Software\n"
      "// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
      "\n"
   };

   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), gpl_notice, strlen(gpl_notice), (gint *)&extext->current_pos);
}

void insert_gpl_noticepython(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;

   gchar *gpl_notice =
   {
      "#\n"
      "# This program is free software; you can redistribute it and/or modify\n"
      "# it under the terms of the GNU General Public License as published by\n"
      "# the Free Software Foundation; either version 2 of the License, or\n"
      "# (at your option) any later version.\n"
      "#\n"
      "# This program is distributed in the hope that it will be useful,\n"
      "# but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
      "# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
      "# GNU Library General Public License for more details.\n"
      "#\n"
      "# You should have received a copy of the GNU General Public License\n"
      "# along with this program; if not, write to the Free Software\n"
      "# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
      "\n"
   };

   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), gpl_notice, strlen(gpl_notice), (gint *)&extext->current_pos);
}

void insert_changelog_string(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;
   gchar buffer[128];
   gchar *full;
   time_t cur_time;
   struct tm *time_ptr;

   extext = GTK_EXTEXT(cur_file->text);
   time(&cur_time);
   time_ptr = localtime(&cur_time);
   strftime(buffer, sizeof(buffer), "%a %b %d %X %Z %Y", time_ptr);
   full = g_strconcat(buffer, " ", general_preferences.name, " <", general_preferences.email, ">", NULL);
   if(full && strlen(full))
     gds_editor_insert_text(GDS_EDITOR(extext), full, strlen(full), (gint *)&extext->current_pos);
   g_free(full);
}

void insert_comment(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), "/*\n *\n */", 9, (gint *)&extext->current_pos);
}

void insert_commentpp(GtkWidget *widget, gpointer data)
{
   GtkExText *extext;
   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(GDS_EDITOR(extext), "//", 2, (gint *)&extext->current_pos);
}

void insert_ascii_char_cb(GtkWidget *widget, gpointer data)
{
   GnomeAsciiList *ascii;
   widget = gnome_ascii_list_new();
   ascii = GNOME_ASCII_LIST(widget);
   gtk_signal_connect(GTK_OBJECT(ascii->insert_button), "clicked", GTK_SIGNAL_FUNC(insert_ascii_char), ascii);
   gtk_signal_connect_object(GTK_OBJECT(ascii->close_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(ascii));
   gtk_widget_show(widget);
}

static void insert_ascii_char(GtkWidget *widget, GnomeAsciiList *ascii)
{
   gchar *text;
   if(!cur_file) return;
   text = gtk_entry_get_text(GTK_ENTRY(ascii->entry));
   gtk_extext_insert_text(GTK_EXTEXT(cur_file->text), text, strlen(text), (gint *)&GTK_EXTEXT(cur_file->text)->current_pos);
}

void insert_file_cb(GtkWidget *widget, gpointer data)
{
   file_selector = last_dir ? gnome_filelist_new_with_path(last_dir) : gnome_filelist_new_with_path(general_preferences.default_dir);
   gnome_filelist_set_title(GNOME_FILELIST(file_selector), _("Insert File..."));
   gtk_signal_connect(GTK_OBJECT(GNOME_FILELIST(file_selector)->ok_button), "clicked", GTK_SIGNAL_FUNC(insert_file_cb_continue), file_selector);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(file_selector)->ok_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(file_selector));
   gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(file_selector)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(file_selector));
   gtk_widget_show(file_selector);
}

static void insert_file_cb_continue(GtkWidget *widget, GnomeFileList *file_list)
{
   gchar *string;
   string = gnome_filelist_get_filename(GNOME_FILELIST(file_list));
   if(check_if_file_exists(string))
   {
      insert_file_real(string);
   }
   else
   {
      read_error(string);
   }
   g_free(string);
}

static void insert_file_real(gchar *filename)
{
   gchar *temp = NULL;
   if(!cur_file) return;
   temp = get_path_from_filename(filename);
   if(temp && strcmp(filename, temp) != 0)
   {
      g_free(last_dir);
      last_dir = g_new(char, strlen(temp)+1);
      strcpy(last_dir, temp);
   }
   g_free(temp);
   open_file_add_text(cur_file->text, filename);
   adjust_sensitivity();
   display_message("File Inserted.", FLASH);
}

void insert_color_cb(GtkWidget *widget, gpointer data)
{
   GtkExText *text;
   gdouble colors[4];
   unsigned int r, g, b;
   gchar *string;
   gboolean colors_found = FALSE;

   text = GTK_EXTEXT(cur_file->text);
   if(text->has_selection && (text->selection_end_pos - text->selection_start_pos) == 7)
   {
      string = gtk_extext_get_chars(text, text->selection_start_pos, text->selection_end_pos);
      if(string && (string[0] == '#') && (sscanf(string, "#%02X%02X%02X", &r, &g, &b) == 3))
      {
         colors_found = TRUE;
         colors[0] = (gdouble)r / 255;
         colors[1] = (gdouble)g / 255;
         colors[2] = (gdouble)b / 255;
      }
      g_free(string);
   }
   widget = gtk_color_selection_dialog_new("Pick a color...");
   if(colors_found) gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(widget)->colorsel), colors);
   gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(insert_color_cb_continue), widget);
   gtk_signal_connect_object(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_signal_connect_object(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_widget_show(widget);
}

static void insert_color_cb_continue(GtkWidget *widget, GtkColorSelectionDialog *colorsel)
{
   gchar string[] = "#000000";
   gdouble colors[4];
   guint16 r, g, b;
   gtk_color_selection_get_color(GTK_COLOR_SELECTION(colorsel->colorsel), colors);
   r = colors[0] * 256;
   g = colors[1] * 256;
   b = colors[2] * 256;
   g_snprintf(string, sizeof(string), "#%02X%02X%02X", r, g, b);
   gtk_extext_insert_text(GTK_EXTEXT(cur_file->text), string, strlen(string), &GTK_EXTEXT(cur_file->text)->current_pos);
   display_message("Color Inserted.", FLASH);
}

void runcmd_cb(GtkWidget *widget, gpointer data)
{
   g_return_if_fail(cur_file != NULL);
   widget = gnome_run_cmd_new("Run Command...", "Command: ");
   gtk_signal_connect(GTK_OBJECT(GNOME_RUN_CMD(widget)->run_button), "clicked", GTK_SIGNAL_FUNC(runcmd_cb_continue), widget);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_RUN_CMD(widget)->run_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_signal_connect_object(GTK_OBJECT(GNOME_RUN_CMD(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gnome_run_cmd_set_history(GNOME_RUN_CMD(widget), build_glist_from_file("run.hist", general_preferences.history));
   gtk_widget_show(widget);
   gtk_widget_grab_focus(GNOME_RUN_CMD(widget)->entry);
}

static void runcmd_cb_continue(GtkWidget *widget, GnomeRunCmd *grc)
{
   gchar *string;
   FILE *exec;
   GtkExText *extext;
   gchar output[384];

   g_return_if_fail(cur_file != NULL);
   if(g_list_index(files_list, (gpointer)cur_file)<0) return;
   extext = GTK_EXTEXT(cur_file->text);
   string = gtk_entry_get_text(GTK_ENTRY(grc->entry));
   add_string_to_file("run.hist", string);

   if(!(exec = popen(string, "r")))
   {
      perror("Error opening process!");
      generic_dialog_with_text("Process could not be opened.");
      g_print("\a");
      return;
   }
   gtk_extext_freeze(extext);
   while(fgets(output, sizeof(output), exec))
   {
      gtk_extext_insert_text(extext, output, strlen(output), (gint *)&extext->current_pos);
      while(gtk_events_pending()) gtk_main_iteration();
   }
   gtk_extext_thaw(extext);
   pclose(exec);
}

void indent_cb(GtkWidget *widget, gpointer data)
{
   g_return_if_fail(cur_file != NULL);
   widget = gnome_run_cmd_new("Indent...", "Indent: ");
   gtk_signal_connect(GTK_OBJECT(GNOME_RUN_CMD(widget)->run_button), "clicked", GTK_SIGNAL_FUNC(indent_cb_continue), widget);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_RUN_CMD(widget)->run_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_signal_connect_object(GTK_OBJECT(GNOME_RUN_CMD(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gnome_run_cmd_set_history(GNOME_RUN_CMD(widget), build_glist_from_file("indent.hist", general_preferences.history));
   gtk_widget_show(widget);
   gtk_widget_grab_focus(GNOME_RUN_CMD(widget)->entry);
}

static void indent_cb_continue(GtkWidget *widget, GnomeRunCmd *grc)
{
   gchar *string;
   FILE *exec;
   FILE *file;
   gchar temp_file[256];
   gchar *full_string;
   GtkExText *extext;
   gint start;
   gint end;
   gchar *text;
   g_return_if_fail(cur_file != NULL);
   extext = GTK_EXTEXT(cur_file->text);
   string = gtk_entry_get_text(GTK_ENTRY(grc->entry));
   add_string_to_file("indent.hist", string);

   g_snprintf(temp_file, sizeof(temp_file), "%s/." PACKAGE "/%s", getenv("HOME"), "indent.x");
   file = fopen(temp_file, "w");
   if(!file) return;
   start = 0;
   end = extext->length;
   if(extext->has_selection)
   {
      start = extext->selection_start_pos;
      end = extext->selection_end_pos;
   }
   text = gtk_extext_get_chars(extext, start, end);
   if(text) fputs(text, file);
   fclose(file);
   g_free(text);
   full_string = g_strconcat("indent ", string, " ", temp_file, NULL);
   if(!(exec = popen(full_string, "r")))
   {
      perror("Error opening process!");
      generic_dialog_with_text("Indent process could not be opened.");
      g_print("\a");
      g_free(full_string);
      return;
   }
   g_free(full_string);
   pclose(exec);
   gtk_extext_freeze(extext);
   gtk_extext_delete_text(extext, start, end);
   gtk_extext_set_position(extext, start);
   open_file_add_text(cur_file->text, temp_file);
   remove(temp_file);
   gtk_extext_thaw(extext);
}
