#include "gnomesearchdialog.h"

/* function declarations */
static void gnome_search_dialog_class_init(GnomeSearchDialogClass *_class);
static void gnome_search_dialog_init(GnomeSearchDialog *search_dialog);
GtkWidget *gnome_search_dialog_new(GnomeSearchDialogType type);
static void gnome_search_dialog_destroy(GtkObject *object);
static gint check_glist_for_multiple_entries(GList *list, gchar *new_entry);
static void kill_combo_history(GList *list);
void activate_find_button(GtkWidget *widget, GnomeSearchDialog *sd);
void set_find_button_sensitive(GtkWidget *widget, GnomeSearchDialog *sd);
/* end function declarations */

static GtkWindowClass *parent_class = NULL;

GtkType gnome_search_dialog_get_type(void)
{
   static GtkType search_dialog_type=0;
   if(!search_dialog_type)
   {
      static const GtkTypeInfo search_dialog_info = 
      {	
         "GnomeSearchDialog",
         sizeof(GnomeSearchDialog),
         sizeof(GnomeSearchDialogClass),
         (GtkClassInitFunc) gnome_search_dialog_class_init,
         (GtkObjectInitFunc) gnome_search_dialog_init,
         NULL,
         NULL,
         (GtkClassInitFunc)NULL,
      };
      search_dialog_type = gtk_type_unique(GTK_TYPE_WINDOW, &search_dialog_info);
   }
   return(search_dialog_type);
}

static void gnome_search_dialog_class_init(GnomeSearchDialogClass *_class)
{
   GtkObjectClass *object_class;
   object_class = (GtkObjectClass*)_class;
   parent_class = gtk_type_class(GTK_TYPE_WINDOW);
   object_class->destroy = gnome_search_dialog_destroy;
}

static void gnome_search_dialog_init(GnomeSearchDialog *search_dialog)
{
   search_dialog->find_combo = NULL;
   search_dialog->replace_combo = NULL;
   search_dialog->find_button = NULL;
   search_dialog->replace_button = NULL;
   search_dialog->cancel_button = NULL;
   search_dialog->find_history = NULL;
   search_dialog->replace_history = NULL;
   search_dialog->startpos = 0;
   search_dialog->endpos = 0;
   search_dialog->max_entries = 10;
}

GtkWidget* gnome_search_dialog_new(GnomeSearchDialogType type)
{
   GnomeSearchDialog *search_dialog;
   GtkWidget *main_box;
   GtkWidget *util_box;
   GtkWidget *frame;
   GtkWidget *options_box;
   GtkWidget *label;
   GtkWidget *hsep;

   search_dialog = gtk_type_new(GNOME_TYPE_SEARCH_DIALOG);
   gtk_container_set_border_width(GTK_CONTAINER(search_dialog), 5);
   search_dialog->dialog_type = type;
   
   main_box = gtk_vbox_new(FALSE, 0);
   gtk_container_add(GTK_CONTAINER(search_dialog), main_box);
   gtk_widget_show(main_box);

   /* find is always there */
   util_box = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(main_box), util_box, TRUE, TRUE, 0);
   gtk_widget_show(util_box);
   label = gtk_label_new("Find String:");
   gtk_box_pack_start(GTK_BOX(util_box), label, FALSE, FALSE, 5);
   gtk_widget_show(label);
   util_box = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(main_box), util_box, TRUE, TRUE, 0);
   gtk_widget_show(util_box);
   search_dialog->find_combo = gtk_combo_new();
   gtk_box_pack_start(GTK_BOX(util_box), search_dialog->find_combo, TRUE, TRUE, 5);
   gtk_combo_disable_activate(GTK_COMBO(search_dialog->find_combo));
   gtk_combo_set_case_sensitive(GTK_COMBO(search_dialog->find_combo), TRUE);
   gtk_signal_connect(GTK_OBJECT(GTK_COMBO(search_dialog->find_combo)->entry), "changed", GTK_SIGNAL_FUNC(set_find_button_sensitive), search_dialog);
   gtk_signal_connect(GTK_OBJECT(GTK_COMBO(search_dialog->find_combo)->entry), "activate", GTK_SIGNAL_FUNC(activate_find_button), search_dialog);
   gtk_widget_show(search_dialog->find_combo);

   if(type == GNOME_SEARCH_REPLACE)
   {
      gtk_window_set_title(GTK_WINDOW(search_dialog), "Find and Replace...");
      util_box = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(main_box), util_box, TRUE, TRUE, 0);
      gtk_widget_show(util_box);
      label = gtk_label_new("Replace String:");
      gtk_box_pack_start(GTK_BOX(util_box), label, FALSE, FALSE, 5);
      gtk_widget_show(label);
      util_box = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(main_box), util_box, TRUE, TRUE, 0);
      gtk_widget_show(util_box);
      search_dialog->replace_combo = gtk_combo_new();
      gtk_box_pack_start(GTK_BOX(util_box), search_dialog->replace_combo, TRUE, TRUE, 5);
      gtk_combo_disable_activate(GTK_COMBO(search_dialog->replace_combo));
      gtk_combo_set_case_sensitive(GTK_COMBO(search_dialog->replace_combo), TRUE);
      gtk_signal_connect(GTK_OBJECT(GTK_COMBO(search_dialog->replace_combo)->entry), "activate", GTK_SIGNAL_FUNC(activate_find_button), search_dialog);
      gtk_widget_show(search_dialog->replace_combo);
   }
   else
   {
      gtk_window_set_title(GTK_WINDOW(search_dialog), "Find...");
   }

   frame = gtk_frame_new("Search Settings");
   gtk_box_pack_start(GTK_BOX(main_box), frame, TRUE, TRUE, 5);
   gtk_widget_show(frame);
   
   options_box = gtk_vbox_new(FALSE, 0);
   gtk_container_add(GTK_CONTAINER(frame), options_box);
   gtk_widget_show(options_box);
   
   search_dialog->check_case_sensitive = gtk_check_button_new_with_label("Case sensitive?");
   gtk_box_pack_start(GTK_BOX(options_box), search_dialog->check_case_sensitive, FALSE, TRUE, 0);
   gtk_widget_show(search_dialog->check_case_sensitive);

   search_dialog->check_search_style = gtk_check_button_new_with_label("Regular expression search?");
   gtk_box_pack_start(GTK_BOX(options_box), search_dialog->check_search_style, FALSE, TRUE, 0);
   gtk_widget_show(search_dialog->check_search_style);

   search_dialog->check_search_start = gtk_check_button_new_with_label("Search from start of file?");
   gtk_box_pack_start(GTK_BOX(options_box), search_dialog->check_search_start, FALSE, TRUE, 0);
   gtk_widget_show(search_dialog->check_search_start);

   search_dialog->check_search_multi = gtk_check_button_new_with_label("Search multiple files?");
   gtk_box_pack_start(GTK_BOX(options_box), search_dialog->check_search_multi, FALSE, TRUE, 0);
   gtk_widget_show(search_dialog->check_search_multi);

   search_dialog->check_stay_open = gtk_check_button_new_with_label("Keep window open?");
   gtk_box_pack_start(GTK_BOX(options_box), search_dialog->check_stay_open, FALSE, TRUE, 0);
   gtk_widget_show(search_dialog->check_stay_open);
   
   hsep = gtk_hseparator_new();
   gtk_box_pack_start(GTK_BOX(main_box), hsep, TRUE, TRUE, 10);
   gtk_widget_show(hsep);

   util_box = gtk_hbutton_box_new();
   gtk_box_pack_start(GTK_BOX(main_box), util_box, TRUE, TRUE, 0);
   gtk_button_box_set_layout(GTK_BUTTON_BOX(util_box), gnome_preferences_get_button_layout());
   gtk_button_box_set_spacing(GTK_BUTTON_BOX(util_box), GNOME_PAD);
   gtk_widget_show(util_box);

   search_dialog->find_button = gnome_pixmap_button(gnome_stock_pixmap_widget_at_size(GTK_WIDGET(search_dialog), GNOME_STOCK_PIXMAP_SEARCH, 20, 20), "Find Next");
   gtk_box_pack_end(GTK_BOX(util_box), search_dialog->find_button, FALSE, FALSE, 5);
   gtk_widget_show(search_dialog->find_button);
   gtk_widget_set_sensitive(search_dialog->find_button, FALSE);
   GTK_WIDGET_SET_FLAGS(search_dialog->find_button, GTK_CAN_DEFAULT);
   gtk_window_set_default(GTK_WINDOW(search_dialog), search_dialog->find_button);

   if(type == GNOME_SEARCH_REPLACE)
   {
      search_dialog->replace_button = gnome_pixmap_button(gnome_stock_pixmap_widget_at_size(GTK_WIDGET(search_dialog), GNOME_STOCK_PIXMAP_SRCHRPL, 20, 20), "Replace");
      gtk_box_pack_end(GTK_BOX(util_box), search_dialog->replace_button, FALSE, FALSE, 5);
      GTK_WIDGET_SET_FLAGS(search_dialog->replace_button, GTK_CAN_DEFAULT);
      gtk_widget_show(search_dialog->replace_button);   

      search_dialog->replace_all = gnome_pixmap_button(gnome_stock_pixmap_widget_at_size(GTK_WIDGET(search_dialog), GNOME_STOCK_PIXMAP_SRCHRPL, 20, 20), "Replace All");
      gtk_box_pack_end(GTK_BOX(util_box), search_dialog->replace_all, FALSE, FALSE, 5);
      GTK_WIDGET_SET_FLAGS(search_dialog->replace_all, GTK_CAN_DEFAULT);
      gtk_widget_show(search_dialog->replace_all);   
   }

   search_dialog->cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CLOSE);
   gtk_box_pack_end(GTK_BOX(util_box), search_dialog->cancel_button, FALSE, FALSE, 5);
   GTK_WIDGET_SET_FLAGS(search_dialog->cancel_button, GTK_CAN_DEFAULT);
   gtk_widget_show(search_dialog->cancel_button);   

   return GTK_WIDGET(search_dialog);
}

static void gnome_search_dialog_destroy(GtkObject *object)
{
   GtkWidget *search_dialog;
   g_return_if_fail(object != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(object));
   search_dialog = GTK_WIDGET(object);
   kill_combo_history(GNOME_SEARCH_DIALOG(search_dialog)->find_history);
   kill_combo_history(GNOME_SEARCH_DIALOG(search_dialog)->replace_history);
   gtk_widget_destroy(search_dialog);
}

void gnome_search_dialog_set_find_text(GnomeSearchDialog *sd, const gchar *text)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(sd->find_combo)->entry), text);
}

void gnome_search_dialog_set_replace_text(GnomeSearchDialog *sd, const gchar *text)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   g_return_if_fail(sd->dialog_type == GNOME_SEARCH_REPLACE);
   gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(sd->find_combo)->entry), text);
}

void gnome_search_dialog_search_selection(GnomeSearchDialog *sd, gboolean tf)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   sd->search_selection = tf;
}

void gnome_search_dialog_set_startpos(GnomeSearchDialog *sd, guint pos)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   sd->startpos = pos;
}

void gnome_search_dialog_set_endpos(GnomeSearchDialog *sd, guint pos)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   sd->endpos = pos;
}

gchar *gnome_search_dialog_get_find_text(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, NULL);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), NULL);
   return(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(sd->find_combo)->entry)));
}

gchar *gnome_search_dialog_get_replace_text(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, NULL);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), NULL);
   g_return_val_if_fail(sd->dialog_type == GNOME_SEARCH_REPLACE, NULL);
   return(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(sd->replace_combo)->entry)));
}

GnomeSearchDialogType gnome_search_dialog_get_dialog_type(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, 0);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), 0);
   return(sd->dialog_type);
}

GnomeSearchStyle gnome_search_dialog_get_search_style(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, 0);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), 0);

   if(GTK_TOGGLE_BUTTON(sd->check_search_style)->active)
      sd->search_style = GNOME_SEARCH_STYLE_REGEX;
   else
      sd->search_style = GNOME_SEARCH_STYLE_NORM;   

   return(sd->search_style);
}

GnomeSearchStart gnome_search_dialog_get_search_start(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, 0);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), 0);

   if(GTK_TOGGLE_BUTTON(sd->check_search_start)->active)
      sd->search_start = GNOME_SEARCH_START_TOP;
   else
      sd->search_start = GNOME_SEARCH_START_CURRENT;   

   return(sd->search_start);
}

gboolean gnome_search_dialog_case_sensitive(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, FALSE);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), FALSE);
   return(GTK_TOGGLE_BUTTON(sd->check_case_sensitive)->active);
}

gboolean gnome_search_dialog_keep_window_open(GnomeSearchDialog *sd)
{
   g_return_val_if_fail(sd != NULL, FALSE);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), FALSE);
   return(GTK_TOGGLE_BUTTON(sd->check_stay_open)->active);
}

void gnome_search_dialog_set_find_history(GnomeSearchDialog *sd, GList *history)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   if(history == NULL)
      return;
   sd->find_history = history;
   gtk_combo_set_popdown_strings(GTK_COMBO(sd->find_combo), sd->find_history);
}

void gnome_search_dialog_set_replace_history(GnomeSearchDialog *sd, GList *history)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   g_return_if_fail(sd->dialog_type == GNOME_SEARCH_REPLACE);
   if(history == NULL) return;
   sd->replace_history = history;
   gtk_combo_set_popdown_strings(GTK_COMBO(sd->replace_combo), sd->replace_history);
}

void gnome_search_dialog_set_max_entries(GnomeSearchDialog *sd, gint entries)
{
   g_return_if_fail(sd != NULL);
   g_return_if_fail(GNOME_IS_SEARCH_DIALOG(sd));
   sd->max_entries = entries;
}

gboolean gnome_search_dialog_add_find_string(GnomeSearchDialog *sd, gchar *string)
{
   GList *ptr;
   gint pos;
   gpointer temp;
   g_return_val_if_fail(sd != NULL, FALSE);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), FALSE);
   if((pos = check_glist_for_multiple_entries(sd->find_history, string)) >= 0)
   {
      temp = g_list_nth_data(sd->find_history, pos);
      sd->find_history = g_list_remove(sd->find_history, temp);
   }
   else
   {
      temp = g_new(char, strlen(string)+2);
      strcpy((gchar *)temp, string);
   }
   sd->find_history = g_list_prepend(sd->find_history, temp);
   while(g_list_length(sd->find_history) > sd->max_entries)
   {
      ptr = g_list_nth(sd->find_history, g_list_length(sd->find_history)-1);
      if(ptr)
      {
         temp = ptr->data;
         sd->find_history = g_list_remove(sd->find_history, ptr->data);
         g_free(ptr->data);
      }
   }
   gtk_combo_set_popdown_strings(GTK_COMBO(sd->find_combo), sd->find_history);
   return(TRUE);
}   

gboolean gnome_search_dialog_add_replace_string(GnomeSearchDialog *sd, gchar *string)
{
   GList *ptr;
   gint pos;
   gpointer temp;
   g_return_val_if_fail(sd != NULL, FALSE);
   g_return_val_if_fail(GNOME_IS_SEARCH_DIALOG(sd), FALSE);
   g_return_val_if_fail(sd->dialog_type == GNOME_SEARCH_REPLACE, FALSE);
   if(!string) return(FALSE);
   if((pos = check_glist_for_multiple_entries(sd->replace_history, string)) >= 0)
   {
      temp = g_list_nth_data(sd->replace_history, pos);
      sd->replace_history = g_list_remove(sd->replace_history, temp);
   }
   else
   {
      temp = g_new(char, strlen(string)+1);
      strcpy((gchar *)temp, string);
   }
   sd->replace_history = g_list_prepend(sd->replace_history, temp);
   while(g_list_length(sd->replace_history) > sd->max_entries)
   {
      ptr = g_list_nth(sd->replace_history, g_list_length(sd->replace_history)-1);
      if(ptr)
      {
         temp = ptr->data;
         sd->replace_history = g_list_remove(sd->replace_history, ptr->data);
         g_free(ptr->data);
      }
   }
   gtk_combo_set_popdown_strings(GTK_COMBO(sd->replace_combo), sd->replace_history);
   return(TRUE);
}   

static gint check_glist_for_multiple_entries(GList *list, gchar *new_entry)
{
   GList *temp;
   gchar *cmp_text;
   gint retval = 0;
   if(list == NULL) return(-1);
   temp = list;
   while(temp != NULL)
   {
      cmp_text = temp->data;
      if(!strcmp(cmp_text, new_entry))
      {
         return(retval);
      }
      temp = g_list_next(temp);
      retval++;
   }
   return(-1);
}

static void kill_combo_history(GList *list)
{
   GList *temp;
   if(list == NULL)
      return;

   temp = list;
   while(temp != NULL)
   {
      g_free(temp->data);
      temp = g_list_next(temp);
   }
   g_list_free(list);
}

void activate_find_button(GtkWidget *widget, GnomeSearchDialog *sd)
{
   gtk_signal_emit_by_name(GTK_OBJECT(sd->find_button), "clicked", NULL);
}

void set_find_button_sensitive(GtkWidget *widget, GnomeSearchDialog *sd)
{
   if(strlen(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(sd->find_combo)->entry))) > 0)
   {
      gtk_widget_set_sensitive(sd->find_button, TRUE);
      if(sd->dialog_type == GNOME_SEARCH_REPLACE)
      {
         gtk_widget_set_sensitive(sd->replace_button, TRUE);
         gtk_widget_set_sensitive(sd->replace_all, TRUE);
      }
   }
   else
   {
      gtk_widget_set_sensitive(sd->find_button, FALSE);   
      if(sd->dialog_type == GNOME_SEARCH_REPLACE)
      {
         gtk_widget_set_sensitive(sd->replace_button, FALSE);
         gtk_widget_set_sensitive(sd->replace_all, FALSE);
      }
   }
}