#include <signal.h>
#include "gdsfile.h"
#include "../gtkextext/gtkextext.h"
#include "gdseditor.h"
#include "gdstearaway.h"
#include "../signals.h"

static void gds_file_init(GdsFile *gds_file);
static void gds_file_class_init(GdsFileClass *klass);
static void gds_file_destroy(GtkObject *object);
static gint button_press_cb(GtkWidget *widget, GdkEventButton *event);
static gint key_press_cb(GtkWidget *widget, GdkEventKey *event, GdsFile *gds_file);
static gchar *make_spaces_string(gint spaces);

void bracket_match_cb(GtkWidget *widget, gint pos, GdsFile *gds_file);

static GtkVBoxClass *parent_class = NULL;

static gint global_undomax = 0;
static gint global_spaces = 0;
static gboolean global_indent = 0;
static gboolean global_bracketmatch = 0;
static gboolean global_tab_stops;

GtkType gds_file_get_type(void)
{
   static GtkType gds_file_type=0;
   if(!gds_file_type)
   {
      static const GtkTypeInfo gds_file_info = 
      {	
         "GdsFile",
         sizeof(GdsFile),
         sizeof(GdsFileClass),
         (GtkClassInitFunc) gds_file_class_init,
         (GtkObjectInitFunc) gds_file_init,
         NULL,
         NULL,
         (GtkClassInitFunc)NULL,
      };
      gds_file_type = gtk_type_unique(GTK_TYPE_VBOX, &gds_file_info);
   }
   return(gds_file_type);
}

static void gds_file_class_init(GdsFileClass *klass)
{
   GtkObjectClass *object_class;
   object_class = (GtkObjectClass*)klass;
   parent_class = gtk_type_class(GTK_TYPE_HBOX);
   object_class->destroy = gds_file_destroy;
}

static void gds_file_init(GdsFile *gds_file)
{
   gds_file->filename = NULL;
   gds_file->tables = NULL;
   gds_file->default_lang = TRUE;
   gds_file->changed_set = FALSE;
   gds_file->modtime = 0;
   gds_file->props_dialog = NULL;
   gds_file->props.over_ride = FALSE;
   gds_file->props.dir = NULL;
   gds_file->props.auto_indent = FALSE;
   gds_file->props.use_spaces = FALSE;
   gds_file->props.spaces = 0;
   gds_file->props.compiler = NULL;
   gds_file->props.debugger = NULL;
   gds_file->props.execution = NULL;
   gds_file->compile_pid = 0;
   gds_file->debug_pid = 0;   
   gds_file->exec_pid = 0;
}

GtkWidget* gds_file_new(gchar *filename, gboolean full)
{
   GdsFile *gds_file;
   GtkWidget *vbox;
   GtkWidget *hbox;
   GtkWidget *hscroll;
   gchar *str;
   
   gds_file = gtk_type_new(GDS_TYPE_FILE);
   gds_file_set_filename(gds_file, filename);

   if(!full && (str = strrchr(filename, '/'))) str++;
   else str = filename;
   gds_file->menu_item = gnome_stock_menu_item(GNOME_STOCK_PIXMAP_NEW, str);
   gtk_widget_show(gds_file->menu_item); 

   gds_file->tear_away = gds_tear_away_new(GTK_POS_LEFT);
   gtk_widget_show(gds_file->tear_away);
   
   gds_file->paned = gtk_vpaned_new();
   gtk_paned_set_gutter_size(GTK_PANED(gds_file->paned), 12);
   gds_tear_away_set_child(GDS_TEAR_AWAY(gds_file->tear_away), gds_file->paned);
   gtk_widget_show(gds_file->paned);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_widget_show(vbox);
   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
   gtk_widget_show(hbox);

   gds_file->text = gds_editor_new();
   gtk_box_pack_start(GTK_BOX(hbox), gds_file->text, TRUE, TRUE, 0);
   gtk_widget_show(gds_file->text);
   gtk_extext_undo_set_max(GTK_EXTEXT(gds_file->text), global_undomax);
   gds_file->vscroller = gtk_vscrollbar_new(GTK_EXTEXT(gds_file->text)->vadj);
   gtk_box_pack_start(GTK_BOX(hbox), gds_file->vscroller, FALSE, FALSE, 0);
   gtk_widget_show(gds_file->vscroller);
   gtk_signal_connect(GTK_OBJECT(gds_file->text), "button_press_event", GTK_SIGNAL_FUNC(button_press_cb), 0);
   gtk_signal_connect(GTK_OBJECT(gds_file->text), "key_press_event", GTK_SIGNAL_FUNC(key_press_cb), gds_file);
   hscroll = gtk_hscrollbar_new(GTK_EXTEXT(gds_file->text)->hadj);
   gtk_box_pack_start(GTK_BOX(vbox), hscroll, FALSE, TRUE, 0);
   gtk_widget_show(hscroll);
   gds_file->file_data = gtk_entry_new();
   gtk_box_pack_end(GTK_BOX(vbox), gds_file->file_data, FALSE, TRUE, 0);
   gtk_widget_show(gds_file->file_data);
   gtk_editable_set_editable(GTK_EDITABLE(gds_file->file_data), FALSE);
   gtk_paned_pack1(GTK_PANED(gds_file->paned), vbox, TRUE, TRUE);
   gtk_signal_connect(GTK_OBJECT(gds_file->text), "move_to_column", GTK_SIGNAL_FUNC(bracket_match_cb), (gpointer)gds_file);
   gds_file->script_box = gtk_vbox_new(FALSE, 0);
   gtk_paned_pack2(GTK_PANED(gds_file->paned), gds_file->script_box, FALSE, FALSE);
   gtk_widget_show(gds_file->script_box);
   return GTK_WIDGET(gds_file);
}

static void gds_file_destroy(GtkObject *object)
{
   GdsFile *gds_file;
   g_return_if_fail(object != NULL);
   g_return_if_fail(GDS_IS_FILE(object));
   gds_file = GDS_FILE(object);
   if(gds_file->props_dialog)
      gtk_widget_destroy(gds_file->props_dialog);
   gtk_widget_destroy(gds_file->menu_item);
   gtk_widget_destroy(gds_file->text);
   gtk_widget_destroy(gds_file->tear_away);
   g_free(gds_file->filename);
   g_free(gds_file->props.dir);
   g_free(gds_file->props.compiler);
   g_free(gds_file->props.debugger);
   g_free(gds_file->props.execution);
   if(gds_file->compile_pid) kill(gds_file->compile_pid, 9);
   if(gds_file->debug_pid) kill(gds_file->debug_pid, 9);
   if(gds_file->exec_pid) kill(gds_file->exec_pid, 9);
	GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

void gds_file_set_filename(GdsFile *gds_file, gchar *filename)
{
   g_return_if_fail(gds_file != NULL);
   g_return_if_fail(GDS_IS_FILE(gds_file));
   if(gds_file->filename)
      g_free(gds_file->filename);
   gds_file->filename = g_new(char, strlen(filename)+1);
   strcpy(gds_file->filename, filename);
}

void gds_file_set_title(GdsFile *gds_file, gchar *title, gboolean full)
{
   GtkWidget *parent;
   gchar *str;
   gint pos = 0;
   g_return_if_fail(gds_file != NULL);
   g_return_if_fail(GDS_IS_FILE(gds_file));
   parent = gds_file->menu_item->parent;
   if(!full && (str = strrchr(title, '/'))) str++;
   else str = title;
   pos = g_list_index(GTK_MENU_SHELL(parent)->children, gds_file->menu_item);
   gtk_widget_destroy(gds_file->menu_item);   

   if(!gds_file->changed_set)
      gds_file->menu_item = gnome_stock_menu_item(GNOME_STOCK_PIXMAP_NEW, str);
   else
      gds_file->menu_item = gnome_stock_menu_item(GNOME_STOCK_PIXMAP_PROPERTIES, str);
   gtk_menu_shell_insert(GTK_MENU_SHELL(parent), gds_file->menu_item, pos);
   gtk_widget_show(gds_file->menu_item);

   gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(files_book), gds_file->tear_away, str);
   gtk_notebook_set_menu_label_text(GTK_NOTEBOOK(files_book), gds_file->tear_away, str);
   gds_tear_away_set_title(GDS_TEAR_AWAY(gds_file->tear_away), str);
}


static gint button_press_cb(GtkWidget *widget, GdkEventButton *event)
{
   GtkExText *text;
   gboolean handled = FALSE;

   text = (GtkExText *)widget;

   if(event->type == GDK_BUTTON_PRESS && event->button == 3)
   {
      if(!GTK_WIDGET_HAS_FOCUS(widget))
         gtk_widget_grab_focus(widget);
      gtk_menu_popup(GTK_MENU(edit_popup_menu), NULL, NULL, NULL, NULL, event->button, event->time);
      handled = TRUE;
   }
   if(handled) gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "button_press_event");
   return(handled);
}

static gint key_press_cb(GtkWidget *widget, GdkEventKey *event, GdsFile *gds_file)
{
   GtkExText *text;
   GtkEditable *editable;
   gint curpos;
   gboolean handled = FALSE;
   gint i = 0;
   gchar *chars = NULL, *spaces = NULL;
   gboolean shift_state;
   gboolean control_state;
   gint key;
   gint abscol;
   gint tabstop;

   shift_state = event->state & GDK_SHIFT_MASK;
   control_state = event->state & GDK_CONTROL_MASK;  
   key=event->keyval;
   
   text = (GtkExText *)widget;
   editable = (GtkEditable *)widget;
   if(!gtk_extext_get_editable(text)) return(handled);

   curpos = editable->current_pos;
   if(event->keyval == GDK_Tab || event->keyval == ' ' || (event->keyval == GDK_Delete && control_state))
   {
      abscol = gtk_extext_get_abs_column(text, curpos);
      tabstop = gtk_extext_get_next_tab_stop(text, curpos);
      if(editable->has_selection && editable->selection_start_pos != editable->selection_end_pos)
      {
         gint start = 0;
         gint end = 0;
         gint first_line = 0;
         gint last_line = 0;
         gint current_line = 0;
         GtkExTextLineData *lineptr = NULL;
         start = editable->selection_start_pos;
         end = editable->selection_end_pos;
         lineptr = gtk_extext_get_line_by_char_pos(text, start);
         first_line = lineptr->line_number;
         g_free(lineptr);
         lineptr = gtk_extext_get_line_by_char_pos(text, end);
         last_line = lineptr->line_number;
         if(end == lineptr->startpos && first_line < last_line) last_line--;
         g_free(lineptr);
         current_line = first_line;
         if(first_line == last_line)
         {
            gtk_extext_freeze(text);
            gtk_editable_delete_text(editable, start, end);
            editable->has_selection = FALSE;
            editable->selection_start_pos = -1;
            editable->selection_end_pos = -1;
            if(event->keyval == GDK_Tab)
            {
               if(!file_emit_scripting_signal("tab-pressed", gds_file))
               {
                  spaces = make_spaces_string(tabstop);
                  if((gds_file->props.over_ride && gds_file->props.use_spaces) || (!gds_file->props.over_ride && global_spaces))
                     gtk_editable_insert_text(editable, spaces, strlen(spaces), &start);
                  else
                     gtk_editable_insert_text(editable, "\t", 1, &start);
                  g_free(spaces);
               }
            }
            else
            {
               gtk_editable_insert_text(editable, " ", 1, &start);
            }
            gtk_extext_thaw(text);
            gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
            return(TRUE);
         }
         gtk_extext_freeze(text);
         editable->has_selection = FALSE;
         editable->selection_start_pos = -1;
         editable->selection_end_pos = -1;
         tabstop = text->default_tab_width;
         spaces = make_spaces_string(tabstop);
         while(current_line <= last_line)
         {
            gtk_extext_set_column(text, 0);
            gtk_extext_set_line(text, current_line);
            chars = gtk_editable_get_chars(editable, editable->current_pos, editable->current_pos + text->default_tab_width);
            if(event->keyval == GDK_Tab && (control_state || shift_state))
            {
               if(chars && !strcmp(chars, spaces))
                  gtk_editable_delete_text(editable, editable->current_pos, editable->current_pos + text->default_tab_width);
               else if(chars && chars[0] == '\t')
                  gtk_editable_delete_text(editable, editable->current_pos, editable->current_pos + 1);
            }
            else if(event->keyval == GDK_Tab)
            {
               if((gds_file->props.over_ride && gds_file->props.use_spaces) || (!gds_file->props.over_ride && global_spaces))
                  gtk_editable_insert_text(editable, spaces, strlen(spaces), &editable->current_pos);
               else
                  gtk_editable_insert_text(editable, "\t", 1, &editable->current_pos);
            }
            else if(event->keyval == ' ' && (control_state || shift_state))
            {
               if(chars[0] == ' ')
                  gtk_editable_delete_text(editable, editable->current_pos, editable->current_pos + 1);
            }
            else if(event->keyval == GDK_Delete && control_state)
            {
               gtk_editable_delete_text(editable, editable->current_pos, editable->current_pos + 1);
            }
            else if(event->keyval == ' ')
            {
               gtk_editable_insert_text(editable, " ", 1, &editable->current_pos);
            }
            g_free(chars);
            current_line++;
         }
         lineptr = gtk_extext_get_line_data(text, first_line, NULL);
         start = lineptr->startpos;
         g_free(lineptr);
         lineptr = gtk_extext_get_line_data(text, last_line, NULL);
         if(GTK_EXTEXT_INDEX(text, lineptr->endpos) == '\n')
            end = lineptr->endpos-1;
         else
            end = lineptr->endpos;
         g_free(lineptr);
         gtk_extext_set_position(editable, end);
         gtk_extext_thaw(text);
         gtk_editable_select_region(editable, start, end);
         g_free(spaces);
         handled = TRUE;
      }
      else if(event->keyval == GDK_Tab)
      {
         if(!file_emit_scripting_signal("tab-pressed", gds_file))
         {
            spaces = make_spaces_string(tabstop);
            if((gds_file->props.over_ride && gds_file->props.use_spaces) || (!gds_file->props.over_ride && global_spaces))
               gtk_editable_insert_text(editable, spaces, strlen(spaces), &editable->current_pos);
            else
               gtk_editable_insert_text(editable, "\t", 1, &editable->current_pos);
            g_free(spaces);
         }
         handled = TRUE;
      }
   }
   else if(event->keyval == GDK_KP_Enter || event->keyval == GDK_Return)
   {
      if((gds_file->props.over_ride && gds_file->props.auto_indent) || (!gds_file->props.over_ride && global_indent))
      {
         if(!file_emit_scripting_signal("enter-pressed", gds_file))
         {
            gtk_extext_set_column(text, 0);
            chars = gtk_editable_get_chars(editable, editable->current_pos, curpos);
            if(chars)
            {
               for(i = 0; i < strlen(chars); i++)
               {
                  if(chars[i] == '\t' || chars[i] == ' ') continue;
                  else break;
               }
            }
            if(i)
            {
               gchar *insert;
               insert = g_new0(char, i+2);
               insert[0] = '\n';
               strncpy(&insert[1], chars, i);
               gtk_editable_insert_text(editable, insert, i+1, &curpos);
               g_free(insert);
            }
            else
            {
               gtk_editable_insert_text(editable, "\n", 1, &curpos);
            }
            g_free(chars);
         }
         handled = TRUE;
      }
   }
   if(handled) gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
   return(handled);
}

static gchar *make_spaces_string(gint spaces)
{
   gchar *string = NULL;
   if(spaces > 10) spaces = 10;
   else if(spaces < 0) spaces = 1;
   string = g_new(char, spaces+1);
   if(string)
   {
      memset(string, ' ', spaces);
      string[spaces] = '\0';
   }
   return(string);
}

void bracket_match_cb(GtkWidget *widget, gint pos, GdsFile *gds_file)
{
   GdsEditor *gds_editor;
   GtkExText *text;
   GtkEditable *editable;
   gboolean found = FALSE;
   g_return_if_fail(widget != NULL);

   gds_editor = GDS_EDITOR(widget);
   text = GTK_EXTEXT(widget);
   editable = GTK_EDITABLE(widget);

   gtk_extext_set_pseudo_select(text, -1, -1);

   if(gds_file->props.over_ride && !gds_file->props.bracketmatch)
      return;
   else if(!gds_file->props.over_ride && !global_bracketmatch)
      return;

   pos = editable->current_pos-1;
   if(pos < 0) return;
   found = find_bracket_match(widget, &pos);

   if(found)
   {
      gtk_extext_set_pseudo_select(text, pos, pos+1);
   }
}

gboolean find_bracket_match(GtkWidget *widget, gint *search)
{
   GdsEditor *gds_editor;
   GtkExText *text;
   GtkEditable *editable;
   gchar base_char = 0;
   gchar search_char = 0;
   gchar cur_char = 0;
   gint addition = -1;
   gint counter = 0;
   gboolean found = FALSE;
   gint pos = 0;

   g_return_val_if_fail(widget != NULL, FALSE);
   if(!search) return(FALSE);

   gds_editor = GDS_EDITOR(widget);
   text = GTK_EXTEXT(widget);
   editable = GTK_EDITABLE(widget);
   pos = *search;
   cur_char = GTK_EXTEXT_INDEX(text, pos);

   if(text->property_current && text->property_current->user_data == GINT_TO_POINTER(SYNTAX_TABLE))
      return(FALSE);

   base_char = cur_char;
   switch((int)base_char)
   {
      case '{': addition = 1;
                search_char = '}';
                break;
      case '(': addition = 1;
                search_char = ')';
                break;
      case '[': addition = 1;
                search_char = ']';
                break;
      case '<': addition = 1;
                search_char = '>';
                break;
      case '}': addition = -1;
                search_char = '{';
                break;
      case ')': addition = -1;
                search_char = '(';
                break;
      case ']': addition = -1;
                search_char = '[';
                break;
      case '>': addition = -1;
                search_char = '<';
                break;
      default : addition = 0;
                break;
   }
   if(!addition) return(FALSE);
   while(pos >= 0 && pos <= text->length)
   {
      pos += addition;
      cur_char = GTK_EXTEXT_INDEX(text, pos);
      if(cur_char == search_char && !counter)
      {
         found = TRUE;
         break;
      }
      if(cur_char == base_char)
         counter++;
      else if(cur_char == search_char)
         counter--;
   }
   if(found) *search = pos;
   return(found);
}

void gds_file_set_global_props(gint undo, gint spaces, gboolean indent, gboolean bracket, gboolean tab_stops)
{
   global_undomax = undo;
   global_spaces = spaces;
   global_indent = indent;
   global_bracketmatch = bracket;
   global_tab_stops = tab_stops;
}
