#include "declarations.h"
#include "insert.h"
#include "macro-language.h"
#include "colors.h"
#include "signals.h"

GSList *highlight_options = NULL;
GList *highlight_tables = NULL;
GdsFileHighlightTable *parse_table = NULL;
gboolean parsing_language = FALSE;

GList *syntax_list = NULL;
GList *pattern_list = NULL;
GList *embedded_list = NULL;

struct highlight_cache HighlightCache;

GtkWidget *build_highlight_menu(void)
{
   FILE *file;
   GList *filenames = NULL;
   GList *names = NULL;
   GList *cur = NULL;
   GtkWidget *menu = NULL;
   GtkWidget *none_item = NULL;
   GtkWidget *menuitem = NULL;
   gchar *text;
   gchar buffer[1024];
   gchar lang_string[] = "language=";
   gchar *filename;
   gchar home_dir[384] = "";
   gchar install_dir[] = LANG_DIR;

   g_snprintf(home_dir, sizeof(home_dir), "%s/." PACKAGE "/languages", getenv("HOME"));   
   menu = gtk_menu_new();
   none_item = gtk_radio_menu_item_new_with_label(NULL, "None");
   gtk_object_set_data(GTK_OBJECT(none_item), "filename", "none"); 
   gtk_signal_connect(GTK_OBJECT(none_item), "activate", GTK_SIGNAL_FUNC(highlight_table_change_cb), GINT_TO_POINTER(1));
   gtk_menu_append(GTK_MENU(menu), none_item);
   gtk_widget_show(none_item);
   gtk_widget_show(menu);
   filenames = build_file_listing(home_dir, filenames);
   filenames = build_file_listing(install_dir, filenames);

   for(cur = g_list_first(filenames); cur; cur = cur->next)
   {
      filename = (gchar *)cur->data;
      if(!(file = fopen(filename, "r"))) continue;
      while(fgets(buffer, sizeof(buffer), file))
      {
         if(buffer[0]=='#') continue;
         if(strstr(buffer, "<settings>")) 
         {
            while(fgets(buffer, sizeof(buffer), file) != NULL)
            {
               if(strstr(buffer, "</settings>")) break;
               if(strstr(buffer, "\n")) buffer[strlen(buffer)-1] = '\0';
               if((text = strstr(buffer, lang_string)))
               {
                  text = g_strdup(text + strlen(lang_string));
                  if(!check_list_for_string(names, text))
                  {
                     menuitem = gtk_radio_menu_item_new_with_label(GTK_RADIO_MENU_ITEM(none_item)->group, text);
                     gtk_object_set_data(GTK_OBJECT(menuitem), "filename", (gpointer)filename); 
                     gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(highlight_table_change_cb), GINT_TO_POINTER(1));
                     gtk_menu_append(GTK_MENU(menu), menuitem);
                     gtk_widget_show(menuitem);
                     names = g_list_append(names, (gpointer)text);
                  }
                  else
                  {
                     g_free(filename);
                     cur->data = NULL;
                     g_free(text);
                  }
               }
            }
         }
      }
      fclose(file);
   }
   g_list_free(filenames);
   for(cur = g_list_first(names); cur; cur = cur->next)
   {
      g_free(cur->data);
   }
   g_list_free(names);
   highlight_options = GTK_RADIO_MENU_ITEM(none_item)->group;
   return(menu);
}

void highlight_table_change_cb(GtkWidget *widget, gint num)
{
   gchar *filename;
   GdsFileHighlightTable *table;
   filename = gtk_object_get_data(GTK_OBJECT(widget), "filename");
   table = cur_file->tables;
   attach_highlight_tables(cur_file, filename);
   if(cur_file->tables != table) cur_file->default_lang = FALSE;
   adjust_sensitivity();
}

GList *build_file_listing(gchar *directory, GList *filenames)
{
   DIR *dir_list = NULL;
   struct dirent *dir;
   gchar *full_path = NULL;

   dir_list = opendir(directory);
   if(dir_list != NULL)
   {
      while((dir = readdir(dir_list)) != NULL)
      {
         if(dir->d_ino > 0)
         {
            full_path = g_strconcat(directory, "/", dir->d_name, NULL);
            filenames = g_list_append(filenames, (gpointer)full_path);
         }
      }
   }
   return(filenames);
}

gint reparse_highlight_table(GdsFileHighlightTable *table)
{
   GList *new_list = NULL;
   GList *temp;
   GdsFile *current;
   GdsFile *old;
   gchar *filename;
   gint retval=0;

   if(table == NULL) return(0);   
   old = cur_file;
   filename = g_strdup(table->filename);

   for(temp = g_list_first(files_list); temp; temp = temp->next)
   {
      current = GDS_FILE(temp->data);
      if(current->tables == table)
      {
         detach_highlight_tables(current);
         new_list = g_list_append(new_list, (gpointer)current);
      }
   }
   for(temp = g_list_first(new_list); temp; temp = temp->next)
   {
      current = GDS_FILE(temp->data);
      cur_file = current;
      attach_highlight_tables(current, filename);
      retval++;
   }
   cur_file = old;
   g_list_free(new_list);
   g_free(filename);
   return(retval);
}

void apply_color_changes(void)
{
   GList *temp;
   GdsFile *current;
   temp = g_list_first(files_list);
   for(temp = g_list_first(files_list); temp; temp = temp->next)
   {
      current = (GdsFile *)temp->data;
      if(current->tables && current->tables->syntax)
         gds_editor_install_table(GDS_EDITOR(current->text), current->tables->syntax);
      if(current->tables && current->tables->pattern)
         gds_editor_install_table(GDS_EDITOR(current->text), current->tables->pattern);
      if(current->tables && current->tables->embedded)
         gds_editor_install_table(GDS_EDITOR(current->text), current->tables->embedded);
      if(current == cur_file) gtk_widget_queue_draw(current->text);
   }
}

void attach_highlight_tables(GdsFile *file_ptr, gchar *path_to_file)
{
   gchar home_dir[384];
   GdsFileHighlightTable *table = NULL;
   gchar *extension = NULL;
   gchar *filename = NULL;   
   gchar install_dir[] = LANG_DIR;
   GList *temp = NULL;
   GdsFileHighlightTable *current;

   g_snprintf(home_dir, sizeof(home_dir), "%s/." PACKAGE "/languages", getenv("HOME"));   
   detach_highlight_tables(file_ptr);

   if(path_to_file)
   {
      for(temp = g_list_first(highlight_tables); temp; temp = temp->next)
      {
         current = (GdsFileHighlightTable *)temp->data;
         if(!strcmp(current->filename, path_to_file))
         {
            table = current;
            break;
         }
      }
      if(!table) table = make_highlight_tables(path_to_file);
   }
   else
   {
      filename = get_file_from_filename(file_ptr->filename);
      extension = strrchr(filename, '.');
      if(extension) extension++;
      else extension = filename;
      table = find_highlight_tables(extension, home_dir);  // try the local path first
      if(!table) table = find_highlight_tables(extension, install_dir);
   }

   if(table)
   {
      file_ptr->tables = table;
      table->refcount++;
      if(table->syntax) gds_editor_install_table(GDS_EDITOR(file_ptr->text), table->syntax);
      if(table->pattern) gds_editor_install_table(GDS_EDITOR(file_ptr->text), table->pattern);
      if(table->embedded) gds_editor_install_table(GDS_EDITOR(file_ptr->text), table->embedded);
      gds_editor_set_highlight(GDS_EDITOR(file_ptr->text), general_preferences.syntax);
      if(table->table_add_callback) table_emit_scripting_signal("table-add", file_ptr->tables);
      if(table->file_focus_callback) file_emit_scripting_signal("focus-in", file_ptr);
   }
   else
   {
      gds_editor_set_highlight(GDS_EDITOR(file_ptr->text), FALSE);
   }
   g_free(filename);
}

gboolean detach_highlight_tables(GdsFile *file_ptr)
{
   if(file_ptr->tables != NULL)
   {
      if(file_ptr->tables->file_unfocus_callback) file_emit_scripting_signal("focus-out", file_ptr);
      if(file_ptr->tables->table_remove_callback) table_emit_scripting_signal("table-remove", file_ptr->tables);
      if(file_ptr->compile_pid == -1 && file_ptr->tables->stop_compile_hook) file_emit_scripting_signal("stop-compile", file_ptr);
      if(file_ptr->debug_pid == -1 && file_ptr->tables->stop_debug_hook) file_emit_scripting_signal("stop-debug", file_ptr);
      if(file_ptr->exec_pid == -1 && file_ptr->tables->stop_execute_hook) file_emit_scripting_signal("stop-execute", file_ptr);
      file_ptr->tables->refcount--;
      delete_highlight_tables(file_ptr->tables);
      file_ptr->tables = NULL;
      GDS_EDITOR(file_ptr->text)->syntax_table = NULL;
      GDS_EDITOR(file_ptr->text)->pattern_table = NULL;
      GDS_EDITOR(file_ptr->text)->embedded_table = NULL;      
      gds_editor_set_highlight(GDS_EDITOR(file_ptr->text), FALSE);      
      return(TRUE);
   }
   else return(FALSE);
}

gboolean delete_highlight_tables(GdsFileHighlightTable *table)
{
   GList *list = NULL;
   if(table->refcount > 0) return(FALSE);
   if(table->syntax) gds_editor_table_free(table->syntax);
   if(table->pattern) gds_editor_table_free(table->pattern);
   if(table->embedded) gds_editor_table_free(table->embedded);
   g_free(table->filename);
   g_list_delete_all(table->extensions);
   g_free(table->props.compiler);
   g_free(table->props.debugger);
   g_free(table->props.execution);
   if(table->table_destroy_callback)
      table_emit_scripting_signal("table-destroy", table);
   remove_signal_hooks(table);
   highlight_tables = g_list_remove(highlight_tables, table);
   table->menu_entries = g_list_reverse(table->menu_entries);
   for(list = g_list_first(table->menu_entries); list; list = list->next)
   {
      if((g_list_index(edit_entries, list->data) >= 0))
      {
         edit_entries = g_list_remove(edit_entries, list->data);
      }
      if((g_list_index(dynamic_build, list->data) >= 0))
      {
         dynamic_build = g_list_remove(dynamic_build, list->data);
      }
      if(list->data)
      {
         gtk_widget_destroy(GTK_WIDGET(list->data));
      }
   }
   g_list_free(table->menu_entries);
   for(list = g_list_first(table->toolbar_entries); list; list = list->next)
   {
      if(list->data)
      {
         gtk_widget_destroy(GTK_WIDGET(list->data));
      }
   }
   g_list_free(table->toolbar_entries);
   g_free(table);
   return(TRUE);
}

GdsFileHighlightTable *find_highlight_tables(gchar *extension, gchar *cur_dir)
{
   GList *temp = NULL;
   GList *ptr = NULL;
   GdsFileHighlightTable *current;
   DIR *dir_list;
   struct dirent *dir;
   gchar buffer[256];
   gchar full_path[384];
   GList *exts = NULL;
   gboolean ext = FALSE;
   GdsFileHighlightTable *new_table;   
   
   for(temp = g_list_first(highlight_tables); temp; temp = temp->next)
   {
      current = (GdsFileHighlightTable *)temp->data;
      for(ptr = g_list_first(current->extensions); ptr; ptr = ptr->next)
      {
         if(!strcmp((gchar *)ptr->data, extension))
         {
            return(current);
         }
      }
   }

   dir_list = opendir(cur_dir);
   if(dir_list != NULL)
   {
      while((dir = readdir(dir_list)) != NULL)
      {
         if(dir->d_ino > 0)
         {
            gboolean found = FALSE;
            for(temp = g_list_first(highlight_tables); temp; temp = temp->next)
            {
               current = (GdsFileHighlightTable *)temp->data;
               if(!strcmp(current->filename, buffer))
               {     
                  found = TRUE;            
                  break;
               }
            }
            if(found) continue;
            g_snprintf(full_path, sizeof(full_path), "%s/%s", cur_dir, dir->d_name);
            if(!(exts = get_exts_list(full_path)))
            {
               continue;
            }
            ext = check_exts_list(extension, exts);
            g_list_delete_all(exts);
            if(ext)
            {
               new_table = make_highlight_tables(full_path);
               return(new_table);
            }
         }
      }
   }
   return(NULL);
}


GList *get_exts_list(gchar *filename)
{
   FILE *file;
   gchar buffer[256];
   const char end_char = '\"';
   gchar name[32];
   gchar ext[24];   
   GList *temp = NULL;
   gchar *new_ext;
   gint success = 0;
   
   if(!(file = fopen(filename, "r")))
      return(NULL);

   while(fgets(buffer, sizeof(buffer), file) != NULL)
   {
      if(strstr(buffer, "<extensions>") != NULL) 
      {
         while(fgets(buffer, sizeof(buffer), file) != NULL)
         {
            if(strstr(buffer, "</extensions>")) break;
            if((success = get_str(buffer, name, "name=\"", end_char, sizeof(name))))
            {
               if((success = get_str(buffer, ext, "ext=\"", end_char, sizeof(ext))))
               {
                  new_ext = g_new(char, strlen(ext)+1);
                  strcpy(new_ext, ext);
                  temp = g_list_append(temp, (gpointer)new_ext);
               }
            }
         }
      }
   }
   fclose(file);
   return(temp);
}

gboolean check_exts_list(gchar *ext, GList *exts)
{
   GList *temp = NULL;
   gchar *current;
   temp = g_list_first(exts);
   while(temp)
   {
      current = (gchar *)temp->data;
      if(!strcmp(ext, current))
         return(TRUE);
      temp = temp->next;
   }
   return(FALSE);
}

GdsFileHighlightTable *make_highlight_tables(gchar *filename)
{
   FILE *ifile;
   FILE *ofile;
   gchar temp_file[384];
   GList *list = NULL;
   GdsFileHighlightTable *new_table;
   gchar buffer[2048];
   gchar name[32];
   gchar start[1024];
   gchar end[1024];
   gchar words[1536];
   gchar color[32];
   gchar cclass[8];
   gchar default_class[] = "d";
   const char end_char = '\"';

   gchar *text;
   gchar compiler_string[] = "compiler=";
   gchar debugger_string[] = "debugger=";
   gchar execution_string[] = "execution=";
   gchar ext[24];   
   gchar *new_ext;
   gint success = FALSE;

   if(!(ifile = fopen(filename, "r"))) return(NULL);

   new_table = (GdsFileHighlightTable *)g_malloc0(sizeof(GdsFileHighlightTable));
   parse_table = new_table;
   new_table->filename = g_strdup(filename);
   new_table->props.over_ride = FALSE;
   new_table->props.auto_indent = general_preferences.auto_indent;
   new_table->props.use_spaces = general_preferences.use_spaces;
   new_table->props.spaces = general_preferences.spaces;
   new_table->props.syntax = general_preferences.syntax;
   new_table->props.bracketmatch = general_preferences.bracketmatch;
   new_table->props.dir = NULL;
   new_table->props.compiler = NULL;
   new_table->props.debugger = NULL;
   new_table->props.execution = NULL;

   while(fgets(buffer, sizeof(buffer), ifile))
   {
      if(buffer[0] == '#') continue;
      if(strstr(buffer, "<settings>")) 
      {
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(strstr(buffer, "</settings>")) break;
            if(strstr(buffer, "\n")) buffer[strlen(buffer)-1] = '\0';
            if((text = strstr(buffer, compiler_string)))
            {
               new_table->props.compiler = g_strdup(text + strlen(compiler_string));
               success++;
               continue;
            }
            else if((text = strstr(buffer, debugger_string)))
            {
               new_table->props.debugger = g_strdup(text + strlen(debugger_string));
               success++;
               continue;
            }
            else if((text = strstr(buffer, execution_string)))
            {
               new_table->props.execution = g_strdup(text + strlen(execution_string));
               success++;
               continue;
            }
            else if(sscanf(buffer, "over_ride=%d", &new_table->props.over_ride))
            {
               success++;
               continue;
            }
            else if(sscanf(buffer, "auto_indent=%d", &new_table->props.auto_indent))
            {
               success++;
               continue;
            }
            else if(sscanf(buffer, "use_spaces=%d", &new_table->props.use_spaces))
            {
               success++;
               continue;
            }
            else if(sscanf(buffer, "tab_stop=%d", &new_table->props.spaces))
            {
               success++;
            }
         }
      }
      else if(strstr(buffer, "<extensions>"))
      {
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(strstr(buffer, "</extensions>")) break;
            if((success = get_str(buffer, name, "name=\"", end_char, sizeof(name))))
            {
               if((success = get_str(buffer, ext, "ext=\"", end_char, sizeof(ext))))
               {
                  new_ext = g_strdup(ext);
                  new_table->extensions = g_list_append(new_table->extensions, (gpointer)new_ext);
               }
            }
         }
      }
      else if(strstr(buffer, "<syntax>"))
      {
         list = NULL;
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(buffer[0] == '#') continue;
            if(strstr(buffer, "</syntax>")) break;
            else
            {
               if((success = get_str(buffer, name, "name=\"", end_char, sizeof(name))))
               {
                  if((success = get_str(buffer, start, "start=\"", end_char, sizeof(start))))
                  {
                     if((success = get_str(buffer, end, "end=\"", end_char, sizeof(end))))
                     {
                        if((success = get_str(buffer, color, "color=\"", end_char, sizeof(color))))
                        {
                           HighlightInfo *info;
                           success = get_str(buffer, cclass, "class=\"", end_char, sizeof(cclass));
                           if(success)
                              info = get_highlight_info(color, cclass);
                           else
                              info = get_highlight_info(color, default_class);
                           syntax_list = gds_editor_syntax_entry_new(syntax_list, name, start, end, info->color, NULL, info->font, info->flags, info->type);
                           g_free(info);
                        }
                     }
                  }
               }
            }
         }
      }	 
      else if(strstr(buffer, "<pattern>"))
      {
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(buffer[0] == '#') continue;
            if(strstr(buffer, "</pattern>")) break;
            else
            {
               if((success = get_str(buffer, name, "name=\"", end_char, sizeof(name))))
               {
                  if((success = get_str(buffer, words, "words=\"", end_char, sizeof(words))))
                  {
                     if((success = get_str(buffer, color, "color=\"", end_char, sizeof(color))))
                     {
                        HighlightInfo *info;
                        success = get_str(buffer, cclass, "class=\"", end_char, sizeof(cclass));
                        if(success)
                           info = get_highlight_info(color, cclass);
                        else
                           info = get_highlight_info(color, default_class);
                        pattern_list = gds_editor_pattern_entry_new(pattern_list, name, words, info->color, NULL, info->font, info->flags, info->type);
                        g_free(info);
                     }
                  }
               }
            }
         }
      }
      else if(strstr(buffer, "<embedded>"))
      {
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(buffer[0] == '#') continue;
            if(strstr(buffer, "</embedded>")) break;
            else
            {
               if((success = get_str(buffer, name, "name=\"", end_char, sizeof(name))))
               {
                  if((success = get_str(buffer, start, "start=\"", end_char, sizeof(start))))
                  {
                     if((success = get_str(buffer, words, "words=\"", end_char, sizeof(words))))
                     {
                        if((success = get_str(buffer, end, "end=\"", end_char, sizeof(end))))
                        {
                           if((success = get_str(buffer, color, "color=\"", end_char, sizeof(color))))
                           {
                              HighlightInfo *info;
                              success = get_str(buffer, cclass, "class=\"", end_char, sizeof(cclass));
                              if(success)
                                 info = get_highlight_info(color, cclass);
                              else
                                 info = get_highlight_info(color, default_class);
                              embedded_list = gds_editor_embedded_entry_new(embedded_list, name, start, end, words, info->color, NULL, info->font, info->flags, info->type);
                              g_free(info);
                           }
                        }
                     }
                  }
               }	
            }
         }
      }	       
   }
   fclose(ifile);
   g_snprintf(temp_file, sizeof(temp_file), "%s/." PACKAGE "/%s", getenv("HOME"), "highlight.py");

   if(!(ifile = fopen(filename, "r"))) return(NULL);
   if(!(ofile = fopen(temp_file, "w")))
   {
   	fclose(ifile);
   	return(NULL);
   }

   while(fgets(buffer, sizeof(buffer), ifile) != NULL)
   {
      if(strstr(buffer, "<python>"))
      {
         while(fgets(buffer, sizeof(buffer), ifile))
         {
            if(strstr(buffer, "</python>")) break;
            fprintf(ofile, "%s", buffer);
         }
      }
   }
   fclose(ifile);
   fclose(ofile);
   ifile = fopen(temp_file, "r");
   if(!ifile) return(NULL);

   parsing_language = TRUE;

   PyRun_SimpleFile(ifile, temp_file);
   fclose(ifile);

   if(syntax_list) new_table->syntax = gds_editor_syntax_table_new(syntax_list);
   if(pattern_list) new_table->pattern = gds_editor_pattern_table_new(pattern_list);
   if(embedded_list) new_table->embedded = gds_editor_embedded_table_new(embedded_list);

   syntax_list = NULL;
   pattern_list = NULL;
   embedded_list = NULL;
   parsing_language = FALSE;
   parse_table = NULL;

   highlight_tables = g_list_append(highlight_tables, (gpointer)new_table);
   new_table->refcount = 0;
   return(new_table);
}

gboolean get_str(gchar *source, gchar *dest, const gchar search[], const gchar end, gint max_chars)
{
   char cur_char;
   char last_char = '\0';
   char *ptr;
   gint iterations = 0;

   if((ptr = strstr(source, search)) == NULL)
      return(FALSE);
   ptr += strlen(search);
   for(iterations = 0; iterations < max_chars; iterations++)
   {
      cur_char = *ptr;
      *dest = cur_char;
      if(cur_char == '\0')
         return(FALSE);
      if(cur_char == end && last_char != '\\')
      {
         *dest = '\0';
         break;
      }
      else if(cur_char == end && last_char == '\\')
      {
         dest--;
         *dest = '\"';
      }
      else if(last_char == '\\' && cur_char == 'n')
      {
         dest--;
         *dest = '\n';
      }
      else if(last_char == '\\' && cur_char == 't')
      {
         dest--;
         *dest = '\t';
      }
      last_char = cur_char;
      ptr++;
      dest++;
   }
   return(TRUE);
}


void g_list_delete_all(GList *list)
{
   GList *ptr = NULL;
   ptr = g_list_first(list);
   while(ptr)
   {
      g_free(ptr->data);
      ptr = ptr->next;
   }
   g_list_free(list);
}

/* Returns a table of highlight info, to be g_free()'d by caller */

HighlightInfo *get_highlight_info(gchar *color, gchar *cclass)
{
   gint i;
   HighlightInfo *info;
   gboolean set = FALSE;

   info = g_new0(HighlightInfo, 1);
   if(strchr(cclass, 'i') && strchr(cclass, 'b'))
      info->type = FONT_BOLD_ITALIC;
   else if(strchr(cclass, 'i'))
      info->type = FONT_ITALIC;
   else if(strchr(cclass, 'b'))
      info->type = FONT_BOLD;
   if(cclass && strcmp(cclass, "d")) set = TRUE; //This means it came from getprefs.C, elsewise it will always be "d" (from parser above)

   for(i = 0; i < EDITOR_COLORS; i++)
   {
      if(!strcmp(editor_color_names[i], color))
      {
         info->color = &HighlightCache.colors[i];
         info->font = HighlightCache.fonts[i];
         if(!set) info->type = HighlightCache.types[i];
         info->flags = HighlightCache.flags[i];
      }
   }

   info->font = get_font(info->type);
   if(strchr(cclass, 'u')) info->flags |= STYLE_UNDER;
   if(strchr(cclass, 's')) info->flags |= STYLE_STRIKE;
   return(info);
}

GdkFont *get_font(gint type)
{
   GdkFont *font = NULL;

   if(type == FONT_BOLD_ITALIC)
   {
      font = general_preferences.boldi;
   }
   else if(type == FONT_ITALIC)
   {
      font = general_preferences.italic;
   }
   else if(type == FONT_BOLD)
   {
      font = general_preferences.bold;
   }
   else
   {
      font = general_preferences.font;
   }
   return(font);
}

void highlight_tables_font_change(GdkFont *nfont, gint type)
{
   GList *tables = NULL;
   GList *entry = NULL;
   GdsFileHighlightTable *table;
   GdsEditorSyntaxEntry *syntax;
   GdsEditorPatternEntry *pattern;
   GdsEditorEmbeddedEntry *embedded;
   
   for(tables = g_list_first(highlight_tables); tables; tables = tables->next)
   {
      table = (GdsFileHighlightTable *)tables->data;
      if(table->syntax)
      for(entry = g_list_first(table->syntax->entries); entry; entry = entry->next)
      {
         syntax = (GdsEditorSyntaxEntry *)entry->data;
         if(syntax->type == type)
            syntax->font = nfont;
      }
      if(table->pattern)
      for(entry = g_list_first(table->pattern->entries); entry; entry = entry->next)
      {
         pattern = (GdsEditorPatternEntry *)entry->data;
         if(pattern->type == type)
            pattern->font = nfont;
      }
      if(table->embedded)
      for(entry = g_list_first(table->embedded->entries); entry; entry = entry->next)
      {
         embedded = (GdsEditorEmbeddedEntry *)entry->data;
         if(embedded->type == type)
            embedded->font = nfont;
      }
   }
}
