#include <unistd.h>
#include "declarations.h"
#include "macro-language.h"
#include "signals.h"

gchar *last_dir = NULL;
GList *recent_files = NULL;

static void main_quit(GtkWidget *widget, gpointer data);

void new_file_real(char *filename)
{
   cur_file = create_file(filename);
   gtk_widget_grab_focus(cur_file->text);

   attach_highlight_tables(cur_file, NULL);
   adjust_sensitivity();
   display_message("New file created.", FLASH);
}

void new_file_cb(GtkWidget *widget, gpointer data)
{
   new_file_real("Untitled");
}

void open_file_cb(GtkWidget *widget, gpointer data)
{
   widget = last_dir ? gnome_filelist_new_with_path(last_dir) : gnome_filelist_new_with_path(general_preferences.default_dir);
   gnome_filelist_set_title(GNOME_FILELIST(widget), _("Open File..."));
   gtk_signal_connect(GTK_OBJECT(GNOME_FILELIST(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(open_file_cb_continue), widget);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_window_set_transient_for(GTK_WINDOW(widget), GTK_WINDOW(window));
   gtk_widget_show(widget);
   gtk_widget_grab_focus(GNOME_FILELIST(widget)->selection_entry);
}

void open_file_cb_continue(GtkWidget *widget, GnomeFileList *file_list)
{
   gchar *string;
   gint result;
   string = gnome_filelist_get_filename(GNOME_FILELIST(file_list));
   if(check_if_file_exists(string))
   {
      gtk_widget_destroy(GTK_WIDGET(file_list));
      open_file_real(string);
   }
   else
   {
      if(check_if_file_can_exist(string))
      {
         remove(string);
         gtk_widget_hide(GTK_WIDGET(file_list));
         result = file_dne_error(string, GTK_SIGNAL_FUNC(open_file_error_cb));
         if(result != 0)
            gtk_widget_show(GTK_WIDGET(file_list));
         else
            gtk_widget_destroy(GTK_WIDGET(file_list));
      }
      else
      {
         gtk_widget_hide(GTK_WIDGET(file_list));
         read_error(string);
         gtk_widget_show(GTK_WIDGET(file_list));
      }
   }
   g_free(string);
}

void open_files_cb(GtkWidget *widget, gpointer data)
{
   GtkWidget *parent;
   widget = last_dir ? gnome_filelist_new_with_path(last_dir) : gnome_filelist_new_with_path(general_preferences.default_dir);
   gnome_filelist_set_title(GNOME_FILELIST(widget), _("Open Files..."));
   gtk_signal_connect(GTK_OBJECT(GNOME_FILELIST(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(open_files_cb_continue), widget);
   gtk_widget_grab_default(GNOME_FILELIST(widget)->ok_button);
   parent = GNOME_FILELIST(widget)->cancel_button->parent;
   gtk_widget_destroy(GNOME_FILELIST(widget)->cancel_button);
   GNOME_FILELIST(widget)->cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CLOSE);
   gtk_box_pack_start(GTK_BOX(parent), GNOME_FILELIST(widget)->cancel_button, FALSE, FALSE, 5);
   GTK_WIDGET_SET_FLAGS(GNOME_FILELIST(widget)->cancel_button, GTK_CAN_DEFAULT);
   gtk_widget_show(GNOME_FILELIST(widget)->cancel_button);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_widget_show(widget);
   gtk_widget_grab_focus(GNOME_FILELIST(widget)->selection_entry);
}

void open_files_cb_continue(GtkWidget *widget, GnomeFileList *file_list)
{
   gchar *string;
   gint result;
   string = gnome_filelist_get_filename(GNOME_FILELIST(file_list));
   if(check_if_file_exists(string))
      open_file_real(string);
   else
   {
      if(check_if_file_can_exist(string))
      {
         remove(string);
         result = file_dne_error(string, GTK_SIGNAL_FUNC(open_file_error_cb));
      }
      else read_error(string);
   }
   g_free(string);
}

void open_file_error_cb(GtkWidget *widget, gchar *filename)
{
   open_file_real(filename);
}

void open_file_real(gchar *filename)
{
   GdsFile *file;
   gchar *temp = NULL;

   if(general_preferences.untitled && (total_files >= 1) && (file = (GdsFile *)g_list_nth_data(files_list, 0)))
   {
      if(strstr(file->filename, "Untitled") && !GDS_EDITOR(file->text)->changed)
         close_file_real(file);
   }
   temp = get_path_from_filename(filename);
   if(temp && strcmp(filename, temp) != 0)
   {
      g_free(last_dir);
      last_dir = g_new(char, strlen(temp)+1);
      strcpy(last_dir, temp);
   }
   g_free(temp);
   cur_file = create_file(filename);
   open_file_add_text(cur_file->text, filename);
   check_if_read_only(filename, cur_file, TRUE, TRUE);
   add_file_to_recent_list(filename);
   attach_highlight_tables(cur_file, NULL);
   temp = cur_file->props.dir;
   if(cur_file->tables) copy_properties(&cur_file->props, &cur_file->tables->props);
   cur_file->props.dir = temp;
   gtk_extext_undo_clear_all(GTK_EXTEXT(cur_file->text));
   GDS_EDITOR(cur_file->text)->changed = FALSE;
   cur_file->modtime = get_time(cur_file->filename);
   adjust_sensitivity();
   adjust_popup_menu();
   display_message("File opened.", FLASH);
   if(general_preferences.auto_save) save_session_cb(NULL, NULL);
}

void revert_current_file(GtkWidget *widget, gpointer data)
{
   if(check_if_file_exists(cur_file->filename) && gtk_extext_get_editable(GTK_EXTEXT(cur_file->text)))
   {
      gtk_editable_delete_text(GTK_EDITABLE(cur_file->text), 0, gtk_extext_get_length(GTK_EXTEXT(cur_file->text)));
      open_file_add_text(cur_file->text, cur_file->filename);
      GDS_EDITOR(cur_file->text)->changed = FALSE;
      cur_file->modtime = get_time(cur_file->filename);
      display_message("File reverted to last saved version.", FLASH);
   }
   else if(strstr(cur_file->filename, "Untitled"))
   {
      gtk_editable_delete_text(GTK_EDITABLE(cur_file->text), 0, gtk_extext_get_length(GTK_EXTEXT(cur_file->text)));
      GDS_EDITOR(cur_file->text)->changed = FALSE;
      cur_file->modtime = get_time(cur_file->filename);
      display_message("File reverted to previous state.", FLASH);
   }
   gtk_signal_emit_by_name(GTK_OBJECT(cur_file->text), "move_to_column", 0, NULL);
   adjust_sensitivity();
   app_emit_scripting_signal("revert-file");
}

void save_file_cb(GtkWidget *widget, gpointer data)
{
   save_file_real(cur_file);
}

void save_file_error_cb(GtkWidget *widget, GdsFile *file)
{
   save_file_real(file);
}

void save_file_real(GdsFile *gds_file)
{
   FILE *file;
   gchar *all_text;
   GdsFile *back_file;

   g_return_if_fail(gds_file != NULL);
   g_return_if_fail(GDS_IS_FILE(gds_file));
   if(strstr(gds_file->filename, "Untitled"))
   {
      save_file_as_cb(NULL, NULL);
      return;
   }
   file = fopen(gds_file->filename, "w");
   if(file == NULL)
   {
      save_error(gds_file->filename);
      return;   
   }
   all_text = gtk_editable_get_chars(GTK_EDITABLE(gds_file->text), 0, gtk_extext_get_length(GTK_EXTEXT(gds_file->text)));
   if(gtk_extext_get_length(GTK_EXTEXT(gds_file->text))) fprintf(file, "%s", all_text);
   fclose(file);
   if(all_text) g_free(all_text);
   GDS_EDITOR(gds_file->text)->changed = FALSE;
   gds_file->modtime = get_time(gds_file->filename);
   gtk_signal_emit_by_name(GTK_OBJECT(gds_file->text), "move_to_column", 0, NULL);
   back_file = cur_file;
   cur_file = gds_file;
   app_emit_scripting_signal("save-file");
   cur_file = back_file;
   display_message("File Saved.", FLASH);
}

void save_file_as_cb(GtkWidget *widget, gpointer data)
{
   gchar *str = NULL;
   str = get_path_from_filename(cur_file->filename);
   widget = gnome_filelist_new_with_path(str);
   g_free(str);
   gnome_filelist_set_title(GNOME_FILELIST(widget), _("Save file with new name..."));
   gtk_signal_connect(GTK_OBJECT(GNOME_FILELIST(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(save_file_as_continue), widget);
   gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
   gtk_window_set_transient_for(GTK_WINDOW(widget), GTK_WINDOW(window));
   gtk_window_set_modal(GTK_WINDOW(widget), TRUE);
   gtk_signal_connect_object(GTK_OBJECT(widget), "destroy", GTK_SIGNAL_FUNC(gtk_grab_remove), GTK_OBJECT(widget));
   gtk_widget_show(widget);
   str = get_file_from_filename(cur_file->filename);
   if(str) gtk_entry_set_text(GTK_ENTRY(GNOME_FILELIST(widget)->selection_entry), str);
   g_free(str);
   gtk_widget_grab_focus(GNOME_FILELIST(widget)->selection_entry);
}

void save_file_as_continue(GtkWidget *widget, GnomeFileList *file_list)
{
   gchar *string;
   gint result;
   string = gnome_filelist_get_filename(GNOME_FILELIST(file_list));
   if(check_if_file_exists(string))
   {
      gtk_widget_hide(GTK_WIDGET(file_list));
      result = file_exists_error(string, GTK_SIGNAL_FUNC(save_file_as_error_cb));
      if(result == 0)
         gtk_widget_destroy(GTK_WIDGET(file_list));
      else
         gtk_widget_show(GTK_WIDGET(file_list));
   }
   else
   {
      gtk_widget_destroy(GTK_WIDGET(file_list));
      save_file_as_real(cur_file, string);
   }
   g_free(string);
}

void save_file_as_error_cb(GtkWidget *widget, gchar *filename)
{
   save_file_as_real(cur_file, filename);
}

void save_file_as_real(GdsFile *gds_file, gchar *filename)
{
   FILE *file;
   gchar *all_text;   
   gchar *temp;
   GdsFile *back_file;
   
   g_return_if_fail(gds_file != NULL);
   g_return_if_fail(GDS_IS_FILE(gds_file));

   if(!(file = fopen(filename, "w")))
   {
      save_error(filename);
      return;   
   }
   temp = get_path_from_filename(filename);
   if(temp && strcmp(filename, temp) != 0)
   {
      g_free(last_dir);
      last_dir = g_new(char, strlen(temp)+1);
      strcpy(last_dir, temp);
   }
   g_free(temp);
   all_text = gtk_editable_get_chars(GTK_EDITABLE(gds_file->text), 0, gtk_extext_get_length(GTK_EXTEXT(gds_file->text)));
   if(all_text) fprintf(file, "%s", all_text);
   fclose(file);
   temp = get_path_from_filename(cur_file->filename);
   if(!gds_file->props.dir || !strcmp(temp, gds_file->props.dir))
   {
      g_free(temp);
      temp = get_path_from_filename(filename);
      g_free(gds_file->props.dir);
      gds_file->props.dir = temp;
   }
   else g_free(temp);
   gds_file_set_filename(gds_file, filename);
   if(!GDS_EDITOR(gds_file->text)->changed && !gds_file->changed_set)
   {
      gds_file_set_title(gds_file, filename, general_preferences.full_filenames);
      gtk_signal_connect(GTK_OBJECT(gds_file->menu_item), "activate", GTK_SIGNAL_FUNC(gds_file_focus_with_raise), gds_file);
   }
   add_file_to_recent_list(filename);
   if(gds_file->default_lang)
   {
      attach_highlight_tables(gds_file, NULL);
      if(gds_file->tables && !gds_file->props.over_ride)
         copy_properties(&gds_file->props, &gds_file->tables->props);
   }
   gds_editor_set_editable(GDS_EDITOR(gds_file->text), TRUE);
   GDS_EDITOR(gds_file->text)->changed = FALSE;
   gds_file->modtime = get_time(gds_file->filename);
   gtk_signal_emit_by_name(GTK_OBJECT(gds_file->text), "move_to_column", 0, NULL);
   adjust_sensitivity();
   set_window_title();
   back_file = cur_file;
   cur_file = gds_file;
   app_emit_scripting_signal("save-as-file");
   cur_file = back_file;
   display_message("File saved with new name.", FLASH);
   if(general_preferences.auto_save) save_session_cb(NULL, NULL);
}

void save_all_files_cb(GtkWidget *widget, gpointer data)
{
   GdsFile *file;
   gint i = 0;

   display_message("Saving all open files...", FLASH);   
   for(i = total_files-1; i >= 0; i--)
   {
      file = (GdsFile *)g_list_nth_data(files_list, i);
      save_file_or_dialog(file);
   }
   display_message("All files saved.", FLASH);   
}

void reload_file_cb(GtkWidget *widget, GdsFile *gds_file)
{
   gtk_editable_delete_text(GTK_EDITABLE(gds_file->text), 0, GTK_EXTEXT(gds_file->text)->length);
	open_file_add_text(gds_file->text, gds_file->filename);
	GDS_EDITOR(gds_file->text)->changed = FALSE;
   gds_file->modtime = get_time(gds_file->filename);
}

static void main_quit(GtkWidget *widget, gpointer data)
{
   gtk_main_quit();
}

void save_file_or_dialog(GdsFile *gds_file)
{
   FILE *file;
   gchar *all_text;   
   GdsFile *back_file;

   g_return_if_fail(gds_file != NULL);
   g_return_if_fail(GDS_IS_FILE(gds_file));
   if(strstr(gds_file->filename, "Untitled"))
   {
      gchar *str = NULL;
      GtkWidget *widget;
      file_change(NULL, gds_file);
      str = get_path_from_filename(gds_file->filename);
      widget = gnome_filelist_new_with_path(str);
      g_free(str);
      gnome_filelist_set_title(GNOME_FILELIST(widget), _("Save file with new name..."));
      gtk_signal_connect(GTK_OBJECT(GNOME_FILELIST(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(save_file_as_continue), widget);
      gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(widget)->ok_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
      gtk_signal_connect_object(GTK_OBJECT(GNOME_FILELIST(widget)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widget));
      gtk_window_set_modal(GTK_WINDOW(widget), TRUE);
      gtk_window_set_transient_for(GTK_WINDOW(widget), GTK_WINDOW(window));
      gtk_widget_show(widget);
      str = get_file_from_filename(gds_file->filename);
      if(str) gtk_entry_set_text(GTK_ENTRY(GNOME_FILELIST(widget)->selection_entry), str);
      g_free(str);
      gtk_widget_grab_focus(GNOME_FILELIST(widget)->selection_entry);
      gtk_signal_connect_object(GTK_OBJECT(widget), "destroy", GTK_SIGNAL_FUNC(gtk_grab_remove), GTK_OBJECT(widget));
      gtk_signal_connect(GTK_OBJECT(widget), "destroy", GTK_SIGNAL_FUNC(main_quit), NULL);
      gtk_main();
      return;
   }
   file = fopen(gds_file->filename, "w");
   if(file == NULL)
   {
      save_error(gds_file->filename);
      return;   
   }
   all_text = gtk_editable_get_chars(GTK_EDITABLE(gds_file->text), 0, gtk_extext_get_length(GTK_EXTEXT(gds_file->text)));
   if(gtk_extext_get_length(GTK_EXTEXT(gds_file->text))) fprintf(file, "%s", all_text);
   fclose(file);
   if(all_text) g_free(all_text);
   GDS_EDITOR(gds_file->text)->changed = FALSE;
   gds_file->modtime = get_time(gds_file->filename);
   gtk_signal_emit_by_name(GTK_OBJECT(gds_file->text), "move_to_column", 0, NULL);
   back_file = cur_file;
   cur_file = gds_file;
   app_emit_scripting_signal("save-file");
   cur_file = back_file;
   display_message("File Saved.", FLASH);
}

void close_file_cb(GtkWidget *widget, gpointer data)
{
   GdsFile *file;
   file = cur_file;

   if(gds_editor_changed(GDS_EDITOR(cur_file->text)) && strstr(cur_file->filename, "Untitled"))
      file_not_saved_save_as(cur_file->filename, cur_file, NULL, NULL);
   else if(gds_editor_changed(GDS_EDITOR(cur_file->text)))
      file_not_saved(cur_file->filename, cur_file, NULL, NULL);
   else
   {
      close_file_real(file);
      if((total_files < 1) && general_preferences.untitled)
         new_file_cb(NULL, NULL);
   }
   if(general_preferences.auto_save) save_session_cb(NULL, NULL);
}

void close_file_error_cb(GtkWidget *widget, GdsFile *file)
{
   g_return_if_fail(file != NULL);
   g_return_if_fail(GDS_IS_FILE(file));
   close_file_real(file);
   if((total_files < 1) && general_preferences.untitled)
      new_file_cb(NULL, NULL);
}

void close_file_real(GdsFile *file)
{
   gint old_page;
   g_return_if_fail(file != NULL);
   g_return_if_fail(GDS_IS_FILE(file));
   if(GDS_TEAR_AWAY(file->tear_away)->torn) gds_tear_away_bust_a_switch(GDS_TEAR_AWAY(file->tear_away));
   app_emit_scripting_signal("close-file");
   detach_highlight_tables(file);
   files_list = g_list_remove(files_list, file);
   total_files--;
   if(!total_files) files_list = NULL;
   old_page = cur_page;
   gtk_widget_destroy(GTK_WIDGET(file));
   if(old_page == 0) { cur_page = old_page; }
   cur_file = NULL;
   gtk_notebook_set_page(GTK_NOTEBOOK(files_book), cur_page);
   set_notebook_page(GTK_NOTEBOOK(files_book), NULL, cur_page, NULL);
   adjust_sensitivity();
   display_message("File closed.", FLASH);   
}

void close_all_files_cb(GtkWidget *widget, gpointer data)
{
   GdsFile *file;
   gint i = 0;

   for(i = total_files-1; i >= 0; i--)
   {
      file = (GdsFile *)g_list_nth_data(files_list, i);
      file_change(widget, file);
      if(gds_editor_changed(GDS_EDITOR(file->text)))
      {
         file_not_saved(file->filename, file, GTK_SIGNAL_FUNC(close_all_files_cb), GTK_SIGNAL_FUNC(close_all_files_cb));
         return;
      }
      else
      {
         close_file_real(file);
      }
   }
}

gboolean open_file_add_text(GtkWidget *text, gchar *filename)
{
#define BUF_SIZE 131072
   FILE *file;
   gpointer buffer;
   gint bytes;
   gint pos;
   pos = gtk_extext_get_position(GTK_EDITABLE(text));
   
   if(!(file = fopen(filename, "rb")))
   {
      return FALSE;
   }
   gtk_extext_freeze(GTK_EXTEXT(text));
   buffer = g_new(char, BUF_SIZE);
   while((bytes = fread(buffer, sizeof(char), BUF_SIZE, file)))
   {
      gtk_editable_insert_text(GTK_EDITABLE(text), (gchar *)buffer, bytes, (gint *)&GTK_EDITABLE(text)->current_pos);
      if(bytes < BUF_SIZE) break;
   }
   g_free(buffer);   
   fclose(file);
   gtk_extext_set_position(GTK_EDITABLE(text), pos);
   gtk_extext_thaw(GTK_EXTEXT(text));
   gtk_widget_grab_focus(text);
   return TRUE;
}

gboolean check_if_file_exists(gchar *filename)
{
   struct stat st;
   gboolean file = FALSE;
   gboolean stated = FALSE;
   FILE *opened;
   if((opened = fopen(filename, "r")))
   {
      file = TRUE;
      fclose(opened);
   }
   stat(filename, &st);
   if(S_ISREG(st.st_mode))
      stated = TRUE;
   return(file && stated ? TRUE : FALSE);
}

gboolean check_if_file_can_exist(gchar *filename)
{
   gboolean success;
   FILE *file;
   if(!(file = fopen(filename, "a")))
   {
      success = FALSE;
   }
   else
   {
      success = TRUE;
      fclose(file);
   }
   return(success);
}

gboolean check_if_read_only(gchar *filename, GdsFile *temp, gboolean set_ro, gboolean do_error)
{
   FILE *file;
   if(!(file = fopen(filename, "a"))) //we arent going to edit it, just goint to check if we could...
   {
      if(set_ro)
      {
         gds_editor_set_editable(GDS_EDITOR(temp->text), FALSE);
      }
      if(do_error) read_only(filename);
      return(TRUE);
   }
   else
   {
      if(set_ro)
      {
         gds_editor_set_editable(GDS_EDITOR(temp->text), TRUE);
      }
      fclose(file);
   }
   return(FALSE);
}

void file_create(GtkWidget *widget, gchar *filename)
{
   FILE *file;
   if(!(file = fopen(filename, "w"))) //create it if it doesnt exist
      fclose(file);
}

GList *build_file_list(void)
{
   GList *temp;
   temp = g_list_copy(files_list);
   return(temp);
}

void setup_recent_files_list(gint max_files)
{
   GList *temp = NULL;
   GList *current = NULL;
   temp = build_glist_from_file("files.hist", max_files);
   build_recent_files_menu(temp);
   
   current = g_list_first(temp);
   while(current)
   {
      g_free(current->data);
      current = current->next;
   }
   g_list_free(temp);
}

void build_recent_files_menu(GList *file_strings)
{
   GList *current = NULL;
   gchar *temp = NULL;
   GtkWidget *menuitem;
   
   kill_old_recent_menu();
   
   current = g_list_first(file_strings);
   while(current)
   {
      temp = (gchar *)current->data;
      menuitem = gtk_menu_item_new_with_label(temp);
      recent_files = g_list_append(recent_files, (gpointer)menuitem);
      gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(recent_files_callback), 0);  
      gtk_menu_append(GTK_MENU(recentmenu), menuitem);
      gtk_widget_show(menuitem);
      current = current->next;
   }
}

void recent_files_callback(GtkWidget *widget, gpointer data)
{
   GtkWidget *accel;
   gchar *string;
   g_return_if_fail(GTK_IS_BIN(widget));
   
   accel = GTK_BIN(widget)->child;
   gtk_label_get(GTK_LABEL(accel), &string);

   if(check_if_file_exists(string))
   {
      open_file_real(string);
   }
   else
   {
      if(check_if_file_can_exist(string))
      {
         remove(string);
         file_dne_error(string, GTK_SIGNAL_FUNC(open_file_error_cb));
      }
      else
      {
         read_error(string);
      }
   }
}

void add_file_to_recent_list(gchar *filename)
{
   add_string_to_file("files.hist", filename);
   setup_recent_files_list(general_preferences.history);
}

void kill_old_recent_menu(void)
{
   GList *temp = NULL;
   GtkWidget *current;
   
   temp = g_list_first(recent_files);
   while(temp)
   {
      current = (GtkWidget *)temp->data;
      if(GTK_IS_WIDGET(current))
         gtk_widget_destroy(current);
      temp = temp->next;
   }
   g_list_free(recent_files);
   recent_files = NULL;
}

gint get_time(gchar *filename)
{
   struct stat st;
   int retval;
   retval = stat(filename, &st);
   if(retval == -1) return(0);
   return(st.st_mtime);
}
