/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * ggv-window.c: the ggv shell
 *
 * Copyright (C) 2002 the Free Software Foundation
 *
 * Author: Jaka Mocnik  <jaka@gnu.org>
 */

#include <config.h>

#include <gnome.h>
#include <bonobo-activation/bonobo-activation.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gconf/gconf-client.h>
#include <bonobo.h>
#include <bonobo/bonobo-ui-main.h>

#include <math.h>
#include <ctype.h>

#include "gtkgs.h"
#include "gtkchecklist.h"
#include "ggv-prefs.h"
#include "ggv-window.h"

#define GGV_POSTSCRIPT_VIEWER_CONTROL_IID "OAFIID:GNOME_GGV_Control"

static BonoboWindowClass *parent_class;

static GList *window_list;

/* what can be dragged in us... */
enum {
        TARGET_URI_LIST,
};

static void
ggv_window_drag_data_received(GtkWidget *widget,
                              GdkDragContext *context,
                              gint x, gint y,
                              GtkSelectionData *selection_data,
                              guint info, guint time)
{
        GgvWindow *win = GGV_WINDOW(widget);

	if (info != TARGET_URI_LIST)
		return;

        win->uris_to_open = g_strsplit(selection_data->data, "\r\n", 0);
        if (context->suggested_action == GDK_ACTION_ASK) {
                GtkWidget *menu = gtk_menu_new ();
		
                bonobo_window_add_popup (BONOBO_WINDOW (win), 
                                         GTK_MENU (menu), 
                                         "/popups/DnD");
                gtk_menu_popup (GTK_MENU (menu),
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                0,
                                GDK_CURRENT_TIME);
        }
        else {
                GtkWidget *newwin;
                gchar **uri = win->uris_to_open;

                if(win->current_page < 0)
                        newwin = GTK_WIDGET(win);
                else
                        newwin = NULL;
                while(*uri) {
                        if(!g_strncasecmp("file:", *uri, 5)) {
                                if(newwin == NULL) {
                                        newwin = ggv_window_new();
                                        gtk_widget_show(newwin);
                                }
                                ggv_window_load(GGV_WINDOW(newwin), (*uri) + 5);
                                newwin = NULL;
                        }
                        uri++;
                }
                g_strfreev(win->uris_to_open);
                win->uris_to_open = NULL;
        }
}

static void
sync_settings_menu_items(GgvWindow *win)
{
        if(win->uic == NULL)
                return;

        bonobo_ui_component_set_prop(win->uic, "/commands/SettingsShowMenus", "state",
                                     win->show_menus?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->uic, "/commands/SettingsShowToolbar", "state",
                                     win->show_toolbar?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->uic, "/commands/SettingsShowSidebar", "state",
                                     win->show_sidebar?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->uic, "/commands/SettingsShowStatusbar", "state",
                                     win->show_statusbar?"1":"0", NULL);
}

static void
sync_settings_popup_items(GgvWindow *win)
{
        if(win->popup_uic == NULL)
                return;

        bonobo_ui_component_set_prop(win->popup_uic, "/commands/SettingsShowMenus", "state",
                                     win->show_menus?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->popup_uic, "/commands/SettingsShowToolbar", "state",
                                     win->show_toolbar?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->popup_uic, "/commands/SettingsShowSidebar", "state",
                                     win->show_sidebar?"1":"0", NULL);
        bonobo_ui_component_set_prop(win->popup_uic, "/commands/SettingsShowStatusbar", "state",
                                     win->show_statusbar?"1":"0", NULL);
}

static void
sync_toolbar_style(GgvWindow *win)
{
        bonobo_ui_component_set_prop(win->uic, "/Toolbar", "look",
                                     ggv_toolbar_labels?"both":"icon", NULL);
}

static void
control_property_changed_handler(BonoboListener    *listener,
                                 char              *event_name, 
                                 CORBA_any         *any,
                                 CORBA_Environment *ev,
                                 gpointer           data)
{
        GgvWindow *win = GGV_WINDOW(data);

        if(!g_strcasecmp(event_name, "Bonobo/Property:change:title")) {
                gchar *title = g_strconcat(_("GGV: "), BONOBO_ARG_GET_STRING(any), NULL);
                gtk_window_set_title(GTK_WINDOW(win), title);
                g_free(title);
        }
        else if(!g_strcasecmp(event_name, "Bonobo/Property:change:status")) {
                gnome_appbar_set_status(GNOME_APPBAR(win->statusbar),
                                        BONOBO_ARG_GET_STRING(any));
        }
        else if(!g_strcasecmp(event_name, "Bonobo/Property:change:page")) {
                GtkTreeSelection *sel;
                gchar path_str[16];
                GtkTreePath *path;

                if(win->current_page == BONOBO_ARG_GET_INT(any) &&
                   win->current_page < 0)
                        return;

                win->current_page = BONOBO_ARG_GET_INT(any);
                if(win->current_page < 0) {
                        GtkWidget *dlg;
                        dlg = gtk_message_dialog_new(GTK_WINDOW(win),
                                                     GTK_DIALOG_MODAL,
                                                     GTK_MESSAGE_ERROR,
                                                     GTK_BUTTONS_OK,
                                                     _("Ghostscript error:\nProbably the file is not a PostScript file."));
                        gtk_widget_show(dlg);
                        gtk_dialog_run(GTK_DIALOG(dlg));
                        gtk_widget_destroy(dlg);
                        return;
                }
        }
        else if(!g_strcasecmp(event_name, "Bonobo/Property:change:page_count")) {
        }
}

static void
file_sel_unmap(GtkWidget *widget)
{
        ggv_get_window_size(widget, &file_sel_width, &file_sel_height);
}

static gboolean
file_sel_delete_event(GtkWidget *widget, GdkEventAny *e, gpointer data)
{
        gtk_widget_hide(widget);
        return FALSE;
}

static void
file_sel_ok_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);
        const gchar *fname;

        gtk_widget_hide(win->file_sel);
        fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(win->file_sel));
        if(!ggv_window_load(win, fname)) {
                GtkWidget *dlg;
                dlg = gtk_message_dialog_new(GTK_WINDOW(win),
                                             GTK_DIALOG_MODAL,
                                             GTK_MESSAGE_ERROR,
                                             GTK_BUTTONS_OK,
                                             _("Unable to load file:\n%s"),
                                             fname);
                gtk_widget_show(dlg);
                gtk_dialog_run(GTK_DIALOG(dlg));
                gtk_widget_destroy(dlg);
        }
}

static void
file_sel_cancel_clicked(GtkWidget *widget, gpointer data)
{
        GgvWindow *win = GGV_WINDOW(data);

        gtk_widget_hide(win->file_sel);
}

static void
verb_FileOpen(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(data);

        if(!win->file_sel) {
                win->file_sel = gtk_file_selection_new(_("Select a PostScript document"));
                gtk_window_set_transient_for(GTK_WINDOW(win->file_sel), GTK_WINDOW(win));
                gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(win->file_sel));
                gtk_window_set_resizable(GTK_WINDOW(win->file_sel), TRUE);
                g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(win->file_sel)->ok_button),
                                 "clicked", G_CALLBACK(file_sel_ok_clicked), win);
                g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(win->file_sel)->cancel_button),
                                 "clicked", G_CALLBACK(file_sel_cancel_clicked), win);
                g_signal_connect(G_OBJECT(win->file_sel), "delete_event",
                                 G_CALLBACK(file_sel_cancel_clicked), win);
                g_signal_connect(G_OBJECT(win->file_sel), "unmap",
                                 G_CALLBACK(file_sel_unmap), NULL);
        }
        if(!GTK_WIDGET_VISIBLE(win->file_sel)) {
                gtk_widget_set_usize(GTK_WIDGET(win->file_sel),
                                     file_sel_width, file_sel_height);
                if(win->filename) {
                        gchar *path = g_path_get_dirname(win->filename);
			gchar *path2 = g_strconcat(path, "/", NULL);
			g_free(path);
                        gtk_file_selection_set_filename(GTK_FILE_SELECTION(win->file_sel), path2);
                        g_free(path2);
                }
                gtk_widget_show(win->file_sel);
		ggv_raise_and_focus_widget(win->file_sel);
        }
}

static void
verb_FileReload(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(data);

        ggv_window_load(win, win->filename);
}

static void
verb_FileClose(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(data);

        ggv_window_close(win);
}

static void
verb_FileNew(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GtkWidget *win = ggv_window_new();
        gtk_widget_show(win);
}

static void
verb_FileExit(BonoboUIComponent *uic, gpointer data, const char *cname)
{
        GList *l;

        for(l = window_list; l != NULL; l = l->next)
                gtk_widget_destroy(GTK_WIDGET(l->data));
        g_list_free(window_list);
        window_list = NULL;
        bonobo_main_quit();
}

static void
listener_SettingsShow (BonoboUIComponent *uic, const char *path,
                       Bonobo_UIComponent_EventType type, const char *state,
                       gpointer user_data)
{
	GgvWindow *window;
        gboolean state_f;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_WINDOW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state)
		return;

        window = GGV_WINDOW(user_data);
        state_f = atoi(state);

        if(!strcmp(path, "SettingsShowMenus")) {
                if(window->show_menus != state_f) {
                        window->show_menus = state_f;
                        bonobo_ui_component_set_prop(window->uic, "/menu", "hidden",
                                                     state_f?"0":"1", NULL);
                }
        }
	else if(!strcmp(path, "SettingsShowSidebar")) {
                if(window->show_sidebar != state_f) {
                        window->show_sidebar = state_f;
                        bonobo_ui_component_set_prop(window->uic, "/Sidebar", "hidden",
                                                     state_f?"0":"1", NULL);
                }
        }
	else if(!strcmp(path, "SettingsShowToolbar")) {
                if(window->show_toolbar != state_f) {
                        window->show_toolbar = state_f;
                        bonobo_ui_component_set_prop(window->uic, "/Toolbar", "hidden",
                                                     state_f?"0":"1", NULL);
                }
        }
	else if(!strcmp(path, "SettingsShowStatusbar")) {
                if(window->show_statusbar != state_f) {
                        window->show_statusbar = state_f;
                        if(state_f)
                                gtk_widget_show(window->statusbar);
                        else
                                gtk_widget_hide(window->statusbar);
                }
        }
        if(uic == window->uic)
                sync_settings_popup_items(window);
        else
                sync_settings_menu_items(window);
}

static void
verb_HelpAbout(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	static GtkWidget *about = NULL;
	static const char *authors[] = {
                "Jaka Mocnik (current maintainer)",
                "Jonathan Blandford",
                "Daniel M. German",
                "Satyajit Kanungo",
                "Dan E. Kelley",
                "Werner Koerner",
                "Tuomas J. Lukka",
                "Johannes Plass",
		"Istvan Szekeres",
                "Tim Theisen",
                "And many more...",
 		NULL
	};
        static GdkPixbuf *logo = NULL;

        if(!logo)
                logo = gdk_pixbuf_new_from_file(GNOMEICONDIR
                                                "/ggv/ggv-splash.png",
                                                NULL);
	if (!about) {
		about = gnome_about_new (
			_("Gnome Ghostview"),
			VERSION,
			_("Copyright (C) 1998-2002 The Free Software Foundation"),
			_("The GNOME PostScript document previewer"),
			authors,
			NULL, /* char **documentors */
			NULL, /* char *translators */
			logo);
                gtk_window_set_transient_for(GTK_WINDOW(about),
                                             GTK_WINDOW(data));
		g_signal_connect (G_OBJECT (about), "destroy",
                                  G_CALLBACK (gtk_widget_destroyed),
                                  &about);
	}

	gtk_widget_show_now (about);
	ggv_raise_and_focus_widget (about);
}

static void
verb_HelpContents(BonoboUIComponent *uic, gpointer data, const char *cname)
{
	GError *error = NULL;
	gnome_help_display ("ggv",NULL,&error);
}

static void
verb_DnDNewWindow(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(user_data);
        gchar **uri;

        uri = win->uris_to_open;
        while(*uri) {
                GtkWidget *newwin;
                if(!g_strncasecmp("file:", *uri, 5)) {
                        newwin = ggv_window_new();
                        gtk_widget_show(newwin);
                        ggv_window_load(GGV_WINDOW(newwin), (*uri) + 5);
                }
                uri++;
        }
        g_strfreev(win->uris_to_open);
        win->uris_to_open = NULL;
}

static void
verb_DnDThisWindow(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(user_data);
        GtkWidget *newwin;
        gchar **uri;

        uri = win->uris_to_open;
        newwin = GTK_WIDGET(win);
        while(*uri) {
                if(!g_strncasecmp("file:", *uri, 5)) {
                        if(newwin == NULL) {
                                newwin = ggv_window_new();
                                gtk_widget_show(newwin);
                        }
                        ggv_window_load(GGV_WINDOW(newwin), (*uri) + 5);
                        newwin = NULL;
                }
                uri++;
        }
        g_strfreev(win->uris_to_open);
        win->uris_to_open = NULL;	
}

static void
verb_DnDCancel(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
        GgvWindow *win = GGV_WINDOW(user_data);
        g_strfreev(win->uris_to_open);
        win->uris_to_open = NULL;	
}

/* our verb list */
static BonoboUIVerb ggv_app_verbs[] = {
        BONOBO_UI_VERB("FileNew", verb_FileNew),
        BONOBO_UI_VERB("FileClose", verb_FileClose),
        BONOBO_UI_VERB("FileOpen", verb_FileOpen),
        BONOBO_UI_VERB("FileReload", verb_FileReload),
        BONOBO_UI_VERB("FileExit", verb_FileExit),
        BONOBO_UI_VERB("HelpAbout", verb_HelpAbout),
        BONOBO_UI_VERB("Help", verb_HelpContents),
        BONOBO_UI_VERB("DnDNewWindow", verb_DnDNewWindow),
        BONOBO_UI_VERB("DnDThisWindow", verb_DnDThisWindow),
        BONOBO_UI_VERB("DnDCancel", verb_DnDCancel),
        BONOBO_UI_VERB_END
};

static void
control_frame_activate_uri(BonoboControlFrame *control_frame, const char *uri,
                           gboolean relative, gpointer data)
{
	GgvWindow *win;
	char *path;

	g_return_if_fail(uri != NULL);

	if(g_strncasecmp("file:", uri, 5) == 0) {
                win = GGV_WINDOW(ggv_window_new());
                ggv_window_load(win, uri + 5);
                gtk_widget_show(GTK_WIDGET(win));
        }
}

static void
ggv_window_destroy(GtkObject *object)
{
        GgvWindow *win;

        g_return_if_fail(object != NULL);
        g_return_if_fail(GGV_IS_WINDOW(object));

        win = GGV_WINDOW(object);

        if(win->ctlframe != NULL) {
                bonobo_object_unref(BONOBO_OBJECT(win->ctlframe));
                win->ctlframe = NULL;
        }

        if(win->filename) {
                g_free(win->filename);
                win->filename = NULL;
        }

        ggv_get_window_size(GTK_WIDGET(object),
                            &ggv_default_width, &ggv_default_height);

        if(GTK_OBJECT_CLASS(parent_class)->destroy)
                GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static gboolean
ggv_window_delete_event(GtkWidget *widget, GdkEventAny *e)
{
        ggv_window_close(GGV_WINDOW(widget));

        return TRUE;
}

static void
ggv_window_class_init(GgvWindowClass *class)
{
	GObjectClass   *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	gobject_class = (GObjectClass *) class;
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;

	parent_class = g_type_class_peek_parent (class);

	object_class->destroy = ggv_window_destroy;

	widget_class->delete_event = ggv_window_delete_event;
	widget_class->drag_data_received = ggv_window_drag_data_received;
}

static void
ggv_window_init (GgvWindow *window)
{
        window_list = g_list_prepend(window_list, window);
}

GType
ggv_window_get_type (void) 
{
	static GType ggv_window_type = 0;
	
	if(!ggv_window_type) {
		static const GTypeInfo ggv_window_info =
		{
			sizeof(GgvWindowClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) ggv_window_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof(GgvWindow),
			0,		/* n_preallocs */
			(GInstanceInitFunc) ggv_window_init,
		};
		
		ggv_window_type = g_type_register_static(BONOBO_TYPE_WINDOW, 
                                                         "GgvWindow", 
                                                         &ggv_window_info, 0);
	}

	return ggv_window_type;
}

GtkWidget *
ggv_window_new(void)
{
        GgvWindow *win;
	BonoboUIContainer *ui_container;
	BonoboUIComponent *uic, *popup_uic;
	GdkGeometry geometry;
        CORBA_Environment ev;
        Bonobo_Control control;
        GtkWidget *widget, *sw, *hbox, *image;
        Bonobo_PropertyBag pb;
        gchar *mask;
        GtkTreeSelection *sel;

	static const GtkTargetEntry drag_types[] = {
		{ "text/uri-list", 0, TARGET_URI_LIST }
	};

        /* get the control */
        CORBA_exception_init(&ev);
        control = bonobo_get_object(GGV_POSTSCRIPT_VIEWER_CONTROL_IID,
                                    "Bonobo/Control", &ev);
        if(BONOBO_EX(&ev) || control == CORBA_OBJECT_NIL) {
                CORBA_exception_free(&ev);
                return NULL;
        }
        CORBA_exception_free(&ev);

	win = GGV_WINDOW(g_object_new(GGV_TYPE_WINDOW, "win_name", "ggv", "title", _("Gnome Ghostview"), NULL));
        win->control = control;

        win->current_page = -2;
        win->show_toolbar = ggv_toolbar;
        win->show_menus = ggv_menubar;
        win->show_statusbar = ggv_statusbar;
        win->show_sidebar = ggv_panel;

        /* a vbox */
        win->vbox = gtk_vbox_new(FALSE, 0);
        gtk_container_set_border_width(GTK_CONTAINER(win->vbox), 2);
        gtk_widget_show(win->vbox);

        /* a hbox at its top*/
        win->hbox = gtk_hbox_new(FALSE, 2);
        gtk_widget_show(win->hbox);
        gtk_box_pack_start(GTK_BOX(win->vbox), win->hbox,
                           TRUE, TRUE, 2);

	/* add statusbar */
	win->statusbar = gnome_appbar_new(FALSE, TRUE,
                                          GNOME_PREFERENCES_NEVER);
        if(ggv_statusbar)
                gtk_widget_show(GTK_WIDGET(win->statusbar));
	gtk_box_pack_end(GTK_BOX(win->vbox), GTK_WIDGET(win->statusbar),
                         FALSE, FALSE, 0);

        bonobo_window_set_contents(BONOBO_WINDOW(win), win->vbox);

	gtk_drag_dest_set (GTK_WIDGET (win),
			   GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
			   drag_types,
			   sizeof (drag_types) / sizeof (drag_types[0]),
			   GDK_ACTION_COPY | GDK_ACTION_ASK);

	/* add menu and toolbar */
	ui_container = bonobo_window_get_ui_container(BONOBO_WINDOW(win));
	BONOBO_OBJREF(ui_container);
	uic = bonobo_ui_component_new("ggv");
        win->uic = uic;
	bonobo_ui_component_set_container(uic, BONOBO_OBJREF(ui_container),
                                          NULL);
	bonobo_ui_util_set_ui(uic, NULL, "ggv-ui.xml", "GGV", NULL);
        sync_toolbar_style(win);
	bonobo_ui_component_add_verb_list_with_data(uic, ggv_app_verbs, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowMenus",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowSidebar",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowToolbar",
                                         listener_SettingsShow, win);
	bonobo_ui_component_add_listener(uic, "SettingsShowStatusbar",
                                         listener_SettingsShow, win);
        sync_settings_menu_items(win);

        if(!win->show_toolbar)
                bonobo_ui_component_set_prop(uic, "/Toolbar", "hidden", "1", NULL);
        if(!win->show_menus)
                bonobo_ui_component_set_prop(uic, "/menu", "hidden", "1", NULL);

	/* add control frame interface and bind it to ggv control */
	win->ctlframe = bonobo_control_frame_new(BONOBO_OBJREF(ui_container));
	bonobo_control_frame_set_autoactivate(win->ctlframe, FALSE);
	g_signal_connect(G_OBJECT(win->ctlframe), "activate_uri",
                         (GtkSignalFunc)control_frame_activate_uri, NULL);
	bonobo_control_frame_bind_to_control(win->ctlframe, control, NULL);
	widget = bonobo_control_frame_get_widget(win->ctlframe);
	gtk_widget_show(widget);
        gtk_box_pack_start(GTK_BOX(win->hbox), widget,
                           TRUE, TRUE, 0);
	bonobo_control_frame_control_activate(win->ctlframe);

        /* now get the control's property bag */
        pb = bonobo_control_frame_get_control_property_bag(win->ctlframe, NULL);
        if(pb == CORBA_OBJECT_NIL) {
                g_warning("Control does not have any properties.");
        }
        else {
                /* TODO: set initial status & title */
                mask =  "Bonobo/Property:change:page,"
                        "Bonobo/Property:change:page_count,"
                        "Bonobo/Property:change:width,"
                        "Bonobo/Property:change:height,"
                        "Bonobo/Property:change:title,"
                        "Bonobo/Property:change:status";
                bonobo_event_source_client_add_listener(pb,
                                                        (BonoboListenerCallbackFn)control_property_changed_handler,
                                                        mask, NULL, win);
                win->pb = pb;
        }

        /* merge our items in the control's popup menu */
        popup_uic = bonobo_control_frame_get_popup_component(win->ctlframe, NULL);
        if(popup_uic == NULL) {
                g_warning("Control does not have a popup component.");
        }
        else {
                bonobo_ui_util_set_ui(popup_uic, NULL, "ggv-ui.xml",
                                      "GGV", NULL);
                bonobo_ui_component_add_listener(popup_uic, "SettingsShowMenus",
                                                 listener_SettingsShow, win);
                bonobo_ui_component_add_listener(popup_uic, "SettingsShowSidebar",
                                                 listener_SettingsShow, win);
                bonobo_ui_component_add_listener(popup_uic, "SettingsShowToolbar",
                                                 listener_SettingsShow, win);
                bonobo_ui_component_add_listener(popup_uic, "SettingsShowStatusbar",
                                                 listener_SettingsShow, win);
                win->popup_uic = popup_uic;
                sync_settings_popup_items(win);
        }

	/* set default geometry */
        gtk_widget_set_usize(GTK_WIDGET(win),
                             ggv_default_width, ggv_default_height);
#if 0
	gtk_window_set_resizable(GTK_WINDOW(win), TRUE);
#else
        gtk_window_set_policy(GTK_WINDOW(win), TRUE, TRUE, FALSE);
#endif

	return GTK_WIDGET(win);
}

gboolean
ggv_window_load(GgvWindow *win, const gchar *filename)
{
        CORBA_Environment ev;
        Bonobo_PersistFile pf;
        gchar *full_path;

        g_return_val_if_fail(win != NULL, FALSE);
        g_return_val_if_fail(GGV_IS_WINDOW(win), FALSE);
        g_return_val_if_fail(filename != NULL, FALSE);

        /* TODO: add handling of stdin: should probably be solved by 
           copying stdin to a stream. */
        g_assert(strcmp(filename, "-"));

        CORBA_exception_init(&ev);
        pf = Bonobo_Unknown_queryInterface(win->control,
                                           "IDL:Bonobo/PersistFile:1.0", &ev);
        if(BONOBO_EX(&ev) || pf == CORBA_OBJECT_NIL) {
                CORBA_exception_free(&ev);
                return FALSE;
        }

        /* we need to pass full path to file to the control as we
           do not really know where it resides.
           NOTE: this still only works when the control is running
           on the same computer as the GGV. I suppose this will mostly
           be the case ;) */
        if(*filename != '/') {
                gchar *cwd;

                cwd = g_get_current_dir();
                full_path = g_strconcat(cwd, "/", filename, NULL);
                g_free(cwd);
        }
        else
                full_path = g_strdup(filename);

        Bonobo_PersistFile_load(pf, full_path, &ev);
        bonobo_object_release_unref(pf, NULL);

        if(win->filename)
                g_free(win->filename);
        win->filename = full_path;

        if(BONOBO_EX(&ev)) {
                CORBA_exception_free(&ev);
                return FALSE;
        }
        CORBA_exception_free(&ev);

        return TRUE;
}

void
ggv_window_close(GgvWindow *win)
{
        g_return_if_fail(win != NULL);
        g_return_if_fail(GGV_IS_WINDOW(win));

        window_list = g_list_remove(window_list, win);

        gtk_widget_destroy(GTK_WIDGET(win));

        if(!window_list)
                bonobo_main_quit();
}

void
ggv_window_apply_prefs(GgvWindow *win)
{
        if(win) {
                sync_toolbar_style(win);
        }
        else {
                GList *node = window_list;
                while(node) {
                        ggv_window_apply_prefs(GGV_WINDOW(node->data));
                        node = node->next;
                }
        }
}
