/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/* Copyright (C) 1998, 1999 Redhat Software Inc. 
 * Authors: Jonathan Blandford <jrb@redhat.com>
 *          Jaka Mocnik <jaka.mocnik@kiss.uni-lj.si>
 */

#include <config.h>
#include <gnome.h>
#include <math.h>
#include <ctype.h>
#include <sys/stat.h>
#include <libgnome/gnome-history.h>

#include "gtkscrollpane.h"
#include "gtkgs.h"
#include "crop.h"
#include "prefs.h"
#include "foot.xpm"
#include "foot_white.xpm"

#include "ggvwindow.h"
#include "callbacks.h"

/* This sizes are good for the worst case: laptops :) */

#define DEFAULT_WINDOW_WIDTH  640
#define DEFAULT_WINDOW_HEIGHT 480

enum {
        GEOMETRY_KEY = -1,
        SPARTAN_KEY = -2,
        NO_MENU_BAR_KEY = -3,
        FULL_SCREEN_KEY = -4,
};

GdkCursor *pan_cursor = NULL;

static gint width = DEFAULT_WINDOW_WIDTH, height = DEFAULT_WINDOW_HEIGHT;

GList *window_list;
GList *recent_files_list;
int active_ggv;
ggv_prefs prefs_dialog = { NULL };

static GdkPixmap *page_marker_pixmap;
static GdkBitmap *page_marker_mask;
static GdkPixmap *page_marker_white_pixmap;
static GdkBitmap *page_marker_white_mask;

static char     *geometry = 0;
static gboolean spartan = FALSE;
static gint     nomenubar = FALSE;
static gboolean fullscreen = FALSE;

static void free_wwd(window_with_data *wwd)
{
        if(wwd->data)
                g_free(wwd->data);
        g_free(wwd);
}

int get_number (const char **geometry)
{
	int value = 0;
	int mult  = 1;
	
	if (**geometry == '-'){
		mult = -1;
		(*geometry)++;
	}
	while (**geometry && isdigit (**geometry)){
		value = value * 10 + (**geometry - '0');
		(*geometry)++;
	}
	return value * mult;
}

gboolean
ggv_parse_geometry (const gchar *geometry, 
                    gint *xpos, gint *ypos, 
                    gboolean *xneg, gboolean *yneg,
                    gint *width, 
                    gint *height)
{
	int subtract;

	g_return_val_if_fail (xpos != NULL, FALSE);
	g_return_val_if_fail (ypos != NULL, FALSE);
	g_return_val_if_fail (width != NULL, FALSE);
	g_return_val_if_fail (height != NULL, FALSE);
	g_return_val_if_fail (yneg != NULL, FALSE);
	g_return_val_if_fail (xneg != NULL, FALSE);
	
	*xpos = *ypos = *width = *height = -1;
        *xneg = FALSE; *yneg = FALSE;

	if (!geometry)
		return FALSE;

	if (*geometry == '=')
		geometry++;
	if (!*geometry)
		return FALSE;
	if (isdigit (*geometry))
		*width = get_number (&geometry);
	if (!*geometry)
		return TRUE;
	if (*geometry == 'x' || *geometry == 'X'){
		geometry++;
		*height = get_number (&geometry);
	}
	if (!*geometry)
		return 1;
	if (*geometry == '+'){
		subtract = 0;
		geometry++;
	} else if (*geometry == '-'){
		subtract = gdk_screen_width ();
		geometry++;
	} else
		return FALSE;
	*xpos = get_number (&geometry);
	if (subtract) {
		*xpos = subtract - *xpos;
                *xneg = TRUE;
        }
	if (!*geometry)
		return TRUE;
	if (*geometry == '+'){
		subtract = 0;
		geometry++;
	} else if (*geometry == '-'){
		subtract = gdk_screen_height ();
		geometry++;
	} else
		return FALSE;
	*ypos = get_number (&geometry);
	if (subtract) {
		*ypos = subtract - *ypos;
                *yneg = TRUE;
        }
	return TRUE;
}

char *escape_underlines (char *str)
{
	char *buf;
	char *p;

	buf = g_new (char, 2 * strlen (str) + 1);
	
	for (p = buf; *str; str++) {
		if (*str == '_')
			*p++ = '_';

		*p++ = *str;
	}

	*p = '\0';

	return buf;
}

void recent_update_menus (ggv_window *ggv, GList *recent_files)
{
	GnomeUIInfo *menu;
	gchar *path, *recent_data;
	int i;
        window_with_data *wwd;
        char *escape;
        
        if (!ggv->show_menus)
                return;

	if (ggv->num_recent > 0)
                gnome_app_remove_menu_range (GNOME_APP(ggv->main_window), 
                                             "_File/", 4, ggv->num_recent + 1);

	if (recent_files == NULL)
		return;

	/* insert a separator at the beginning */
	menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
	menu->type = GNOME_APP_UI_SEPARATOR;

	(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
	gnome_app_insert_menus (GNOME_APP(ggv->main_window),
                                "_File/<Separator>", menu);

	for (i = g_list_length (recent_files) - 1; i >= 0;  i--)
	{
                recent_data = (gchar *)g_list_nth_data(recent_files, i);
                wwd = g_malloc0(sizeof(window_with_data));

		menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
	
                escape = escape_underlines (recent_data);
		menu->label =  g_strdup_printf("_%i. %s", i+1, escape);
                g_free (escape);

		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
                
		menu->moreinfo = (gpointer) recent_callback;

                wwd->data = g_strdup(recent_data);
                wwd->ggv = ggv;

		menu->user_data = wwd;
		menu->unused_data = NULL;
		menu->pixmap_type = 0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;

		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
	
		gnome_app_insert_menus (GNOME_APP(ggv->main_window),
                                        "_File/<Separator>", menu);

                gtk_object_set_data_full(GTK_OBJECT(menu->widget), "wwd",
                                         wwd, (GtkDestroyNotify)free_wwd);
	}
	ggv->num_recent = g_list_length (recent_files);
}

void recent_update()
{
	GList *dirlist = NULL;
        GList *filelist = NULL;
	GList *gnome_recent_list;
        GList *ggv_node;
	GnomeHistoryEntry histentry;
	char *filename;
	int i, j;
	gboolean duplicate = FALSE;

	filelist = NULL;
	gnome_recent_list = gnome_history_get_recently_used ();

        if(recent_files_list) {
                g_list_foreach(recent_files_list, (GFunc)g_free, NULL);
                g_list_free(recent_files_list);
        }

	if (g_list_length (gnome_recent_list) > 0)                {
                for (i = g_list_length (gnome_recent_list) - 1; i >= 0; i--) {
                        histentry = g_list_nth_data (gnome_recent_list, i);
			if (strcmp ("ggv", histentry->creator) == 0) {
				/* This is to make sure you don't have more than one
				   file of the same name in the recent list
				*/
				if (g_list_length (filelist) > 0)
					for (j = g_list_length (filelist) - 1; j >= 0; j--)
						if (strcmp (histentry->filename, g_list_nth_data (filelist, j)) == 0)
							filelist = g_list_remove (filelist, g_list_nth_data (filelist, j));
                                
				filename = g_strdup(histentry->filename);
				filelist = g_list_append (filelist, filename);
                                
                                /* For recent-directories, not yet fully implemented...
                                   end_path = strrchr (histentry->filename, '/');
                                   if (end_path)
                                   {
                                   for (i = 0; i < strlen (histentry->filename); i++)
                                   if ( (histentry->filename + i) == end_path)
                                   break;
                                   directory = g_malloc0 (i + 2);
                                   strcat (directory, histentry->filename, i);
                                   }
                                */
				if (g_list_length (filelist) == MAX_RECENT)
					break;
			}
		}
	}

	gnome_history_free_recently_used_list (gnome_recent_list);

        recent_files_list = filelist;

        ggv_node = window_list;
        while(ggv_node) {
                recent_update_menus ((ggv_window *)ggv_node->data,
                                     recent_files_list);
                ggv_node = ggv_node->next;
        }
}

void recent_add(char *filename)
{
        gnome_history_recently_used (filename, "text/plain", "ggv", "");
}

/* Useful functions */
gboolean load_gs(ggv_window *ggv, char *fname)
{
	char *title;
	guint i,n;
	gchar *texts[3], *flash;
        gint page;
        gboolean is_structured;

        if(ggv->loaded && strcmp(fname, GTK_GS(ggv->gs)->gs_filename) == 0)
                page = GTK_GS(ggv->gs)->current_page;
        else
                page = 0;
        flash = g_strdup_printf(_("Opening %s..."), fname);
        flash_message(ggv, flash);
	if (!gtk_gs_load (GTK_GS (ggv->gs), fname)) {
		GtkWidget *w;
                gtk_clist_clear (GTK_CLIST (ggv->pagelist));
                gtk_window_set_title (GTK_WINDOW(ggv->main_window), "ggv");
                gtk_widget_set_sensitive (ggv->zoomoutbutton, FALSE);
                gtk_widget_set_sensitive (ggv->zoominbutton, FALSE);
                gtk_widget_set_sensitive (ggv->printitem, FALSE);
                gtk_widget_set_sensitive (ggv->printmarked, FALSE);
                gtk_widget_set_sensitive (ggv->save, FALSE);
                /*                gtk_widget_set_sensitive (ggv->saveas, FALSE); */
                gtk_widget_set_sensitive (ggv->toggleakt, FALSE);
                gtk_widget_set_sensitive (ggv->toggleall, FALSE);
                gtk_widget_set_sensitive (ggv->toggleeven, FALSE);
                gtk_widget_set_sensitive (ggv->toggleodd, FALSE);
                gtk_widget_set_sensitive (ggv->clearall, FALSE);
                gtk_widget_set_sensitive (ggv->toggleakt_pu, FALSE);
                gtk_widget_set_sensitive (ggv->toggleall_pu, FALSE);
                gtk_widget_set_sensitive (ggv->toggleeven_pu, FALSE);
                gtk_widget_set_sensitive (ggv->toggleodd_pu, FALSE);
                gtk_widget_set_sensitive (ggv->clearall_pu, FALSE);
                flash = g_strdup_printf(_("Unable to load %s."), fname);
                error_message(ggv, flash);
                return FALSE;
        }
        fflush(NULL);
        gtk_gs_set_pagemedia(GTK_GS (ggv->gs), -1, 0);
        fflush(NULL);

	title = gtk_gs_document_title (GTK_GS (ggv->gs));
	if (title) {
		title = g_strjoin("", "ggv: ", title, NULL);
		gtk_window_set_title(GTK_WINDOW(ggv->main_window), title);
		g_free(title);
	}
        else 
                gtk_window_set_title (GTK_WINDOW(ggv->main_window), "ggv");

        /* FIXME:
           Resize window if necessary, we 
           must verify whether automatic resizing is allowed (add option to
           menu on whether to auto resize 
           finally, at most it should be as large as the screen
           
           We also have to zoom the document to the specific zoom factor
           according to the menu. 

           gtk_gs_set_zoom (GTK_GS (ggv->gs), ggv->zoom_value); 

           So far, the user can't make the window smaller. I don't 
           know how to fix it.

        */

	/*
        gtk_widget_set_usize (ggv->gs, 
                              GTK_GS(ggv->gs)->width,GTK_GS(ggv->gs)->height);
	*/

	is_structured = GTK_GS(ggv->gs)->structured_doc;

        if(ggv->pagelist) {
                gtk_clist_freeze(GTK_CLIST(ggv->pagelist));
                gtk_clist_clear(GTK_CLIST(ggv->pagelist));
                if(is_structured) {
                        n = gtk_gs_document_numpages(GTK_GS (ggv->gs));

                        if (n > 0) {
                                for (i=1; i<=n ; i++) {
                                        texts[0] = "";
                                        texts[1] = g_strdup(gtk_gs_document_page_label(GTK_GS (ggv->gs), i));
                                        gtk_clist_append(GTK_CLIST(ggv->pagelist), texts);
                                }
                        }
                }
                gtk_clist_thaw(GTK_CLIST(ggv->pagelist));
        }
        goto_page(ggv, page);
        gtk_widget_set_sensitive(ggv->zoomoutbutton, TRUE);
        gtk_widget_set_sensitive(ggv->zoominbutton, TRUE);
        gtk_widget_set_sensitive(ggv->printitem, TRUE);
        gtk_widget_set_sensitive(ggv->printmarked, is_structured);
        gtk_widget_set_sensitive(ggv->reloaditem, TRUE);
        gtk_widget_set_sensitive(ggv->save, is_structured);

        /* this is not implemented yet 
           gtk_widget_set_sensitive(ggv->saveas, TRUE); */

        set_page_sensitivities(ggv, page);

        gtk_widget_set_sensitive (ggv->toggleakt, is_structured);
        gtk_widget_set_sensitive (ggv->toggleall, is_structured);
        gtk_widget_set_sensitive (ggv->toggleeven, is_structured);
        gtk_widget_set_sensitive (ggv->toggleodd, is_structured);
        gtk_widget_set_sensitive (ggv->clearall, is_structured);
        gtk_widget_set_sensitive (ggv->toggleakt_pu, is_structured);
        gtk_widget_set_sensitive (ggv->toggleall_pu, is_structured);
        gtk_widget_set_sensitive (ggv->toggleeven_pu, is_structured);
        gtk_widget_set_sensitive (ggv->toggleodd_pu, is_structured);
        gtk_widget_set_sensitive (ggv->clearall_pu, is_structured);

        flash = g_strdup_printf(_("Successfully loaded %s."), fname);
        flash_message(ggv, flash);

        recent_add(fname);

        return TRUE;
}

void save_marked_pages(ggv_window *ggv, gchar *file)
{
        gchar *error_msg, *flash;
        gint i;
        FILE *file_handle;

        g_print("Pages marked %d\n",
                gtk_gs_count_marked_pages(GTK_GS(ggv->gs)));

        if((file_handle = fopen(file, "w")) != NULL) {

                if (GTK_GS (ggv->gs)->gs_filename_unc != NULL)
                        /* Use uncompress file */
                        pscopydoc(file_handle, GTK_GS (ggv->gs)->gs_filename_unc,
                          GTK_GS(ggv->gs)->doc, 
                          GTK_GS(ggv->gs)->pages_marked); 
                else
                        pscopydoc(file_handle, GTK_GS (ggv->gs)->gs_filename,
                                  GTK_GS(ggv->gs)->doc, 
                                  GTK_GS(ggv->gs)->pages_marked);
                flash = g_strdup_printf(_("Successfully saved marked pages to file %s."), file);
                flash_message(ggv, flash);
        }
        else {
                error_msg = g_strdup_printf(_("Unable to create %s."), file);
                error_message(ggv,error_msg);
        }
}

void set_page_sensitivities(ggv_window *ggv, int page)
{
        gboolean prev = FALSE, next = FALSE;

        if(page > 0)
                prev = TRUE;
        if(page < GTK_GS(ggv->gs)->doc->numpages - 1 ||
           GTK_GS(ggv->gs)->structured_doc == FALSE)
                next = TRUE;
        gtk_widget_set_sensitive(ggv->nextbutton, next);
        gtk_widget_set_sensitive(ggv->prevbutton, prev);
        gtk_widget_set_sensitive(ggv->nextitem, next);
        gtk_widget_set_sensitive(ggv->previtem, prev);
        gtk_widget_set_sensitive(ggv->nextpu, next);
        gtk_widget_set_sensitive(ggv->prevpu, prev);
}        

void goto_page(ggv_window *ggv, int page)
{
        gtk_gs_goto_page ( GTK_GS (ggv->gs), page);

        if (GTK_GS(ggv->gs)->structured_doc) {
                page = GTK_GS (ggv->gs)->current_page; /* Make sure */
                set_page_sensitivities(ggv, page);
                if(ggv->pagelist) {
                        gtk_clist_select_row (GTK_CLIST (ggv->pagelist), page,1);
                        if( !gtk_clist_row_is_visible (GTK_CLIST(ggv->pagelist), page)) {
                                gtk_clist_moveto (GTK_CLIST (ggv->pagelist), GTK_GS (ggv->gs)->current_page, 0, 0.5, 0.5);
                        }
                }
        }
}

/* Convenience function to load a pixmap and mask from xpm data */
void create_pixmap (char **data, GdkPixmap **pixmap, GdkBitmap **mask)
{
	GdkImlibImage *im;

	im = gdk_imlib_create_image_from_xpm_data (data);
	gdk_imlib_render (im, im->rgb_width, im->rgb_height);
	*pixmap = gdk_imlib_copy_image (im);
	*mask = gdk_imlib_copy_mask (im);
	gdk_imlib_destroy_image (im);
}

void toggle_pages(ggv_window *ggv, gint what_to_toggle, gint page_number)
{
        gint n;

	g_assert (ggv);
	g_assert (ggv->gs);

        if((!ggv->loaded) || (!GTK_GS(ggv->gs)->structured_doc))
                return;

        gtk_clist_freeze(GTK_CLIST(ggv->pagelist));

        switch (what_to_toggle) {
        case TOGGLE_CURRENT_PAGE:
                if (GTK_GS(ggv->gs)->pages_marked[page_number] = !GTK_GS(ggv->gs)->pages_marked[page_number])
                        gtk_clist_set_pixmap(GTK_CLIST(ggv->pagelist), page_number, 0, 
                                             page_marker_pixmap, page_marker_mask);
                else
                        gtk_clist_set_pixmap(GTK_CLIST(ggv->pagelist), page_number, 0, 
                                             page_marker_white_pixmap, page_marker_white_mask);
                break;
        case TOGGLE_ALL_PAGES:
        case TOGGLE_EVEN_PAGES:
        case TOGGLE_ODD_PAGES:
                n = gtk_gs_document_numpages(GTK_GS (ggv->gs));
                for (page_number=(what_to_toggle==TOGGLE_EVEN_PAGES)?1:0;
                     page_number<n;
                     page_number+=(what_to_toggle==TOGGLE_ALL_PAGES)?1:2)
                        if (GTK_GS(ggv->gs)->pages_marked[page_number] = !GTK_GS(ggv->gs)->pages_marked[page_number])
                                gtk_clist_set_pixmap(GTK_CLIST(ggv->pagelist), page_number, 0, 
                                                     page_marker_pixmap, page_marker_mask);
                        else
                                gtk_clist_set_pixmap(GTK_CLIST(ggv->pagelist), page_number, 0, 
                                                     page_marker_white_pixmap, page_marker_white_mask);
                break;
        case UNMARK_ALL_PAGES:
                n = gtk_gs_document_numpages(GTK_GS (ggv->gs));
                for (page_number=0; page_number<n; page_number++) {
                        GTK_GS(ggv->gs)->pages_marked[page_number]  = FALSE;
                        gtk_clist_set_pixmap(GTK_CLIST(ggv->pagelist), page_number, 0, 
                                             page_marker_white_pixmap, page_marker_white_mask);
                        
                }
                break;
        }

        gtk_clist_thaw(GTK_CLIST(ggv->pagelist));
}

double compute_zoom(gint zoom_spec)
{
	return pow(1.2, zoom_spec); /* The Knuth magstep formula rules */
}

void zoom_to(ggv_window *ggv, gint magstep) 
{
        double z;
        ggv->zoom_magstep = magstep;

        z = compute_zoom(magstep);

        /* Return if no active doc */
        if (!ggv->loaded) {
                g_assert(ggv->gs);
                GTK_GS(ggv->gs)->zoom_factor = z;
                return;
        }
        gtk_gs_set_zoom (GTK_GS (ggv->gs), z);
        gtk_gs_goto_page (GTK_GS (ggv->gs), GTK_GS(ggv->gs)->current_page);        
}

void create_popup_menus(ggv_window *ggv)
{
        /* the below #defines are supposed to be synced with actual indexes
           of the corresponding items in the popupMenu[] array! */
        GnomeUIInfo popupMenu[] = {
#define PU_NEXT_PAGE_INDEX 0
                {GNOME_APP_UI_ITEM, N_("Next Page"), NULL, next_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
#define PU_PREV_PAGE_INDEX 1
                {GNOME_APP_UI_ITEM, N_("Previous Page"), NULL, previous_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Zoom In"), NULL, zoomin_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Zoom Out"), NULL, zoomout_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Recenter Page"), NULL, recenter_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
#define PU_PAGE_MENUS_INDEX 6
                {GNOME_APP_UI_ITEM, N_("Toggle _current page"), NULL, toggle_current_page_callback, NULL ,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _all pages"), NULL, toggle_all_pages_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _even pages"), NULL, toggle_even_pages_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _odd pages"), NULL, toggle_odd_pages_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Unmark all pages"), NULL, unmark_all_pages_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
#define PU_SHOW_MENUS_INDEX 12
                {GNOME_APP_UI_TOGGLEITEM, N_("Show Menus"), NULL, show_menubar_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
#define PU_SHOW_PANEL_INDEX 13
                {GNOME_APP_UI_TOGGLEITEM, N_("Show Side Panel"), NULL, hide_panel_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ENDOFINFO}
        };
                
        ggv->popup_menu = gnome_popup_menu_new(popupMenu);
        ggv->nextpu = popupMenu[PU_NEXT_PAGE_INDEX].widget;
        ggv->prevpu = popupMenu[PU_PREV_PAGE_INDEX].widget;
        ggv->menus_vis_pu = popupMenu[PU_SHOW_MENUS_INDEX].widget;
        ggv->panel_vis_pu = popupMenu[PU_SHOW_PANEL_INDEX].widget;
        ggv->toggleakt_pu = popupMenu[PU_PAGE_MENUS_INDEX].widget;
        ggv->toggleall_pu = popupMenu[PU_PAGE_MENUS_INDEX+1].widget;
        ggv->toggleeven_pu = popupMenu[PU_PAGE_MENUS_INDEX+2].widget;
        ggv->toggleodd_pu = popupMenu[PU_PAGE_MENUS_INDEX+3].widget;
        ggv->clearall_pu = popupMenu[PU_PAGE_MENUS_INDEX+4].widget;

        gtk_widget_set_sensitive (GTK_WIDGET(ggv->prevpu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->nextpu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->toggleakt_pu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->toggleall_pu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->toggleeven_pu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->toggleodd_pu), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->clearall_pu), FALSE);
}
      
void create_menus(ggv_window *retval, window_with_data *temp)
{
        GnomeUIInfo orientationMenu2[] = {
                GNOMEUIINFO_ITEM_DATA(N_("_Automatic"), N_("Use document's orientation"), orientation_callback, &temp[0], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Portrait"), N_("Portrait orientation"), orientation_callback, &temp[1], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Landscape"), N_("Landscape orientation"), orientation_callback, &temp[2], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Seascape"), N_("Seascape orientation"), orientation_callback, &temp[3], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Upside Down"), N_("Upside down orientation"), orientation_callback, &temp[4], NULL),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo orientationMenu[] = {
                GNOMEUIINFO_RADIOLIST(orientationMenu2),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo zoomMenu2[] = {
                GNOMEUIINFO_RADIOITEM_DATA("1:1", NULL, zoom_callback,  &temp[10], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:10" , NULL, zoom_callback, &temp[5] , NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:8" , NULL, zoom_callback, &temp[6], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:4", NULL, zoom_callback, &temp[7], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:2", NULL, zoom_callback, &temp[8], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:1.4", NULL, zoom_callback, &temp[9], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1.4:1", NULL, zoom_callback, &temp[11] , NULL),
                GNOMEUIINFO_RADIOITEM_DATA("2:1", NULL, zoom_callback, &temp[12], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("4:1", NULL, zoom_callback, &temp[13], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("8:1", NULL, zoom_callback, &temp[14], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("10:1", NULL, zoom_callback, &temp[15], NULL),
                GNOMEUIINFO_END,
        };

        GnomeUIInfo zoomMenu[] = {
                GNOMEUIINFO_RADIOLIST(zoomMenu2),
                GNOMEUIINFO_END,
        };

        GnomeUIInfo settingsMenu[] = {
                GNOMEUIINFO_MENU_PREFERENCES_ITEM(preferences_callback, retval),
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_TOGGLEITEM, N_("_Show Menus"), N_("Toggle menu visibility"), show_menubar_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_TOGGLEITEM, N_("S_how Side Panel"), N_("Toggle side panel visibility"), hide_panel_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                GNOMEUIINFO_END
        };

        GnomeUIInfo fileMenu[] = {
                GNOMEUIINFO_MENU_NEW_ITEM(N_("_New Window"), N_("Open a new window"), new_callback, NULL),
                {GNOME_APP_UI_ITEM, N_("_Open..."), N_("Load a postscript document"), open_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 'o', GDK_CONTROL_MASK, NULL},
                {GNOME_APP_UI_ITEM, N_("_Reload"), N_("Reload current document from disk"), reload_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH, 'r', GDK_CONTROL_MASK, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("_Print"), N_("Print the document"), print_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 'p', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_ITEM, N_("Print marked pages"), N_("Print marked pages"), print_marked_pages_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 0, 0, NULL },
                {GNOME_APP_UI_ITEM, N_("Save _marked pages..."), N_("Save marked pages to a new file"), save_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL},
                /* Not implemented yet 
                {GNOME_APP_UI_ITEM, N_("Save _As..."), N_("Save document to a new file"), save_as_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL},
                */
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("_Close"), N_("Close this window"), close_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 'w', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("E_xit"), N_("Exit GGv"), exit_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'q', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_ENDOFINFO, NULL, NULL, NULL, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL}
        };

        GnomeUIInfo documentMenu[] = {
                {GNOME_APP_UI_ITEM, N_("_Next Page"), N_("Show next page"), next_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Previous Page"), N_("Show previous page"), previous_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Recenter Page"), N_("Center page"), recenter_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("Toggle _current page"), NULL, toggle_current_page_callback, retval,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _all pages"), NULL, toggle_all_pages_callback, retval,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _even pages"), NULL, toggle_even_pages_callback, retval,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Toggle _odd pages"), NULL, toggle_odd_pages_callback, retval,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Unmark all pages"), NULL, unmark_all_pages_callback, retval,NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_SUBTREE, N_("_Orientation"), NULL, orientationMenu, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SUBTREE, N_("_Zoom"), NULL, zoomMenu, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ENDOFINFO}
        };
        
        GnomeUIInfo helpMenu[] = {
                GNOMEUIINFO_HELP("ggv"),
                GNOMEUIINFO_MENU_ABOUT_ITEM(about_callback, retval),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo mainMenu[] = {
                GNOMEUIINFO_MENU_FILE_TREE(fileMenu),
                GNOMEUIINFO_SUBTREE(N_("_Document"), documentMenu),
                GNOMEUIINFO_MENU_SETTINGS_TREE(settingsMenu),
                GNOMEUIINFO_MENU_HELP_TREE(helpMenu),
                GNOMEUIINFO_END
        };

        /* We set up the main window... */
        gnome_app_create_menus (GNOME_APP (retval->main_window), mainMenu);

	gnome_app_install_menu_hints(GNOME_APP(retval->main_window),
                                     mainMenu);

        retval->nextitem = documentMenu[0].widget;
        retval->previtem = documentMenu[1].widget;
        retval->reloaditem = fileMenu[2].widget;
        retval->printitem = fileMenu[4].widget;
        retval->printmarked = fileMenu[5].widget;
        retval->save = fileMenu[6].widget;
        /*
        retval->saveas = fileMenu[7].widget;
        */
        retval->menus_vis = settingsMenu[2].widget;
        retval->panel_vis = settingsMenu[3].widget;
        retval->toggleakt = documentMenu[4].widget;
        retval->toggleall = documentMenu[5].widget;
        retval->toggleeven = documentMenu[6].widget;
        retval->toggleodd = documentMenu[7].widget;
        retval->clearall = documentMenu[8].widget;

        gtk_widget_set_sensitive (GTK_WIDGET(retval->previtem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->nextitem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->printitem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->printmarked), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->reloaditem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->save), FALSE);
        /*        gtk_widget_set_sensitive (GTK_WIDGET(retval->saveas), FALSE); */
        gtk_widget_set_sensitive (GTK_WIDGET(retval->toggleakt), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->toggleall), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->toggleeven), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->toggleodd), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->clearall), FALSE);

        if(!recent_files_list)
                recent_update();
        recent_update_menus(retval, recent_files_list);
}

void create_sidebar(ggv_window *retval)
{
        GtkWidget *pic;
        GtkWidget *npbhbox;
        GtkWidget *zbhbox;

        retval->coordinates = gtk_label_new (" ");
        gtk_widget_show (retval->coordinates);

        retval->scrollpane = gtk_scrollpane_new(GTK_ADJUSTMENT (retval->hadj), GTK_ADJUSTMENT (retval->vadj), 1.0);
        gtk_signal_connect (GTK_OBJECT(retval->scrollpane), "middle_clicked", GTK_SIGNAL_FUNC(scrollpane_middle_click_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->scrollpane), "right_clicked", GTK_SIGNAL_FUNC(scrollpane_right_click_callback), retval);
        gtk_widget_show (retval->scrollpane);
        
        npbhbox = gtk_hbox_new (TRUE, 0);
        retval->nextbutton = gtk_button_new ();
        retval->prevbutton = gtk_button_new ();

        pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_BACK);
        gtk_widget_show (pic);
        gtk_container_add (GTK_CONTAINER (retval->prevbutton), pic);
        pic = gnome_stock_pixmap_widget_new(retval->nextbutton, GNOME_STOCK_MENU_FORWARD);
        gtk_widget_show (pic);
        gtk_container_add (GTK_CONTAINER (retval->nextbutton), pic);
        gtk_signal_connect (GTK_OBJECT(retval->nextbutton), "clicked", GTK_SIGNAL_FUNC(next_page_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->prevbutton), "clicked", GTK_SIGNAL_FUNC(previous_page_callback), retval);
        gtk_container_add (GTK_CONTAINER (npbhbox), retval->prevbutton);
        gtk_container_add (GTK_CONTAINER (npbhbox), retval->nextbutton);

        gtk_widget_set_sensitive (GTK_WIDGET(retval->prevbutton),FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->nextbutton),FALSE);

        gtk_widget_show (retval->prevbutton);
        gtk_widget_show (retval->nextbutton);
        gtk_widget_show (npbhbox);
        
        zbhbox = gtk_hbox_new (TRUE, 0);
        retval->zoominbutton = gtk_button_new_with_label ( _("+") );
        retval->zoomoutbutton = gtk_button_new_with_label ( _("-") );
        gtk_widget_set_sensitive (GTK_WIDGET(retval->zoomoutbutton), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->zoominbutton), FALSE);
	/*  We need stock buttons for these...
         * pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_BACK);
         * gtk_widget_show (pic);
         * gtk_container_add (GTK_CONTAINER (retval->prevbutton), pic);
         * pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_FORWARD);
         * gtk_widget_show (pic);
         * gtk_container_add (GTK_CONTAINER (retval->nextbutton), pic);
	 */
        gtk_signal_connect (GTK_OBJECT(retval->zoominbutton), "clicked", GTK_SIGNAL_FUNC(zoomin_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->zoomoutbutton), "clicked", GTK_SIGNAL_FUNC(zoomout_callback), retval);
        gtk_container_add (GTK_CONTAINER (zbhbox), retval->zoomoutbutton);
        gtk_container_add (GTK_CONTAINER (zbhbox), retval->zoominbutton);
        gtk_widget_show (retval->zoomoutbutton);
        gtk_widget_show (retval->zoominbutton);
        gtk_widget_show (zbhbox);

	retval->pagelistscroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(retval->pagelistscroll),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_placement(GTK_SCROLLED_WINDOW(retval->pagelistscroll),
					GTK_CORNER_BOTTOM_RIGHT);

        retval->pagelist = gtk_clist_new(2);
	gtk_widget_set_usize (retval->pagelist, 80, 100);

	gtk_clist_set_column_width (GTK_CLIST (retval->pagelist), 0, 18);
	gtk_clist_set_column_width (GTK_CLIST (retval->pagelist), 1, 20);
	gtk_clist_set_selection_mode (GTK_CLIST (retval->pagelist), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_justification (GTK_CLIST (retval->pagelist), 0, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_justification (GTK_CLIST (retval->pagelist), 1, GTK_JUSTIFY_RIGHT);

        gtk_clist_set_shadow_type (GTK_CLIST (retval->pagelist),GTK_SHADOW_ETCHED_IN);

        gtk_signal_connect(GTK_OBJECT(retval->pagelist), "select_row", GTK_SIGNAL_FUNC(select_page_callback),retval);

        gtk_signal_connect(GTK_OBJECT(retval->pagelist), "button_press_event", GTK_SIGNAL_FUNC(select_page_button_press_callback),retval);

	gtk_container_add (GTK_CONTAINER (retval->pagelistscroll), retval->pagelist);
	gtk_widget_show (retval->pagelist);
	gtk_widget_show (retval->pagelistscroll);

        gtk_box_pack_start (GTK_BOX (retval->sidebar), retval->coordinates, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (retval->sidebar), retval->scrollpane, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (retval->sidebar), zbhbox, FALSE, FALSE, 2 );
        gtk_box_pack_start (GTK_BOX (retval->sidebar), npbhbox, FALSE, FALSE, 2 );
        gtk_container_add (GTK_CONTAINER (retval->sidebar), retval->pagelistscroll);
        gtk_widget_show (retval->sidebar); 
}

/* main functions */
ggv_window *create_ggv_window()
{
        /* DnD stuff */
	static GtkTargetEntry drop_types [] = {
		{ "text/uri-list", 0, TARGET_URI_LIST}
	};
	static gint n_drop_types = sizeof (drop_types) / sizeof(drop_types[0]);

        gboolean bxneg, byneg;
        gint i;
        GtkWidget *hbox;
        GtkWidget *vbox;
        GtkWidget *gsframe;

        ggv_window *retval = g_new0 (ggv_window, 1);

        if (nomenubar)
                retval->show_menus = FALSE;
        else
                retval->show_menus = gs_menubar;

        if (spartan)
                retval->show_panel = FALSE;
        else
                retval->show_panel = gs_panel;

        retval->pane_auto_jump = gs_auto_jump;
        retval->loaded = FALSE;
        retval->pan = FALSE;

        /* create the main window. */
        retval->main_window = gnome_app_new("ggv", "ggv:");

        if (geometry){
                int xpos, ypos;
         
                ggv_parse_geometry (geometry, &xpos, &ypos, &bxneg, &byneg, &width, &height);
                g_print("Geometry %d %d %d %d\n", xpos, ypos, width, height);
                
                if (xpos != -1 && ypos != -1) {
                        if (bxneg) xpos -= width;
                        /* FIXME we need to know the vertical size of
                           the current window bar */
                        if (byneg) ypos -= height + 18;
                        gtk_widget_set_uposition (GTK_WIDGET (retval->main_window), xpos, ypos);
                }
                if (width == -1 || height == -1) {
                        width = DEFAULT_WINDOW_WIDTH;
                        height = DEFAULT_WINDOW_HEIGHT;
                }
         /* Only the first window gets --geometry treatment for now */
                geometry = NULL;
        }
        if (fullscreen) {
                gtk_widget_set_uposition (GTK_WIDGET (retval->main_window), 0, 0);
                /* FIXME we need to know the vertical size of
                   the current window bar */
                height = gdk_screen_height () - 18;
                width = gdk_screen_width ();

        }
        gtk_window_set_default_size(GTK_WINDOW(retval->main_window), 
                                    width, height);
        gtk_window_set_policy(GTK_WINDOW(retval->main_window), TRUE, TRUE, FALSE);
        retval->cd = NULL;
        /* make it a drop zone for files */
	gtk_drag_dest_set (GTK_WIDGET (retval->main_window),
	                   GTK_DEST_DEFAULT_MOTION |
			   GTK_DEST_DEFAULT_HIGHLIGHT |
			   GTK_DEST_DEFAULT_DROP,
			   drop_types, n_drop_types,
			   GDK_ACTION_COPY);
        gtk_signal_connect (GTK_OBJECT (retval->main_window), "drag_data_received",
			    GTK_SIGNAL_FUNC(drop_callback), retval);
        /* take care of its deletion */
	gtk_signal_connect (GTK_OBJECT (retval->main_window), "delete_event",
                            GTK_SIGNAL_FUNC (delete_callback), retval);

        /* Set up the callback values for the orientation */
        for (i = 0; i < 16; i++)
                retval->menudata[i].ggv = retval;

        retval->menudata[0].data = GINT_TO_POINTER(GTK_GS_ORIENTATION_DEFAULT);
        retval->menudata[1].data = GINT_TO_POINTER(GTK_GS_ORIENTATION_PORTRAIT);
        retval->menudata[2].data = GINT_TO_POINTER(GTK_GS_ORIENTATION_LANDSCAPE);
        retval->menudata[3].data = GINT_TO_POINTER(GTK_GS_ORIENTATION_SEASCAPE);
        retval->menudata[4].data = GINT_TO_POINTER(GTK_GS_ORIENTATION_UPSIDEDOWN);

	/* We are lying a little here - the magstep does not give
	 * these ratios exactly but who's going to know ;)
	 */

        retval->menudata[5].data = GINT_TO_POINTER(-13);
        retval->menudata[6].data = GINT_TO_POINTER(-12);
        retval->menudata[7].data = GINT_TO_POINTER(-8); 
        retval->menudata[8].data = GINT_TO_POINTER(-4); 
        retval->menudata[9].data = GINT_TO_POINTER(-2); 
        retval->menudata[10].data = GINT_TO_POINTER(0);
        retval->menudata[11].data = GINT_TO_POINTER(2);
        retval->menudata[12].data = GINT_TO_POINTER(4);
        retval->menudata[13].data = GINT_TO_POINTER(8);
        retval->menudata[14].data = GINT_TO_POINTER(12);
        retval->menudata[15].data = GINT_TO_POINTER(13);

        /* Statusbar */
        retval->appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
        gtk_widget_show(retval->appbar);
        gnome_app_set_statusbar(GNOME_APP(retval->main_window), retval->appbar);

        /* Menus */
        if (retval->show_menus)
                create_menus (retval, retval->menudata);
        create_popup_menus(retval);

        /* We set up the layout */
        hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
        vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

        retval->hadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
        retval->vadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);

        /* we set up the post script display */
        gsframe = gtk_frame_new (NULL);
        gtk_frame_set_shadow_type (GTK_FRAME (gsframe), GTK_SHADOW_IN);
        retval->gs = gtk_gs_new (GTK_ADJUSTMENT (retval->hadj),
                                 GTK_ADJUSTMENT (retval->vadj));
        gtk_widget_set_events (retval->gs, 
                               GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK);
        set_gs_prefs (retval);
        gtk_container_add (GTK_CONTAINER (gsframe), retval->gs);
        gtk_widget_show (retval->gs);
        gtk_widget_show (gsframe);

        /* we prepare for the crop data */
        retval->cd = g_malloc (sizeof (crop_data));

        /* We set up the sidebar */
        retval->zoom_magstep = 0; /* 1 : 1 */

        retval->sidebar = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

        if(retval->show_panel)
                create_sidebar(retval);

        /* We layout the page */
        gtk_box_pack_start (GTK_BOX (hbox), retval->sidebar, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), gsframe, TRUE, TRUE, 0);
        gtk_widget_show(hbox);
        gtk_widget_show(vbox);

	gnome_app_set_contents(GNOME_APP(retval->main_window), hbox);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "button_press_event",
                            GTK_SIGNAL_FUNC (button_press_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "button_release_event",
                            GTK_SIGNAL_FUNC (button_release_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "motion_notify_event",
                            GTK_SIGNAL_FUNC (motion_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "interpreter_message",
                            GTK_SIGNAL_FUNC (interpreter_message_callback),
                            retval);
        gtk_signal_connect (GTK_OBJECT(retval->main_window), "key_press_event",
                            GTK_SIGNAL_FUNC(key_pressed_event_callback), retval);

        retval->file_sel = NULL;
        retval->save_file_sel = NULL;

        if (retval->show_menus)
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(retval->menus_vis),
                                        retval->show_menus);
        if(retval->show_panel)
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(retval->panel_vis),
                                                retval->show_panel);

        return retval;
}

void flash_message(ggv_window *ggv, gchar *flash)
{
        if(flash) {
                gnome_app_flash(GNOME_APP(ggv->main_window), flash);
                g_free(flash);
        }
}

void error_message(ggv_window *ggv, gchar *errormsg)
{
        if(errormsg) {
                gnome_app_error(GNOME_APP(ggv->main_window), errormsg);
                g_free(errormsg);
        }
}

void open_window(gchar *filename)
{
        if(filename == NULL) {
                ggv_window *ggv = create_ggv_window ();
                active_ggv++;
                window_list = g_list_append(window_list, ggv);
                gtk_widget_show (ggv->main_window);
        }
        else {
                ggv_window *ggv = create_ggv_window ();
                gtk_widget_show (ggv->main_window);

                if (load_gs(ggv, filename)) {                                
                        ggv->loaded = TRUE;
                        active_ggv ++;
                        window_list = g_list_append (window_list, ggv);
                }
                else {
                        gtk_widget_destroy(ggv->main_window);
                        g_free(ggv);
                }
        }
}

void close_window(ggv_window *ggv)
{
        /* do we want to save changes??? */
	window_list = g_list_remove (window_list, ggv);
	gtk_widget_destroy (GTK_WIDGET (ggv->main_window));
        g_free(ggv);

        active_ggv--;

	if (active_ggv == 0)
		gtk_main_quit ();
}

void open_prefs_dialog(ggv_prefs *pd)
{
        GtkWidget *table, *label, *menu;
        gint i;

        if(pd->pbox)
                return;

        pd->pbox = GNOME_PROPERTY_BOX(gnome_property_box_new());
	gtk_signal_connect(GTK_OBJECT(pd->pbox), "destroy",
                           GTK_SIGNAL_FUNC(prefs_destroy_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->pbox), "apply",
                           GTK_SIGNAL_FUNC(prefs_apply_callback), pd);

        /* Document page */
        table = gtk_table_new(2, 3, FALSE);

        /* media choice menu */
        label = gtk_label_new(_("Fallback media type"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->media = gtk_option_menu_new();
        gtk_widget_show(pd->media);
        menu = gtk_menu_new();
        for(i = 0; papersizes[i].name != NULL; i++) {
                pd->media_choice[i] = gtk_menu_item_new_with_label(papersizes[i].name);
                gtk_widget_show(pd->media_choice[i]);
                gtk_menu_append(GTK_MENU(menu), pd->media_choice[i]);
        }
        gtk_option_menu_set_menu(GTK_OPTION_MENU(pd->media), menu);
        i = gs_def_media;
        gtk_option_menu_set_history(GTK_OPTION_MENU(pd->media), i);
        for(i = 0; papersizes[i].name != NULL; i++)
                gtk_signal_connect(GTK_OBJECT(pd->media_choice[i]),
                                   "activate", GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_table_attach(GTK_TABLE(table), pd->media,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        /* override document media */
        pd->override_media = gtk_check_button_new_with_label(_("Override document media"));
        gtk_table_attach(GTK_TABLE(table), pd->override_media,
                         0, 2, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->override_media), gs_override_media);
        gtk_signal_connect(GTK_OBJECT(pd->override_media), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->override_media);
        /* antialiasing */
        pd->aa = gtk_check_button_new_with_label(_("Antialiasing"));
        gtk_table_attach(GTK_TABLE(table), pd->aa,
                         0, 2, 2, 3,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->aa), gs_antialiased);
        gtk_signal_connect(GTK_OBJECT(pd->aa), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->aa);

        label = gtk_label_new(_("Document"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* Layout page */
        table = gtk_table_new(3, 1, FALSE);

        /* show toolbar */
        pd->tbar = gtk_check_button_new_with_label(_("Show side panel by default"));
        gtk_table_attach(GTK_TABLE(table), pd->tbar,
                         0, 1, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->tbar), gs_panel);
        gtk_signal_connect(GTK_OBJECT(pd->tbar), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->tbar);
        /* show menubar */
        pd->mbar = gtk_check_button_new_with_label(_("Show menubar by default"));
        gtk_table_attach(GTK_TABLE(table), pd->mbar,
                         0, 1, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->mbar), gs_menubar);
        gtk_signal_connect(GTK_OBJECT(pd->mbar), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->mbar);
        /* auto jump to beginning of page */
        pd->auto_jump= gtk_check_button_new_with_label(_("Jump to beginning of page when using keyboard"));
        gtk_table_attach(GTK_TABLE(table), pd->auto_jump,
                         0, 1, 2, 3,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->auto_jump), gs_auto_jump);
        gtk_signal_connect(GTK_OBJECT(pd->auto_jump), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->auto_jump);

        label = gtk_label_new(_("Layout"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* GhostScript page */
        table = gtk_table_new(2, 3, FALSE);

        /* interpreter */
        label = gtk_label_new(_("Interpreter"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->gs = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->gs), gs_cmd);
	gtk_entry_set_position(GTK_ENTRY(pd->gs), 0);
        gtk_table_attach(GTK_TABLE(table), pd->gs,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->gs), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->gs), "changed",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->gs);
        /* scan PDF command */
        label = gtk_label_new(_("Scan PDF"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 1, 2,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->scan_pdf = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->scan_pdf), gs_scan_pdf_cmd);
	gtk_entry_set_position(GTK_ENTRY(pd->scan_pdf), 0);
        gtk_table_attach(GTK_TABLE(table), pd->scan_pdf,
                         1, 2, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->scan_pdf), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->scan_pdf), "changed",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->scan_pdf);
        /* uncompress command */
        label = gtk_label_new(_("Uncompress"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 2, 3,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->uncompress = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->uncompress), gs_uncompress_cmd);
	gtk_entry_set_position(GTK_ENTRY(pd->uncompress), 0);
        gtk_table_attach(GTK_TABLE(table), pd->uncompress,
                         1, 2, 2, 3,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->uncompress), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->uncompress), "changed",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->uncompress);

        label = gtk_label_new(_("Ghostscript"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* Printing page */
        table = gtk_table_new(1, 2, FALSE);

        /* print command */
        label = gtk_label_new(_("Print command"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->print = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->print), gs_print_cmd);
	gtk_entry_set_position(GTK_ENTRY(pd->print), 0);
        gtk_table_attach(GTK_TABLE(table), pd->print,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->print), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->print), "changed",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->print);

        label = gtk_label_new(_("Printing"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

	gtk_window_position (GTK_WINDOW(pd->pbox), GTK_WIN_POS_MOUSE);
	
	gtk_widget_show(GTK_WIDGET(pd->pbox));
}

void set_gs_prefs(ggv_window *ggv)
{
        GtkGS *gs = GTK_GS(ggv->gs);
        gboolean redisplay = FALSE;

        if(gs->antialiased != gs_antialiased && ggv->loaded)
                redisplay = TRUE;

        set_prefs(ggv);
        if(redisplay)
                reload_callback(NULL, ggv);
}

void apply_gs_prefs(GList *windows)
{
        GtkGS *gs;

        while(windows) {
                set_gs_prefs((ggv_window *)windows->data);
                windows = windows->next;
        }
}

void parse_an_arg (poptContext state,
           enum poptCallbackReason reason,
           const struct poptOption *opt,
           const char *arg, void *data)
{
     struct terminal_config *cfg = data;
 
     int key = opt->val;
 
     switch (key){
     case GEOMETRY_KEY:
             geometry = (char *)arg;
             break;
     case SPARTAN_KEY:
             spartan = TRUE;
             break;
     case NO_MENU_BAR_KEY:
             nomenubar = TRUE;
             break;
     case FULL_SCREEN_KEY:
             fullscreen = TRUE;
             break;
     }
}

struct poptOption options_callback [] = {
        { NULL, '\0', POPT_ARG_CALLBACK, parse_an_arg, 0},
        
        { "geometry", '\0', POPT_ARG_STRING, NULL, GEOMETRY_KEY,
          N_("Specifies the geometry for the main window"), N_("GEOMETRY")},
      
	{ "spartan", '\0', POPT_ARG_NONE, NULL, SPARTAN_KEY,
	  N_("Do not use sidebar"), N_("SPARTAN") },

	{ "nomenubar", '\0', POPT_ARG_NONE, NULL, NO_MENU_BAR_KEY,
	  N_("Do not use a menu bar"), N_("NOMENUBAR") },

	{ "fullscreen", '\0', POPT_ARG_NONE, NULL, FULL_SCREEN_KEY,
	  N_("Use the whole screen for the main window"), N_("FULLSCREEN") },
        
        { NULL, '\0', 0, NULL, 0}
};


gboolean ggv_file_readable(char *FileName)
{
        struct stat stat_rec;

        if (stat(FileName,&stat_rec) == 0) 
                return TRUE;

        return FALSE;
}

/* and we finally start the main prog. */
int main(int argc, char *argv[])
{
        GList *list;
	poptContext ctx;
	char **startups;
	int i;
        char *FileName, *AlterFileName;
        gboolean Done;

        active_ggv=0;

        bindtextdomain (PACKAGE, GNOMELOCALEDIR);
        textdomain (PACKAGE);

	gnome_init_with_popt_table("gnome-ghostview", 
                                   VERSION, argc, argv, options_callback, 
                                   0, &ctx);

        window_list = NULL;

        load_prefs("/ggv/");

	create_pixmap (foot_xpm, &page_marker_pixmap, &page_marker_mask);
	create_pixmap (foot_white_xpm, &page_marker_white_pixmap, &page_marker_white_mask);

        startups = poptGetArgs(ctx);
        if(startups) {
                for(i = 0; startups[i]; i++) {
                        FileName = startups[i];
                        AlterFileName = NULL;
                        Done = FALSE;
                        if (!ggv_file_readable(FileName)) {
                                AlterFileName = g_malloc(strlen(startups[i])+8);
                                sprintf(AlterFileName, "%s.ps", FileName);
                        }
                        else
                                Done = TRUE;
                        if (!Done && !ggv_file_readable(AlterFileName)) 
                                sprintf(AlterFileName, "%s.ps.gz", FileName);
                        else
                                Done = TRUE;

                        if (!Done && !ggv_file_readable(AlterFileName)) 
                                sprintf(AlterFileName, "%s.pdf", FileName);
                        else
                                Done = TRUE;


                        if (AlterFileName) FileName = AlterFileName;
                        if (Done) {
                                open_window(FileName);
                                g_print("opening %s [%s]\n", FileName,AlterFileName);
                        }
                        if (AlterFileName) g_free(AlterFileName);
                }
        }

	poptFreeContext(ctx);

        if (window_list == NULL)
                open_window(NULL);

	gtk_main ();

        save_prefs("/ggv/");

        return 0;
}

