/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/* Copyright (C) 1998, 1999 Redhat Software Inc. 
 * Authors: Jonathan Blandford <jrb@redhat.com>
 *          Jaka Mocnik <jaka.mocnik@kiss.uni-lj.si>
 */
#include <config.h>
#include <gdk/gdk.h>
#include <gnome.h>
#include <math.h>
#include <sys/stat.h>
#include <libgnome/gnome-history.h>

#include <gdk/gdkkeysyms.h>
#include "gtkscrollpane.h"
#include "gtkgs.h"
#include "crop.h"
#include "prefs.h"

enum {
		TARGET_URI_LIST,
};

typedef struct _window_with_data window_with_data;
typedef struct _ggv_window ggv_window;
typedef struct _ggv_prefs ggv_prefs;

static GdkCursor *pan_cursor = NULL;

struct _window_with_data {
        ggv_window *ggv;
        gint data;
};

struct _ggv_window {
        GtkWidget *main_window;
	GtkWidget *pagelistscroll;
        GtkWidget *pagelist;
        GtkWidget *coordinates;
        GtkWidget *scrollpane;
        GtkWidget *popup_menu;
        GtkWidget *sidebar;
        GtkWidget *file_sel;
        GtkWidget *appbar;
        GtkObject *hadj, *vadj;
        GtkWidget *gs;
        gboolean show_menus, show_panel;
        gboolean loaded;
        gboolean pan;
        gdouble prev_x, prev_y;
        gchar *startup_file;
        crop_data *cd;
        gint zoom_magstep;
        window_with_data menudata[16];

        /* store some pointers to widgets from GnomeUIInfo trees here
           as we need them for setting sensitivity and stuff like that */
        GtkWidget *nextbutton, *nextitem, *nextpu;  /* next button, menu item and popup menu item */
        GtkWidget *prevbutton, *previtem, *prevpu;  /* prev ... */
        GtkWidget *printitem, *reloaditem;          /* menu items */
        GtkWidget *zoominbutton;                    /* zoom buttons */
        GtkWidget *zoomoutbutton;
        GtkWidget *panel_vis, *menus_vis;           /* menu/side panel visibility check items in menus */
        GtkWidget *panel_vis_pu, *menus_vis_pu;     /* and in popup menu */
        gboolean bRecentHasBeenInstantiated;        /* Recent menu has been intantiated? so we can
                                                       remove it before we add new entries */
};

struct _ggv_prefs
{
        GnomePropertyBox *pbox;
        GtkWidget *gs, *scan_pdf, *uncompress, *print; /* entries */
        GtkWidget *media;                              /* option menus */
        GtkWidget *aa, *override_media;                /* checkbuttons */
        GtkWidget *tbar, *mbar;                        /* checkbuttons */
        GtkWidget *media_choice[PAPER_SIZE_COUNT];     /* menu items */
};

GList *window_list;
static int active_ggv;
static ggv_prefs prefs_dialog = { NULL };

/* prototypes */
static gboolean load_gs(ggv_window *ggv, char *fname);
static ggv_window *create_ggv_window(void);
static void create_sidebar(ggv_window *retval);
static void create_menus(ggv_window *retval, window_with_data *temp);
static void create_popup_menus(ggv_window *ggv);
static void open_window(gchar *filename);
static void close_window(ggv_window *ggv);
static void set_gs_prefs(ggv_window *ggv);
static void apply_gs_prefs(GList *windows);
static void open_prefs_dialog(ggv_prefs *pd);
static void goto_page(ggv_window *ggv, int page);
static void set_page_sensitivities(ggv_window *ggv, int page);
static void flash_message(ggv_window *ggv, gchar *flash);
static void error_message(ggv_window *ggv, gchar *errormsg);

/* callback prototypes */
static void about_callback(GtkWidget *widget, gpointer data);
static void new_callback(GtkWidget *widget, gpointer data);
static void button_press_callback(GtkWidget *widget, GdkEventButton *event, gpointer data);
static void button_release_callback(GtkWidget *widget, GdkEventButton *event, gpointer data);
static void close_callback (GtkWidget *widget, ggv_window *ggv);
static gint delete_callback (GtkWidget *widget, GdkEventAny *event, gpointer data);
static gint exit_callback(GtkWidget *widget, gpointer data);
static void file_open_cancel_callback(GtkWidget *widget, gpointer data);
static void file_open_delete_callback(GtkWidget *widget, GdkEventAny *e, gpointer data);
static void file_open_ok_callback(GtkWidget *widget, gpointer data);
static void show_menubar_callback(GtkWidget *widget, gpointer data);
static void hide_panel_callback(GtkWidget *widget, gpointer data);
static void motion_callback(GtkWidget *widget, GdkEventMotion *event, gpointer data);
static void key_pressed_event_callback(GtkWidget *widget, GdkEventKey *event, gpointer data);
static void next_page_callback(GtkWidget *widget, gpointer data);
static void open_callback(GtkWidget *widget, gpointer data);
static void orientation_callback(GtkWidget *widget, gpointer data);
static void preferences_callback(GtkWidget *widget, gpointer data);
static void previous_page_callback(GtkWidget *widget, gpointer data);
static void print_callback(GtkWidget *widget, gpointer data);
static void recenter_page_callback(GtkWidget *widget, gpointer data);
static void reload_callback(GtkWidget *widget, gpointer data);
static void scrollpane_middle_click_callback(GtkWidget *widget, gpointer data);
static void scrollpane_right_click_callback(GtkWidget *widget, gpointer data);
static void save_callback(GtkWidget *widget, gpointer data);
static void save_as_callback(GtkWidget *widget, gpointer data);
static void select_page_callback(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer data);
static void zoom_callback(GtkWidget *widget, gpointer data);
static void drop_callback(GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *selection_data, guint info, guint time, gpointer data);
static double compute_zoom(gint zoom_spec);
static void prefs_destroy_callback(GtkWidget *w, ggv_prefs *p);
static void prefs_apply_callback(GtkWidget *w, gint page, ggv_prefs *p);
static void prefs_changed_callback(GtkWidget *w, ggv_prefs *p);
static void zoomin_callback(GtkWidget *widget, gpointer *data);
static void zoomout_callback(GtkWidget *widget, gpointer *data);
static void recent_add(char *filename);
static void recent_update_menus (ggv_window *ggv, GList *recent_files);
static void recent_cb(GtkWidget *w, gpointer *data);

static void
recent_cb(GtkWidget *w, gpointer *data)
{
        char *ErrorMessage;
        window_with_data *wwd = (window_with_data *) data;
        struct stat StatRecord;

        /*
        g_print("TO open file %s\n",(char*)wwd->data);
        */
        if (stat((char*)wwd->data,&StatRecord) == 0) {
                if (S_ISREG(StatRecord.st_mode)) {
                        wwd->ggv->loaded = load_gs(wwd->ggv, (char*)wwd->data);
                        return;
                }
                ErrorMessage = g_strdup_printf(_("%s is not a file."), (char*)wwd->data);

        }
        else {
                ErrorMessage = g_strdup_printf(_("%s does not exist."), (char*)wwd->data);
        }

        error_message(wwd->ggv,ErrorMessage);

        /*        g_free((char*)wwd->data); */
}


static void
recent_update_menus (ggv_window *ggv, GList *recent_files)
{
	GnomeUIInfo *menu;
        /*	gE_data *data; */
	gchar *path;
	int i;
        window_with_data *wwd;

        /*
        g_print("Num recent %d\n", gs_num_recent);
        */
	if (gs_num_recent && ggv->bRecentHasBeenInstantiated) {
                gnome_app_remove_menu_range (GNOME_APP(ggv->main_window), 
                                             _("_File/"), 4, gs_num_recent + 1);
                g_print("Removing...\n");
        }

	if (recent_files == NULL)
		return;


	/* insert a separator at the beginning */
	
	menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
	path = g_new (gchar, strlen (_("_File")) + strlen ("<Separator>") + 3 );
	sprintf (path, "%s/%s", _("_File"), "<Separator>");
	menu->type = GNOME_APP_UI_SEPARATOR;

	(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
	gnome_app_insert_menus (GNOME_APP(ggv->main_window), path, menu);


	for (i = g_list_length (recent_files) - 1; i >= 0;  i--)
	{
                wwd = g_malloc0(sizeof(window_with_data));

		menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
	
		/* data = g_malloc0 (sizeof (gE_data));
                   data->temp1 = g_strdup (g_list_nth_data (recent_files, i)); */
		/*data->window = window;*/
		menu->label = g_new (gchar, 
			strlen (g_list_nth_data (recent_files, i)) + 5);
		sprintf (menu->label, "_%i. %s", i+1, (gchar*)g_list_nth_data (recent_files, i));
		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
                
		menu->moreinfo = (gpointer) recent_cb;

                wwd->data = (int)g_strdup(g_list_nth_data (recent_files, i));
                wwd->ggv = ggv;

		menu->user_data = wwd;
		menu->unused_data = NULL;
		menu->pixmap_type = 0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;

		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
	
		gnome_app_insert_menus (GNOME_APP(ggv->main_window), path, menu);
		g_free (g_list_nth_data (recent_files, i));
	}
	gs_num_recent = g_list_length (recent_files);

        ggv->bRecentHasBeenInstantiated = TRUE;



	g_list_free (recent_files);
}


void recent_update (ggv_window *ggv)
{
	GList *filelist = NULL;
	GList *dirlist = NULL;
	
	GList *gnome_recent_list;
	GnomeHistoryEntry histentry;
	char *filename;
	int i, j;
	gboolean duplicate = FALSE;
	
	filelist = NULL;
	gnome_recent_list = gnome_history_get_recently_used ();
	

	if (g_list_length (gnome_recent_list) > 0)                {
                for (i = g_list_length (gnome_recent_list) - 1; i >= 0; i--) {
                        histentry = g_list_nth_data (gnome_recent_list, i);
			if (strcmp ("ggv", histentry->creator) == 0) {
				/* This is to make sure you don't have more than one
				   file of the same name in the recent list
				*/
				if (g_list_length (filelist) > 0)
					for (j = g_list_length (filelist) - 1; j >= 0; j--)
						if (strcmp (histentry->filename, g_list_nth_data (filelist, j)) == 0)
							filelist = g_list_remove (filelist, g_list_nth_data (filelist, j));
                                
				filename = g_malloc0 (strlen (histentry->filename) + 1);
				strcpy (filename, histentry->filename);
				filelist = g_list_append (filelist, filename);
                                
                                /* For recent-directories, not yet fully implemented...
                                   end_path = strrchr (histentry->filename, '/');
                                   if (end_path)
                                   {
                                   for (i = 0; i < strlen (histentry->filename); i++)
                                   if ( (histentry->filename + i) == end_path)
                                   break;
                                   directory = g_malloc0 (i + 2);
                                   strcat (directory, histentry->filename, i);
                                   }
                                */
				if (g_list_length (filelist) == MAX_RECENT)
					break;
			}
		}
	}
	gnome_history_free_recently_used_list (gnome_recent_list);

	gs_num_recent = g_list_length (filelist);

	
	recent_update_menus (ggv, filelist);
}


static void recent_add(char *filename)
{
        gnome_history_recently_used (filename, "text/plain", "ggv", "");
}

/* Useful functions */
static gboolean
load_gs(ggv_window *ggv, char *fname)
{
	char *title;
	guint i,n;
	gchar *texts[3], *flash;
        gint page;

        if(ggv->loaded && strcmp(fname, GTK_GS(ggv->gs)->gs_filename) == 0)
                page = GTK_GS(ggv->gs)->current_page;
        else
                page = 0;
        flash = g_strdup_printf(_("Opening %s..."), fname);
        flash_message(ggv, flash);
	if (!gtk_gs_load (GTK_GS (ggv->gs), fname)) {
		GtkWidget *w;
                gtk_clist_clear (GTK_CLIST (ggv->pagelist));
                gtk_window_set_title (GTK_WINDOW(ggv->main_window), "ggv");
                gtk_widget_set_sensitive (ggv->zoomoutbutton, FALSE);
                gtk_widget_set_sensitive (ggv->zoominbutton, FALSE);
                gtk_widget_set_sensitive (ggv->printitem, FALSE);
                flash = g_strdup_printf(_("Unable to load %s."), fname);
                error_message(ggv, flash);
                return FALSE;
        }
        fflush(NULL);
        gtk_gs_set_pagemedia(GTK_GS (ggv->gs), -1, 0);
        fflush(NULL);

	title = gtk_gs_document_title (GTK_GS (ggv->gs));
	if (title) {
		title = g_strjoin("", "ggv: ", title, NULL);
		gtk_window_set_title(GTK_WINDOW(ggv->main_window), title);
		g_free(title);
	}
        else 
                gtk_window_set_title (GTK_WINDOW(ggv->main_window), "ggv");

        /* FIXME:
           Resize window if necessary, we 
           must verify whether automatic resizing is allowed (add option to
           menu on whether to auto resize 
           finally, at most it should be as large as the screen
           
           We also have to zoom the document to the specific zoom factor
           according to the menu. 

           gtk_gs_set_zoom (GTK_GS (ggv->gs), ggv->zoom_value); 

           So far, the user can't make the window smaller. I don't 
           know how to fix it.

        */

	/*
        gtk_widget_set_usize (ggv->gs, 
                              GTK_GS(ggv->gs)->width,GTK_GS(ggv->gs)->height);
	*/

        if(ggv->pagelist && GTK_GS(ggv->gs)->structured_doc) {
                gtk_clist_freeze(GTK_CLIST(ggv->pagelist));
                gtk_clist_clear(GTK_CLIST(ggv->pagelist));
                n = gtk_gs_document_numpages(GTK_GS (ggv->gs));

                if (n > 0) {
                        for (i=1; i<=n ; i++) {
                                texts[0] = "";
                                texts[1] = g_strdup(gtk_gs_document_page_label(GTK_GS (ggv->gs), i));
                                gtk_clist_append(GTK_CLIST(ggv->pagelist), texts);
                        }
                }
                gtk_clist_thaw(GTK_CLIST(ggv->pagelist));
		goto_page( ggv, page );
        }
        else
		goto_page( ggv, page );
        gtk_widget_set_sensitive (ggv->zoomoutbutton, TRUE);
        gtk_widget_set_sensitive (ggv->zoominbutton, TRUE);
        gtk_widget_set_sensitive (ggv->printitem, TRUE);
        gtk_widget_set_sensitive (ggv->reloaditem, TRUE);

        flash = g_strdup_printf(_("Successfully loaded %s."), fname);
        flash_message(ggv, flash);

        return TRUE;
}


  /****************/
 /* Callbacks... */
/****************/
static void about_callback(GtkWidget *widget, gpointer data) 
{
	GtkWidget *about = NULL;

	gchar *authors[] = {
		"Szekeres Istvan <szekeres@cyberspace.mht.bme.hu>",
                "Jonathan Blandford <jrb@redhat.com>",
                "Jaka Mocnik <jaka.mocnik@kiss.uni-lj.si>",
		NULL
	};

	about = gnome_about_new(_("GNOME Ghostview"), VERSION,
				"Copyright (C) 1998, 1999 the Free Software Foundation",
				(const gchar **) authors,
				_("Postscript(TM) document viewer.\nBased on Tim Theisen's excellent Ghostview application."),
				NULL);
        gtk_widget_show(about);
}

static void new_callback(GtkWidget *widget, gpointer data)
{
        open_window(NULL);
}

static void button_press_callback(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;
        gint button = event->button;
        gint x = 0;
        gint y = 0;

        if (event->window == GTK_GS (ggv->gs)->pstarget) {
                gdk_window_get_position (event->window, &x, &y);
        }
        x += event->x;
        y += event->y;
        switch(button) {
        case 1:
                if(!ggv->pan) {
                        gint wx = 0, wy = 0;

                        gdk_window_get_pointer(widget->window, &wx, &wy, NULL);

                        ggv->pan = TRUE;
                        if(pan_cursor == NULL)
                                pan_cursor = gdk_cursor_new(GDK_FLEUR);

                        gtk_grab_add(widget);
                        gdk_pointer_grab(widget->window, FALSE,
                                         GDK_POINTER_MOTION_MASK |
                                         GDK_BUTTON_RELEASE_MASK, NULL,
                                         pan_cursor, GDK_CURRENT_TIME);
                        ggv->prev_x = wx;
                        ggv->prev_y = wy;
                }
                break;
        case 2:
                /* uncomment this to mark a section to crop.
                 * it has been removed until it clears correctly */
                   
                /*                clear_crop (ggv->cd, ggv->gs->window);
                 *                crop_button_clicked (ggv->cd, x, y);
                 *                draw_crop (ggv->cd, ggv->gs->window);
                 */
                break;
        case 3:
                gnome_popup_menu_do_popup(ggv->popup_menu, NULL, NULL, event, ggv);
                if(ggv->menus_vis_pu)
                        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(ggv->menus_vis_pu),
                                                       ggv->show_menus);
                if(ggv->panel_vis_pu)
                        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(ggv->panel_vis_pu),
                                                       ggv->show_panel);
                break;
        default:
                break;
        }
}

static void
button_release_callback(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;

        switch(event->button) {
        case 1:
                if(ggv->pan) {
                        ggv->pan = FALSE;
                        gdk_pointer_ungrab(GDK_CURRENT_TIME);
                        gtk_grab_remove(widget);
                }
                break;
        case 2:
                ggv->cd->incrop = 0;
                break;
        default:
                break;
        }
}

static void
close_callback (GtkWidget *widget, ggv_window *ggv)
{
        close_window(ggv);
}

static gint
exit_callback(GtkWidget *widget, gpointer data) 
{

        ggv_window *ggv;
        GList *node;

        /* do we want to save anything before we exit??? */        

        while (window_list) {
                node = window_list;
                window_list = window_list->next;
                if(window_list)
                        window_list->prev = NULL;
                node->next = NULL;
                ggv = (ggv_window *)node->data;
                g_list_free(node);
                gtk_widget_destroy (GTK_WIDGET (ggv->main_window));
                g_free(ggv);
        }

	gtk_main_quit();

	return 0;
}

static gint
delete_callback(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
        close_window((ggv_window *)data);

        return 0;
}

static void file_open_cancel_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;

        gtk_widget_destroy(ggv->file_sel);
        ggv->file_sel = NULL;
}

static void file_open_delete_callback(GtkWidget *widget, GdkEventAny *e, gpointer data)
{
        file_open_cancel_callback(widget, data);
}

static void file_open_ok_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;
        gchar *file;
        gchar *baseFileName;
        char *ErrorMessage;
        struct stat StatRecord;

	file = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(ggv->file_sel)));

        /* We have to verify that this file exists */
        
        if (stat(file,&StatRecord) == 0) {
                
                if (S_ISDIR(StatRecord.st_mode)) {
                        /* This is a directory, reset window and
                           try again */
                        if (strlen(file) > 0 && file[strlen(file)-1] != '/')
                                strcat(file,"/");
                        gtk_file_selection_set_filename(GTK_FILE_SELECTION(ggv->file_sel),
                                                        file);
                        return;
                }
                
        }
        else {
                /* We have to check if there is a wildcard in the directory name
                 */
                
                baseFileName = g_basename(file);
                if (strchr(baseFileName, '?' ) != NULL ||
                    strchr(baseFileName, '*' ) != NULL) {
                        gtk_file_selection_complete(GTK_FILE_SELECTION(ggv->file_sel),
                                                    file);
                }
                           
                /* FIXME:  currently , if the file does not exist,
                   the window does not close, but it does not send a message.
                   If we send a message, the focus does not return to the
                   File window after the user clicks on it

                  ErrorMessage = g_strdup_printf(_("File does not exist %s."), file);
                  error_message(ggv,ErrorMessage);
                */
                return;
        }
        gtk_widget_destroy(ggv->file_sel);
        ggv->file_sel = NULL;

        if ((ggv->loaded = load_gs(ggv, file))) {
                recent_add (file);
                recent_update (ggv);

        }
        g_free(file);
}




static void
show_menubar_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        ggv->show_menus = GTK_CHECK_MENU_ITEM(widget)->active;

        if (ggv->show_menus) {
                if((GNOME_APP(ggv->main_window))->menubar == NULL) {
                        create_menus(ggv, ggv->menudata);
                        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(ggv->menus_vis),
                                                        ggv->show_menus);
                        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(ggv->panel_vis),
                                                        ggv->show_panel);
                }
                gtk_widget_show ((GNOME_APP(ggv->main_window))->menubar->parent);
        } else {
                gtk_widget_hide ((GNOME_APP(ggv->main_window))->menubar->parent);
		gtk_widget_queue_resize (ggv->main_window);
        }
        if(ggv->menus_vis &&
           GTK_CHECK_MENU_ITEM(ggv->menus_vis)->active != ggv->show_menus)
                gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(ggv->menus_vis),
                                               ggv->show_menus);
}

static void
hide_panel_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        ggv->show_panel = GTK_CHECK_MENU_ITEM(widget)->active;

        if (ggv->show_panel) {
                if(ggv->scrollpane == NULL)
                        create_sidebar(ggv);
                gtk_widget_show (ggv->sidebar);
        } else {
                gtk_widget_hide (ggv->sidebar);
		gtk_widget_queue_resize (ggv->main_window);
        }
        if(ggv->panel_vis &&
           GTK_CHECK_MENU_ITEM(ggv->panel_vis)->active != ggv->show_panel)
                gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(ggv->panel_vis),
                                               ggv->show_panel);
}

static void
motion_callback(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        if(ggv->pan) {
                gtk_gs_scroll(GTK_GS(ggv->gs), -event->x + ggv->prev_x, -event->y + ggv->prev_y);;
                ggv->prev_x = event->x;
                ggv->prev_y = event->y;
        }
        else {
#if 0
                mouse_moved (ggv->cd, ggv->gs->window, 
                             event->x, event->y,
                             ggv->main_window->allocation.width,
                             ggv->main_window->allocation.height);
#endif
        }
}

/* Utility function */
static void
set_page_sensitivities(ggv_window *ggv, int page)
{
        gboolean prev = FALSE, next = FALSE;

        if(page > 0)
                prev = TRUE;
        if(page < GTK_GS(ggv->gs)->doc->numpages - 1)
                next = TRUE;
        gtk_widget_set_sensitive(ggv->nextbutton, next);
        gtk_widget_set_sensitive(ggv->prevbutton, prev);
        gtk_widget_set_sensitive(ggv->nextitem, next);
        gtk_widget_set_sensitive(ggv->previtem, prev);
        gtk_widget_set_sensitive(ggv->nextpu, next);
        gtk_widget_set_sensitive(ggv->prevpu, prev);
}        

static void
goto_page(ggv_window *ggv, int page)
{
        gtk_gs_goto_page ( GTK_GS (ggv->gs), page);

        if (GTK_GS(ggv->gs)->structured_doc) {
                page = GTK_GS (ggv->gs)->current_page; /* Make sure */
                set_page_sensitivities(ggv, page);
                if(ggv->pagelist) {
                        gtk_clist_select_row (GTK_CLIST (ggv->pagelist), page,1);
                        if( !gtk_clist_row_is_visible (GTK_CLIST(ggv->pagelist), page)) {
                                gtk_clist_moveto (GTK_CLIST (ggv->pagelist), GTK_GS (ggv->gs)->current_page, 0, 0.5, 0.5);
                        }
                }
        }
}

static void
key_pressed_event_callback(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;
        gint key = event->keyval;
        gint orientation = gtk_gs_get_orientation (GTK_GS (ggv->gs));

        if (!ggv->loaded)
                return;
        /*ugh. the possibilities! */
        switch (key) {
        case GDK_space:
                switch (orientation) {
                case GTK_GS_ORIENTATION_PORTRAIT:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane), 
                                                 GTK_SCROLLPANE_SCROLL_DOWN, TRUE)) {
                                /* Since we are doing wrapping, go to top of the page */
                                gtk_scrollpane_goto_edge(GTK_SCROLLPANE (ggv->scrollpane),
                                                         GTK_SCROLLPANE_GOTOEDGE_LOWER, 
                                                         GTK_SCROLLPANE_GOTOEDGE_LOWER);
                                g_print("Scroll Return false\n");
				goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_LANDSCAPE:
                        if (!gtk_scrollpane_step (GTK_SCROLLPANE (ggv->scrollpane),
                                                       GTK_SCROLLPANE_SCROLL_LEFT,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_SEASCAPE:
                        if (!gtk_scrollpane_step (GTK_SCROLLPANE (ggv->scrollpane),
                                                  GTK_SCROLLPANE_SCROLL_RIGHT,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_UPSIDEDOWN:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),
                                                     GTK_SCROLLPANE_SCROLL_UP,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
                        }
                        break;
                }
                break;
        case GDK_BackSpace:
        case GDK_Delete:
                switch (orientation) {
                case GTK_GS_ORIENTATION_PORTRAIT:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),
                                                 GTK_SCROLLPANE_SCROLL_UP,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_LANDSCAPE:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_RIGHT,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_SEASCAPE:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_LEFT,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
                        }
                        break;
                case GTK_GS_ORIENTATION_UPSIDEDOWN:
                        if (!gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_DOWN,TRUE)) {
				goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
                        }
                        break;
                }
                break;
        case GDK_Left:
                gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_LEFT,FALSE);
                break;
        case GDK_Right:
                gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_RIGHT,FALSE);
                break;
        case GDK_Up:
                gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_UP,FALSE);
                break;
        case GDK_Down:
                gtk_scrollpane_step(GTK_SCROLLPANE (ggv->scrollpane),GTK_SCROLLPANE_SCROLL_DOWN,FALSE);
                break;
        case GDK_Page_Up:
		goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
                break;
        case GDK_Page_Down:
		goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
                break;
	case GDK_plus:
		zoomin_callback(ggv->gs, (gpointer *)ggv);
		break;
	case GDK_minus:
		zoomout_callback(ggv->gs, (gpointer *)ggv);
		break;
	default:
		goto DONT_STOP_SIGNAL;
        }
	gtk_signal_emit_stop_by_name (GTK_OBJECT(ggv->main_window), "key_press_event");
 DONT_STOP_SIGNAL:
	;
}

static void next_page_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;
        g_print("Going to next page [%d]\n", ggv->loaded);
        if (ggv->loaded) {
		goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
        }
}


static void open_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;
        gchar *dirname;

        if(ggv->file_sel)
                return;

	ggv->file_sel = gtk_file_selection_new(_("ggv: Open File"));
	gtk_signal_connect(GTK_OBJECT(ggv->file_sel), "delete_event",
			   GTK_SIGNAL_FUNC(file_open_delete_callback), data);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(ggv->file_sel)->cancel_button),
			   "clicked",
			   GTK_SIGNAL_FUNC(file_open_cancel_callback), data);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(ggv->file_sel)->ok_button),
			   "clicked",
			   GTK_SIGNAL_FUNC(file_open_ok_callback), data);

        gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(ggv->file_sel));

        /* Switch to the directory where the current 
           file came from */

        if (GTK_GS (ggv->gs)->gs_filename != NULL &&
            (dirname=g_dirname(GTK_GS (ggv->gs)->gs_filename)) != NULL)
                gtk_file_selection_complete(GTK_FILE_SELECTION(ggv->file_sel),
                                                    dirname);

        gtk_widget_show(ggv->file_sel);
}

static void orientation_callback(GtkWidget *widget, gpointer data)
{
        window_with_data *wwd = (window_with_data *) data;
        ggv_window *ggv = wwd->ggv;

        if (gtk_gs_set_orientation (GTK_GS(ggv->gs), wwd->data))
                gtk_gs_goto_page (GTK_GS(ggv->gs), GTK_GS(ggv->gs)->current_page);
}

static void preferences_callback(GtkWidget *widget, gpointer data) {
        open_prefs_dialog(&prefs_dialog);
}

static void previous_page_callback(GtkWidget *widget, gpointer data) 
{
        ggv_window *ggv = (ggv_window *) data;

        if (ggv->loaded) {
		goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
        }
}

static void print_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;
        gchar *print_command;

        if(ggv->loaded) {
                print_command = g_strdup_printf(gs_print_cmd,
                                                GTK_GS(ggv->gs)->gs_filename);
                if(print_command) {
                        gnome_execute_shell(NULL, print_command);
                        g_free(print_command);
                }
        }
}

static void recenter_page_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        if (ggv->loaded)
                gtk_gs_center_page (GTK_GS (ggv->gs));
}

static void reload_callback(GtkWidget *widget, gpointer data) 
{
        ggv_window *ggv = (ggv_window *) data;

        if (ggv->loaded) {
                gchar *fname = g_strdup (GTK_GS (ggv->gs)->gs_filename);
                ggv->loaded = load_gs (ggv, fname);
                g_free (fname);
        }
}

static void save_callback(GtkWidget *widget, gpointer data) {}

static void save_as_callback(GtkWidget *widget, gpointer data) {}

static void
scrollpane_middle_click_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        if (ggv->loaded) {
		goto_page( ggv, GTK_GS(ggv->gs)->current_page - 1);
        }
}

static void
scrollpane_right_click_callback(GtkWidget *widget, gpointer data)
{
        ggv_window *ggv = (ggv_window *) data;

        if (ggv->loaded) {
		goto_page( ggv, GTK_GS(ggv->gs)->current_page + 1);
        }
}

static void 
select_page_callback(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer data)
{
        ggv_window *ggv = (ggv_window *)data;

        if(GTK_GS(ggv->gs)->current_page != row) {
                gtk_gs_goto_page(GTK_GS(ggv->gs), row);
                row = GTK_GS(ggv->gs)->current_page; /* make sure */
                set_page_sensitivities(ggv, row);
        }
}

static void 
drop_callback(GtkWidget *widget, GdkDragContext *context,
                          gint x, gint y, GtkSelectionData *selection_data,
                          guint info, guint time, gpointer data) {
	GList *names, *list;
        ggv_window *ggv = (ggv_window *)data;

	switch (info) {
	case TARGET_URI_LIST:
		list = names = gnome_uri_list_extract_filenames (selection_data->data);
		while (names) {
                        if(ggv->loaded)
                                open_window(names->data);
                        else if (load_gs (ggv, names->data))
                                ggv->loaded = TRUE;

			names = names->next;
		}
		gnome_uri_list_free_strings (list);
		break;
	default:
                break;
	}
}

static double compute_zoom(gint zoom_spec)
{
	return pow(1.2, zoom_spec); /* The Knuth magstep formula rules */
}

static void zoom_to(ggv_window *ggv, gint magstep) 
{
        double z;
        ggv->zoom_magstep = magstep;

        z = compute_zoom(magstep);

        g_print("Changing zoom mag: %f\n", z);

        /* Return if no active doc */
        if (!ggv->loaded) {
                g_assert(ggv->gs);
                GTK_GS(ggv->gs)->zoom_factor = z;
                return;
        }
        gtk_gs_set_zoom (GTK_GS (ggv->gs), z);
        gtk_gs_goto_page (GTK_GS (ggv->gs), GTK_GS(ggv->gs)->current_page);        
}

static void zoom_callback(GtkWidget *widget, gpointer data)
{
        window_with_data *wwd = (window_with_data *) data;
        ggv_window *ggv = wwd->ggv;

        if (ggv->zoom_magstep == wwd->data) 
                return;
	zoom_to(ggv, wwd->data);
}

static void zoomin_callback(GtkWidget *widget, gpointer *data)
{
        ggv_window *ggv = (ggv_window *)data;
	zoom_to(ggv, ggv->zoom_magstep+1);
}

static void zoomout_callback(GtkWidget *widget, gpointer *data)
{
        ggv_window *ggv = (ggv_window *)data;
	zoom_to(ggv, ggv->zoom_magstep-1);
}

static void
prefs_destroy_callback(GtkWidget *w, ggv_prefs *p)
{
        p->pbox = NULL;
}

static void
prefs_apply_callback(GtkWidget *w, gint page, ggv_prefs *p)
{
        GtkWidget *active;
        gint i;

        if( page != -1 ) return;

        if(gs_cmd)
                g_free(gs_cmd);
        gs_cmd = g_strdup(gtk_entry_get_text(GTK_ENTRY(p->gs)));
        if(gs_scan_pdf_cmd)
                g_free(gs_scan_pdf_cmd);
        gs_scan_pdf_cmd = g_strdup(gtk_entry_get_text(GTK_ENTRY(p->scan_pdf)));
        if(gs_uncompress_cmd)
                g_free(gs_uncompress_cmd);
        gs_uncompress_cmd = g_strdup(gtk_entry_get_text(GTK_ENTRY(p->uncompress)));  
        if(gs_print_cmd)
                g_free(gs_print_cmd);
        gs_print_cmd = g_strdup(gtk_entry_get_text(GTK_ENTRY(p->print)));

        active = gtk_menu_get_active(GTK_MENU(gtk_option_menu_get_menu(GTK_OPTION_MENU(p->media))));
        for(i = 0; i < PAPER_SIZE_COUNT; i++)
                if(active == p->media_choice[i]) {
                        gs_def_media = i;
                        break;
                }
        
        gs_antialiased = GTK_TOGGLE_BUTTON(p->aa)->active;
        gs_override_media = GTK_TOGGLE_BUTTON(p->override_media)->active;
        gs_panel = GTK_TOGGLE_BUTTON(p->tbar)->active;
        gs_menubar = GTK_TOGGLE_BUTTON(p->mbar)->active;

        apply_gs_prefs(window_list);
}

static void
prefs_changed_callback(GtkWidget *w, ggv_prefs *p)
{
	gnome_property_box_changed(p->pbox);
}

static void
create_popup_menus(ggv_window *ggv)
{
        GnomeUIInfo popupMenu[] = {
                {GNOME_APP_UI_ITEM, N_("Next Page"), NULL, next_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Previous Page"), NULL, previous_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Zoom In"), NULL, zoomin_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Zoom Out"), NULL, zoomout_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("Recenter Page"), NULL, recenter_page_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_TOGGLEITEM, N_("Show Menus"), NULL, show_menubar_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_TOGGLEITEM, N_("Show Side Panel"), NULL, hide_panel_callback, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ENDOFINFO}
        };
                
        ggv->popup_menu = gnome_popup_menu_new(popupMenu);
        ggv->nextpu = popupMenu[0].widget;
        ggv->prevpu = popupMenu[1].widget;
        ggv->menus_vis_pu = popupMenu[6].widget;
        ggv->panel_vis_pu = popupMenu[7].widget;
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->prevpu),FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(ggv->nextpu),FALSE);
}
       
static void
create_menus(ggv_window *retval, window_with_data *temp)
{
        GnomeUIInfo orientationMenu2[] = {
                GNOMEUIINFO_ITEM_DATA(N_("_Automatic"), N_("Use document's orientation"), orientation_callback, &temp[0], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Portrait"), N_("Portrait orientation"), orientation_callback, &temp[1], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Landscape"), N_("Landscape orientation"), orientation_callback, &temp[2], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Seascape"), N_("Seascape orientation"), orientation_callback, &temp[3], NULL),
                GNOMEUIINFO_ITEM_DATA(N_("_Upside Down"), N_("Upside down orientation"), orientation_callback, &temp[4], NULL),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo orientationMenu[] = {
                GNOMEUIINFO_RADIOLIST(orientationMenu2),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo zoomMenu2[] = {
                GNOMEUIINFO_RADIOITEM_DATA("1:1", NULL, zoom_callback,  &temp[10], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:10" , NULL, zoom_callback, &temp[5] , NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:8" , NULL, zoom_callback, &temp[6], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:4", NULL, zoom_callback, &temp[7], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:2", NULL, zoom_callback, &temp[8], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1:1.4", NULL, zoom_callback, &temp[9], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("1.4:1", NULL, zoom_callback, &temp[11] , NULL),
                GNOMEUIINFO_RADIOITEM_DATA("2:1", NULL, zoom_callback, &temp[12], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("4:1", NULL, zoom_callback, &temp[13], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("8:1", NULL, zoom_callback, &temp[14], NULL),
                GNOMEUIINFO_RADIOITEM_DATA("10:1", NULL, zoom_callback, &temp[15], NULL),
                GNOMEUIINFO_END,
        };

        GnomeUIInfo zoomMenu[] = {
                GNOMEUIINFO_RADIOLIST(zoomMenu2),
                GNOMEUIINFO_END,
        };

        GnomeUIInfo settingsMenu[] = {
                GNOMEUIINFO_MENU_PREFERENCES_ITEM(preferences_callback, retval),
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_TOGGLEITEM, N_("_Show Menus"), N_("Toggle menu visibility"), show_menubar_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_TOGGLEITEM, N_("S_how Side Panel"), N_("Toggle side panel visibility"), hide_panel_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                GNOMEUIINFO_END
        };

        GnomeUIInfo fileMenu[] = {
                GNOMEUIINFO_MENU_NEW_ITEM(N_("_New Window"), N_("Open a new window"), new_callback, NULL),
                {GNOME_APP_UI_ITEM, N_("_Open"), N_("Load a postscript document"), open_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 'o', GDK_CONTROL_MASK, NULL},
                {GNOME_APP_UI_ITEM, N_("_Reload"), N_("Reload current document from disk"), reload_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REVERT, 'r', GDK_CONTROL_MASK, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("_Print..."), N_("Print the document"), print_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 'p', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_ITEM, N_("_Save"), NULL, save_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("S_ave as..."), NULL, save_as_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("_Close"), N_("Close this window"), close_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 'w', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_ITEM, N_("E_xit"), N_("Exit GGv"), exit_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'q', GDK_CONTROL_MASK, NULL },
                {GNOME_APP_UI_ENDOFINFO, NULL, NULL, NULL, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL}
        };
        
        GnomeUIInfo documentMenu[] = {
                {GNOME_APP_UI_ITEM, N_("_Next Page"), N_("Show next page"), next_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Previous Page"), N_("Show previous page"), previous_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK, 0, 0, NULL},
                {GNOME_APP_UI_ITEM, N_("_Recenter Page"), N_("Center page"), recenter_page_callback, retval, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SEPARATOR},
                {GNOME_APP_UI_SUBTREE, N_("_Orientation"), NULL, orientationMenu, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_SUBTREE, N_("_Zoom"), NULL, zoomMenu, NULL, NULL,
                 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
                {GNOME_APP_UI_ENDOFINFO}
        };
        
        GnomeUIInfo helpMenu[] = {
                GNOMEUIINFO_HELP("ggv"),
                GNOMEUIINFO_MENU_ABOUT_ITEM(about_callback, retval),
                GNOMEUIINFO_END
        };
        
        GnomeUIInfo mainMenu[] = {
                GNOMEUIINFO_MENU_FILE_TREE(fileMenu),
                GNOMEUIINFO_SUBTREE(N_("_Document"), documentMenu),
                GNOMEUIINFO_MENU_SETTINGS_TREE(settingsMenu),
                GNOMEUIINFO_MENU_HELP_TREE(helpMenu),
                GNOMEUIINFO_END
        };

        /* We set up the main window... */
        gnome_app_create_menus (GNOME_APP (retval->main_window), mainMenu);

	gnome_app_install_menu_hints(GNOME_APP(retval->main_window),
                                     mainMenu);

        retval->nextitem = documentMenu[0].widget;
        retval->previtem = documentMenu[1].widget;
        retval->reloaditem = fileMenu[2].widget;
        retval->printitem = fileMenu[4].widget;
        retval->menus_vis = settingsMenu[2].widget;
        retval->panel_vis = settingsMenu[3].widget;
        gtk_widget_set_sensitive (GTK_WIDGET(retval->previtem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->nextitem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->printitem), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->reloaditem), FALSE);

        gtk_widget_set_sensitive (GTK_WIDGET(fileMenu[5].widget), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(fileMenu[6].widget), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(fileMenu[8].widget), FALSE);
}        

static void
create_sidebar(ggv_window *retval)
{
        GtkWidget *pic;
        GtkWidget *npbhbox;
        GtkWidget *zbhbox;

        retval->coordinates = gtk_label_new (" ");
        gtk_widget_show (retval->coordinates);

        retval->scrollpane = gtk_scrollpane_new(GTK_ADJUSTMENT (retval->hadj), GTK_ADJUSTMENT (retval->vadj), 1.0);
        gtk_signal_connect (GTK_OBJECT(retval->scrollpane), "middle_clicked", GTK_SIGNAL_FUNC(scrollpane_middle_click_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->scrollpane), "right_clicked", GTK_SIGNAL_FUNC(scrollpane_right_click_callback), retval);
        gtk_widget_show (retval->scrollpane);
        
        npbhbox = gtk_hbox_new (TRUE, 0);
        retval->nextbutton = gtk_button_new ();
        retval->prevbutton = gtk_button_new ();

        pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_BACK);
        gtk_widget_show (pic);
        gtk_container_add (GTK_CONTAINER (retval->prevbutton), pic);
        pic = gnome_stock_pixmap_widget_new(retval->nextbutton, GNOME_STOCK_MENU_FORWARD);
        gtk_widget_show (pic);
        gtk_container_add (GTK_CONTAINER (retval->nextbutton), pic);
        gtk_signal_connect (GTK_OBJECT(retval->nextbutton), "clicked", GTK_SIGNAL_FUNC(next_page_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->prevbutton), "clicked", GTK_SIGNAL_FUNC(previous_page_callback), retval);
        gtk_container_add (GTK_CONTAINER (npbhbox), retval->prevbutton);
        gtk_container_add (GTK_CONTAINER (npbhbox), retval->nextbutton);

        gtk_widget_set_sensitive (GTK_WIDGET(retval->prevbutton),FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->nextbutton),FALSE);

        gtk_widget_show (retval->prevbutton);
        gtk_widget_show (retval->nextbutton);
        gtk_widget_show (npbhbox);
        
        zbhbox = gtk_hbox_new (TRUE, 0);
        retval->zoominbutton = gtk_button_new_with_label ( _("+") );
        retval->zoomoutbutton = gtk_button_new_with_label ( _("-") );
        gtk_widget_set_sensitive (GTK_WIDGET(retval->zoomoutbutton), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET(retval->zoominbutton), FALSE);
	/*  We need stock buttons for these...
         * pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_BACK);
         * gtk_widget_show (pic);
         * gtk_container_add (GTK_CONTAINER (retval->prevbutton), pic);
         * pic = gnome_stock_pixmap_widget_new(retval->prevbutton, GNOME_STOCK_MENU_FORWARD);
         * gtk_widget_show (pic);
         * gtk_container_add (GTK_CONTAINER (retval->nextbutton), pic);
	 */
        gtk_signal_connect (GTK_OBJECT(retval->zoominbutton), "clicked", GTK_SIGNAL_FUNC(zoomin_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->zoomoutbutton), "clicked", GTK_SIGNAL_FUNC(zoomout_callback), retval);
        gtk_container_add (GTK_CONTAINER (zbhbox), retval->zoomoutbutton);
        gtk_container_add (GTK_CONTAINER (zbhbox), retval->zoominbutton);
        gtk_widget_show (retval->zoomoutbutton);
        gtk_widget_show (retval->zoominbutton);
        gtk_widget_show (zbhbox);


	retval->pagelistscroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(retval->pagelistscroll),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_placement(GTK_SCROLLED_WINDOW(retval->pagelistscroll),
					GTK_CORNER_BOTTOM_RIGHT);

        retval->pagelist = gtk_clist_new(2);
	gtk_widget_set_usize (retval->pagelist, 80, 100);
	gtk_clist_set_column_width (GTK_CLIST (retval->pagelist), 1, 10);
	gtk_clist_set_column_width (GTK_CLIST (retval->pagelist), 2, 20);
	gtk_clist_set_selection_mode (GTK_CLIST (retval->pagelist), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_justification (GTK_CLIST (retval->pagelist), 1, GTK_JUSTIFY_CENTER);
	gtk_clist_set_column_justification (GTK_CLIST (retval->pagelist), 2, GTK_JUSTIFY_RIGHT);
        gtk_signal_connect(GTK_OBJECT(retval->pagelist), "select_row", GTK_SIGNAL_FUNC(select_page_callback),retval);
	gtk_container_add (GTK_CONTAINER (retval->pagelistscroll), retval->pagelist);
	gtk_widget_show (retval->pagelist);
	gtk_widget_show (retval->pagelistscroll);

        gtk_box_pack_start (GTK_BOX (retval->sidebar), retval->coordinates, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (retval->sidebar), retval->scrollpane, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (retval->sidebar), zbhbox, FALSE, FALSE, 2 );
        gtk_box_pack_start (GTK_BOX (retval->sidebar), npbhbox, FALSE, FALSE, 2 );
        gtk_container_add (GTK_CONTAINER (retval->sidebar), retval->pagelistscroll);
        gtk_widget_show (retval->sidebar); 
}

/* main functions */

static
ggv_window *create_ggv_window()
{
        /* DnD stuff */
	static GtkTargetEntry drop_types [] = {
		{ "text/uri-list", 0, TARGET_URI_LIST}
	};
	static gint n_drop_types = sizeof (drop_types) / sizeof(drop_types[0]);

        gint i;
        GtkWidget *hbox;
        GtkWidget *vbox;
        GtkWidget *gsframe;

        ggv_window *retval = g_new0 (ggv_window, 1);

        retval->show_menus = gs_menubar;
        retval->show_panel = gs_panel;
        retval->loaded = FALSE;
        retval->pan = FALSE;

        /* create the main window. */
        retval->main_window = gnome_app_new("ggv", "ggv:");
        gtk_window_set_default_size(GTK_WINDOW(retval->main_window), 640, 480);
        gtk_window_set_policy(GTK_WINDOW(retval->main_window), TRUE, TRUE, FALSE);
        retval->cd = NULL;
        /* make it a drop zone for files */
	gtk_drag_dest_set (GTK_WIDGET (retval->main_window),
	                   GTK_DEST_DEFAULT_MOTION |
			   GTK_DEST_DEFAULT_HIGHLIGHT |
			   GTK_DEST_DEFAULT_DROP,
			   drop_types, n_drop_types,
			   GDK_ACTION_COPY);
        gtk_signal_connect (GTK_OBJECT (retval->main_window), "drag_data_received",
			    GTK_SIGNAL_FUNC(drop_callback), retval);
        /* take care of its deletion */
	gtk_signal_connect (GTK_OBJECT (retval->main_window), "delete_event",
                            GTK_SIGNAL_FUNC (delete_callback), retval);

        /* Set up the callback values for the orientation */
        for (i = 0; i < 16; i++)
                retval->menudata[i].ggv = retval;

        retval->menudata[0].data = GTK_GS_ORIENTATION_DEFAULT;
        retval->menudata[1].data = GTK_GS_ORIENTATION_PORTRAIT;
        retval->menudata[2].data = GTK_GS_ORIENTATION_LANDSCAPE;
        retval->menudata[3].data = GTK_GS_ORIENTATION_SEASCAPE;
        retval->menudata[4].data = GTK_GS_ORIENTATION_UPSIDEDOWN;

	/* We are lying a little here - the magstep does not give
	 * these ratios exactly but who's going to know ;)
	 */

        retval->menudata[5].data = -13;
        retval->menudata[6].data = -12;
        retval->menudata[7].data = -8; 
        retval->menudata[8].data = -4; 
        retval->menudata[9].data = -2; 
        retval->menudata[10].data = 0;
        retval->menudata[11].data = 2;
        retval->menudata[12].data = 4;
        retval->menudata[13].data = 8;
        retval->menudata[14].data = 12;
        retval->menudata[15].data = 13;

        /* Statusbar */
        retval->appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
        gtk_widget_show(retval->appbar);
        gnome_app_set_statusbar(GNOME_APP(retval->main_window), retval->appbar);

        /* Menus */
        if(gs_menubar)
                create_menus (retval, retval->menudata);
        create_popup_menus(retval);

        /* We set up the layout */
        hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
        vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

        retval->hadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
        retval->vadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);

        /* we set up the post script display */
        gsframe = gtk_frame_new (NULL);
        gtk_frame_set_shadow_type (GTK_FRAME (gsframe), GTK_SHADOW_IN);
        retval->gs = gtk_gs_new (GTK_ADJUSTMENT (retval->hadj),
                                 GTK_ADJUSTMENT (retval->vadj));
        gtk_widget_set_events (retval->gs, 
                               GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK);
        set_gs_prefs (retval);
        gtk_container_add (GTK_CONTAINER (gsframe), retval->gs);
        gtk_widget_show (retval->gs);
        gtk_widget_show (gsframe);

        /* we prepare for the crop data */
        retval->cd = g_malloc (sizeof (crop_data));

        /* We set up the sidebar */
        retval->zoom_magstep = 0; /* 1 : 1 */

        retval->sidebar = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
        if(gs_panel)
                create_sidebar(retval);

        /* We layout the page */
        gtk_box_pack_start (GTK_BOX (hbox), retval->sidebar, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), gsframe, TRUE, TRUE, 0);
        gtk_widget_show(hbox);
        gtk_widget_show(vbox);

	gnome_app_set_contents(GNOME_APP(retval->main_window), hbox);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "button_press_event", GTK_SIGNAL_FUNC (button_press_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "button_release_event", GTK_SIGNAL_FUNC (button_release_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->gs), "motion_notify_event", GTK_SIGNAL_FUNC (motion_callback), retval);
        gtk_signal_connect (GTK_OBJECT(retval->main_window), "key_press_event", GTK_SIGNAL_FUNC(key_pressed_event_callback), retval);

        retval->file_sel = NULL;

        if(gs_menubar) {
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(retval->menus_vis),
                                                retval->show_menus);
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(retval->panel_vis),
                                                retval->show_panel);
        }
        retval->bRecentHasBeenInstantiated = FALSE;
        return retval;
}

static void
flash_message(ggv_window *ggv, gchar *flash)
{
        if(flash) {
                gnome_app_flash(GNOME_APP(ggv->main_window), flash);
                g_free(flash);
        }
}

static void
error_message(ggv_window *ggv, gchar *errormsg)
{
        if(errormsg) {
                gnome_app_error(GNOME_APP(ggv->main_window), errormsg);
                g_free(errormsg);
        }
}

static void
open_window(gchar *filename)
{
        if(filename == NULL) {
                ggv_window *ggv = create_ggv_window ();
                active_ggv ++;
                window_list = g_list_append(window_list, ggv);
                gtk_widget_show (ggv->main_window);
                recent_update(ggv);

        }
        else {
                ggv_window *ggv = create_ggv_window ();
                gtk_widget_show (ggv->main_window);

                if (load_gs (ggv, filename)) {                                
                        ggv->loaded = TRUE;
                        active_ggv ++;
                        window_list = g_list_append (window_list, ggv);
                }
                else {
                        gtk_widget_destroy(ggv->main_window);
                        g_free(ggv);
                }
        }
}

static void
close_window(ggv_window *ggv)
{
        /* do we want to save changes??? */
	window_list = g_list_remove (window_list, ggv);
	gtk_widget_destroy (GTK_WIDGET (ggv->main_window));
        g_free(ggv);

        active_ggv--;

	if (active_ggv == 0)
		gtk_main_quit ();
}

static void
open_prefs_dialog(ggv_prefs *pd)
{
        GtkWidget *table, *label, *menu;
        gint i;

        if(pd->pbox)
                return;

        pd->pbox = GNOME_PROPERTY_BOX(gnome_property_box_new());
	gtk_signal_connect(GTK_OBJECT(pd->pbox), "destroy",
                           GTK_SIGNAL_FUNC(prefs_destroy_callback), pd);
	gtk_signal_connect(GTK_OBJECT(pd->pbox), "apply",
                           GTK_SIGNAL_FUNC(prefs_apply_callback), pd);

        /* Document page */
        table = gtk_table_new(2, 3, FALSE);

        /* media choice menu */
        label = gtk_label_new(_("Fallback media type"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->media = gtk_option_menu_new();
        gtk_widget_show(pd->media);
        menu = gtk_menu_new();
        for(i = 0; papersizes[i].name != NULL; i++) {
                pd->media_choice[i] = gtk_menu_item_new_with_label(papersizes[i].name);
                gtk_widget_show(pd->media_choice[i]);
                gtk_menu_append(GTK_MENU(menu), pd->media_choice[i]);
        }
        gtk_option_menu_set_menu(GTK_OPTION_MENU(pd->media), menu);
        i = gs_def_media;
        gtk_option_menu_set_history(GTK_OPTION_MENU(pd->media), i);
        for(i = 0; papersizes[i].name != NULL; i++)
                gtk_signal_connect(GTK_OBJECT(pd->media_choice[i]),
                                   "activate", GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_table_attach(GTK_TABLE(table), pd->media,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        /* override document media */
        pd->override_media = gtk_check_button_new_with_label(_("Override document media"));
        gtk_table_attach(GTK_TABLE(table), pd->override_media,
                         0, 2, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->override_media), gs_override_media);
        gtk_signal_connect(GTK_OBJECT(pd->override_media), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->override_media);
        /* antialiasing */
        pd->aa = gtk_check_button_new_with_label(_("Antialiasing"));
        gtk_table_attach(GTK_TABLE(table), pd->aa,
                         0, 2, 2, 3,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->aa), gs_antialiased);
        gtk_signal_connect(GTK_OBJECT(pd->aa), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->aa);

        label = gtk_label_new(_("Document"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* Layout page */
        table = gtk_table_new(2, 1, FALSE);

        /* show toolbar */
        pd->tbar = gtk_check_button_new_with_label(_("Show side panel by default"));
        gtk_table_attach(GTK_TABLE(table), pd->tbar,
                         0, 1, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->tbar), gs_panel);
        gtk_signal_connect(GTK_OBJECT(pd->tbar), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->tbar);
        /* show menubar */
        pd->mbar = gtk_check_button_new_with_label(_("Show menubar by default"));
        gtk_table_attach(GTK_TABLE(table), pd->mbar,
                         0, 1, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->mbar), gs_menubar);
        gtk_signal_connect(GTK_OBJECT(pd->mbar), "clicked",
                           prefs_changed_callback, pd);
        gtk_widget_show(pd->mbar);

        label = gtk_label_new(_("Layout"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* GhostScript page */
        table = gtk_table_new(2, 3, FALSE);

        /* interpreter */
        label = gtk_label_new(_("Interpreter"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->gs = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->gs), gs_cmd);
        gtk_table_attach(GTK_TABLE(table), pd->gs,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->gs), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->gs);
        /* scan PDF command */
        label = gtk_label_new(_("Scan PDF"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 1, 2,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->scan_pdf = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->scan_pdf), gs_scan_pdf_cmd);
        gtk_table_attach(GTK_TABLE(table), pd->scan_pdf,
                         1, 2, 1, 2,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->scan_pdf), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->scan_pdf);
        /* uncompress command */
        label = gtk_label_new(_("Uncompress"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 2, 3,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->uncompress = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->uncompress), gs_uncompress_cmd);
        gtk_table_attach(GTK_TABLE(table), pd->uncompress,
                         1, 2, 2, 3,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->uncompress), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->uncompress);

        label = gtk_label_new(_("GhostScript"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

        /* Printing page */
        table = gtk_table_new(1, 2, FALSE);

        /* print command */
        label = gtk_label_new(_("Print command"));
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, 0, 1,
                         GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
        gtk_widget_show(label);
        pd->print = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(pd->print), gs_print_cmd);
        gtk_table_attach(GTK_TABLE(table), pd->print,
                         1, 2, 0, 1,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                         GNOME_PAD_SMALL, GNOME_PAD_SMALL);
	gtk_signal_connect(GTK_OBJECT(pd->print), "activate",
                           GTK_SIGNAL_FUNC(prefs_changed_callback), pd);
        gtk_widget_show(pd->print);

        label = gtk_label_new(_("Printing"));
        gtk_widget_show(label);
        gtk_widget_show(table);
        gnome_property_box_append_page(pd->pbox, table, label);

	gtk_window_position (GTK_WINDOW(pd->pbox), GTK_WIN_POS_MOUSE);
	
	gtk_widget_show(GTK_WIDGET(pd->pbox));
}

static void
set_gs_prefs(ggv_window *ggv)
{
        GtkGS *gs = GTK_GS(ggv->gs);
        gboolean redisplay = FALSE;

        if(gs->antialiased != gs_antialiased && ggv->loaded)
                redisplay = TRUE;
        set_prefs(GTK_GS(ggv->gs));
        if(redisplay)
                reload_callback(NULL, ggv);
}

static void
apply_gs_prefs(GList *windows)
{
        GtkGS *gs;

        while(windows) {
                set_gs_prefs((ggv_window *)windows->data);
                windows = windows->next;
        }
}

/* and we finally start the main prog. */
int
main(int argc, char *argv[])
{
        GList *list;
	poptContext ctx;
	char **startups;
	int i;
        
        active_ggv=0;

        bindtextdomain (PACKAGE, GNOMELOCALEDIR);
        textdomain (PACKAGE);

	gnome_init_with_popt_table("gnome-ghostview", VERSION, argc, argv, NULL, 0, &ctx);

        window_list = NULL;

        load_prefs("/ggv/");

        startups = poptGetArgs(ctx);
        if(startups) {
                for(i = 0; startups[i]; i++)
                        open_window(startups[i]);
        }
	poptFreeContext(ctx);

        if (window_list == NULL)
                open_window(NULL);

	gtk_main ();

        save_prefs("/ggv/");

        return 0;
}

