/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/*
 * application/ps Bonobo bonobo_object.
 *
 * based on text/plain bonobo_object from Nat Friedman  <nat@gnome-support.com>
 *
 * Author:
 *   Jaka Mocnik  <jaka.mocnik@kiss.uni-lj.si>
 */

#include <config.h>
#include <gnome.h>
#include <libgnorba/gnorba.h>
#include <bonobo/gnome-bonobo.h>

#include "gtkgs.h"
#include "prefs.h"

static GdkCursor *pan_cursor = NULL;

CORBA_Environment ev;
CORBA_ORB orb;

/*
 * BonoboObject data
 */
typedef struct {
	GnomeBonoboObject *bonobo_object;

	gchar *tmp_name;

	GList *views;
} bonobo_object_data_t;

/*
 * View data
 */
typedef struct {
	bonobo_object_data_t *bonobo_object_data;

	GtkWidget        *gs;
	GtkObject        *hadj, *vadj;

	gint magstep;
	gboolean pan;
	gdouble prev_x, prev_y;
} view_data_t;

static void
view_destroy_cb (GnomeView *view, view_data_t *view_data)
{
	view_data->bonobo_object_data->views = g_list_remove (view_data->bonobo_object_data->views, view_data);
	gtk_object_unref (GTK_OBJECT (view_data->gs));

	gtk_object_destroy(view_data->hadj);
	gtk_object_destroy(view_data->vadj);

	g_free (view_data);
}

static void
update_all_views (bonobo_object_data_t *bonobo_object_data)
{
	GList *l;
	
	for (l = bonobo_object_data->views; l; l = l->next)
	{
		view_data_t *view_data = l->data;
		GtkGS *gs = GTK_GS(view_data->gs);

		if(bonobo_object_data->tmp_name == NULL ||
		   !gtk_gs_load(gs, bonobo_object_data->tmp_name))
			continue;

		gtk_gs_set_pagemedia(GTK_GS (gs), -1, 0);

		gtk_gs_goto_page ( GTK_GS (gs), 0);
	}
}

static void
stream_read (GNOME_Stream stream, bonobo_object_data_t *bonobo_object_data)
{
	GNOME_Stream_iobuf *buffer;
	CORBA_Environment ev;
	CORBA_long bytes_read, bytes_written;

	FILE *tmp_file;

	if(bonobo_object_data->tmp_name == NULL) {
		bonobo_object_data->tmp_name = g_malloc(256);
		tmpnam(bonobo_object_data->tmp_name);
	}

	buffer = GNOME_Stream_iobuf__alloc ();

#if 1
	/* FIXME: GNOME_Stream_copy_to() is not implemented yet. */
	CORBA_exception_init (&ev);

	GNOME_Stream_copy_to (stream, bonobo_object_data->tmp_name,
						  -1, &bytes_read, &bytes_written, &ev);
	g_print("read %d, written %d\n", bytes_read, bytes_written);
	CORBA_exception_free (&ev);
#else
	tmp_file = fopen(bonobo_object_data->tmp_name, "w");
	if(tmp_file == NULL) {
		g_print("Couldn't create a temporary file");
		return;
	}

	do {
		CORBA_exception_init (&ev);

#define READ_CHUNK_SIZE 65536
		bytes_read = GNOME_Stream_read (stream, READ_CHUNK_SIZE,
										&buffer, &ev);
		CORBA_exception_free (&ev);

		fwrite (buffer->_buffer, buffer->_length, 1, tmp_file);

		CORBA_free (buffer);

	} while (bytes_read > 0);

	fclose(tmp_file);
#endif

   	update_all_views (bonobo_object_data);
}

/*
 * Loads text from a GNOME_Stream
 */
static int
load_ps_from_stream (GnomePersistStream *ps, GNOME_Stream stream,
					   void *data)
{
	bonobo_object_data_t *bonobo_object_data = data;

	stream_read (stream, bonobo_object_data);

	return 0;
}

static void
view_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        view_data_t *v = (view_data_t *)data;

        if(event->button == 1 && !v->pan) {
			gint wx = 0, wy = 0;
			
			gdk_window_get_pointer(widget->window, &wx, &wy, NULL);
			
			v->pan = TRUE;
			if(pan_cursor == NULL)
				pan_cursor = gdk_cursor_new(GDK_FLEUR);
			
			gtk_grab_add(widget);
			gdk_pointer_grab(widget->window, FALSE,
							 GDK_POINTER_MOTION_MASK |
							 GDK_BUTTON_RELEASE_MASK, NULL,
							 pan_cursor, GDK_CURRENT_TIME);
			v->prev_x = wx;
			v->prev_y = wy;
		}
}

static void
view_button_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        view_data_t *v = (view_data_t *)data;

        if(event->button == 1 && v->pan) {
			v->pan = FALSE;
			gdk_pointer_ungrab(GDK_CURRENT_TIME);
			gtk_grab_remove(widget);
		}
}

static void
view_motion_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
        view_data_t *v = (view_data_t *) data;

        if(v->pan) {
                gtk_gs_scroll(GTK_GS(v->gs), -event->x + v->prev_x, -event->y + v->prev_y);;
                v->prev_x = event->x;
                v->prev_y = event->y;
        }
}

static GnomeView *
view_factory (GnomeBonoboObject *bonobo_object, void *data)
{
	GnomeView *view;
	bonobo_object_data_t *bonobo_object_data = data;
	view_data_t *view_data = g_new0 (view_data_t, 1);
	GtkGS *gs;

	view_data->hadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
	view_data->vadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
	view_data->pan = FALSE;
	view_data->magstep = 0; /* 1:1 */

	view_data->gs = gtk_gs_new (GTK_ADJUSTMENT (view_data->hadj),
								GTK_ADJUSTMENT (view_data->vadj));
	gs = GTK_GS(view_data->gs);

	set_prefs(gs);

	gtk_widget_set_events (GTK_WIDGET(gs), 
						   GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
						   GDK_POINTER_MOTION_MASK);
	gtk_signal_connect (GTK_OBJECT (gs), "button_press_event",
						GTK_SIGNAL_FUNC (view_button_press_cb), view_data);
	gtk_signal_connect (GTK_OBJECT (gs), "button_release_event",
						GTK_SIGNAL_FUNC (view_button_release_cb), view_data);
	gtk_signal_connect (GTK_OBJECT (gs), "motion_notify_event",
						GTK_SIGNAL_FUNC (view_motion_cb), view_data);

	gtk_widget_show (view_data->gs);

	view = gnome_view_new (view_data->gs);
	gtk_signal_connect (GTK_OBJECT (view), "destroy",
						GTK_SIGNAL_FUNC (view_destroy_cb), view_data);
	
	bonobo_object_data->views = g_list_prepend (bonobo_object_data->views,
											view_data);
	
	return view;
}

static void
bonobo_object_destroy_cb(GnomeBonoboObject *bonobo_object, bonobo_object_data_t *data)
{
	g_message("destroy");

	if(data->tmp_name) {
		unlink(data->tmp_name);
		g_free(data->tmp_name);
	}
}

static void
verb_next_page(bonobo_object_data_t *data)
{
	GList *v;
	GtkGS *gs;

	v = data->views;
	while(v) {
		gs = GTK_GS(((view_data_t *)v->data)->gs);
		gtk_gs_goto_page(gs, gs->current_page + 1);
		v = v->next;
	}
}

static void
verb_prev_page(bonobo_object_data_t *data)
{
	GList *v;
	GtkGS *gs;

	v = data->views;
	while(v) {
		gs = GTK_GS(((view_data_t *)v->data)->gs);
		gtk_gs_goto_page(gs, gs->current_page - 1);
		v = v->next;
	}
}

static const char *verbs[] = {
	"Next page", "Previous page", NULL,
};

typedef void (*PSVerbCallback)(bonobo_object_data_t *);

static const PSVerbCallback verb_actions[] = {
	verb_next_page,
	verb_prev_page,
	NULL,
};

static void
do_verb_cb(GnomeBonoboObject *bonobo_object, const char *verb_name)
{
	bonobo_object_data_t *data;
	gint i;

	g_print("%s\n", verb_name);
	
	data = gtk_object_get_data(GTK_OBJECT(bonobo_object), "bonobo-object-data");
	if(data)
		for(i = 0; verbs[i] != NULL; i++)
			if(strcmp(verbs[i], verb_name) == 0) {
				verb_actions[i](data);
				break;
			}
}

static GnomeBonoboObject *
bonobo_object_factory (GnomeBonoboObjectFactory *this, const char *path, void *data)
{
	GnomeBonoboObject *bonobo_object;
	GnomePersistStream *stream;
	bonobo_object_data_t *bonobo_object_data = data;

	bonobo_object_data = g_new0 (bonobo_object_data_t, 1);
	if (!bonobo_object_data)
		return NULL;

	bonobo_object_data->tmp_name = NULL;

	/*
	 * Creates the BonoboObject server
	 */
	bonobo_object = gnome_bonobo_object_new (view_factory, bonobo_object_data);
	if (bonobo_object == NULL){
		g_free (bonobo_object_data);
		return NULL;
	}

	/*
	 * Interface GNOME::PersistStream 
	 */
	stream = gnome_persist_stream_new (load_ps_from_stream, NULL,
									   bonobo_object_data);
	if (stream == NULL){
		gtk_object_unref (GTK_OBJECT (bonobo_object));
		g_free (bonobo_object_data);
		return NULL;
	}

	bonobo_object_data->bonobo_object = bonobo_object;

	/*
	 * Add the verbs
	 */
	gnome_bonobo_object_add_verbs(bonobo_object, verbs);

	/*
	 * attach our bonobo_object_data to the GnomeBonoboObject
	 */
	gtk_object_set_data(GTK_OBJECT(bonobo_object), "bonobo-object-data",
						bonobo_object_data);

	/*
	 * Bind the interfaces
	 */
	gtk_object_add_interface (GTK_OBJECT (bonobo_object),
							  GTK_OBJECT (stream));

	gtk_signal_connect(GTK_OBJECT(bonobo_object), "destroy",
					   GTK_SIGNAL_FUNC(bonobo_object_destroy_cb), bonobo_object_data);
	gtk_signal_connect(GTK_OBJECT(bonobo_object), "do_verb",
					   GTK_SIGNAL_FUNC(do_verb_cb), bonobo_object_data);

	return bonobo_object;
}

static void
init_bonobo_application_ps_factory (void)
{
	GnomeBonoboObjectFactory *factory;
	
	factory = gnome_bonobo_object_factory_new (
		"bonobo-object-factory:application-ps",
		bonobo_object_factory, NULL);
}

static void
init_server_factory (int argc, char **argv)
{
	gnome_CORBA_init_with_popt_table (
		"bonobo-application-ps", VERSION,
		&argc, argv, NULL, 0, NULL, GNORBA_INIT_SERVER_FUNC, &ev);

	orb = gnome_CORBA_ORB ();
	if (bonobo_init (orb, NULL, NULL) == FALSE)
		g_error (_("I could not initialize Bonobo"));
}

int
main (int argc, char **argv)
{

	CORBA_exception_init (&ev);

	init_server_factory (argc, argv);
	init_bonobo_application_ps_factory ();

	load_prefs("/bonobo-application-ps/");

	gtk_main ();

	CORBA_exception_free (&ev);

	return 0;
}
