/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/* Ghostscript widget for GTK/GNOME
 * Copyright (C) 1998 the Free Software Foundation
 * Author: Jonathan Blandford <jrb@redhat.com>
 * Based on code by: Federico Mena (Quartic), Szekeres Istvan (Pista)
 */

#include "config.h"
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#include <X11/Intrinsic.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include "gtkgs.h"
#include "file.h"

#define SIGVAL void

#define RES 75
typedef void (* GtkGSSignal0) (GtkObject *object, gpointer data);
enum { FOO, LAST_SIGNAL };
static Boolean broken_pipe = False;

static SIGVAL catchPipe(int i)
{
        g_print ("in catch Pipe!!!\n");
        broken_pipe = True;
#ifdef SIGNALRETURNSINT
        g_print ("in catch Pipe!!! - returning 0\n");
        return 0;
#endif
}



/* Forward declarations */
static void gtk_gs_init (GtkGS *gs);
static void gtk_gs_class_init (GtkGSClass *klass);
static void gtk_gs_destroy (GtkObject *object);
static void gtk_gs_realize (GtkWidget *widget);
static void gtk_gs_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static gint gtk_gs_widget_event(GtkWidget *widget, GdkEvent *event, gpointer data);
static void gtk_gs_value_adjustment_changed (GtkAdjustment *adjustment, gpointer data);
static void gtk_gs_marshal_signal_0 (GtkObject *object, GtkSignalFunc func, gpointer func_data, GtkArg *args);
static void send_ps(GtkGS *gs, long begin, unsigned int len, gboolean close);
static void set_up_page (GtkGS *gs);
static void close_pipe (int p[2]);
static void interpreter_failed (GtkGS *gs);
static float compute_xdpi();
static float compute_ydpi();
static gboolean compute_size(GtkGS *gs);
static void output (gpointer data, gint source, GdkInputCondition condition);
static void input (gpointer data, gint source, GdkInputCondition condition);


static void stop_interpreter (GtkGS *gs);
static gint start_interpreter (GtkGS *gs);
gboolean computeSize(void);


static GtkWidgetClass *parent_class;
static GtkGSClass *gs_class;
static gint ghostview_signals[LAST_SIGNAL] = { 0 };



/* Static, private functions */

static void
gtk_gs_init (GtkGS *gs)
{
        gs->bpixmap     = NULL;
        gs->use_bpixmap = TRUE;

        gs->disable_start   = FALSE;
        gs->interpreter_pid = -1;

        gs->width  = -1;
        gs->height = -1;
        gs->busy      = FALSE;
        gs->changed   = FALSE;
        gs->gs_gs_cmd_scan_pdf = "gs -dNODISPLAY -dQUIET -sPDFname=%s -sDSCname=%s pdf2dsc.ps -c quit";
        gs->gs_uncompress_command = "gzip -cd %s > %s";
        gs->gs_scanstyle = 0;
	gs->gs_filename_raw = 0;
	gs->gs_filename_dsc = 0;
	gs->gs_filename_unc = 0;
        gs->gs_interpreter = GS_PATH;

        broken_pipe = FALSE;
        /* FIXME: */
        gs->structured_doc = TRUE; 

	gs->interpreter_input = -1;
	gs->interpreter_output = -1;
	gs->interpreter_err = -1;
	gs->interpreter_input_id = 0;
	gs->interpreter_output_id = 0;
	gs->interpreter_error_id = 0;


        gs->ps_input = NULL;
        gs->input_buffer = NULL;
        gs->input_buffer_ptr = NULL;
        gs->bytes_left = 0;
        gs->buffer_bytes_left = 0;

        gs->llx = 0;
        gs->lly = 0;
        gs->urx = 0;
        gs->ury = 0;
        gs->xdpi = compute_xdpi();
        gs->ydpi = compute_ydpi();

        gs->left_margin = 0;
        gs->top_margin = 0;
        gs->right_margin = 0;
        gs->bottom_margin = 0;
        gs->orientation = GTK_GS_ORIENTATION_DEFAULT;
        gs->zoom_factor = 1.0;
        gs->default_page_media = 1;
        gs->antialiased = TRUE;
        gs->override_media = FALSE;
}

static void
gtk_gs_class_init (GtkGSClass *klass)
{
        GtkObjectClass *object_class;
        GtkWidgetClass *widget_class;

        object_class    = (GtkObjectClass *) klass;
        widget_class    = (GtkWidgetClass *) klass;
        parent_class    = gtk_type_class(gtk_widget_get_type());
        gs_class = klass;

        object_class->destroy = gtk_gs_destroy;

        widget_class->realize       = gtk_gs_realize;
        widget_class->size_allocate = gtk_gs_size_allocate;
        /* Create atoms */
        klass->gs_atom        = gdk_atom_intern ("GHOSTVIEW", FALSE);
        klass->gs_colors_atom = gdk_atom_intern ("GHOSTVIEW_COLORS", FALSE);
        klass->next_atom      = gdk_atom_intern ("NEXT", FALSE);
        klass->page_atom      = gdk_atom_intern ("PAGE", FALSE);
        klass->done_atom      = gdk_atom_intern ("DONE", FALSE);
        klass->string_atom    = gdk_atom_intern ("STRING", FALSE);
}


static void
gtk_gs_destroy (GtkObject *object)
{
        GtkGS *gs;
  
        g_return_if_fail (object != NULL);
        g_return_if_fail (GTK_IS_GS (object));
        
        gs = GTK_GS (object);

        stop_interpreter(gs);
  
        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
gtk_gs_realize (GtkWidget *widget)
{
        GtkGS  *gs;
        GdkWindowAttr  attributes;
        gint           attributes_mask;

        g_return_if_fail (widget != NULL);
        g_return_if_fail (GTK_IS_GS (widget));

        gs = GTK_GS (widget);

        /* we set up the main widget! */
        GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
        attributes.window_type = GDK_WINDOW_CHILD;
        attributes.x           = widget->allocation.x;
        attributes.y           = widget->allocation.y;
        attributes.width       = widget->allocation.width;
        attributes.height      = widget->allocation.height;
        attributes.wclass      = GDK_INPUT_OUTPUT;
        attributes.visual      = gtk_widget_get_visual (widget);
        attributes.colormap    = gtk_widget_get_colormap (widget);
        attributes.event_mask  = gtk_widget_get_events (widget);
        attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

        widget->window = gdk_window_new (widget->parent->window, &attributes, attributes_mask);
        gdk_window_set_user_data (widget->window, gs);
        widget->style = gtk_style_attach (widget->style, widget->window);
        gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);


        /* now we set up the child window.  This is the one that ps actually draws too. */
        attributes.x = 0;
        attributes.y = 0;

        gs->pstarget = gdk_window_new (widget->window, &attributes, attributes_mask);
        gdk_window_set_user_data (gs->pstarget, widget);
        gdk_window_show (gs->pstarget);
        gdk_window_clear (gs->pstarget);
        gtk_style_set_background (widget->style, 
                                  gs->pstarget,
                                  GTK_STATE_ACTIVE);

        gs->width = attributes.width;
        gs->height = attributes.height;
        /* XXX: how to handle X BadAlloc and stuff? */
        gtk_signal_connect(GTK_OBJECT(widget), "event",
                           GTK_SIGNAL_FUNC(gtk_gs_widget_event), gs);
}

static void
gtk_gs_size_allocate (GtkWidget     *widget,
                      GtkAllocation *allocation)
{
        GtkGS *gs = GTK_GS (widget);
        GdkEventConfigure event;
        gint x, y;
        
        g_return_if_fail (widget != NULL);
        g_return_if_fail (GTK_IS_GS (widget));
        g_return_if_fail (allocation != NULL);

        widget->allocation = *allocation;
        
        if (GTK_WIDGET_REALIZED (widget)) {
                gdk_window_move_resize (widget->window,
                                        allocation->x, allocation->y,
                                        allocation->width, allocation->height);

                event.type = GDK_CONFIGURE;
                event.window = widget->window;
                event.x = allocation->x;
                event.y = allocation->y;
                event.width = allocation->width;
                event.height = allocation->height;

                gtk_widget_event (widget, (GdkEvent*) &event);
        }
        /* 
         * update the adjustment if necessary (ie. a resize);
         */
        if ((gs->width > 0) && (gs->height >0)) {
                gdk_window_get_position (gs->pstarget, &x, &y);

                /* 
                 * This is a bit messy:
                 * we want to make sure that we do the right thing if dragged.
                 */

                if (gs->widget.allocation.width >= gs->width) {
                        x = (gs->widget.allocation.width - gs->width)/2;
                        gs->hadj->value = (gs->hadj->upper - gs->hadj->lower)/2;
                        gs->hadj->page_size = gs->hadj->upper - gs->hadj->lower;
                } else {
                        if (x > 0)
                                x = 0;
                        else if (gs->widget.allocation.width > x + gs->width) 
                                x = gs->widget.allocation.width - gs->width;
                        gs->hadj->page_size = ((gfloat) gs->widget.allocation.width)/gs->width;
                        gs->hadj->value = (gs->hadj->upper-gs->hadj->lower) *
                                ((gfloat) (gs->widget.allocation.width/2 - x))/gs->width;
                }
                if (gs->widget.allocation.height >= gs->height) {
                        y = (gs->widget.allocation.height - gs->height)/2;
                        gs->vadj->value = (gs->vadj->upper - gs->vadj->lower)/2;
                        gs->vadj->page_size = gs->vadj->upper - gs->vadj->lower;
                } else {
                        if (y > 0)
                                y = 0;
                        else if (gs->widget.allocation.height > y + gs->height) 
                                y = gs->widget.allocation.height - gs->height;
                        gs->vadj->page_size = ((gfloat) gs->widget.allocation.height)/gs->height;
                        gs->vadj->value = (gs->vadj->upper-gs->vadj->lower) *
                                ((gfloat) (gs->widget.allocation.height/2 - y))/gs->height;
                }

                /* now we want to update the slider. */
                gdk_window_move (gs->pstarget, x, y);
                gtk_adjustment_changed (gs->hadj);
                gtk_adjustment_changed (gs->vadj);
        }

}

static gboolean 
gtk_gs_widget_event(GtkWidget *widget,
                    GdkEvent *event,
                    gpointer data)
{
        GtkGS *gs = (GtkGS *)data;
        
	if (event->type != GDK_CLIENT_EVENT) 
                return FALSE;
	
	if (event->client.message_type == gs_class->page_atom) {
                gs->busy = FALSE;
	}
        return TRUE;
}
static void
gtk_gs_value_adjustment_changed (GtkAdjustment *adjustment, gpointer data)
{
        GtkGS *gs;
        gint x, y, width, height, depth;
        gint newx, newy;
        
        g_return_if_fail (adjustment != NULL);
        g_return_if_fail (data != NULL);
        gs = GTK_GS (data);
        if (gs->bpixmap == NULL)
                return;
        
        if (adjustment->value + adjustment->page_size/2 > adjustment->upper)
                g_print ("value = %f\npage_size = %f\nupper = %f\n", adjustment->value,adjustment->page_size,adjustment->upper);
        gdk_window_get_geometry (gs->pstarget, &x, &y, &width, &height, &depth);
        if (gs->width <= gs->widget.allocation.width)
                newx = (gs->widget.allocation.width-gs->width)/2;
        else
                newx = gs->widget.allocation.width/2 -
                        (gs->hadj->value - gs->hadj->lower) * width /
                        (gs->hadj->upper - gs->hadj->lower);
        if (gs->height <= gs->widget.allocation.height)
                newy = (gs->widget.allocation.height-gs->height)/2;
        else
                newy = gs->widget.allocation.height/2 -
                        (gs->vadj->value - gs->vadj->lower) * height /
                        (gs->vadj->upper - gs->vadj->lower);
        gdk_window_move (gs->pstarget, newx, newy);
}

static void
gtk_gs_marshal_signal_0 (GtkObject     *object,
                         GtkSignalFunc  func,
                         gpointer       func_data,
                         GtkArg        *args)
{
        /*        GtkGSSignal0 rfunc;*/
        /*        rfunc = (GtkGSSignal0) func;*/
        
        /*        (* rfunc) (object, func_data);*/
}

void 
gtk_gs_center_page(GtkGS *gs)
{
	g_return_if_fail (gs != NULL);

        gdk_window_move (gs->pstarget,
                         (gs->widget.allocation.width - gs->width)/2,
                         (gs->widget.allocation.height - gs->height)/2);
        gs->hadj->page_size = ((gfloat) gs->widget.allocation.width)/gs->width;
        gs->vadj->page_size = ((gfloat) gs->widget.allocation.height)/gs->height;
        gs->hadj->value = (gs->hadj->upper - gs->hadj->lower)/2;
        gs->vadj->value = (gs->vadj->upper - gs->vadj->lower)/2;
        gtk_adjustment_changed (gs->hadj);
        gtk_adjustment_changed (gs->vadj);
}

static void
send_ps(GtkGS *gs, long begin, unsigned int len, gboolean close)
{
        struct record_list *ps_new;
        
        if (gs->interpreter_input < 0) {
                g_print ("no pipe to gs\nerror in send_ps\n");
                return;
        }

        ps_new = (struct record_list *) g_malloc(sizeof (struct record_list));
        ps_new->fp = gs->gs_psfile;
        ps_new->begin = begin;
        ps_new->len = len;
        ps_new->seek_needed = TRUE;
        ps_new->close = close;
        ps_new->next = NULL;
        
        if (gs->input_buffer == NULL) {
                gs->input_buffer = g_malloc(MAX_BUFSIZE);
        }
        
        if (gs->ps_input == NULL) {
                gs->input_buffer_ptr = gs->input_buffer;
                gs->bytes_left = len;
                gs->buffer_bytes_left = 0;
                gs->ps_input = ps_new;
                gs->interpreter_input_id = 
                        gdk_input_add(gs->interpreter_input,
                                      GDK_INPUT_WRITE,
                                      input,
                                      gs);
        } else {
                struct record_list *p = gs->ps_input;
                while (p->next != NULL) {
                        p = p->next;
                }
                p->next = ps_new;
        }
}

static void
set_up_page (GtkGS *gs) 
     /* 
      * This is used to prepare the widget internally for
      * a new document. It sets gs->pstarget to the
      * correct size and position, and updates the 
      * adjustments appropriately.
      *
      * It is not meant to be used every time a specific page
      * is selected.
      *
      * NOTE: It expects the widget is realized.
      */
{ 
        guint orientation;
        char               buf[1024];
        GdkPixmapPrivate  *pprivate;

        stop_interpreter(gs);
        if (gs->doc == NULL) {
                /* 
                 * FIXME:  need to support uncommented files
                 */
                g_print ("ERROR: gs->doc is null!!!\n");
                /* create_default_page (gs); */
                return;
        }
        
        if (gs->orientation == GTK_GS_ORIENTATION_DEFAULT)
                /* FIXME: hmmmm. many things don't seem to set their
                   default orientation.... */
                /*orientation = gs->doc->default_page_orientation;*/
                orientation = GTK_GS_ORIENTATION_PORTRAIT;
        else
                orientation = gs->orientation;
        /* do we need to make a new pixmap??? */
        if (!gs->use_bpixmap) {
                
        }else if ((gs->use_bpixmap) && compute_size (gs)) {
                if (gs->bpixmap) {
                        gdk_pixmap_unref(gs->bpixmap);
                        gs->bpixmap = NULL;
                }
                gs->bpixmap = gdk_pixmap_new(gs->pstarget, gs->width, gs->height, -1);
                gdk_window_set_back_pixmap(gs->pstarget, gs->bpixmap, FALSE);
        }

        gdk_window_resize (gs->pstarget, gs->width, gs->height);
        
        pprivate = (GdkPixmapPrivate *) gs->bpixmap;
        snprintf(buf, 1024, "%ld %d %d %d %d %d %f %f %d %d %d %d",
                 (pprivate ? pprivate->xwindow : 0L),
                 orientation,
                 gs->llx,gs->lly,
                 gs->urx,gs->ury,
                 gs->xdpi * gs->zoom_factor,
                 gs->ydpi * gs->zoom_factor,
                 gs->left_margin,
                 gs->bottom_margin,
                 gs->right_margin,
                 gs->top_margin);
        
        gdk_property_change(gs->pstarget,
                            gs_class->gs_atom,
                            gs_class->string_atom,
                            8,
                            GDK_PROP_MODE_REPLACE,
                            buf,
                            strlen(buf));
        gtk_gs_center_page (gs);
}

static void
close_pipe(int p[2])
{
	close(p[0]);
	close(p[1]);
}

static gboolean
is_interpreter_ready(GtkGS *gs)
{
        return (gs->interpreter_pid != -1 &&
                !gs->busy &&
                gs->ps_input == NULL);
}

/* GhostviewIsInterpreterRunning:
 * Returns True if the interpreter is running.
 */
static gboolean
is_interpreter_running(GtkGS *gs)
{
    return (gs->interpreter_pid != -1);
}

static void
interpreter_failed (GtkGS *gs)
{
        stop_interpreter (gs);
}

static void
output (gpointer data, gint source, GdkInputCondition condition)
{
        char buf[MAX_BUFSIZE+1];
        guint bytes = 0;
        GtkGS *gs = GTK_GS (data);
        
        if (source == gs->interpreter_output) {
                bytes = read(gs->interpreter_output, buf, MAX_BUFSIZE);
                if (bytes == 0) { /* EOF occurred */
                        close (gs->interpreter_output);
                        gs->interpreter_output = -1;
                        gdk_input_remove (gs->interpreter_output_id);
                        return;
                } else if (bytes == -1) {
                        /* trouble... */
                        interpreter_failed(gs);
                        return;
                }
        } else if (source == gs->interpreter_err) {
                bytes = read(gs->interpreter_err, buf, MAX_BUFSIZE);
                if (bytes == 0) { /* EOF occurred */
                        close(gs->interpreter_err);
                        gs->interpreter_err = -1;
                        gdk_input_remove (gs->interpreter_error_id);
                        return;
                } else if (bytes == -1) {
                        /* trouble... */
                        interpreter_failed(gs);
                        return;
                }
        }
        if (bytes > 0) {
                buf[bytes] = '\0';
                /* 
                 * FIXME:  er, we actually need to do something with 
                 * the message...
                 */
                g_print ("we got a message from ghostscript:\n");
                g_print ("*****\n\t%s\n*****\n", buf);
        }
}

static void
input (gpointer data, gint source, GdkInputCondition condition)
{
        GtkGS *gs = GTK_GS (data);
        int bytes_written;
	SIGVAL (*oldsig)(int);
	oldsig = signal(SIGPIPE, catchPipe);
                
        do {
                if (gs->buffer_bytes_left == 0) {
                        /* Get a new section if required */
                        if (gs->ps_input && gs->bytes_left == 0) {
                                struct record_list *ps_old = gs->ps_input;
                                gs->ps_input = ps_old->next;
                                if (ps_old->close) 
                                        fclose(ps_old->fp);
                                g_free((char *)ps_old);
                        }
                        /* Have to seek at the beginning of each section */
                        if (gs->ps_input && gs->ps_input->seek_needed) {

                                        fseek(gs->ps_input->fp,
                                              gs->ps_input->begin, SEEK_SET);
                                gs->ps_input->seek_needed = FALSE;
                                gs->bytes_left = gs->ps_input->len;
                        }

                        if (gs->bytes_left > MAX_BUFSIZE) {
                                gs->buffer_bytes_left =
                                        fread(gs->input_buffer,
                                              sizeof (char), MAX_BUFSIZE,
                                              gs->ps_input->fp);
                        } else if (gs->bytes_left > 0) {
                                gs->buffer_bytes_left =
                                        fread(gs->input_buffer,
                                              sizeof (char), gs->bytes_left,
                                              gs->ps_input->fp);
                        } else {
                                gs->buffer_bytes_left = 0;
                        }
                        if (gs->bytes_left > 0 && gs->buffer_bytes_left == 0) {
                                interpreter_failed(gs);	/* Error occurred */
                        }
                        gs->input_buffer_ptr = gs->input_buffer;
                        gs->bytes_left -= gs->buffer_bytes_left;
                }

                if (gs->buffer_bytes_left > 0) {
                        /*                        g_print (" writing: %s\n",gs->input_buffer_ptr);*/
                        
                        bytes_written = write(gs->interpreter_input,
                                              gs->input_buffer_ptr,
                                              gs->buffer_bytes_left);

                        if (broken_pipe) {
                                g_print ("the pipe is broken\n");
                                broken_pipe = FALSE;
                                interpreter_failed(gs);
                        } else if (bytes_written == -1) {
                                if ((errno != EWOULDBLOCK) && (errno != EAGAIN)) {
                                        interpreter_failed(gs);	/* Something bad happened */
                                }
                        } else {
                                gs->buffer_bytes_left -= bytes_written;
                                gs->input_buffer_ptr += bytes_written;
                        }
                }
        } while(gs->ps_input &&
                gs->buffer_bytes_left == 0);
        
        signal(SIGPIPE, oldsig);

        if (gs->ps_input == NULL && gs->buffer_bytes_left == 0) {
                if (gs->interpreter_input_id != 0) {
                        gdk_input_remove(gs->interpreter_input_id);
                        gs->interpreter_input_id = 0;
                }
        }
}


static int
start_interpreter (GtkGS *gs)
{
	int std_in[2]; /* pipe to interp stdin */
	int std_out[2]; /* pipe from interp stdout */
	int std_err[2]; /* pipe from interp stderr */
	char buffer[1024];
#define NUM_ARGS 100
        char *argv[NUM_ARGS];
        int argc = 0;

        if (!gs->gs_filename) 
                return 0;

	stop_interpreter(gs);
        
        /* FIXME: we need to clear the window first... */

	if (gs->disable_start == TRUE)
                return 0;
        
        /* set up the args... */
        argv[argc++] = gs->gs_interpreter;
        if( gs->antialiased ) 
		argv[argc++] = "-sDEVICE=x11alpha";
        else 
		argv[argc++] = "-sDEVICE=x11";
        argv[argc++] = "-dNOPAUSE";
        argv[argc++] = "-dQUIET";
        /* I assume we do _not_ want to change this... (: */
        argv[argc++] = "-dSAFER";
        argv[argc++] = "-";
        argv[argc++] = NULL;


        /* set up the pipes */
        if (pipe (std_in) == -1)
		exit (1);
	if (pipe (std_out) == -1) {
		close_pipe(std_in);
		return -1;
	}
	if (pipe (std_err) == -1) {
		close_pipe(std_in);
		close_pipe(std_out);
		return -1;
	}

        gs->busy = TRUE;
	gs->interpreter_pid = fork();
	switch (gs->interpreter_pid) {
	case -1:
		close_pipe(std_in);
		close_pipe(std_out);
		close_pipe(std_err);
		return -2;
		break;
	case 0: /* child */
		close(std_out[0]);
		close(std_err[0]);
                dup2(std_out[1], 1);
                close(std_out[1]);
                dup2(std_err[1], 2);
                close (std_err[1]);
                
                sprintf(buffer, "GHOSTVIEW=%ld",
                        ((GdkWindowPrivate*)(gs->pstarget))->xwindow);
                
		putenv(g_strdup(buffer));
                
                if (gs->gs_filename == NULL) {
                        close(std_in[1]);
                        dup2(std_in[0], 0);
                        close(std_in[0]);
                } else if (strcmp(gs->gs_filename, "-")) {
                        dup2(std_in[0], 0);
                        close(std_in[0]);
                }

                execvp(argv[0], argv);
		_exit(1);
		break;
	default: /* parent */
                if (gs->gs_filename == NULL) {
#ifdef NON_BLOCKING_IO
                        int result;
#endif
                        close(std_in[0]);

#ifdef NON_BLOCKING_IO
                        result = fcntl(std_in[1], F_GETFL, 0);
                        result = result | O_NONBLOCK;
                        result = fcntl(std_in[1], F_SETFL, result);
#endif
                        gs->interpreter_input = std_in[1];
                        gs->interpreter_input_id = 0;
                } else if (strcmp(gs->gs_filename, "-")) {
                        close(std_in[0]);
                }
                close (std_out[1]);
		gs->interpreter_input = std_in[1];
		gs->interpreter_output = std_out[0];
		gs->interpreter_err = std_err[0];
                close (std_err[1]);
                gs->interpreter_output_id =
                        gdk_input_add(std_out[0],
                                      GDK_INPUT_READ,
                                      output,
                                      gs);
                gs->interpreter_error_id = 
                        gdk_input_add(std_err[0],
                                      GDK_INPUT_READ,
                                      output,
                                      gs);
		break;
	}
        return TRUE;
}

static void
stop_interpreter (GtkGS *gs)
{
        if (gs->interpreter_pid >= 0) {
                kill(gs->interpreter_pid, SIGTERM);
                wait(0);
		gs->interpreter_pid = -1;
	}

	if (gs->interpreter_input >= 0) {
		close(gs->interpreter_input);
		gs->interpreter_input = -1;
                if (gs->interpreter_input_id != 0) {
                        gdk_input_remove(gs->interpreter_input_id);
                        gs->interpreter_input_id = 0;
                }
                while (gs->ps_input) {
                        struct record_list *ps_old = gs->ps_input;
                        gs->ps_input = gs->ps_input->next;
                        if (ps_old->close)
                                fclose (ps_old->fp);
                        g_free ((char *)ps_old);
                }
	}

	if (gs->interpreter_output >= 0) {
		close(gs->interpreter_output);
		gs->interpreter_output = -1;
                if (gs->interpreter_output_id) {
                        gdk_input_remove(gs->interpreter_output_id);
                        gs->interpreter_input_id = 0;
                }
	}
	
	if (gs->interpreter_err >= 0) {
		close(gs->interpreter_err);
		gs->interpreter_err = -1;
                if (gs->interpreter_error_id) {
                        gdk_input_remove(gs->interpreter_error_id);
                        gs->interpreter_input_id = 0;
                }
	}

	gs->busy = FALSE;
}

#if 0
static void
Message(GtkGS *gs, Xevent event)
{
        gvw->ghostview.mwin = event->xclient.data.l[0];
    if (event->xclient.message_type ==
	XmuInternAtom(XtDisplay(w), gvc->ghostview_class.page)) {
	gvw->ghostview.busy = False;
	XDefineCursor(XtDisplay(gvw), XtWindow(gvw), gvw->ghostview.cursor);
	XtCallCallbackList(w, gvw->ghostview.message_callback, "Page");
    } else if (event->xclient.message_type ==
	       XmuInternAtom(XtDisplay(w), gvc->ghostview_class.done)) {
	StopInterpreter(w);
	XtCallCallbackList(w, gvw->ghostview.message_callback, "Done");
    }
}

#endif

/* publicly accessible functions */

guint
gtk_gs_get_type (void)
{
        static guint gs_type = 0;

        if (!gs_type) {
                GtkTypeInfo gs_info = { "GtkGS",
                                        sizeof (GtkGS),
                                        sizeof (GtkGSClass),
                                        (GtkClassInitFunc) gtk_gs_class_init,
                                        (GtkObjectInitFunc) gtk_gs_init,
                                        (GtkArgSetFunc) NULL,
                                        (GtkArgGetFunc) NULL };
                gs_type = gtk_type_unique (gtk_widget_get_type (), &gs_info);
        }

        return gs_type;
}

GtkWidget *
gtk_gs_new (GtkAdjustment *hadj, GtkAdjustment *vadj)
{
	GtkGS *gs = gtk_type_new(gtk_gs_get_type());
        gs->hadj = hadj;
        gs->vadj = vadj;

        /* we set the adjustment widget to a default size... */
        hadj->lower = 0.0;
        hadj->upper = 1.0;
        hadj->value = 0.5;
        hadj->page_size = 1.0;
        vadj->lower = 0.0;
        vadj->upper = 1.0;
        vadj->value = 0.5;
        vadj->page_size = 1.0;

        gtk_signal_connect (GTK_OBJECT (hadj), "value_changed",
                            (GtkSignalFunc) gtk_gs_value_adjustment_changed,
                            (gpointer) gs);
        gtk_signal_connect (GTK_OBJECT (vadj), "value_changed",
                            (GtkSignalFunc) gtk_gs_value_adjustment_changed,
                            (gpointer) gs);


	return GTK_WIDGET(gs);
}


GtkWidget *
gtk_gs_new_from_file(GtkAdjustment *hadj, GtkAdjustment *vadj, char *fname)
{
	GtkWidget *gs = gtk_gs_new(hadj, vadj);
	gtk_gs_load(GTK_GS(gs), fname);
	return gs;
}

void
gtk_gs_load (GtkGS *gs, char *fname)
{
        struct stat sbuf;
	
	g_assert(gs != NULL);
        printf (" in gs_load\n");
	if (fname == NULL) {
                /* clear drawing area? */
	} else {
                /*
                 * We need to make sure that the file is loadable/exists!
                 * otherwise we want to exit without loading new stuff...
                 */


		/* clean up previous document */
                gtk_gs_disable_interpreter(gs);
		if (gs->gs_filename)
                        g_free(gs->gs_filename);
		if (gs->gs_filename_raw)
                        g_free(gs->gs_filename_raw);
                if (gs->doc)
                        psfree(gs->doc);
                gs->gs_filename_dsc = NULL;
                gs->gs_filename_unc = NULL;
		gs->doc = NULL;

		/* prepare this document */
		gs->gs_psfile = fopen(fname, "r");
                gs->gs_filename = g_strdup(fname);
                printf (" filename is: %s\n", gs->gs_filename);
		gs->gs_filename_raw = g_strdup(fname);
                gs->current_page = -1;

                /* why do we have this??? */
                gs->gs_filename_raw = file_getUsefulName(gs->gs_filename_raw);
		
                /* we grab the vital statistics!!! */
                gs->doc = psscan(&gs->gs_psfile, fname,
                                 "/tmp/",
                                 &gs->gs_filename_dsc, gs->gs_gs_cmd_scan_pdf,
                                 &gs->gs_filename_unc, gs->gs_uncompress_command);
                stat(gs->gs_filename, &sbuf);
                gs->mtime = sbuf.st_mtime;
	}
}


gboolean
gtk_gs_next_page(GtkGS *gs)
{
	XEvent event;
	GdkWindowPrivate *pw;
	
	g_return_val_if_fail (gs != NULL, FALSE);

	if (gs->interpreter_pid == 0) { /* no interpreter active */ 
                g_print ("pid = 0\n");
		return FALSE;
        }

	if (gs->busy) { /* interpreter is busy */
                g_print ("int busy\n");
		return FALSE;
        }

	gs->busy = TRUE;
	pw = (GdkWindowPrivate *)gs->widget.window;

	event.xclient.type          = ClientMessage;
	event.xclient.display       = gdk_display;
	event.xclient.window        = pw->xwindow;
	event.xclient.message_type  = gs_class->next_atom;
	event.xclient.format        = 32;
        g_print ("sending event.");
	gdk_send_xevent(pw->xwindow, FALSE, 0, &event);
        g_print ("..sent\n");
        gdk_flush();
	return TRUE;
}

gboolean
gtk_gs_goto_page (GtkGS *gs, gint page) 
{
        struct stat sbuf;
	g_assert (gs != NULL);

        if (!gs->gs_filename) {
                g_print ("filename is null???\n");
                return FALSE;
        }
        if (!gs->structured_doc)
                return FALSE;
        g_assert (GTK_WIDGET_REALIZED (gs));
        
        if (gs->gs_psfile) {
        /* has it changed? */
                if (!stat(gs->gs_filename, &sbuf) && gs->mtime != sbuf.st_mtime) {
                        g_print ("file has changed!!!\n");
                        fclose(gs->gs_psfile);
                        gtk_gs_load (gs, gs->gs_filename);
                }
        }

        /* range checking... */
        if (page < 0)
                page = 0;
        if (page >= gs->doc->numpages)
                page = gs->doc->numpages - 1;
        
        if (page == gs->current_page && !gs->changed)
                  return TRUE;

        gs->changed = FALSE;
        gs->current_page = page;

        
        /* BIG BIG BIG FIXME!!!!! */
        /* 
         * For some reason, ghost script is unable to accept new input in
         * the next section.  So, we stop the interpreter and restart it.
         * I will look for this error later on!
         */
        stop_interpreter (gs);
        /* End truly bad hack */

        if (gs->structured_doc) {
                if (is_interpreter_ready (gs)) {
                        /*g_print ("the interpreter is ready!\n");*/
                        gtk_gs_next_page (gs);
                } else {
                        /*g_print ("the interpreter is not ready -- starting now\n");*/
                        gtk_gs_enable_interpreter (gs);
                        send_ps(gs, gs->doc->beginprolog, 
                                gs->doc->lenprolog, FALSE);
                        send_ps(gs, gs->doc->beginsetup, 
                                gs->doc->lensetup, FALSE);
                }
		
                send_ps(gs, gs->doc->pages[gs->current_page].begin,
                        gs->doc->pages[gs->current_page].len, FALSE);
        }        
        /* FIXME: */
        /* handle non-structured docs, too... */
        return TRUE;
}

/*
 * set pagemedia sets the media from
 * if set_override is TRUE, then override_media is set to true.
 * if set_default is TRUE, then the pagemedia used is set to be the default one.
 * if new_pagemedia is -1, then it is set to either
 *  a) the default settings of pageid, if they exist, or if pageid != -1.
 *  b) the default setting of the document, if it exists.
 *  c) the default setting of the widget.
 * otherwise, the new_pagemedia is used as the pagemedia
 */
gboolean
gtk_gs_set_pagemedia(GtkGS *gs, gint new_pagemedia, gboolean set_default, gboolean set_override, gint pageid)
{
        gint new_llx;
        gint new_lly;
        gint new_urx;
        gint new_ury;
        gboolean changed = FALSE;
        gboolean from_doc = FALSE;

	g_assert (gs != NULL);

        if (set_override)
                gs->override_media = TRUE;
        if (new_pagemedia == -1) {
                if (gs->doc) {
                        if ((pageid >= 0) && (gs->doc->pages[pageid].media != NULL)) {
                                new_pagemedia = gs->doc->pages[pageid].media - gs->doc->media;
                                from_doc = TRUE;
                        } else if (gs->doc->default_page_media != NULL) {
                                new_pagemedia = gs->doc->default_page_media - gs->doc->media;
                                from_doc = TRUE;
                        } else {
                                new_pagemedia = gs->default_page_media;
                        }
                } else {
                        new_pagemedia = gs->default_page_media;
                }
        }
        if (set_default) {
                gs->default_page_media = new_pagemedia;
        }
        
        /* Compute bounding box */
        if (gs->doc && gs->doc->epsf 
            && (gs->doc->boundingbox[URX] > gs->doc->boundingbox[LLX]) &&
            (gs->doc->boundingbox[URY] > gs->doc->boundingbox[LLY])) {
                new_llx = gs->doc->boundingbox[LLX];
                new_lly = gs->doc->boundingbox[LLY];
                new_urx = gs->doc->boundingbox[URX];
                new_ury = gs->doc->boundingbox[URY];
        } else {

                new_llx = new_lly = 0;
                if (new_pagemedia < gs->doc->nummedia) {
                        new_urx = gs->doc->media[new_pagemedia].width;
                        new_ury = gs->doc->media[new_pagemedia].height;
                } else {
                        new_urx = papersizes[new_pagemedia-gs->doc->nummedia].width;
                        new_ury = papersizes[new_pagemedia-gs->doc->nummedia].height;
                }
        }

        /* If bounding box changed, setup for new size. */
        if ((new_llx != gs->llx) || (new_lly != gs->lly) ||
            (new_urx != gs->urx) || (new_ury != gs->ury)) {
                //                gtk_gs_disable_interpreter (gs);
                gs->llx = new_llx;
                gs->lly = new_lly;
                gs->urx = new_urx;
                gs->ury = new_ury;
                changed = TRUE;
                g_print ("setting up the page\n");

                set_up_page(gs);
        }
        g_print ("\tall done.\n");
        return changed;
}

gboolean
gtk_gs_set_orientation (GtkGS *gs, gint orientation) {
	g_assert (gs != NULL);
        g_assert ((orientation == GTK_GS_ORIENTATION_DEFAULT) ||
                  (orientation == GTK_GS_ORIENTATION_PORTRAIT) ||
                  (orientation == GTK_GS_ORIENTATION_LANDSCAPE) ||
                  (orientation == GTK_GS_ORIENTATION_UPSIDEDOWN) ||
                  (orientation == GTK_GS_ORIENTATION_SEASCAPE));
        if (gs->orientation != orientation) { 
                gs->orientation = orientation;
                gs->changed = TRUE;
                set_up_page(gs);
                return TRUE;
        }
        return FALSE;
}

void
gtk_gs_set_zoom (GtkGS *gs, gfloat zoom) {
	g_assert (gs != NULL);

        g_print ("zoom factor is %f\n",zoom);
        
        if (gs->zoom_factor != zoom) { 
                gs->zoom_factor = zoom;
                set_up_page(gs);
                gs->changed = TRUE;
        }
}

static float
compute_xdpi()
{
        return 25.4 * gdk_screen_width()/gdk_screen_width_mm();
}

static float
compute_ydpi()
{
        return 25.4 * gdk_screen_height()/gdk_screen_height_mm();
}

/* Compute new size of window, sets xdpi and ydpi if necessary.
 * returns True if new window size is different */
static gboolean
compute_size(GtkGS *gs)
{
        guint new_width;
        guint new_height;
        gboolean change;

	/* width and height can be changed, calculate window size according */
	/* to xpdi and ydpi */
	switch (gs->orientation) {
                /* FIXME: what if the default orientation isn't normal??? */
        case GTK_GS_ORIENTATION_DEFAULT:
        case GTK_GS_ORIENTATION_PORTRAIT:
	case GTK_GS_ORIENTATION_UPSIDEDOWN:
                new_width = (gs->urx - gs->llx) / 72.0 *
                        gs->xdpi + 0.5;
                new_height = (gs->ury - gs->lly) / 72.0 *
                        gs->ydpi + 0.5;
                break;
	case GTK_GS_ORIENTATION_LANDSCAPE:
	case GTK_GS_ORIENTATION_SEASCAPE:
                new_width = (gs->ury - gs->lly) / 72.0 *
                        gs->xdpi + 0.5;
                new_height = (gs->urx - gs->llx) / 72.0 *
                        gs->ydpi + 0.5;
                break;
                
	}
        change = (new_width != gs->width * gs->zoom_factor) || (new_height != gs->height * gs->zoom_factor);
        gs->width = (gint) (new_width * gs->zoom_factor);
        gs->height = (gint) (new_height * gs->zoom_factor);
        return (change);
}

gint
gtk_gs_enable_interpreter (GtkGS *gs)
{
	g_assert (gs != NULL);
        if (!gs->gs_filename)
                return 0;

	gs->disable_start = FALSE;
	if (GTK_WIDGET_REALIZED (gs)) {
		return start_interpreter (gs);
	} else {
		return 0;
	}
}


void
gtk_gs_disable_interpreter (GtkGS *gs)
{
	g_assert (gs != NULL);
	gs->disable_start = TRUE;
	if (GTK_WIDGET_REALIZED (GTK_WIDGET (gs)))
		stop_interpreter (gs);
}

gint
gtk_gs_get_orientation(GtkGS *gs)
{
	g_return_val_if_fail(gs != NULL, -1);
	g_return_val_if_fail(GTK_IS_GS(gs), -1);
        
        if (gs->orientation != GTK_GS_ORIENTATION_DEFAULT)
                return gs->orientation;
        /* FIXME: we want a way to get the orientation from the page */
        return GTK_GS_ORIENTATION_PORTRAIT;
}

gchar *
gtk_gs_document_title(GtkGS *gs)
{
	g_return_val_if_fail(gs != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(gs), NULL);

        if (gs->doc && gs->doc->title)
		return gs->doc->title;
        
	return NULL;
}

guint
gtk_gs_document_numpages(GtkGS *widget)
{
	g_return_val_if_fail(widget != NULL, 0);
	g_return_val_if_fail(GTK_IS_GS(widget), 0);
        
        if (widget->doc) 
                return widget->doc->numpages;
        
	return 0;
}

gchar *
gtk_gs_document_page_label(GtkGS *widget, int page)
{
	g_return_val_if_fail(widget != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(widget), NULL);

        if (widget->doc && widget->doc->numpages >= page)
		return widget->doc->pages[page-1].label;
        
	return NULL;
}
