# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import xml.dom.minidom
from xml.sax.saxutils import escape

import gtk
import gobject

from gazpacho import util
from gazpacho.gaction import GAction, get_gaction_group_from_gtk_action_group
from gazpacho.gaction import get_gaction_from_gtk_action
from gazpacho.context import Context
from gazpacho.loader import tags, widgettree

from gazpacho import uim

from os.path import basename, splitext

class Project(gobject.GObject):

    __gsignals__ = {
        'add_widget':          (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_widget':       (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'widget_name_changed': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'project_changed':     (gobject.SIGNAL_RUN_LAST, None, ()),
        'selection_changed':   (gobject.SIGNAL_RUN_LAST, None, ()),
        'add_action':          (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_action':       (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'action_name_changed': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        }
    
    project_counter = 1
    
    def __init__(self, untitled, app):
        gobject.GObject.__init__(self)

        self._app = app

        self.name = None
        
        if untitled:
            # The name of the project like network-conf
            self.name = _('Untitled %d') % Project.project_counter
            Project.project_counter += 1

        # The full path of the xml file for this project
        self.path = None

        # The menu entry in the /Project menu
        self.entry = None
        
        # A list of GtkWidgets that make up this project. The widgets are
        # stored in no particular order.
        self.widgets = []

        # We need to keep the selection in the project because we have multiple
        # projects and when the user switchs between them, he will probably
        # not want to loose the selection. This is a list of GtkWidget items.
        self.selection = []

        #  A stack with the last executed commands
        self.undo_stack = []

        #  Points to the item previous to the redo items
        self.prev_redo_item = -1

        # widget -> old name of the widget
        self._widget_old_names = {}
        self.tooltips = gtk.Tooltips()

        # A flag that is set when a project has changes if this flag is not set
        # we don't have to query for confirmation after a close or exit is
        # requested
        self._changed = False

        # There is a UIManager in each project which holds the information
        # about menus and toolbars
        self.uim = uim.GazpachoUIM(self)
        
        # This is the id for the menuitem entry on Gazpacho main menu for this
        # project
        self.uim_id = -1
        
        # create our context
        self.context = Context(self)

    def get_changed(self):
        return self._changed

    def set_changed(self, value):
        if self._changed != value:
            self._changed = value
            self.emit('project_changed')

    changed = property(get_changed, set_changed)
    
    def selection_changed(self):
        self.emit('selection_changed')

    def _on_widget_notify(self, gwidget, arg):
        if arg.name == 'name':
            old_name = self._widget_old_names[gwidget]
            self.widget_name_changed(gwidget, old_name)
            self._widget_old_names[gwidget] = gwidget.name
        elif arg.name == 'project':
            self.remove_widget(gwidget.gtk_widget)

    def add_widget(self, gtk_widget):
        from gazpacho import widget, placeholder
        # we don't list placeholders
        if isinstance(gtk_widget, placeholder.Placeholder):
            return

        # If it's a container, add the children as well
        if isinstance(gtk_widget, gtk.Container):
            for child in gtk_widget.get_children():
                self.add_widget(child)

        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        self.changed = True
        # The internal widgets (e.g. the label of a GtkButton) are handled
        # by gtk and don't have an associated GladeWidget: we don't want to
        # add these to our list. It would be nicer to have a flag to check
        # (as we do for placeholders) instead of checking for the associated
        # GladeWidget, so that we can assert that if a widget is _not_
        # internal, it _must_ have a corresponding GladeWidget...
        # Anyway this suffice for now.
        if not gwidget:
            return

        gwidget.project = self
        self._widget_old_names[gwidget] = gwidget.name

        gwidget.connect('notify', self._on_widget_notify)
        self.widgets.append(gtk_widget)

        self.emit('add_widget', gwidget)

    def remove_widget(self, gtk_widget):
        from gazpacho import widget, placeholder
        if isinstance(gtk_widget, placeholder.Placeholder):
            return

        if isinstance(gtk_widget, gtk.Container):
            for child in gtk_widget.get_children():
                self.remove_widget(child)

        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        if not gwidget:
            return

        if gtk_widget in self.selection:
            self.selection_remove(gtk_widget, False)
            self.selection_changed()

        self.release_widget_name(gwidget, gwidget.name)
        self.widgets.remove(gtk_widget)

        self.changed = True
        self.emit('remove_widget', gwidget)

    def add_action_group(self, action_group):
        self.uim.add_action_group(action_group)

        self.emit('add_action', action_group)
        self.changed = True
    
    def remove_action_group(self, action_group):
        self.uim.remove_action_group(action_group)
        
        self.changed = True
        self.emit('remove_action', action_group)
        
    def _add_action_cb(self, action_group, action):
        self.changed = True
        self.emit('add_action', action)
    
    def _remove_action_cb(self, action_group, action):
        self.changed = True
        self.emit('remove_action', action)
    
    def change_action_name(self, action):
        self.changed = True
        self.emit('action_name_changed', action)
         
    def release_widget_name(self, gwidget, name):
        pass # XXX TODO

    def widget_name_changed(self, gwidget, old_name):
        self.release_widget_name(gwidget, old_name)
        self.emit('widget_name_changed', gwidget)
        self.changed = True
    
    def selection_clear(self, emit_signal):
        if not self.selection:
            return
        
        for gtk_widget in self.selection:
            util.remove_nodes(gtk_widget)
            
        self.selection = []
        
        if emit_signal:
            self.selection_changed()

    def selection_remove(self, gtk_widget, emit_signal):
        """ Remove the widget from the current selection """
        if not util.has_nodes(gtk_widget):
            return

        util.remove_nodes(gtk_widget)

        self.selection.remove(gtk_widget)
        if emit_signal:
            self.selection_changed()

    def selection_add(self, gtk_widget, emit_signal):
        if util.has_nodes(gtk_widget):
            return

        util.add_nodes(gtk_widget)

        self.selection.insert(0, gtk_widget)
        if emit_signal:
            self.selection_changed()
        
    def selection_set(self, gtk_widget, emit_signal):
        # If the widget is already selected we don't have to bother
        if util.has_nodes(gtk_widget):
            # But we need to emit the selection_changed signal if the
            # widget isn't in the active project
            if self._app._project != self and emit_signal:
                self.selection_changed()
            return

        self.selection_clear(False)
        self.selection_add(gtk_widget, emit_signal)

    def delete_selection(self):
        """ Delete (remove from the project) the widgets in the selection """
        from gazpacho import widget, placeholder

        newlist = list(self.selection)
        for gtk_widget in newlist:
            gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
            if gwidget is not None:
                self._app.command_manager.delete(gwidget)
            elif isinstance(gtk_widget, placeholder.Placeholder):
                self._app.command_manager.delete_placeholder(gtk_widget)

    def new_widget_name(self, base_name, blacklist=[]):
        """Create a new name based on the base_name. The name should not
        exist neither in the project neither in the blacklist"""
        i = 1
        while True:
            name = '%s%d' % (base_name, i)
            if self.get_widget_by_name(name) == None and name not in blacklist:
                return name
            i += 1

    def get_widget_by_name(self, name):
        from gazpacho import widget
        for w in self.widgets:
            gw = widget.get_widget_from_gtk_widget(w)
            if gw.name == name:
                return gw

    def write(self, xml_doc, path):
        from gazpacho import widget
        """ Returns the root node of a newly created xml representation of the
        project and its contests. """
        node = xml_doc.appendChild(xml_doc.createElement(tags.XML_TAG_PROJECT))

        # check what modules are the widgets using
        modules = []
        for w in self.widgets:
            klass = widget.get_widget_from_gtk_widget(w).klass
            libglade_module = klass.library.libglade_module
            if libglade_module and libglade_module not in modules:
                modules.append(libglade_module)

        for module in modules:
            n = xml_doc.createElement(tags.XML_TAG_REQUIRES)
            n.setAttribute(tags.XML_TAG_LIB, module)
            node.appendChild(n)

        # Append uimanager
        ui_widgets = [widget.get_widget_from_gtk_widget(w) \
                      for w in self.widgets if isinstance(w, (gtk.Toolbar,
                                                              gtk.MenuBar))]
        if ui_widgets:
            node.appendChild(self.uim.save(xml_doc, ui_widgets))
        
        # Append toplevel widgets. Each widget then takes care of
        # appending its children
        for w in self.widgets:
            gwidget = widget.get_widget_from_gtk_widget(w)
            if gwidget.is_toplevel():
                if gwidget.klass.save_function is not None:
                    wnode = gwidget.klass.save_function(self.context, xml_doc,
                                                        gwidget)
                else:
                    wnode = gwidget.write(xml_doc)
                    
                node.appendChild(wnode)
        
        return node

    def open(path, app):
        from gazpacho import widget, placeholder
        from gazpacho.widgetregistry import widget_registry
        
        class ObjectGlue:
            def __init__(self, project):
                self.project = project
                
            def load_custom_property(self, object, name, value):
                widget_class = widget_registry.get_by_type (object.__gtype__)

                if not widget_class:
                    return False

                for prop_class in widget_class.properties:
                    if prop_class.id == name:
                        if prop_class._load_function:
                            context = self.project.context
                            prop_class._load_function(context, object, value)
                	    return True

		return False

            def get_internal_child(self, ancestor, internal_child):
                widget_class = widget_registry.get_by_type(ancestor.__gtype__)

                if widget_class and widget_class.get_internal_child:
                    context = self.project.context
                    return widget_class.get_internal_child(context,
                                                           ancestor,
                                                           internal_child)

        prj = Project(False, app)
        
        wt = widgettree.WidgetTree(path, placeholder.Placeholder, app,
                                   ObjectGlue(prj))
        
        # Load the UI Manager
        prj.uim.create()
        prj.uim.load(wt)
        
        # Load the widgets
        for key, gtk_widget in wt._widgets.items():
            if isinstance(gtk_widget, gtk.Widget):
                if gtk_widget.flags() & gtk.TOPLEVEL:
                    widget.load_widget_from_gtk_widget(gtk_widget, prj, wt)
                    prj.add_widget(gtk_widget)

        if prj is not None:
            prj.path = path
            prj.name = basename(path)
            prj.changed = False

        return prj

    open = staticmethod(open)
    
    def save(self, path):
        xml_doc = xml.dom.getDOMImplementation().createDocument(None, None,
                                                                None)
        self.path = path
        self.name = basename(path)
        
        root = self.write(xml_doc, path)
        if not root:
            return False

        # save the xml tree
        f = file(path, 'w')
        f.write('<?xml version="1.0" ?>\n')
        util.write_xml(f, xml_doc.documentElement)
        f.close()

        self.changed = False
        return True

gobject.type_register(Project)
