# Copyright (C) 2004,2005 by SICEm S.L. and Imendio
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import xml.dom.minidom
from xml.parsers import expat
from xml.sax.saxutils import unescape

import gtk
import gobject

from gazpacho.loader import tags
from gazpacho.loader import compat
from gazpacho.gaction import GAction
from gazpacho import util
from gazpacho.choice import enum_from_string, flags_from_string


class GladeParseError(Exception): pass

def dump_widget(widget, indent=0):
    print ' '*indent, widget
    if isinstance(widget, gtk.Container):
        for child in widget.get_children():
            dump_widget(child, indent+2)

    if isinstance(widget, gtk.MenuItem):
        submenu = widget.get_submenu()
        if submenu is not None:
            dump_widget(submenu, indent+2)
            
class WidgetTree(object):
    def __init__(self, path=None, placeholder=None, app=None,
                 object_glue=None, path_resolver=None):

        self._reset()
        self._placeholder = placeholder
        self._app = app

        self._object_glue = object_glue
        
        # see the docstring of set_path_resolver for information about this
        self._path_resolver = path_resolver
        
        if path is not None:
            self.load_from_file(path)
        
    def _reset(self):
        self._modules = {}
        # put at least the gtk module so existing glade files don't break
        self._modules['gtk'] = __import__('gtk')
        
        self._widgets = {}
        self._signals = []
        self._actions = {}
        self._uimanager = None
        self._ui_definitions = {}

        self._deprecated_widgets = []

        # here we store a list of functions we need to call after the
        # loading of a toplevel is done. For example setting some properties
        # that need the object to be added to the window ('has-default')
        self._postload_tasks = []

    def load_from_file(self, path):
        self._reset()
        
        f = file(path)
        try:
            doc = xml.dom.minidom.parse(f)
        except  expat.ExpatError, e:
            raise GladeParseError('XML Parse error in %s (at %d:%d): %s' %
                                  (path, e.lineno, e.offset, e.args[0]))

        self._read_xml_tree(doc.documentElement)

        self._set_up_accel_groups()
        
    def load_from_stream(self, stream):
        self._reset()

        try:
            doc = xml.dom.minidom.parseString(stream)
        except expat.ExpatError, e:
            raise GladeParseError('XML Parse error in stream (at %d:%d): %s' %
                                  (e.lineno, e.offset, e.args[0]))
        
        self.load_from_xml_document_instance(doc)

    def load_from_xml_document_instance(self, document):
        self._read_xml_tree(document.documentElement)
        self._set_up_accel_groups()

    def __getattr__(self, name):
        if name in self._widgets.keys():
            return self._widgets[name]
        raise AttributeError('The widget %s was not found' % name)
    
    def _read_xml_tree(self, xml_root_node):
        if xml_root_node.tagName != tags.XML_TAG_PROJECT:
            raise GladeParseError("The root node should be %s but is %s" % \
                                  (tags.XML_TAG_PROJECT, xml_root_node.tagName))

        # read the toplevel widgets (each toplevel read its own children)
        for child_node in util.xml_filter_nodes(xml_root_node.childNodes,
                                                xml_root_node.ELEMENT_NODE):
            self._postload_tasks = []
            if child_node.tagName == tags.XML_TAG_REQUIRES:
                self._add_requirement(child_node)

            elif child_node.tagName == tags.XML_TAG_OBJECT:
                obj = self._build_object(child_node)
                if obj is not None:
                    self._run_postload_tasks(obj)
            
            elif child_node.tagName == tags.XML_TAG_WIDGET:
                widget = self._build_object(child_node)
                if widget is not None:
                    self._run_postload_tasks(widget)
                    # make window management easier by making created windows
                    # transient for the editor window
                    if self._app is not None \
                           and isinstance(widget, gtk.Window):
                        widget.set_transient_for(self._app.window)

        # check if we have any deprecated widget
        if self._deprecated_widgets:
            if self._app is not None:
                dialog = compat.DeprecatedWidgetsDialog(self._app.window,
                                                        self._deprecated_widgets)
                dialog.run()
                dialog.destroy()

    def _run_postload_tasks(self, toplevel):
        for task in self._postload_tasks:
            f = task[0]
            args = task[1:]
            f(*args)
            
    def _add_requirement(self, xml_node):
        assert xml_node.tagName == tags.XML_TAG_REQUIRES

        module = xml_node.getAttribute(tags.XML_TAG_LIB)
        if not module in self._modules.keys():
            module = xml_node.getAttribute(tags.XML_TAG_LIB)
            mod = __import__(module, globals(), locals())
            submodules = module.split('.')
            for submodule in submodules[1:]:
                mod = getattr(mod, submodule)
            self._modules[module.lower()] = mod
            
    def _build_instance(self, class_name):
        """Look for class_name into the read modules and create an
        instance of this class
        """
        obj = None
        try:
            gtype = gobject.type_from_name(class_name)
            klass = util.class_from_gtype(gtype, self._modules.values())
            if klass is not None:
                obj = klass()
            else:
                obj = gobject.new(gtype)
        except RuntimeError:
            print 'There is no registered gtype for this class %s' % \
                  class_name
        except TypeError:
            print 'Unable to find class for object %s' % class_name

        return obj
    
    def _build_object(self, xml_node):
        assert xml_node.tagName in (tags.XML_TAG_OBJECT, tags.XML_TAG_WIDGET)

        class_name = xml_node.getAttribute(tags.XML_TAG_CLASS)
        func = getattr(self, '_build_%s' % class_name, None)
        if func:
            obj = func(xml_node)
        else:
            # Default loader, only create the instance and set
            # the properties
            obj = self._build_instance(class_name)

            # Set the name
            name = xml_node.getAttribute(tags.XML_TAG_ID)
            obj.set_property('name', name)
            self._fill_object_from_node(obj, xml_node)

        if obj is not None:
            try:
                self._widgets[obj.get_property('name')] = obj
            except TypeError:
                pass # not all the objects have a name

        return obj

    def _fill_object_from_node(self, obj, xml_node):
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            # read the properties
            if child_node.tagName == tags.XML_TAG_PROPERTY:
                self._read_property(obj, child_node)

            # read the signals
            elif child_node.tagName == tags.XML_TAG_SIGNAL:
                self._read_signal(obj, child_node)
                
            # build the children
            elif child_node.tagName == tags.XML_TAG_CHILD:
                child = self._build_child_object(obj, child_node)
                if child is not None and child not in obj.get_children():
                    obj.add(child)
                    self._read_packing_properties(obj, child, child_node)
    
    def _build_child_object(self, parent, xml_node):
        assert xml_node.tagName == tags.XML_TAG_CHILD

        obj = None
        
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_PLACEHOLDER \
                   and self._placeholder is not None:
                obj = self._placeholder(self._app)
                break
            
            elif child_node.tagName == tags.XML_TAG_WIDGET:
                internal_child = xml_node.getAttribute(tags.XML_TAG_INTERNAL_CHILD)
                if internal_child:
                    # search for the ancestor that we are internal child of
                    # note that this is not always the direct parent
                    ancestor = parent

                    if self._object_glue:
                        gic = self._object_glue.get_internal_child
                    else:
                        gic = lambda a, ic: getattr(a, ic, None)
                                
                    while gic(ancestor, internal_child) is None:
                        ancestor = ancestor.get_parent()
                    
                    obj = gic(ancestor, internal_child)
                    
                    self._fill_object_from_node(obj, child_node)
                    # usually the name is set in _build_object but as this
                    # is an internal widget we do it manually
                    name = child_node.getAttribute(tags.XML_TAG_ID)
                    obj.set_name(name)
                else:
                    obj = self._build_object(child_node)
                break
                        
            elif child_node.tagName == tags.XML_TAG_OBJECT:
                obj = self._build_object(child_node)
                break

        return obj

    def _read_packing_properties(self, parent, child, xml_node):
        # now load the packing properties
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_PACKING:
                for prop_node in util.xml_filter_nodes(child_node.childNodes,
                                                       child_node.ELEMENT_NODE):
                    if prop_node.tagName == tags.XML_TAG_PROPERTY:
                        self._read_child_property(child, prop_node, parent)
    
    # Custom widget builders
    def _build_GtkUIManager(self, xml_node):
        self._uimanager = gtk.UIManager()
        # we can't use fill object_from_node because we use
        # insert_action_group instead of add for adding our
        # children
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_CHILD:
                child = self._build_child_object(self._uimanager, child_node)
                self._uimanager.insert_action_group(child, 0)
            elif child_node.tagName == tags.XML_TAG_UI:
                # read this ui definition
                ui_name = child_node.getAttribute(tags.XML_TAG_ID)
                value = child_node.childNodes[0]
                self._uimanager.add_ui_from_string(value.data)
                self._ui_definitions[ui_name] = value.data

        self._uimanager.ensure_update()
        return self._uimanager

    def _build_GtkActionGroup(self, xml_node):
        name = xml_node.getAttribute(tags.XML_TAG_ID)
        action_group = gtk.ActionGroup(name)
        # we can't use fill_object_from_node because we use
        # add_action instead of add for adding our children
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_CHILD:
                action = self._build_child_object(action_group, child_node)

                accel = action.get_data('accelerator')
                # If there is no accelerator GTK+ will try
                # to use the stock one
                action_group.add_action_with_accel(action, accel)
                    
                # register this action with our actions dictionary
                self._actions[action.get_name()] = action

                callback = action.get_data(tags.SIGNAL_HANDLER)
                if callback is not None:
                    # register the signal handler
                    self._signals.append([action, 'activate', callback, None])
                    
        return action_group

    def _build_GtkAction(self, xml_node):
        action_name = xml_node.getAttribute(tags.XML_TAG_ID)
        # set some default values
        props = {'label': '', 'tooltip': '', 'stock-id': None}
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_PROPERTY:
                name, value = util.xml_read_raw_property(child_node)
                props[name] = value
        
        action = gtk.Action(action_name, props['label'], props['tooltip'],
                            props['stock-id'])
        if 'accelerator' in props.keys():
            if props['accelerator']:
                action.set_data('accelerator', props['accelerator'])
        if 'callback' in props.keys():
            if props['callback']:
                action.set_data(tags.SIGNAL_HANDLER, props['callback'])
            
        return action
    
    def _build_GtkMenuBar(self, xml_node):
        name = xml_node.getAttribute(tags.XML_TAG_ID)

        constructor = xml_node.getAttribute('constructor')

        if self._uimanager is None:
            self._uimanager = gtk.UIManager()
            
        if constructor:
            obj = self._uimanager.get_widget('ui/%s' % name)
            if obj is None:
                raise RuntimeError("There is no widget in the uimanager for "
                                   "%s" % name)
            self._fill_object_from_node(obj, xml_node)
        else:
            # no ui property, it's probably a Glade-2 menubar/toolbar
            obj = compat.build_deprecated_widget(xml_node, self._uimanager,
                                                 self._ui_definitions,
                                                 self._signals)
            class_name = xml_node.getAttribute(tags.XML_TAG_CLASS)
            class_name = "%s not using UI Manager" % class_name
            self._deprecated_widgets.append((obj, class_name))
            
        obj.set_property('name', name)
        return obj
    
    # Same as MenuBar
    _build_GtkToolbar = _build_GtkMenuBar

    def _build_GtkFrame(self, xml_node):
        name = xml_node.getAttribute(tags.XML_TAG_ID)        
        obj = gtk.Frame()
        obj.set_property('name', name)
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            # read the properties
            if child_node.tagName == tags.XML_TAG_PROPERTY:
                self._read_property(obj, child_node)

            # read the signals
            elif child_node.tagName == tags.XML_TAG_SIGNAL:
                self._read_signal(obj, child_node)
                
            # build the children
            elif child_node.tagName == tags.XML_TAG_CHILD:
                child = self._build_child_object(obj, child_node)
                if child is not None:
                    type_node = util.xml_find_node(child_node,
                                                    tags.XML_TAG_PROPERTY,
                                                    name='type')
                    if type_node is not None:
                        value = util.xml_read_raw_property(type_node)[1]
                        if value == 'label_item':
                            obj.set_label_widget(child)
                            continue

                    obj.add(child)
                    self._read_packing_properties(obj, child, child_node)    

        return obj

    def _build_GtkButton(self, xml_node):
        """We need a custom build button because the special case where
        a button is a container with alignment and image inside"""
        obj_id = xml_node.getAttribute(tags.XML_TAG_ID)
        obj = gtk.Button()
        obj.set_property('name', obj_id)
        has_child = False
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            # read the properties
            if child_node.tagName == tags.XML_TAG_PROPERTY:
                self._read_property(obj, child_node)

            # read the signals
            elif child_node.tagName == tags.XML_TAG_SIGNAL:
                self._read_signal(obj, child_node)
                
            # build the children
            elif child_node.tagName == tags.XML_TAG_CHILD:
                if has_child:
                    msg = "A button is supposed to have one child at most: %s"
                    raise ValueError, msg % obj_id
                
                child = self._build_child_object(obj, child_node)
                if child is not None:
                    # remove the current label
                    obj.remove(obj.get_child())
                    # add this new child
                    obj.add(child)
                    self._read_packing_properties(obj, child, child_node)
                    has_child = True
                
        return obj

    def _build_GtkTreeView(self, xml_node):
        """We need to create an empty model for a treeview or we will have
        problems later"""
        obj_id = xml_node.getAttribute(tags.XML_TAG_ID)
        obj = gtk.TreeView()
        model = gtk.ListStore(str)
        obj.set_model(model)
        obj.set_property('name', obj_id)
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            # read the properties
            if child_node.tagName == tags.XML_TAG_PROPERTY:
                self._read_property(obj, child_node)

            # read the signals
            elif child_node.tagName == tags.XML_TAG_SIGNAL:
                self._read_signal(obj, child_node)

            # treeviews does not have any child right now
        return obj

    def _build_GtkCombo(self, xml_node):
        name = xml_node.getAttribute(tags.XML_TAG_ID)
        obj = gtk.ComboBox()
        self._deprecated_widgets.append((obj, 'GtkCombo'))
        obj.set_property('name', name)
        return obj
    
    def __get_adjustment(self, value):
        values = [float(v) for v in value.split()]
        return gtk.Adjustment(*values)
        
    # Custom property loaders
    def _load_has_default(self, obj, value, pspec):
        """Can't set the default until the object is added to the toplevel
        window"""
        value = self._value_from_string(value, pspec)
        if value is not None:
            self._postload_tasks.append((self._set_has_default, obj, value))

    def _load_adjustment(self, obj, value, pspec):
        obj.set_property(pspec.name, self.__get_adjustment(value))

    # all the adjustment are the same
    _load_hadjustment = _load_adjustment
    _load_vadjustment = _load_adjustment
        
    def _load_visible(self, obj, value, pspec):
        value = self._value_from_string(value, pspec)
        if isinstance(obj, gtk.Window):
            self._postload_tasks.append((obj.set_property, 'visible', value))
        else:
            obj.set_property('visible', value)

    def _load_group(self, obj, value, pspec):
        """Can't set the group yet because maybe the other radiobuttons are
        not loaded yet"""
        self._postload_tasks.append((self._set_group, obj, value))

    def _load_file(self, obj, value, pspec):
        """Read the file with the image filename for an gtkimage"""
        if not value: return
        if self._path_resolver is not None:
            value = self._path_resolver(value)
        obj.set_from_file(value)
        obj.set_data('image-file-name', value)
        
    def _set_has_default(self, obj, value):
        obj.set_property('has-default', value)

    def _set_group(self, obj, value):
        # get the radiobutton with name == value
        rb = self._widgets.get(value, None)
        if rb is not None and rb is not obj:
            obj.set_group(rb)

    def _read_property(self, obj, xml_node):
        name, value = util.xml_read_raw_property(xml_node)
        is_custom = False
        if self._object_glue is not None:
            is_custom = self._object_glue.load_custom_property(obj,
                                                               name, value)
        if not is_custom:
            for pspec in gobject.list_properties(obj):
                if pspec.name == name:
                    name = name.replace('-', '_')
                    func = getattr(self, '_load_%s' % name, None)
                    if func:
                        func(obj, value, pspec)
                    else:
                        value = self._value_from_string(value, pspec)
                        if value is not None:
                            obj.set_property(name, value)
                            break

        # Read any i18n metadata and set as object data so the
        # wrappers can access it when the wrapper widget tree is
        # created
        value = xml_node.getAttribute(tags.XML_TAG_TRANSLATABLE)
        if value is not None:
            val_bool = util.get_bool_from_string_with_default(value, True)
            util.gtk_widget_set_metadata(obj,
                                         util.METADATA_I18N_IS_TRANSLATABLE,
                                         name,
                                         val_bool)

        value = xml_node.getAttribute(tags.XML_TAG_CONTEXT)
        if value is not None:
            val_bool = util.get_bool_from_string_with_default(value, True)            
            util.gtk_widget_set_metadata(obj,
                                         util.METADATA_I18N_HAS_CONTEXT,
                                         name,
                                         val_bool)

        value = xml_node.getAttribute(tags.XML_TAG_COMMENT)
        if value is not None and value != '':
            util.gtk_widget_set_metadata(obj,
                                         util.METADATA_I18N_COMMENT,
                                         name,
                                         value)


    def _read_child_property(self, widget, xml_node, parent):
        assert xml_node.tagName == tags.XML_TAG_PROPERTY
        name, value = util.xml_read_raw_property(xml_node)
        for pspec in gtk.container_class_list_child_properties(parent):
            if pspec.name == name:
                value = self._value_from_string(value, pspec)
                if value is not None:
                    parent.child_set_property(widget, name, value)
                    break

    def _read_signal(self, widget, xml_node):
        assert xml_node.tagName == tags.XML_TAG_SIGNAL

        name = xml_node.getAttribute(tags.XML_TAG_NAME)
        tmp = xml_node.getAttribute(tags.XML_TAG_AFTER)
        after = util.get_bool_from_string_with_default(tmp, False)
        handler = xml_node.getAttribute(tags.XML_TAG_HANDLER)

        self._signals.append([widget, name, handler, after])
        
    def _value_from_string(self, value, pspec):
        
        if hasattr(pspec, 'flags_class'):
            return flags_from_string(value, pspec)
        elif hasattr(pspec, 'enum_class'):
            return enum_from_string(value, pspec)

        if pspec.value_type in [gobject.TYPE_INT, gobject.TYPE_UINT]:
            # can be GParamUnichar
            try:
                return int(value)
            except ValueError:
                return value
        elif pspec.value_type in [gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE]:
            return float(value)
        elif pspec.value_type == gobject.TYPE_BOOLEAN:
            if value in ['True', 'TRUE', 'yes', '1']:
                return True
            else:
                return False
        elif pspec.value_type == gobject.TYPE_STRING:
            return value

        else:
            # let the object deal with this value
            return value

    def signal_autoconnect(self, dic):
        for widget, signal, handler, after in self._signals:
            try:
                if after:
                    widget.connect_after(signal, dic[handler])
                else:
                    widget.connect(signal, dic[handler])
            except KeyError:
                pass

    def _set_up_accel_groups(self):
        # now for menubars and toolbar we need to set the accel groups to their
        # windows
        for widget in self._widgets.values():
            if isinstance(widget, (gtk.MenuBar, gtk.Toolbar)):
                window = widget.get_toplevel()
                # don't assume window will be a window. This fails if the
                # toolbar is not yet attached to a window as in the case
                # we are pasting in Gazpacho
                if isinstance(window, gtk.Window):
                    accel_group = self._uimanager.get_accel_group()
                    accel_groups = gtk.accel_groups_from_object(window)
                    if accel_group not in accel_groups:
                        window.add_accel_group(accel_group)        

    def _widget_iterator(self):
        for widget in self._widgets.values():
            yield widget

    widgets = property(_widget_iterator)

    def get_actions(self):
        return self._actions
    actions = property(get_actions)

    def get_widget(self, widget_name):
        return self._widgets.get(widget_name, None)

    def get_widgets(self):
        return self._widgets.values()

    def set_path_resolver(self, resolver):
        """The path resolver is a callable that translate a path in the
        Glade file into another path.
        This is useful when loading images whose path is hardcoded in the
        glade file.
        The resolver has the following signature:

        def resolver(path): -> new_path
        """
        self._path_resolver = resolver
        
if __name__ == '__main__':
    import sys
    wt = WidgetTree(sys.argv[1])
    wt.signal_autoconnect(globals())
    
    print 'Widgets read:'
    for key, value in wt._widgets.items():
        print key, value

    gtk.main()
