# Copyright (C) 2004,2005 by SICEm S.L. and Imendio
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from gazpacho.widget import Widget, copy_widget
from gazpacho.util import get_parent
from gazpacho.gaction import GAction, GActionGroup
from gazpacho.l10n import _

class CommandManager(object):
    """This class is the entry point accesing the commands.
    Every undoable action in Gazpacho is wrapped into a command.
    The stack of un/redoable actions is stored in the Project class
    so each project is independent on terms of undo/redo.
    
    The CommandManager knows how to perform the undo and redo actions.
    There is also a method for every type of command supported so
    you don't have to worry about creating the low level command
    objects.
    """
    def __init__(self, app):
        self.app = app
        
    def undo(self, project):
        """Undo the last command if there is such a command"""
        if project.undo_stack is None:
            return
        if project.prev_redo_item == -1:
            return
        
        cmd = project.undo_stack[project.prev_redo_item]
        cmd.undo()

        project.prev_redo_item -= 1
    
    def redo(self, project):
        """Redo the last undo command if there is such a command"""
        if not project.undo_stack: return
        # There is no more items to redo
        if project.prev_redo_item + 1 == len(project.undo_stack): return
    
        project.prev_redo_item += 1

        cmd = project.undo_stack[project.prev_redo_item]
        cmd.redo()

    #
    # for every possible command we have a method here
    #

    def delete(self, widget):
        # internal children cannot be deleted. Should we notify the user?
        if widget.internal:
            return
        description = _("Delete %s") % widget.name
        cmd = CommandCreateDelete(widget, None, widget.get_parent(), False,
                                  description, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

    def create(self, klass, placeholder, project, parent=None, interactive=True):
        if placeholder:
            parent = get_parent(placeholder)
            if parent is None:
                return

        if project is None:
            project = parent.project

        widget = Widget(klass, project, interactive=interactive)
        if widget is None:
            return

        description = _("Create %s") % widget.name

        cmd = CommandCreateDelete(widget, placeholder, parent, True, description,
                                  self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

        self.app.palette.unselect_widget()

    def delete_placeholder(self, placeholder):
        parent = get_parent(placeholder)
        if len(parent.widget.get_children()) == 1:
            return
        
        description = _("Delete placeholder")
        cmd = CommandDeletePlaceholder(placeholder, parent, description, self.app)
        cmd.execute()
        cmd.push_undo(parent.project)
        
    def box_insert_placeholder(self, box, pos):
        description = _("Insert after")

        cmd = CommandInsertPlaceholder(box, pos, description, self.app)
        cmd.execute()
        cmd.push_undo(box.project)

    def set_property(self, property, value):
        gwidget = property.widget
        dsc = _('Setting %s of %s') % (property.klass.name, gwidget.name)
        cmd = CommandSetProperty(property, value, dsc, self.app)
        cmd.execute()
        cmd.push_undo(gwidget.project)

    def set_translatable_property(self, property, value, comment,
                                  is_translatable, has_context):
        gwidget = property.widget
        dsc = _('Setting %s of %s') % (property.klass.name, gwidget.name)
        cmd = CommandSetTranslatableProperty(property, value,
                                             comment, is_translatable,
                                             has_context,
                                             dsc, self.app)
        cmd.execute()
        cmd.push_undo(gwidget.project)

    def set_name(self, widget, name):
        dsc = _('Renaming %s to %s') % (widget.name, name)
        cmd = CommandSetName(widget, widget.name, name, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

    def add_signal(self, widget, signal):
        dsc = _('Add signal handler %s') % signal['handler']
        cmd = CommandAddRemoveSignal(True, signal, widget, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

    def remove_signal(self, widget, signal):
        dsc = _('Remove signal handler %s') % signal['handler']
        cmd = CommandAddRemoveSignal(False, signal, widget, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

    def copy(self, widget):
        dsc = _('Copy widget %s into the clipboard') % widget.name
        op = CommandCopyCutPaste.COPY
        copy = copy_widget(widget)
        cmd = CommandCopyCutPaste(copy, None, op, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)
        
    def cut(self, widget):
        dsc = _('Cut widget %s into the clipboard') % widget.name
        op = CommandCopyCutPaste.CUT
        cmd = CommandCopyCutPaste(widget, None, op, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)
        
    def paste(self, placeholder):
        clipboard = self.app.get_clipboard()
        widget = clipboard.get_selected_widget()
        dsc = _('Paste widget %s from the clipboard') % widget.name
        op = CommandCopyCutPaste.PASTE
        cmd = CommandCopyCutPaste(widget, placeholder, op, dsc, self.app)
        cmd.execute()
        cmd.push_undo(widget.project)

    def add_action(self, values, parent, project):
        gaction = GAction(parent, values['name'], values['label'],
                          values['tooltip'], values['stock_id'],
                          values['callback'], values['accelerator'])
        dsc = _('Add action %s') % gaction.name
        cmd = CommandAddRemoveAction(parent, gaction, True, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def remove_action(self, gaction, project):
        dsc = _('Remove action %s') % gaction.name
        cmd = CommandAddRemoveAction(gaction.parent, gaction, False, dsc,
                                     self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def edit_action(self, gaction, new_values, project):
        dsc = _('Edit action %s') % gaction.name
        cmd = CommandEditAction(gaction, new_values, dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        project.change_action_name(gaction)

    def add_action_group(self, name, project):
        gaction_group = GActionGroup(name)
        dsc = _('Add action group %s') % gaction_group.name
        cmd = CommandAddRemoveActionGroup(gaction_group, project, True,
                                          dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
    
    def remove_action_group(self, gaction_group, project):
        dsc = _('Remove action group %s') % gaction_group.name
        cmd = CommandAddRemoveActionGroup(gaction_group, project, False,
                                          dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        
    def edit_action_group(self, gaction_group, new_name, project):
        dsc = _('Edit action group %s') % gaction_group.name
        cmd = CommandEditActionGroup(gaction_group, new_name,
                                     dsc, self.app)
        cmd.execute()
        cmd.push_undo(project)
        project.change_action_name(gaction_group)
    
class Command(object):
    """A command is the minimum unit of undo/redoable actions.

    It has a description so the user can see it on the Undo/Redo
    menu items.

    Every Command subclass should implement the 'execute' method in
    the following way:
        - After a command is executed, if the execute method is called
        again, their effects will be undone.
        - If we keep calling the execute method, the action will
        be undone, then redone, then undone, etc...
        - To acomplish this every command constructor will probably
        need to gather more data that it may seems necessary.

    After you execute a command in the usual way you should put that
    command in the command stack of that project and that's what
    the push_undo method does. Otherwise no undo will be available.
    
    Some commands unifies themselves. This means that if you execute
    several commands of the same type one after the other, they will be
    treated as only one big command in terms of undo/redo. In other words,
    they will collapse. For example, every time you change a letter on a
    widget's name it is a command but all these commands unifies to one
    command so if you undo that all the name will be restored to the old one.
    """
    def __init__(self, description=None, app=None):
        self._description = description
        self._app = app

    def get_description(self): return self._description
    def set_description(self, value): self._description = value
    description = property(get_description, set_description)

    def __repr__(self):
        return self._description

    def execute(self):
        """ This is the main method of the Command class.
        Note that it does not have any arguments so all the
        necessary data should be provided in the constructor.
        """
        pass

    def undo(self):
        """Convenience method that just call execute"""
        self.execute()
    
    def redo(self):
        """Convenience method that just call redo"""
        self.execute()
    
    def unifies(self, other):
        """True if self unifies with 'other'
        Unifying means that both commands can be treated as they
        would be only one command
        """
        return False
    
    def collapse(self, other):
        """Combine self and 'other' to form only one command.
        'other' should unifies with self but this method does not
        check that.
        """
        return False

    def push_undo(self, project):
        """Put the command in the project command stack
        It tries to collapse the command with the previous command
        """
        # If there are no "redo" items, and the last "undo" item unifies with
        # us, then we collapse the two items in one and we're done
        if project.prev_redo_item == len(project.undo_stack) - 1 and \
               project.prev_redo_item != -1:
            cmd = project.undo_stack[project.prev_redo_item]
            if cmd.unifies(self):
                cmd.collapse(self)
                return

        # We should now free al the 'redo' items
        project.undo_stack = project.undo_stack[:project.prev_redo_item + 1]

        # and then push the new undo item
        project.undo_stack.append(self)
        project.prev_redo_item += 1

        self._app.refresh_undo_and_redo()

class CommandCreateDelete(Command):
    def __init__(self, widget=None, placeholder=None, parent=None, create=True,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self._widget = widget
        self._placeholder = placeholder
        self._parent = parent
        self._create = create
        self._initial_creation = create
        
    def _create_execute(self):
        from gazpacho.placeholder import Placeholder
        if not self._widget.widget.flags() & gtk.TOPLEVEL:
            if self._placeholder is None:
                for child in self._parent.widget.get_children():
                    if isinstance(child, Placeholder):
                        self._placeholder = child
                        break
            self._widget.replace(self._placeholder, self._widget.widget,
                                 self._parent)

        if isinstance(self._widget.widget, gtk.Widget):
            self._widget.get_glade_property('visible').value = True
            
        self._widget.project.add_widget(self._widget.widget)
        self._widget.project.selection_set(self._widget.widget, True)

        if isinstance(self._widget.widget, gtk.Widget):
            self._widget.widget.show_all()

        if self._widget.is_toplevel():
            # we have to attach the accelerators groups so key shortcuts
            # keep working when this window has the focus. Only do
            # this the first time when creating a window, not when
            # redoing the creation since the accel group is already
            # set by then
            if self._initial_creation:
                for ag in self._app.get_accel_groups():
                    self._widget.widget.add_accel_group(ag)
                self._initial_creation = False
                
            # make window management easier by making created windows
            # transient for the editor window
            self._widget.widget.set_transient_for(self._app.get_window())
        
    def _delete_execute(self):
        from gazpacho.placeholder import Placeholder
        if self._parent:
            if self._placeholder is None:
                self._placeholder = Placeholder(self._app)
            self._widget.replace(self._widget.widget, self._placeholder,
                                 self._parent)

        self._widget.widget.hide()
        self._widget.project.remove_widget(self._widget.widget)

        return True

    def execute(self):
        if self._create:
            retval = self._create_execute()
        else:
            retval = self._delete_execute()

        self._create = not self._create

        return retval

class CommandDeletePlaceholder(Command):
    def __init__(self, placeholder=None, parent=None, description=None, application=None):
        Command.__init__(self, description, application)

        self._placeholder = placeholder
        self._parent = parent
        self._create = False

        children = self._parent.widget.get_children()
        self._pos = children.index(placeholder)
        
    def _create_execute(self):
        from gazpacho.placeholder import Placeholder
        # create a placeholder and insert it at self._pos
        self._placeholder = Placeholder(self._app)
        self._parent.widget.add(self._placeholder)
        self._parent.widget.reorder_child(self._placeholder, self._pos)

    def _delete_execute(self):
        from gazpacho.placeholder import Placeholder
        self._placeholder.destroy()
        self._placeholder = None
        return True

    def execute(self):
        if self._create:
            retval = self._create_execute()
        else:
            retval = self._delete_execute()

        self._create = not self._create

        return retval

class CommandInsertPlaceholder(Command):
    def __init__(self, box, pos, description=None, application=None):
        Command.__init__(self, description, application)

        self._box = box
        self._pos = pos
        self._placeholder = None
        self._insert = True
        

    def _insert_execute(self):
        from gazpacho.placeholder import Placeholder
        # create a placeholder and insert it at self._pos
        self._placeholder = Placeholder(self._app)
        self._box.widget.add(self._placeholder)
        self._box.widget.reorder_child(self._placeholder, self._pos)

    def _delete_execute(self):
        from gazpacho.placeholder import Placeholder
        self._placeholder.destroy()
        self._placeholder = None
        return True

    def execute(self):
        if self._insert:
            retval = self._insert_execute()
        else:
            retval = self._delete_execute()

        self._insert = not self._insert

        return retval

class CommandSetProperty(Command):
    def __init__(self,  property=None, value=None, description=None,
                 application=None):
        Command.__init__(self, description, application)

        self._property = property
        self._value = value

    def execute(self):
        new_value = self._value
        # store the current value for undo
        self._value = self._property.value
        self._property.set(new_value)

        return True

    def unifies(self, other):
        if isinstance(other, CommandSetProperty):
            return self._property == other._property
        return False

    def collapse(self, other):
        self._description = other._description
        other._description = None

        self._app.refresh_undo_and_redo()

class CommandSetTranslatableProperty(Command):
    def __init__(self,  property=None, value=None, comment=None,
                 is_translatable=False, has_context=False, description=None,
                 application=None):
        Command.__init__(self, description, application)

        self._property = property
        self._value = value
        self._comment = comment
        self._is_translatable = is_translatable
        self._has_context = has_context

    def execute(self):
        new_value = self._value
        new_comment = self._comment
        new_is_translatable = self._is_translatable
        new_has_context = self._has_context

        # store the current value for undo
        self._value = self._property.value
        self._comment = self._property.i18n_comment
        self._is_translatable = self._property.is_translatable
        self._has_context = self._property.has_i18n_context

        self._property.set(new_value)
        self._property.is_translatable = new_is_translatable
        self._property.i18n_comment = new_comment
        self._property.has_i18n_context = new_has_context

        return True

    def unifies(self, other):
        return False

    def collapse(self, other):
        return False

class CommandSetName(Command):
    def __init__(self, widget=None, old_name=None, name=None,
                 description=None, application=None):
        Command.__init__(self, description, application)
        self._widget = widget
        self._old_name = old_name
        self._name = name

    def execute(self):
        self._widget.name = self._name
        self._old_name, self._name = self._name, self._old_name
        return True

    def unifies(self, other):
        if isinstance(other, CommandSetName):
            return self._widget == other._widget
        return False

    def collapse(self, other):
        if not isinstance(other, CommandSetName):
            return

        self._old_name = other._old_name
        other._old_name = None
        self._description = _('Rename %s to %s') % (self._name,
                                                    self._old_name)

        self._app.refresh_undo_and_redo()
        
class CommandAddRemoveSignal(Command):
    def __init__(self, add=True, signal=None, widget=None,
                 description=None, application=None):
        Command.__init__(self, description, application)
        self._add = add
        self._signal = signal
        self._widget = widget

    def execute(self):
        if self._add:
            self._widget.add_signal_handler(self._signal)
        else:
            self._widget.remove_signal_handler(self._signal)

        self._add = not self._add
        return True

class CommandCopyCutPaste(Command):
    COPY, CUT, PASTE = range(3)
    
    def __init__(self, widget, placeholder, operation,
                 description=None, application=None):
        Command.__init__(self, description, application)
        
        self._widget = widget
        self._placeholder = placeholder
        self._operation = operation
        self._clipboard = application.get_clipboard()
        self._undo = False
        
    def execute(self):
        if self._operation == CommandCopyCutPaste.COPY:
            self._execute_copy()
        elif self._operation == CommandCopyCutPaste.CUT:
            if self._undo:
                self._execute_paste()
            else:
                self._execute_cut()
        elif self._operation == CommandCopyCutPaste.PASTE:
            if self._undo:
                self._execute_cut()
            else:
                self._execute_paste()

        self._undo = not self._undo
        
    def _execute_copy(self):
        if self._undo:
            self._clipboard.remove_widget(self._widget)
        else:
            self._clipboard.add_widget(self._widget)
    
    def _execute_cut(self):
        from gazpacho.placeholder import Placeholder
        self._clipboard.add_widget(self._widget)
        parent = self._widget.get_parent()
        if parent is not None:
            if self._placeholder is None:
                self._placeholder = Placeholder(self._app)
            self._widget.replace(self._widget.widget, self._placeholder, parent)
        
        self._widget.widget.hide()
        self._widget.project.remove_widget(self._widget.widget)
    
    def _execute_paste(self):
        parent = get_parent(self._placeholder)
        project = parent.project
        
        if not self._widget.is_toplevel():
            self._widget.replace(self._placeholder, self._widget.widget, parent)
            
        project.add_widget(self._widget.widget)
        project.selection_set(self._widget.widget, True)
        
        self._widget.widget.show_all()
        
        self._clipboard.remove_widget(self._widget)

class CommandAddRemoveAction(Command):
    def __init__(self, parent, gaction, add=True,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self.add = add
        self.gaction = gaction
        self.parent = parent
        
    def execute(self):
        if self.add:
            self._add_execute()
        else:
            self._remove_execute()

        self.add = not self.add
    
    def _add_execute(self):
        self.parent.append(self.gaction)
    
    def _remove_execute(self):
        self.parent.remove(self.gaction)

class CommandEditAction(Command):
    def __init__(self, gaction, new_values, description=None,
                 application=None):
        Command.__init__(self, description, application)
        
        self.new_values = new_values
        self.gaction = gaction

    def execute(self):
        old_values = {
            'name' : self.gaction.name,
            'label': self.gaction.label,
            'stock_id': self.gaction.stock_id,
            'tooltip': self.gaction.tooltip,
            'accelerator': self.gaction.accelerator,
            'callback': self.gaction.callback,
            }
        self.gaction.name = self.new_values['name']
        self.gaction.label = self.new_values['label']
        self.gaction.stock_id = self.new_values['stock_id']
        self.gaction.tooltip = self.new_values['tooltip']
        self.gaction.accelerator = self.new_values['accelerator']
        self.gaction.callback = self.new_values['callback']
        self.gaction.parent.update_action(self.gaction, old_values['name'])
        self.new_values = old_values

class CommandAddRemoveActionGroup(Command):
    def __init__(self, gaction_group, project, add=True,
                 description=None, application=None):
        Command.__init__(self, description, application)

        self.add = add
        self.project = project
        self.gaction_group = gaction_group
        
    def execute(self):
        if self.add:
            self._add_execute()
        else:
            self._remove_execute()

        self.add = not self.add
    
    def _add_execute(self):
        self.project.add_action_group(self.gaction_group)
    
    def _remove_execute(self):
        self.project.remove_action_group(self.gaction_group)

class CommandEditActionGroup(Command):
    def __init__(self, gaction_group, new_name, description=None,
                 application=None):
        Command.__init__(self, description, application)
        
        self.new_name = new_name
        self.gaction_group = gaction_group

    def execute(self):
        old_name = self.gaction_group.name
        self.gaction_group.name = self.new_name
        self.new_name = old_name
        
class CommandStackView(gtk.ScrolledWindow):
    """This class is just a little TreeView that knows how
    to show the command stack of a project.
    It shows a plain list of all the commands performed by
    the user and also it mark the current command that
    would be redone if the user wanted so.
    Older commands are under newer commands on the list.
    """
    def __init__(self):
        gtk.ScrolledWindow.__init__(self)
        self._project = None

        self._model = gtk.ListStore(bool, str)
        self._treeview = gtk.TreeView(self._model)
        self._treeview.set_headers_visible(False)
        
        column = gtk.TreeViewColumn()
        renderer1 = gtk.CellRendererPixbuf()
        column.pack_start(renderer1, expand=False)
        column.set_cell_data_func(renderer1, self._draw_redo_position)

        renderer2 = gtk.CellRendererText()
        column.pack_start(renderer2, expand=True)
        column.add_attribute(renderer2, 'text', 1)
        
        self._treeview.append_column(column)

        self.add(self._treeview)

    def set_project(self, project):
        self._project = project
        self.update()
        
    def update(self):
        self._model.clear()
        if self._project is None:
            return
        
        i = 0
        for cmd in self._project.undo_stack:
            if i == self._project.prev_redo_item:
                redo = True
            else:
                redo = False
            self._model.insert(0, (redo, cmd.description))
            i += 1

    def _draw_redo_position(self, column, cell, model, iter):
        is_the_one = model.get_value(iter, 0)

        if is_the_one:
            stock_id = gtk.STOCK_JUMP_TO
        else:
            stock_id = None

        cell.set_property('stock-id', stock_id)
        
gobject.type_register(CommandStackView)
