# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from os.path import join, exists

from gazpacho import palette, editor, project, catalog
from gazpacho.placeholder import Placeholder
from gazpacho.path import pixmaps_dir
from gazpacho.command import CommandManager, CommandStackView
from gazpacho.clipboard import Clipboard, ClipboardView
from gazpacho.gaction import GActionsView, GAction, GActionGroup
from gazpacho.gaction import GActionDialog, GActionGroupDialog
from gazpacho.widget import Widget
from gazpacho.l10n import _

class Application(object):
    def __init__(self):
        # see catalog.py
        self._catalogs = catalog.load_all()

        # The WidgetClass that we are about to add to a container. None if no
        # class is to be added. This also has to be in sync with the depressed
        # button in the Palette
        self._add_class = None

        # This is our current project
        self._project = None

        # debugging windows
        self._command_stack_window = None
        self._clipboard_window = None

        self._clipboard = Clipboard()
        
        # here we put views that should update when changing the project
        self._project_views = []

        self._window = self._application_window_create()

        self._active_view = None

        self._projects = []
        
        self._command_manager = CommandManager(self)

    def _widget_tree_view_create(self):
        from gazpacho.widgettreeview import WidgetTreeView
        view = WidgetTreeView(self)
        self._project_views.insert(0, view)
        view.set_project(self._project)
        view.set_size_request(150, 300)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
 
    def _command_stack_view_create(self):
        view = CommandStackView()
        self._project_views.insert(0, view)
        view.set_project(self._project)
        view.set_size_request(300, 200)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
        
    def _clipboard_view_create(self):
        view = ClipboardView(self._clipboard)
        view.set_size_request(300, 200)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
    
    def _gactions_view_create(self):
        view = GActionsView()
        view.set_project(self._project)
        self._project_views.insert(0, view)        
        view.set_size_request(150, -1)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
    
    def _application_window_create(self):
        application_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        application_window.move(0, 0)
        application_window.set_default_size(700, -1)
        gtk.window_set_default_icon_from_file(join(pixmaps_dir,
                                                   'gazpacho-icon.png'))
        application_window.connect('delete-event', self._delete_event)

        # Create the different widgets
        menubar, toolbar = self._construct_menu_and_toolbar(application_window)

        self._palette = palette.Palette(self._catalogs)
        self._palette.connect('toggled', self._palette_button_clicked)

        self._editor = editor.Editor(self)

        widget_view = self._widget_tree_view_create()

        self.gactions_view = self._gactions_view_create()
        
        self._statusbar = self._construct_statusbar()

        # Layout them on the window
        main_vbox = gtk.VBox()
        application_window.add(main_vbox)

        main_vbox.pack_start(menubar, False)
        main_vbox.pack_start(toolbar, False)

        hbox = gtk.HBox(spacing=6)
        frame = self._create_frame(_('Palette'), self._palette)
        hbox.pack_start(frame, False, False)

        vpaned = gtk.VPaned()
        hbox.pack_start(vpaned, False, False)
        
        frame = self._create_frame(_('Widget Tree'), widget_view)
        vpaned.add1(frame)
        
        frame = self._create_frame(_('Actions'), self.gactions_view)
        vpaned.add2(frame)
        
        frame = self._create_frame(_('Editor'), self._editor)
        hbox.pack_start(frame, True, True)

        main_vbox.pack_start(hbox)
        
        main_vbox.pack_end(self._statusbar, False)

        self.refresh_undo_and_redo()

        return application_window

    def _expander_activated(self, expander, other_expander):
        # Make the widget tree and property editor use all available
        # space when the other one is collapsed.
        box = expander.parent

        # When both are collapsed, neither should expand to get them
        # neatly packed at the top.
        if expander.get_expanded() and not other_expander.get_expanded():
            expand = False
            other_expand = False
        else:
            expand = not expander.get_expanded()
            other_expand = not expand
        
        box.set_child_packing(expander,
                              expand,
                              True,
                              0,
                              gtk.PACK_START)
        box.set_child_packing(other_expander,
                              other_expand,
                              True,
                              0,
                              gtk.PACK_START)
            
    def _create_expander(self, label, widget, expanded=True):
        expander = gtk.Expander('<span size="large" weight="bold">%s</span>' % label)
        expander.set_use_markup(True)
        expander.set_expanded(expanded)
        expander.add(widget)
        return expander
    
    def _create_frame(self, label, widget):
        frame = gtk.Frame()
        label = gtk.Label('<span size="large" weight="bold">%s</span>' % label)
        label.set_use_markup(True)
        frame.set_label_widget(label)
        frame.set_shadow_type(gtk.SHADOW_NONE)
        frame.add(widget)
        return frame
    
    def _construct_menu_and_toolbar(self, application_window):
        actions =(
            ('FileMenu', None, _('_File')),
            ('New', gtk.STOCK_NEW, _('_New'), '<control>N',
             _('New Project'), self._new_cb),
            ('Open', gtk.STOCK_OPEN, _('_Open'), '<control>O',
             _('Open Project'), self._open_cb),
            ('Save', gtk.STOCK_SAVE, _('_Save'), '<control>S',
             _('Save Project'), self._save_cb),
            ('SaveAs', gtk.STOCK_SAVE_AS, _('_Save As...'),
             '<shift><control>S', _('Save project with different name'),
             self._save_as_cb),
            ('Close', gtk.STOCK_CLOSE, _('_Close'), '<control>W',
             _('Close Project'), self._close_cb),
            ('Quit', gtk.STOCK_QUIT, _('_Quit'), '<control>Q', _('Quit'),
             self._quit_cb),
            ('EditMenu', None, '_Edit'),
            ('Cut', gtk.STOCK_CUT, _('C_ut'), '<control>X', _('Cut'),
             self._cut_cb),
            ('Copy', gtk.STOCK_COPY, _('_Copy'), '<control>C', _('Copy'),
             self._copy_cb),
            ('Paste', gtk.STOCK_PASTE, _('_Paste'), '<control>V', _('Paste'),
             self._paste_cb),
            ('Delete', gtk.STOCK_DELETE, _('_Delete'), '<control>D',
             _('Delete'), self._delete_cb),
            ('ActionMenu', None, _('_Actions')),
            ('AddAction', gtk.STOCK_ADD, _('_Add action'), '<control>A',
             _('Add an action'), self._add_action_cb),
            ('RemoveAction', gtk.STOCK_REMOVE, _('_Remove action'), None,
             _('Remove action'), self._remove_action_cb),
            ('EditAction', None, _('_Edit action'), None, _('Edit Action'),
             self._edit_action_cb),
            ('ProjectMenu', None, _('_Project')),
            ('DebugMenu', None, _('_Debug')),
            ('HelpMenu', None, _('_Help')),
            ('About', None, _('_About'), None, _('About'), self._about_cb)
            )

        toggle_actions = (
            ('ShowCommandStack', None, _('Show _command stack'), 'F3',
             _('Show the command stack'), self._show_command_stack_cb, False),
            ('ShowClipboard', None, _('Show _clipboard'), 'F4',
             _('Show the clipboard'), self._show_clipboard_cb, False),
            )
        
        undo_action = (
            ('Undo', gtk.STOCK_UNDO, _('_Undo'), '<control>Z',
             _('Undo last action'), self._undo_cb),
            )

        redo_action = (
            ('Redo', gtk.STOCK_REDO, _('_Redo'), '<control>R',
             _('Redo last action'), self._redo_cb),
            )
        
        ui_description = """
<ui>
  <menubar name="MainMenu">
    <menu action="FileMenu">
      <menuitem action="New"/>
      <menuitem action="Open"/>
      <separator name="FM1"/>
      <menuitem action="Save"/>
      <menuitem action="SaveAs"/>
      <separator name="FM2"/>
      <menuitem action="Close"/>
      <menuitem action="Quit"/>
    </menu>
    <menu action="EditMenu">
      <menuitem action="Undo"/>
      <menuitem action="Redo"/>
      <separator name="EM1"/>
      <menuitem action="Cut"/>
      <menuitem action="Copy"/>
      <menuitem action="Paste"/>
      <menuitem action="Delete"/>
    </menu>
    <menu action="ActionMenu">
      <menuitem action="AddAction"/>
      <menuitem action="RemoveAction"/>
      <menuitem action="EditAction"/>
    </menu>
    <menu action="ProjectMenu">
    </menu>
    <menu action="DebugMenu">
      <menuitem action="ShowCommandStack"/>
      <menuitem action="ShowClipboard"/>
    </menu>
    <menu action="HelpMenu">
      <menuitem action="About"/>
    </menu>
  </menubar>
  <toolbar name="MainToolbar">
    <toolitem action="Quit"/>
    <toolitem action="Open"/>
    <toolitem action="Save"/>
    <separator name="MT1"/>
    <toolitem action="Undo"/>
    <toolitem action="Redo"/>    
    <separator name="MT2"/>
    <toolitem action="Cut"/>
    <toolitem action="Copy"/>
    <toolitem action="Paste"/>
    <toolitem action="Delete"/>
  </toolbar>
</ui>
"""
        self._ui_manager = gtk.UIManager()

        action_group = gtk.ActionGroup('MenuActions')
        action_group.add_actions(actions)
        action_group.add_toggle_actions(toggle_actions)
        self._ui_manager.insert_action_group(action_group, 0)

        action_group = gtk.ActionGroup('UndoAction')
        action_group.add_actions(undo_action)
        self._ui_manager.insert_action_group(action_group, 0)
        
        action_group = gtk.ActionGroup('RedoAction')
        action_group.add_actions(redo_action)
        self._ui_manager.insert_action_group(action_group, 0)
        
        self._ui_manager.add_ui_from_string(ui_description)
        
        application_window.add_accel_group(self._ui_manager.get_accel_group())

        menu = self._ui_manager.get_widget('/MainMenu')
        toolbar = self._ui_manager.get_widget('/MainToolbar')
        return (menu, toolbar)
    
    def _construct_statusbar(self):
        statusbar = gtk.Statusbar()
        self._statusbar_menu_context_id = statusbar.get_context_id("menu")
        self._statusbar_actions_context_id = statusbar.get_context_id("actions")
        return statusbar

    # some properties
    def get_window(self): return self._window
    window = property(get_window)
    
    def _push_statusbar_hint(self, menuitem, tip):
        self._statusbar.push(self._statusbar_menu_context_id, tip)

    def _pop_statusbar_hint(self, menuitem, data=None):
        self._statusbar.pop(self._statusbar_menu_context_id)
        
    def refresh_undo_and_redo(self):
        undo_item = redo_item = None
        if self._project is not None:
            pri = self._project.prev_redo_item
            if pri != -1:
                undo_item = self._project.undo_stack[pri]
            if pri + 1 < len(self._project.undo_stack):
                redo_item = self._project.undo_stack[pri + 1]

        undo_action = self._ui_manager.get_action('/MainToolbar/Undo')
        undo_group = undo_action.get_property('action-group')
        undo_group.set_sensitive(undo_item is not None)
        undo_widget = self._ui_manager.get_widget('/MainMenu/EditMenu/Undo')
        label = undo_widget.get_child()
        if undo_item is not None:
            label.set_text_with_mnemonic(_('_Undo: %s') % \
                                         undo_item.description)
        else:
            label.set_text_with_mnemonic(_('_Undo: Nothing'))
            
        redo_action = self._ui_manager.get_action('/MainToolbar/Redo')
        redo_group = redo_action.get_property('action-group')
        redo_group.set_sensitive(redo_item is not None)
        redo_widget = self._ui_manager.get_widget('/MainMenu/EditMenu/Redo')
        label = redo_widget.get_child()
        if redo_item is not None:
            label.set_text_with_mnemonic(_('_Redo: %s') % \
                                         redo_item.description)
        else:
            label.set_text_with_mnemonic(_('_Redo: Nothing'))

        if self._command_stack_window is not None:
            command_stack_view = self._command_stack_window.get_child()
            command_stack_view.update()

    def show_all(self):
        self._window.show_all()
        
    # projects
    def new_project(self):
        prj = project.Project(True, self)
        self._add_project(prj)

    def open_project(self, path):
        prj = project.Project.open(path, self)

        if prj is not None:
            self._add_project(prj)
                
    def _add_project(self, project):
        # if the project was previously added, don't reload
        for prj in self._projects:
            if prj.path and prj.path == project.path:
                self._set_project(prj)
                return

        self._projects.insert(0, project)

        # add the project in the /Project menu
        project_action= gtk.Action(project.name, project.name, project.name,
                                   '')

        project_action.connect('activate', self._set_project, project)
        project_ui = """
        <ui>
          <menubar name="MainMenu">
            <menu action="ProjectMenu">
              <menuitem action="%s"/>
            </menu>
          </menubar>
        </ui>
        """ % project.name
        action_group = self._ui_manager.get_action_groups()[0]
        action_group.add_action(project_action)
        
        project.uim_id = self._ui_manager.add_ui_from_string(project_ui)

        # connect to the project signals so that the editor can be updated
        project.connect('widget_name_changed', self._widget_name_changed_cb)
        project.connect('selection_changed',
                         self._project_selection_changed_cb)

        # make sure the palette is sensitive
        self._palette.set_sensitive(True)

        self._set_project(project)

    def _set_project(self, project):
        if not project in self._projects:
            print _('Could not set project because it could not be found in the list')
            return

        # clear the selection in the previous project
        if self._project:
            self._project.selection_clear(False)

        self._project = project
        self._refresh_title()

        for view in self._project_views:
            view.set_project(self._project)
            
        # trigger the selection changed signal to update the editor
        self._project.selection_changed()
        
    def _refresh_title(self):
        if self._project:
            title = 'Gazpacho - %s' % self._project.name
        else:
            title = 'Gazpacho'
        self._window.set_title(title)

    def _refresh_project_entry(self, project):
        item = self._ui_manager.get_widget(project.entry.path)
        label = item.get_child()
        # change the menu item's label
        label.set(project.name)
        project.entry.path = "/Project/%s" % project.name
        
    # callbacks
    def _delete_event(self, window, event):
        self._quit_cb()

        # return TRUE to stop other handlers
        return True
    
    def _new_cb(self, action):
        self.new_project()

    def _open_cb(self, action):
        filechooser = gtk.FileChooserDialog(_('Open ...'), self._window,
                                            gtk.FILE_CHOOSER_ACTION_OPEN,
                                            (gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_CANCEL,
                                             gtk.STOCK_OPEN,
                                             gtk.RESPONSE_OK))
        file_filter = gtk.FileFilter()
        file_filter.add_pattern("*.glade")
        filechooser.set_filter(file_filter)
        if filechooser.run() == gtk.RESPONSE_OK:
            path = filechooser.get_filename()
            if path:
                self.open_project(path)

        filechooser.destroy()

    def _save(self, project, path):
        """ Internal save """
        project.save(path)
#        self._refresh_project_entry(project)
        self._refresh_title()
        
    def _save_cb(self, action):
        project = self._project
        if project.path is not None:
            project.save(project.path)
            return

        # If instead we don't have a path yet, fire up a file chooser
        self._save_as_cb(None)

    def _save_as_cb(self, action):
        project = self._project
        if action is None:
            # we were called from the _save_cb callback
            title = _('Save')
        else:
            title = _('Save as')
        self._project_save_as(title, project)

    def _project_save_as(self, title, project):
        saved = False
        filechooser = gtk.FileChooserDialog(title, self._window,
                                            gtk.FILE_CHOOSER_ACTION_SAVE,
                                            (gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_CANCEL,
                                             gtk.STOCK_SAVE,
                                             gtk.RESPONSE_OK))
        filechooser.set_current_name("%s.glade" % project.name)
        while True:
            if filechooser.run() == gtk.RESPONSE_OK:
                path = filechooser.get_filename()
                if exists(path):
                    msg = gtk.MessageDialog(parent=self._window,
                                            flags=gtk.DIALOG_MODAL \
                                            | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            type=gtk.MESSAGE_WARNING,
                                            buttons=(gtk.BUTTONS_YES_NO),
                                            message_format=_('There is a file with that name already.\nWould you like to overwrite it?'))
                    result = msg.run()
                    msg.destroy()
                    # the user want to overwrite the file
                    if result == gtk.RESPONSE_YES:
                        self._save(project, path)
                        saved = True
                        break

                # the file does not exists. we can save it without worries
                else:
                    self._save(project, path)
                    saved = True
                    break
            # maybe the user changed his/her opinion and don't want to
            # save (by clicking cancel on the file chooser)
            else:
                break
                
        filechooser.destroy()
        return saved

    def _confirm_close_project(self, project):
        submsg1 = _('Save changes to project \"%s\" before closing?') % \
                  project.name
        submsg2 = _("Your changes will be lost if you don't save them")
        msg = "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s\n" % \
              (submsg1, submsg2)
        dialog = gtk.MessageDialog(self._window, gtk.DIALOG_MODAL,
                                    gtk.MESSAGE_QUESTION, gtk.BUTTONS_NONE,
                                    msg)
        dialog.label.set_use_markup(True)
        dialog.add_buttons(_("Close without Saving"), gtk.RESPONSE_NO,
                            gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                            gtk.STOCK_SAVE, gtk.RESPONSE_YES)
        dialog.set_default_response(gtk.RESPONSE_YES)
        while True:
            ret = dialog.run()
            if ret == gtk.RESPONSE_YES:
                if project.path:
                    project.save(project.path)
                    close = True
                else:
                    title = _('Save as')
                    saved = self._project_save_as(title, project)
                    if not saved:
                        continue
            elif ret == gtk.RESPONSE_NO:
                close = True
            else:
                close = False
            break
        dialog.destroy()
        return close
    
    def _close_cb(self, action):
        if not self._project:
            return

        if self._project.changed:
            close = self._confirm_close_project(self._project)
            if not close:
                return

        self._project.selection_clear(False)
        self._project.selection_changed()
        
        for widget in self._project.widgets:
            widget.destroy()

        self._ui_manager.remove_ui(self._project.uim_id)
        self._projects.remove(self._project)

        # If no more projects
        if not self._projects:
            for view in self._project_views:
                view.set_project(None)
            self._project = None
            self._refresh_title()
            self._palette.set_sensitive(False)
            return

        # this is needed to prevent clearing the selection of a closed project
        self._project = None
        self._set_project(self._projects[0])

    def _quit_cb(self, action=None):
        projects = self._projects
        for project in projects:
            if project.changed:
                if not self._confirm_close_project(project):
                    return
                self._projects.remove(project)
        gtk.main_quit()

    def _undo_cb(self, action):
        self._command_manager.undo(self._project)
        self.refresh_undo_and_redo()
        self._editor.refresh()

    def _redo_cb(self, action):
        self._command_manager.redo(self._project)
        self.refresh_undo_and_redo()
        self._editor.refresh()

    def _real_cut(self):
        selection = self._project.selection
        if selection:
            gwidget = Widget.get_from_gtk_widget(selection[0])
            self._command_manager.cut(gwidget)
            
    def _cut_cb(self, action):
        self._real_cut()

    def _real_copy(self):
        selection = self._project.selection
        if selection:
            gwidget = Widget.get_from_gtk_widget(selection[0])
            self._command_manager.copy(gwidget)
    
    def _copy_cb(self, action):
        self._real_copy()

    def _real_paste(self):
        if self._clipboard.get_selected_widget() is None:
            print _('There is nothing in the clipboard to paste')
            return
        selection = self._project.selection
        if selection and isinstance(selection[0], Placeholder):
            self._command_manager.paste(selection[0])
        else:
            print _('Please, select a placeholder to paste the widget on')
            return
        
    def _paste_cb(self, action):
        self._real_paste()
        
    def _delete_cb(self, action):
        if self._project is not None:
            self._project.delete_selection()

    def _about_cb(self, action):
        dialog = gtk.Dialog(_('About Gazpacho'), self._window,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        dialog.set_resizable(False)
        dialog.set_border_width(12)
        dialog.vbox.set_spacing(6)
        image = gtk.Image()
        image.set_from_file(join(pixmaps_dir, 'gazpacho.png'))

        version = gtk.Label('<span size="xx-large"><b>Gazpacho 0.5.0</b></span>')
        version.set_use_markup(True)
        description = gtk.Label(_('Gazpacho is a User Interface Builder for GTK+.\n\n')+'\n\n<small>(C) 2004,2005 SICEm S.L.</small>')
        description.set_use_markup(True)
        description.set_justify(gtk.JUSTIFY_FILL)
        description.set_line_wrap(True)
        
        dialog.vbox.pack_start(image)
        dialog.vbox.pack_start(version)
        dialog.vbox.pack_start(description)
        
        dialog.vbox.show_all()
        dialog.run()
        dialog.destroy()

    def _palette_button_clicked(self, palette):
        klass = palette.current

        # klass may be None if the selector was pressed
        self._add_class = klass
        if klass and klass.is_toplevel():
            self._command_manager.create(klass, None, self._project)
            self._palette.unselect_widget()
            self._add_class = None

    def _widget_name_changed_cb(self, project, widget):
        self._editor.update_widget_name()

    def _project_selection_changed_cb(self, project):
        if self._project != project:
            self._set_project(project)
            return

        if self._editor:
            children = self._project.selection
            if len(children) == 1 and not isinstance(children[0], Placeholder):
                self._editor.load_widget(
                    Widget.get_from_gtk_widget(children[0])
                    )
            else:
                self._editor.load_widget(None)

    # debugging windows
    def _delete_event_for_debugging_window(self, window, event, action):
        # this will hide the window
        action.activate()
        # we don't want the window to be destroyed
        return True
    
    def _create_debugging_window(self, view, title, action):
        win = gtk.Window()
        win.set_title(title)
        win.set_transient_for(self.window)
        win.add(view)
        view.show_all()
        win.connect('delete-event', self._delete_event_for_debugging_window, action)
        return win
    
    def _show_command_stack_cb(self, action):
        if self._command_stack_window is None:
            view = self._command_stack_view_create()
            title = _('Command Stack')
            action = self._ui_manager.get_action('/MainMenu/DebugMenu/ShowCommandStack')
            self._command_stack_window = self._create_debugging_window(view,
                                                                       title,
                                                                       action)
            self._command_stack_window.show()
        else:
            if self._command_stack_window.get_property('visible'):
                self._command_stack_window.hide()
            else:
                self._command_stack_window.show_all()
                
    def _show_clipboard_cb(self, action):
        if self._clipboard_window is None:
            view = self._clipboard_view_create()
            title = _('Clipboard contents')
            action = self._ui_manager.get_action('/MainMenu/DebugMenu/ShowClipboard')
            self._clipboard_window = self._create_debugging_window(view,
                                                                   title,
                                                                   action)
            self._clipboard_window.show()
        else:
            if self._clipboard_window.get_property('visible'):
                self._clipboard_window.hide()
            else:
                self._clipboard_window.show_all()
                
    # useful properties
    def get_add_class(self): return self._add_class
    add_class = property(get_add_class)

    def get_palette(self): return self._palette
    palette = property(get_palette)

    def get_command_manager(self): return self._command_manager
    command_manager = property(get_command_manager)
    
    # some accesors needed by the unit tests
    def get_action(self, action_name):
        action = self._ui_manager.get_action('ui/' + action_name)
        return action

    def get_current_project(self):
        return self._project

    def get_accel_groups(self):
        return (self._ui_manager.get_accel_group(),)
    
    def get_clipboard(self):
        return self._clipboard
    
    def get_clipboard_view(self):
        if self._clipboard_window is not None:
            return self._clipboard_window.get_child()
        else:
            return None

    # actions and action groups
    def _real_add_action(self):
        """Can add a simple action or an action group depending on
        the current selection in the ActionsView"""
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            if isinstance(gaction, GActionGroup):
                # create an action with the selected action group
                # as the parent
                parent = gaction
            else:
                # create a brother action of the selected action
                parent = gaction.parent
            
            dialog = GActionDialog(self.window, None)
            if dialog.run() == gtk.RESPONSE_OK:
                values = dialog.get_values()
                self._command_manager.add_action(values, parent, self._project)
            dialog.destroy()
        else:
            # nothing is selected, we create an action group
            dialog = GActionGroupDialog(self.window, None)
            if dialog.run() == gtk.RESPONSE_OK:
                name = dialog.get_action_group_name()
                self._command_manager.add_action_group(name, self._project)
            dialog.destroy()


    def _add_action_cb(self, action):
        self._real_add_action()

    def _real_remove_action(self, gaction):
        if isinstance(gaction, GAction):
            self._command_manager.remove_action(gaction, self._project)
        else:
            self._command_manager.remove_action_group(gaction, self._project)

    def _remove_action_cb(self, action):
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            self._real_remove_action(gaction)

    def _real_edit_action(self, gaction):
        if isinstance(gaction, GAction):
            dialog = GActionDialog(self.window, gaction)
            if dialog.run() == gtk.RESPONSE_OK:
                new_values = dialog.get_values()
                self._command_manager.edit_action(gaction, new_values,
                                                  self._project)
            dialog.destroy()
        else:
            dialog = GActionGroupDialog(self.window, gaction)
            if dialog.run() == gtk.RESPONSE_OK:
                new_name = dialog.get_action_group_name()
                self._command_manager.edit_action_group(gaction, new_name,
                                                        self._project)
            dialog.destroy()
        
    def _edit_action_cb(self, action):
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            self._real_edit_action(gaction)
