/* Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* DO NOT USE THIS OBJECT AS A BASIS FOR A NEW OBJECT. */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <math.h>
#include <string.h>

#include "intl.h"
#include "object.h"
#include "element.h"
#include "connectionpoint.h"
#include "diarenderer.h"
#include "attributes.h"
#include "widgets.h"
#include "properties.h"

#include "pixmaps/attribute.xpm"

#define DEFAULT_WIDTH 2.0
#define DEFAULT_HEIGHT 1.0
#define FONT_HEIGHT 0.8
#define MULTIVALUE_BORDER_WIDTH_X 0.4
#define MULTIVALUE_BORDER_WIDTH_Y 0.2
#define TEXT_BORDER_WIDTH_X 1.0
#define TEXT_BORDER_WIDTH_Y 0.5

typedef struct _Attribute Attribute;

struct _AttributeState {
  ObjectState obj_state;

  gchar *name;
  real name_width;

  gboolean key;
  gboolean weakkey;
  gboolean derived;
  gboolean multivalue;

  real border_width;
  Color border_color;
  Color inner_color;
};

struct _Attribute {
  Element element;

  DiaFont *font;
  real font_height;
  gchar *name;
  real name_width;

  ConnectionPoint connections[8];

  gboolean key;
  gboolean weakkey;
  gboolean derived;
  gboolean multivalue;

  real border_width;
  Color border_color;
  Color inner_color;
};

static real attribute_distance_from(Attribute *attribute, Point *point);
static void attribute_select(Attribute *attribute, Point *clicked_point,
			   DiaRenderer *interactive_renderer);
static void attribute_move_handle(Attribute *attribute, Handle *handle,
				  Point *to, HandleMoveReason reason, 
				  ModifierKeys modifiers);
static void attribute_move(Attribute *attribute, Point *to);
static void attribute_draw(Attribute *attribute, DiaRenderer *renderer);
static void attribute_update_data(Attribute *attribute);
static Object *attribute_create(Point *startpoint,
			      void *user_data,
			      Handle **handle1,
			      Handle **handle2);
static void attribute_destroy(Attribute *attribute);
static Object *attribute_copy(Attribute *attribute);
static PropDescription *
attribute_describe_props(Attribute *attribute);
static void
attribute_get_props(Attribute *attribute, GPtrArray *props);
static void
attribute_set_props(Attribute *attribute, GPtrArray *props);

static void attribute_save(Attribute *attribute, ObjectNode obj_node,
			   const char *filename);
static Object *attribute_load(ObjectNode obj_node, int version,
			      const char *filename);

static ObjectTypeOps attribute_type_ops =
{
  (CreateFunc) attribute_create,
  (LoadFunc)   attribute_load,
  (SaveFunc)   attribute_save
};

ObjectType attribute_type =
{
  "ER - Attribute",   /* name */
  0,                      /* version */
  (char **) attribute_xpm,  /* pixmap */
  
  &attribute_type_ops       /* ops */
};

ObjectType *_attribute_type = (ObjectType *) &attribute_type;

static ObjectOps attribute_ops = {
  (DestroyFunc)         attribute_destroy,
  (DrawFunc)            attribute_draw,
  (DistanceFunc)        attribute_distance_from,
  (SelectFunc)          attribute_select,
  (CopyFunc)            attribute_copy,
  (MoveFunc)            attribute_move,
  (MoveHandleFunc)      attribute_move_handle,
  (GetPropertiesFunc)   object_create_props_dialog,
  (ApplyPropertiesFunc) object_apply_props_from_dialog,
  (ObjectMenuFunc)      NULL,
  (DescribePropsFunc)   attribute_describe_props,
  (GetPropsFunc)        attribute_get_props,
  (SetPropsFunc)        attribute_set_props,
};

static PropDescription attribute_props[] = {
  ELEMENT_COMMON_PROPERTIES,
  { "name", PROP_TYPE_STRING, PROP_FLAG_VISIBLE,
    N_("Name:"), NULL, NULL },
  { "key", PROP_TYPE_BOOL, PROP_FLAG_VISIBLE,
    N_("Key:"), NULL, NULL },
  { "weakkey", PROP_TYPE_BOOL, PROP_FLAG_VISIBLE,
    N_("Weak key:"), NULL, NULL },
  { "derived", PROP_TYPE_BOOL, PROP_FLAG_VISIBLE,
    N_("Derived:"), NULL, NULL },
  { "multivalue", PROP_TYPE_BOOL, PROP_FLAG_VISIBLE,
    N_("Multivalue:"), NULL, NULL },
  PROP_STD_LINE_WIDTH,
  PROP_STD_LINE_COLOUR,
  PROP_STD_FILL_COLOUR,
  PROP_STD_TEXT_FONT,
  PROP_STD_TEXT_HEIGHT,
  PROP_DESC_END
};

static PropDescription *
attribute_describe_props(Attribute *attribute)
{
  if (attribute_props[0].quark == 0)
    prop_desc_list_calculate_quarks(attribute_props);
  return attribute_props;
}

static PropOffset attribute_offsets[] = {
  ELEMENT_COMMON_PROPERTIES_OFFSETS,
  { "name", PROP_TYPE_STRING, offsetof(Attribute, name) },
  { "key", PROP_TYPE_BOOL, offsetof(Attribute, key) },
  { "weakkey", PROP_TYPE_BOOL, offsetof(Attribute, weakkey) },
  { "derived", PROP_TYPE_BOOL, offsetof(Attribute, derived) },
  { "multivalue", PROP_TYPE_BOOL, offsetof(Attribute, multivalue) },
  { "line_width", PROP_TYPE_REAL, offsetof(Attribute, border_width) },
  { "line_colour", PROP_TYPE_COLOUR, offsetof(Attribute, border_color) },
  { "fill_colour", PROP_TYPE_COLOUR, offsetof(Attribute, inner_color) },
  { "text_font", PROP_TYPE_FONT, offsetof(Attribute, font) },
  { "text_height", PROP_TYPE_REAL, offsetof(Attribute, font_height) },
  { NULL, 0, 0}
};


static void
attribute_get_props(Attribute *attribute, GPtrArray *props)
{
  object_get_props_from_offsets(&attribute->element.object, 
                                attribute_offsets, props);
}

static void
attribute_set_props(Attribute *attribute, GPtrArray *props)
{
  object_set_props_from_offsets(&attribute->element.object, 
                                attribute_offsets, props);
  attribute_update_data(attribute);
}

static real
attribute_distance_from(Attribute *attribute, Point *point)
{
  Object *obj = &attribute->element.object;
  return distance_rectangle_point(&obj->bounding_box, point);
}

static void
attribute_select(Attribute *attribute, Point *clicked_point,
	       DiaRenderer *interactive_renderer)
{
  element_update_handles(&attribute->element);
}

static void
attribute_move_handle(Attribute *attribute, Handle *handle,
		      Point *to, HandleMoveReason reason, 
		      ModifierKeys modifiers)
{
  assert(attribute!=NULL);
  assert(handle!=NULL);
  assert(to!=NULL);

  assert(handle->id < 8);
  element_move_handle(&attribute->element, handle->id, to, reason);
  attribute_update_data(attribute);
}

static void
attribute_move(Attribute *attribute, Point *to)
{
  attribute->element.corner = *to;
  attribute_update_data(attribute);
}

static void
attribute_draw(Attribute *attribute, DiaRenderer *renderer)
{
  DiaRendererClass *renderer_ops = DIA_RENDERER_GET_CLASS (renderer);
  Point center;
  Point start, end;
  Point p;
  Element *elem;
  real width;
  
  assert(attribute != NULL);
  assert(renderer != NULL);

  elem = &attribute->element;

  center.x = elem->corner.x + elem->width/2;
  center.y = elem->corner.y + elem->height/2;
  
  renderer_ops->set_fillstyle(renderer, FILLSTYLE_SOLID);
  renderer_ops->fill_ellipse(renderer, &center,
			      elem->width, elem->height,
			      &attribute->inner_color);

  renderer_ops->set_linewidth(renderer, attribute->border_width);
  if (attribute->derived) {
    renderer_ops->set_linestyle(renderer, LINESTYLE_DASHED);
    renderer_ops->set_dashlength(renderer, 0.3);
  } else {
    renderer_ops->set_linestyle(renderer, LINESTYLE_SOLID);
  }

  renderer_ops->draw_ellipse(renderer, &center,
			      elem->width, elem->height,
			      &attribute->border_color);

  if(attribute->multivalue) {
    renderer_ops->draw_ellipse(renderer, &center,
				elem->width - 2*MULTIVALUE_BORDER_WIDTH_X,
				elem->height - 2*MULTIVALUE_BORDER_WIDTH_Y,
				&attribute->border_color);
  }

  p.x = elem->corner.x + elem->width / 2.0;
  p.y = elem->corner.y + (elem->height - attribute->font_height)/2.0 +
         dia_font_ascent(attribute->name,
                         attribute->font, attribute->font_height);

  renderer_ops->set_font(renderer,  attribute->font, attribute->font_height);
  renderer_ops->draw_string(renderer, attribute->name, 
			     &p, ALIGN_CENTER, 
			     &color_black);

  if (attribute->key || attribute->weakkey) {
    if (attribute->weakkey) {
      renderer_ops->set_linestyle(renderer, LINESTYLE_DASHED);
      renderer_ops->set_dashlength(renderer, 0.3);
    } else {
      renderer_ops->set_linestyle(renderer, LINESTYLE_SOLID);
    }
    width = dia_font_string_width(attribute->name,
                                  attribute->font, attribute->font_height);
    start.x = center.x - width / 2;
    start.y = center.y + 0.4;
    end.x = center.x + width / 2;
    end.y = center.y + 0.4;
    renderer_ops->draw_line(renderer, &start, &end, &color_black);
  }
}

static void
attribute_update_data(Attribute *attribute)
{
  Element *elem = &attribute->element;
  Object *obj = &elem->object;
  Point center;
  ElementBBExtras *extra = &elem->extra_spacing;
  real half_x, half_y;

  attribute->name_width =
    dia_font_string_width(attribute->name,
                          attribute->font, attribute->font_height);

  elem->width = attribute->name_width + 2*TEXT_BORDER_WIDTH_X;
  elem->height = attribute->font_height + 2*TEXT_BORDER_WIDTH_Y;

  center.x = elem->corner.x + elem->width / 2.0;
  center.y = elem->corner.y + elem->height / 2.0;
  
  half_x = elem->width * M_SQRT1_2 / 2.0;
  half_y = elem->height * M_SQRT1_2 / 2.0;
    
  /* Update connections: */
  attribute->connections[0].pos.x = center.x - half_x;
  attribute->connections[0].pos.y = center.y - half_y;
  attribute->connections[1].pos.x = center.x;
  attribute->connections[1].pos.y = elem->corner.y;
  attribute->connections[2].pos.x = center.x + half_x;
  attribute->connections[2].pos.y = center.y - half_y;
  attribute->connections[3].pos.x = elem->corner.x;
  attribute->connections[3].pos.y = center.y;
  attribute->connections[4].pos.x = elem->corner.x + elem->width;
  attribute->connections[4].pos.y = elem->corner.y + elem->height / 2.0;
  attribute->connections[5].pos.x = center.x - half_x;
  attribute->connections[5].pos.y = center.y + half_y;
  attribute->connections[6].pos.x = elem->corner.x + elem->width / 2.0;
  attribute->connections[6].pos.y = elem->corner.y + elem->height;
  attribute->connections[7].pos.x = center.x + half_x;
  attribute->connections[7].pos.y = center.y + half_y;

  extra->border_trans = attribute->border_width/2.0;
  element_update_boundingbox(elem);

  obj->position = elem->corner;

  element_update_handles(elem);
  
}

static Object *
attribute_create(Point *startpoint,
	       void *user_data,
  	       Handle **handle1,
	       Handle **handle2)
{
  Attribute *attribute;
  Element *elem;
  Object *obj;
  int i;

  attribute = g_malloc0(sizeof(Attribute));
  elem = &attribute->element;
  obj = &elem->object;
  
  obj->type = &attribute_type;
  obj->ops = &attribute_ops;

  elem->corner = *startpoint;
  elem->width = DEFAULT_WIDTH;
  elem->height = DEFAULT_WIDTH;

  attribute->border_width =  attributes_get_default_linewidth();
  attribute->border_color = attributes_get_foreground();
  attribute->inner_color = attributes_get_background();

  element_init(elem, 8, 8);

  for (i=0;i<8;i++) {
    obj->connections[i] = &attribute->connections[i];
    attribute->connections[i].object = obj;
    attribute->connections[i].connected = NULL;
  }

  attribute->key = FALSE;
  attribute->weakkey = FALSE;
  attribute->derived = FALSE;
  attribute->multivalue = FALSE;
  attribute->font = dia_font_new_from_style(DIA_FONT_MONOSPACE,FONT_HEIGHT);
  attribute->font_height = FONT_HEIGHT;
  attribute->name = g_strdup(_("Attribute"));

  attribute->name_width =
    dia_font_string_width(attribute->name,
                          attribute->font, attribute->font_height);

  attribute_update_data(attribute);

  for (i=0;i<8;i++) {
    obj->handles[i]->type = HANDLE_NON_MOVABLE;
  }

  *handle1 = NULL;
  *handle2 = obj->handles[0];
  return &attribute->element.object;
}

static void
attribute_destroy(Attribute *attribute)
{
  element_destroy(&attribute->element);
  g_free(attribute->name);
}

static Object *
attribute_copy(Attribute *attribute)
{
  int i;
  Attribute *newattribute;
  Element *elem, *newelem;
  Object *newobj;
  
  elem = &attribute->element;
  
  newattribute = g_malloc0(sizeof(Attribute));
  newelem = &newattribute->element;
  newobj = &newelem->object;

  element_copy(elem, newelem);

  newattribute->border_width = attribute->border_width;
  newattribute->border_color = attribute->border_color;
  newattribute->inner_color = attribute->inner_color;

  for (i=0;i<8;i++) {
    newobj->connections[i] = &newattribute->connections[i];
    newattribute->connections[i].object = newobj;
    newattribute->connections[i].connected = NULL;
    newattribute->connections[i].pos = attribute->connections[i].pos;
    newattribute->connections[i].last_pos = attribute->connections[i].last_pos;
  }
  
  newattribute->font = dia_font_ref(attribute->font);
  newattribute->font_height = attribute->font_height;
  newattribute->name = g_strdup(attribute->name);
  newattribute->name_width = attribute->name_width;

  newattribute->key = attribute->key;
  newattribute->weakkey = attribute->weakkey;
  newattribute->derived = attribute->derived;
  newattribute->multivalue = attribute->multivalue;

  return &newattribute->element.object;
}


static void
attribute_save(Attribute *attribute, ObjectNode obj_node,
	       const char *filename)
{
  element_save(&attribute->element, obj_node);

  data_add_real(new_attribute(obj_node, "border_width"),
		attribute->border_width);
  data_add_color(new_attribute(obj_node, "border_color"),
		 &attribute->border_color);
  data_add_color(new_attribute(obj_node, "inner_color"),
		 &attribute->inner_color);
  data_add_string(new_attribute(obj_node, "name"),
		  attribute->name);
  data_add_boolean(new_attribute(obj_node, "key"),
		   attribute->key);
  data_add_boolean(new_attribute(obj_node, "weak_key"),
		   attribute->weakkey);
  data_add_boolean(new_attribute(obj_node, "derived"),
		   attribute->derived);
  data_add_boolean(new_attribute(obj_node, "multivalued"),
		   attribute->multivalue);
  data_add_font (new_attribute (obj_node, "font"),
		 attribute->font);
  data_add_real(new_attribute(obj_node, "font_height"),
		attribute->font_height);
}

static Object *attribute_load(ObjectNode obj_node, int version,
			      const char *filename)
{
  Attribute *attribute;
  Element *elem;
  Object *obj;
  int i;
  AttributeNode attr;

  attribute = g_malloc0(sizeof(Attribute));
  elem = &attribute->element;
  obj = &elem->object;
  
  obj->type = &attribute_type;
  obj->ops = &attribute_ops;

  element_load(elem, obj_node);

  attribute->border_width = 0.1;
  attr = object_find_attribute(obj_node, "border_width");
  if (attr != NULL)
    attribute->border_width =  data_real( attribute_first_data(attr) );

  attribute->border_color = color_black;
  attr = object_find_attribute(obj_node, "border_color");
  if (attr != NULL)
    data_color(attribute_first_data(attr), &attribute->border_color);
  
  attribute->inner_color = color_white;
  attr = object_find_attribute(obj_node, "inner_color");
  if (attr != NULL)
    data_color(attribute_first_data(attr), &attribute->inner_color);
  
  attribute->name = NULL;
  attr = object_find_attribute(obj_node, "name");
  if (attr != NULL)
    attribute->name = data_string(attribute_first_data(attr));

  attr = object_find_attribute(obj_node, "key");
  if (attr != NULL)
    attribute->key = data_boolean(attribute_first_data(attr));

  attr = object_find_attribute(obj_node, "weak_key");
  if (attr != NULL)
    attribute->weakkey = data_boolean(attribute_first_data(attr));
  
  attr = object_find_attribute(obj_node, "derived");
  if (attr != NULL)
    attribute->derived = data_boolean(attribute_first_data(attr));

  attr = object_find_attribute(obj_node, "multivalued");
  if (attr != NULL)
    attribute->multivalue = data_boolean(attribute_first_data(attr));

  dia_font_unref(attribute->font);
  attribute->font = NULL;
  attr = object_find_attribute (obj_node, "font");
  if (attr != NULL)
	  attribute->font = data_font (attribute_first_data (attr));

  attribute->font_height = FONT_HEIGHT;
  attr = object_find_attribute (obj_node, "font_height");
  if (attr != NULL)
    attribute->font_height = data_real( attribute_first_data(attr) );

  element_init(elem, 8, 8);

  for (i=0;i<8;i++) {
    obj->connections[i] = &attribute->connections[i];
    attribute->connections[i].object = obj;
    attribute->connections[i].connected = NULL;
  }

  if (attribute->font == NULL) {
	  attribute->font = dia_font_new_from_style(DIA_FONT_MONOSPACE,
                                              attribute->font_height);
  }

  attribute->name_width = dia_font_string_width(attribute->name,
                                                attribute->font,
                                                attribute->font_height);
  attribute_update_data(attribute);

  for (i=0;i<8;i++) {
    obj->handles[i]->type = HANDLE_NON_MOVABLE;
  }

  return &attribute->element.object;
}
