/* session-properties.c - Edit session properties.

   Copyright 1999 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. 

   Authors: Tom Tromey, Felix Bellaby */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include "capplet-widget.h"
#include "session.h"

/* Config prefix holding the current session details */
static char      *prefix; 

static GtkWidget *capplet;
static GtkWidget *clist;
static GtkWidget *entry;
static GtkWidget *add_button;
static GtkWidget *remove_button;
static GdkFont   *font;
static gint       width;

/* CORBA callbacks and intialization */
static void capplet_build (void);
static void capplet_revert (void);
static void capplet_cancel (void);
static void capplet_ok (void);

/* internal callbacks */
static void program_row_selected_cb (GtkWidget *widget, gint row);
static void program_row_unselected_cb (GtkWidget *widget, gint row);
static void program_entry_changed_cb (GtkWidget *widget);
static void program_add_cb (GtkWidget *widget);
static void program_remove_cb (GtkWidget *widget);

static void
capplet_build (void)
{
  GtkWidget *table;
  GtkWidget *scrolled_window;

  /* capplet */
  capplet = capplet_widget_new ();
  gtk_signal_connect (GTK_OBJECT (capplet), "revert",
		      GTK_SIGNAL_FUNC (capplet_revert), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "cancel",
		      GTK_SIGNAL_FUNC (capplet_cancel), NULL);
  gtk_signal_connect (GTK_OBJECT (capplet), "ok",
		      GTK_SIGNAL_FUNC (capplet_ok), NULL);

  /* program entry box */
  entry = gtk_entry_new ();
  gtk_signal_connect (GTK_OBJECT (entry), "activate", 
		      GTK_SIGNAL_FUNC (program_add_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "changed", 
		      GTK_SIGNAL_FUNC (program_entry_changed_cb), NULL);

  /* add/remove buttons */
  add_button = gnome_stock_button (GNOME_STOCK_PIXMAP_ADD);
  gtk_widget_set_sensitive (GTK_WIDGET (add_button), FALSE);
  gtk_signal_connect (GTK_OBJECT (add_button), "clicked",
		      GTK_SIGNAL_FUNC (program_add_cb), NULL);
  remove_button = gnome_stock_button (GNOME_STOCK_PIXMAP_REMOVE);
  gtk_widget_set_sensitive (GTK_WIDGET (remove_button), FALSE);
  gtk_signal_connect(GTK_OBJECT (remove_button), "clicked",
		     GTK_SIGNAL_FUNC (program_remove_cb), NULL);

  /* startup program list */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  clist = gtk_clist_new (1);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist);
  gtk_clist_set_selection_mode (GTK_CLIST (clist), 
				GTK_SELECTION_MULTIPLE);
  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		     GTK_SIGNAL_FUNC (program_row_selected_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
		     GTK_SIGNAL_FUNC (program_row_unselected_cb), NULL);

  /* table */
  table = gtk_table_new (3, 4, FALSE);
  gtk_table_attach (GTK_TABLE (table), gtk_label_new (_("Program:")),
		    0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), gtk_hbox_new (FALSE, 0), 
		    1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), add_button, 
		    2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0); 
  gtk_table_attach (GTK_TABLE (table), remove_button, 
		    3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), entry, 
		    0, 4, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), scrolled_window, 
		    0, 4, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND,0, 0);

  gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD);
  gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
  gtk_container_add (GTK_CONTAINER(capplet), table);
}

int
main (int argc, char *argv[])
{
  gchar* name;

  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  switch(gnome_capplet_init("session-properties", VERSION, argc, argv,
			    NULL, 0, NULL)) {
  case 1:
	  return 0;
  case -1:
	  error (1);
  }

  gnome_client_set_restart_style (gnome_master_client (), GNOME_RESTART_NEVER);
  gtk_signal_connect (GTK_OBJECT (gnome_master_client ()), "die",
		      GTK_SIGNAL_FUNC (capplet_cancel), NULL);

  gnome_config_push_prefix (GSM_CONFIG_PREFIX);
  name = gnome_config_get_string (CURRENT_SESSION_KEY "=" DEFAULT_SESSION);
  gnome_config_pop_prefix ();

  prefix = g_strconcat (CONFIG_PREFIX, name, "/", NULL);
  g_free (name);

  capplet_build ();
  capplet_revert ();

  gtk_widget_show_all (GTK_WIDGET (capplet));

  capplet_gtk_main();

  return 0;
}

/* CORBA CALLBACKS */
static void
capplet_revert (void)
{
  int i, count;
  gboolean unaware, corrupt;

  font = gtk_widget_get_style (GTK_WIDGET (clist))->font;
  width = 40 * gdk_string_width (font, "n");

  gnome_config_push_prefix (prefix);
  count = gnome_config_get_int (CLIENT_COUNT_KEY "=0");
  gnome_config_pop_prefix ();

  gtk_clist_freeze (GTK_CLIST (clist));
  gtk_clist_clear (GTK_CLIST (clist));

  /* scan the session for session unaware clients.
   * These are with RestartCommands but no client ids. */
  for (i = 0; i < count; ++i)
    {
      gchar *value, buf[128];

      g_snprintf (buf, sizeof(buf), "%s%d,", prefix, i);
      gnome_config_push_prefix (buf);

      g_free (gnome_config_get_string_with_default ("id", &unaware));

      if (unaware)
	{
	  value = gnome_config_get_string_with_default ("RestartCommand", 
							&corrupt);
	  /* let gnome-session deal with the nasty cases */
	  if (corrupt)
	      exit (0);

	  gtk_clist_append (GTK_CLIST (clist), (gchar **) &value);
	  width = MAX (gdk_string_width (font, value), width);
	  g_free (value);
	}
      gnome_config_pop_prefix ();
    }
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, width);
  gtk_clist_thaw (GTK_CLIST (clist));
}

static void
capplet_ok (void)
{
  int old_i, new_i, last_i, i;
  void * iter;
  gchar *key, *value, name[1024];
  gboolean aware = FALSE;
  
  /* Only write to the session file when gnome-session is idle.
   * This is a fail safe against an unlikely but dangerous race.
   * gnome-session will wait for us to finish before starting a save. */
  if (gnome_master_client()->state != GNOME_CLIENT_DISCONNECTED &&
      gnome_master_client()->state != GNOME_CLIENT_IDLE)
    return;

  /* The algorithm to write this file is an awful lot easier when
   * we can iterate forwards through its contents rather than backwards.
   * Therefore, we copy the session over into another section to
   * reverse its order before doing the real work */
  iter = gnome_config_init_iterator (prefix);
  gnome_config_push_prefix (RESERVED_SECTION);
  
  /* Step over CLIENT_COUNT_KEY */
  iter = gnome_config_iterator_next (iter, &key, &value);

  while ((iter = gnome_config_iterator_next (iter, &key, &value)))
      gnome_config_set_string (key, value);

  gnome_config_pop_prefix ();
  gnome_config_clean_section (prefix);

  new_i = last_i = -1;

  iter = gnome_config_init_iterator (RESERVED_SECTION);
  gnome_config_push_prefix (prefix);

  /* Remove all non-aware clients from the saved default session */
  while ((iter = gnome_config_iterator_next (iter, &key, &value)))
    {
      sscanf (key, "%d,%1023s", &old_i, name);
      
      if (old_i != last_i)
	{
	  last_i = old_i;
	  
	  if ((aware = !strcmp (name, "id")))
	    ++new_i;
	}
      
      if (aware)
	{
	  if (old_i != new_i)
	    {
	      g_free (key);
	      key = g_strdup_printf ("%d,%s", new_i, name);
	    }
	  gnome_config_set_string (key, value);
	}
      g_free (key);
      g_free (value);
    }

  /* Replace them with the ones in our list. */
  for (i = 0; i < GTK_CLIST(clist)->rows; i++)
    {
      char buf[40];

      gtk_clist_get_text (GTK_CLIST(clist), i, 0, &value);
      g_snprintf (buf, sizeof(buf), "%d,RestartCommand", ++new_i);
      gnome_config_set_string (buf, value);
    }

  gnome_config_set_int (CLIENT_COUNT_KEY, ++new_i);
  gnome_config_pop_prefix ();
  gnome_config_clean_section (RESERVED_SECTION);
  gnome_config_sync ();

}

static void
capplet_cancel (void)
{
  gtk_main_quit ();
}

/* INTERNAL CALLBACKS */
static void
program_row_selected_cb (GtkWidget *widget, gint row)
{
  gtk_widget_set_sensitive (GTK_WIDGET (remove_button), TRUE);
}


static void
program_row_unselected_cb (GtkWidget *widget, gint row)
{
  if (GTK_CLIST (clist)->selection == NULL)
    {
      gtk_widget_set_sensitive (GTK_WIDGET (remove_button), FALSE);
    }
}

static void
program_entry_changed_cb (GtkWidget *widget)
{
  gtk_widget_set_sensitive (GTK_WIDGET (add_button), TRUE);
}

static void
program_add_cb (GtkWidget *widget)
{
  gchar *value;

  value = gtk_entry_get_text (GTK_ENTRY (entry));
  if (strcmp (value, ""))
    {
      gtk_clist_freeze (GTK_CLIST (clist));
      gtk_clist_append (GTK_CLIST (clist), (gchar **) &value);
      width = MAX(gdk_string_width(font, value), 
			       width);
      gtk_clist_set_column_width (GTK_CLIST (clist), 
				  0, width);
      gtk_clist_thaw (GTK_CLIST (clist));
      gtk_entry_set_text (GTK_ENTRY (entry), "");
      capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (add_button), FALSE);
    }
}

static void
program_remove_cb (GtkWidget *widget)
{
  gtk_clist_freeze (GTK_CLIST (clist));

  while (GTK_CLIST (clist)->selection != NULL)
    {
      gint row = GPOINTER_TO_INT (GTK_CLIST (clist)->selection->data);
      /* This modifies clist->selection, so eventually the loop will
       * terminate.
       */
      gtk_clist_remove (GTK_CLIST (clist), row);
    }

  gtk_clist_thaw (GTK_CLIST (clist));
  capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}
