/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * This file is part of capuchin-glib. 
 *
 * Copyright (C) Sebastian Pölsterl 2008 <marduk@k-d-w.org>
 * 
 * capuchin-glib is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 * 
 * capuchin-glib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with capuchin-glib.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <dbus/dbus-glib.h>
#include <capuchin-glib/capuchin-g-app-object-manager.h>
#include <capuchin-glib/capuchin-glib-globals.h>

#define CAPUCHIN_G_APP_OBJECT_MANAGER_METHOD_GetAppObject "GetAppObject"

#define CAPUCHIN_G_APP_OBJECT_MANAGER_PRIVATE(o)(G_TYPE_INSTANCE_GET_PRIVATE ((o), CAPUCHIN_TYPE_G_APP_OBJECT_MANAGER, CapuchinGAppObjectManagerPrivate))



struct _CapuchinGAppObjectManagerPrivate
{
	DBusGProxy* proxy;
};

G_DEFINE_TYPE (CapuchinGAppObjectManager, capuchin_g_app_object_manager, G_TYPE_OBJECT);

static void
capuchin_g_app_object_manager_init (CapuchinGAppObjectManager *object)
{
	object->priv = CAPUCHIN_G_APP_OBJECT_MANAGER_PRIVATE (object);
	object->priv->proxy = NULL;
}

static void
capuchin_g_app_object_manager_finalize (GObject *object)
{
	CapuchinGAppObjectManager *manager;
	
	manager = CAPUCHIN_G_APP_OBJECT_MANAGER (object);
	
	g_object_unref (manager->priv->proxy);

	G_OBJECT_CLASS (capuchin_g_app_object_manager_parent_class)->finalize (object);
}

static void
capuchin_g_app_object_manager_class_init (CapuchinGAppObjectManagerClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	GObjectClass* parent_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (CapuchinGAppObjectManagerPrivate));

	object_class->finalize = capuchin_g_app_object_manager_finalize;
}

/**
 * capuchin_g_app_object_manager_new
 * @returns: a new #CapuchinGAppObjectManager instance or NULL if
 * an error occured
 */
CapuchinGAppObjectManager*
capuchin_g_app_object_manager_new (void)
{
	DBusGConnection *conn;
	GError *error = NULL;
	DBusGProxy *proxy;
	CapuchinGAppObjectManager *manager;
	
	conn = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
	if (conn == NULL) {
		g_critical ("Failed to open connection to bus: %s\n", error->message);
		g_error_free (error);
		return NULL;
	}
	
	proxy = dbus_g_proxy_new_for_name(conn,
			CAPUCHIN_G_DBUS_SERVICE,
			CAPUCHIN_G_APP_OBJECT_MANAGER_DBUS_PATH,
			CAPUCHIN_G_APP_OBJECT_MANAGER_DBUS_INTERFACE);
	
	manager = CAPUCHIN_G_APP_OBJECT_MANAGER (g_object_new (CAPUCHIN_TYPE_G_APP_OBJECT_MANAGER, NULL));
	manager->priv->proxy = proxy;
	
	return manager;
}	

/**
 * capuchin_g_app_object_manager_get_appobject
 * @manager: #CapuchinGAppObjectManager object
 * @url: URL where the XML repository file is located
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: a #CapuchinGAppObject for the repository or NULL if an error occured
 *
 * Get a application specific #CapuchinGAppObject for the repository located at given URL
 */
CapuchinGAppObject*
capuchin_g_app_object_manager_get_appobject (CapuchinGAppObjectManager *manager,
											 const gchar *url, GError **error)
{
	DBusGProxy *proxy;
	CapuchinGAppObject *appobject;
		
	if (!dbus_g_proxy_call(manager->priv->proxy, CAPUCHIN_G_APP_OBJECT_MANAGER_METHOD_GetAppObject , error,
			G_TYPE_STRING, url, G_TYPE_INVALID,
			DBUS_TYPE_G_PROXY, &proxy, G_TYPE_INVALID)) {
		return NULL;
	}
	
	dbus_g_proxy_set_interface (proxy, CAPUCHIN_G_APP_OBJECT_DBUS_INTERFACE);
	
	appobject = capuchin_g_app_object_new (proxy);
	
	return appobject;
}
