/* Copyright 2014 Yorba Foundation
 *
 * This software is licensed under the GNU Lesser General Public License
 * (version 2.1 or later).  See the COPYING file in this distribution.
 */

namespace California.Backing {

/**
 * Subscribe to all {@link CalendarSource}s and their {@link Component.Instance}s for a specific
 * span of time.
 *
 * This class manages the signals and {@link CalendarSourceSubscription}s for all registered
 * calendars and converts their important events into a set of symmetric signals.  It also
 * automatically subscribes to new calendars (and unsubscribes to dropped ones) notifying of changes
 * through the same set of signals  Callers should simply subscribe to those signals and let them
 * drive the application.
 *
 * The time span {@link window} cannot be altered once the object is created.
 */

public class CalendarSubscriptionManager : BaseObject {
    /**
     * The time span for all managed subscriptions.
     */
    public Calendar.ExactTimeSpan window { get; private set; }
    
    /**
     * Set to true when {@link start_async} begins.
     */
    public bool is_started { get; private set; default = false; }
    
    /**
     * Indicates a {@link CalendarSource} was added to the manager, either listed when first
     * created or detected at runtime afterwards.
     */
    public signal void calendar_added(Backing.CalendarSource calendar);
    
    /**
     * Indicates the {@link CalendarSource} was removed from the manager.
     */
    public signal void calendar_removed(Backing.CalendarSource calendar);
    
    /**
     * Indicates the {@link Component.Instance} was generated by one of the managed subscriptions,
     * either generated (discovered) when first opened or added later.
     */
    public signal void instance_added(Component.Instance instance);
    
    /**
     * Indicates the {@link Component.Instance} was removed by one of the managed subscriptions,
     * either due to the {@link CalendarSource} being made unavailable or removal by the user.
     */
    public signal void instance_removed(Component.Instance instance);
    
    /**
     * An error was returned when attempting to subscribe to the {@link CalendarSource}.
     */
    public signal void subscription_error(Backing.CalendarSource calendar, Error err);
    
    private Gee.ArrayList<Backing.CalendarSourceSubscription> subscriptions = new Gee.ArrayList<
        Backing.CalendarSourceSubscription>();
    private Cancellable cancellable = new Cancellable();
    
    /**
     * Create a new {@link CalendarSubscriptionManager}.
     *
     * The {@link window} cannot be modified once created.
     *
     * Events will not be signalled until {@link start_async} is called.
     */
    public CalendarSubscriptionManager(Calendar.ExactTimeSpan window) {
        this.window = window;
    }
    
    ~CalendarSubscriptionManager() {
        // cancel any outstanding subscription starts
        cancellable.cancel();
        
        // drop signals on objects that will persist after this object's destruction
        foreach (Backing.Store store in Backing.Manager.instance.get_stores()) {
            store.source_added.disconnect(on_source_added);
            store.source_removed.disconnect(on_source_removed);
        }
    }
    
    /**
     * Generate subscriptions and begin firing signals.
     *
     * There is no "stop" method.  Destroying the object will cancel all subscriptions, although
     * signals will not be fired at that time.
     */
    public async void start_async() {
        // to prevent reentrancy
        if (is_started)
            return;
        
        is_started = true;
        
        foreach (Backing.Store store in Backing.Manager.instance.get_stores()) {
            // watch each store for future added sources
            store.source_added.connect(on_source_added);
            store.source_removed.connect(on_source_removed);
            
            foreach (Backing.Source source in store.get_sources_of_type<Backing.CalendarSource>())
                yield add_calendar_async((Backing.CalendarSource) source, cancellable);
        }
    }
    
    private void on_source_added(Backing.Source source) {
        Backing.CalendarSource? calendar = source as Backing.CalendarSource;
        if (calendar != null)
            add_calendar_async.begin(calendar, cancellable);
    }
    
    private async void add_calendar_async(Backing.CalendarSource calendar, Cancellable? cancellable) {
        // report calendar as added to subscription
        calendar_added(calendar);
        
        // start generating instances on this calendar
        try {
            // Since this might be called after the dtor has finished (cancelling the operation), don't
            // touch the "this" ref unless the Error is shown not to be a cancellation
            Backing.CalendarSourceSubscription subscription = yield calendar.subscribe_async(window,
                cancellable);
            
            // okay to use "this" ref
            subscriptions.add(subscription);
            
            subscription.instance_discovered.connect(on_instance_added);
            subscription.instance_added.connect(on_instance_added);
            subscription.instance_removed.connect(on_instance_removed);
            subscription.instance_dropped.connect(on_instance_removed);
            subscription.start_failed.connect(on_error);
            
            // this will start signals firing for event changes
            subscription.start();
        } catch (Error err) {
            debug("Unable to subscribe to %s: %s", calendar.to_string(), err.message);
            
            // only fire -- or even touch "this" -- if not a cancellation
            if (!(err is IOError.CANCELLED))
                subscription_error(calendar, err);
        }
    }
    
    private void on_instance_added(Component.Instance instance) {
        instance_added(instance);
    }
    
    private void on_instance_removed(Component.Instance instance) {
        instance_removed(instance);
    }
    
    private void on_error(CalendarSourceSubscription subscription, Error err) {
        subscription_error(subscription.calendar, err);
    }
    
    // Don't need to do much here as all instances are dropped prior to the source being removed
    private void on_source_removed(Backing.Source source) {
        Backing.CalendarSource? calendar = source as Backing.CalendarSource;
        if (calendar == null)
            return;
        
        // drop all related subscriptions ... their instances should've been dropped via the
        // "instance-dropped" signal, so no signal their removal here
        Gee.Iterator<CalendarSourceSubscription> iter = subscriptions.iterator();
        while (iter.next()) {
            if (iter.get().calendar == calendar)
                iter.remove();
        }
        
        calendar_removed(calendar);
    }
    
    public override string to_string() {
        return "%s window=%s".printf(get_class().get_type().name(), window.to_string());
    }
}

}

