/**
 * bonobo-property-editor.c:
 *
 * Author:
 *   Dietmar Maurer (dietmar@ximian.com)
 *
 * Copyright 2001 Ximian, Inc.
 */

#include <config.h>
#include <gtk/gtksignal.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-property-bag-client.h>
#include <liboaf/liboaf.h>

#include "bonobo-property-editor.h"

#define CLASS(o) BONOBO_PROPERTY_EDITOR_CLASS (GTK_OBJECT(o)->klass)

struct _BonoboPropertyEditorPrivate {
	Bonobo_Property               property;
	BonoboPropertyEditorSetFn     set_cb;
	Bonobo_EventSource_ListenerId id;
};

#define  TYPE_PRINT(tckind)			                        \
	case tckind:							\
                printf (#tckind "\n");           			\
                break;

void
print_typecode (CORBA_TypeCode tc, const gchar *name, int level)
{
        int i, j;

	for (i=0;i<level;i++) printf(" ");          
	if (name) printf ("(%s)", name);
	printf ("(%d)(%p)", ORBIT_ROOT_OBJECT(tc)->refs, tc);

	switch (tc->kind) {

		TYPE_PRINT(CORBA_tk_null);
		TYPE_PRINT(CORBA_tk_void);
		TYPE_PRINT(CORBA_tk_short);
		TYPE_PRINT(CORBA_tk_long);
		TYPE_PRINT(CORBA_tk_ushort);
		TYPE_PRINT(CORBA_tk_ulong);
		TYPE_PRINT(CORBA_tk_float);
		TYPE_PRINT(CORBA_tk_double);
		TYPE_PRINT(CORBA_tk_boolean);
		TYPE_PRINT(CORBA_tk_char);
		TYPE_PRINT(CORBA_tk_octet);
		TYPE_PRINT(CORBA_tk_string);
		TYPE_PRINT(CORBA_tk_longlong);
		TYPE_PRINT(CORBA_tk_ulonglong);
		TYPE_PRINT(CORBA_tk_longdouble);
		TYPE_PRINT(CORBA_tk_wchar);
		TYPE_PRINT(CORBA_tk_wstring);
		TYPE_PRINT(CORBA_tk_objref);
		TYPE_PRINT(CORBA_tk_any);
		TYPE_PRINT(CORBA_tk_TypeCode);
		
	case CORBA_tk_struct:
		for (i=0;i<level;i++) printf(" "); 
		printf ("CORBA_tk_struct %s\n", tc->repo_id);
		for (i = 0; i < tc->sub_parts; i++) {
			print_typecode (tc->subtypes [i], tc->subnames [i],
					level+2);
		}
		break;

	case CORBA_tk_sequence:
		for (i=0;i<level;i++) printf(" "); 
		printf ("CORBA_tk_sequence\n");
		print_typecode (tc->subtypes [0], NULL, level+2);
		break;

	case CORBA_tk_alias:
		for (i=0;i<level;i++) printf(" "); 
		printf ("CORBA_tk_alias %p %p %s\n", tc, tc->repo_id,
			tc->repo_id);
		print_typecode (tc->subtypes [0], NULL, level+2);
		break;

	case CORBA_tk_enum:
		for (i=0;i<level;i++) printf(" "); 
		printf ("CORBA_tk_enum %p %p %s\n", tc, tc->repo_id,
			tc->repo_id);
		for (j = 0; j < tc->sub_parts; j++) {
			for (i=0;i<level+2;i++) printf(" "); 
			printf ("%s\n", tc->subnames [j]);
		}
		break;

	default:
		for (i=0;i<level;i++) printf(" "); 
		printf ("Unknown Type\n");
	
	}

}

static void
int_set_value (BonoboPropertyEditor *editor, CORBA_any *any,
	       CORBA_Environment *ev)
{
	CORBA_any nv;

	/* fixme: support nested aliases */

	if (any->_type->kind == CORBA_tk_alias) {

		nv._type =  any->_type->subtypes [0];
		nv._value = any->_value;

		if (editor->priv->set_cb)
			editor->priv->set_cb (editor, &nv, ev);
		else if (CLASS (editor)->set_value)
			CLASS (editor)->set_value (editor, &nv, ev);

	} else {
		if (editor->priv->set_cb)
			editor->priv->set_cb (editor, any, ev);
		else if (CLASS (editor)->set_value)
			CLASS (editor)->set_value (editor, any, ev);
	}
}

static void 
value_changed_cb (BonoboListener    *listener,
		  char              *event_name, 
		  CORBA_any         *any,
		  CORBA_Environment *ev,
		  gpointer           user_data)
{
	BonoboPropertyEditor *editor = BONOBO_PROPERTY_EDITOR (user_data);
	
	if (!bonobo_arg_type_is_equal (any->_type, editor->tc, ev)) {
		bonobo_exception_set 
			(ev, ex_Bonobo_Property_InvalidValue);
		g_warning ("property type change (changed cb) %d %d", 
			   any->_type->kind, editor->tc->kind);
		return;
	}

	int_set_value (editor, any, ev);
}

void                
bonobo_property_editor_set_property (BonoboPropertyEditor *editor,
				     Bonobo_PropertyBag    bag,
				     const char           *name,
				     CORBA_TypeCode        tc, 
				     CORBA_any            *defval)
{
	CORBA_Environment  ev;
	Bonobo_Property    property;
	BonoboArg         *arg;

	g_return_if_fail (editor != NULL);
	g_return_if_fail (bag != CORBA_OBJECT_NIL);
	g_return_if_fail (name != NULL);
	g_return_if_fail (tc != CORBA_OBJECT_NIL);

	CORBA_exception_init (&ev);

	property = Bonobo_PropertyBag_getPropertyByName (bag, name, &ev);
	
	if (BONOBO_EX (&ev) || property == CORBA_OBJECT_NIL) {
		CORBA_exception_free (&ev);
		return;
	}

	arg = Bonobo_Property_getValue (property, &ev);

	if (BONOBO_EX (&ev) || arg == NULL) {
		bonobo_object_release_unref (property, NULL);
		CORBA_exception_free (&ev);
		return;
	}

	if (bonobo_arg_type_is_equal (arg->_type, TC_null, NULL) ||
	    !bonobo_arg_type_is_equal (arg->_type, tc, NULL)) {
		CORBA_free (arg);
		
		if (defval) {
			Bonobo_Property_setValue (property, defval, &ev);
			arg = bonobo_arg_copy (defval);
		} else {
			arg = bonobo_arg_new (tc);
			Bonobo_Property_setValue (property, arg, &ev);
		}
	}
	
	/* fixme: use bonobo_object_release_unref instead */
	if (editor->priv->property) 
		CORBA_Object_release (property, NULL);
	   
	editor->priv->property = property;

	if (editor->priv->id && editor->priv->property != CORBA_OBJECT_NIL)
		bonobo_event_source_client_remove_listener 
			(editor->priv->property, editor->priv->id, NULL);

	editor->priv->id = bonobo_event_source_client_add_listener 
		(property, value_changed_cb, NULL, &ev, editor);

	if (bonobo_arg_type_is_equal (arg->_type, TC_null, NULL)) {
		bonobo_arg_release (arg);
		CORBA_exception_free (&ev);
		return;
	}

	if (editor->tc == CORBA_OBJECT_NIL)
		editor->tc = (CORBA_TypeCode)CORBA_Object_duplicate 
			((CORBA_Object) arg->_type, NULL);

	int_set_value (editor, arg, &ev);
		
	CORBA_exception_free (&ev);
	bonobo_arg_release (arg);
}


static void
bonobo_property_editor_destroy (GtkObject *object)
{
	BonoboPropertyEditor *editor = BONOBO_PROPERTY_EDITOR (object);
	CORBA_Environment ev;
	
	CORBA_exception_init (&ev);

	if (editor->priv->id && editor->priv->property != CORBA_OBJECT_NIL)
		bonobo_event_source_client_remove_listener (
	                editor->priv->property, editor->priv->id, NULL);
		
	/* fixme: use bonobo_object_release_unref instead */
	if (editor->priv->property)
		CORBA_Object_release ((CORBA_Object) editor->priv->property, 
				      &ev);

	if (editor->tc)
		CORBA_Object_release ((CORBA_Object) editor->tc, &ev);


	CORBA_exception_free (&ev);	
}

static void 
real_property_editor_set_value (BonoboPropertyEditor *editor,
				BonoboArg            *value,
				CORBA_Environment    *ev)
{
	g_warning ("calling default set_value implementation");
}

static void
pe_size_request (GtkWidget      *widget,
		 GtkRequisition *requisition)
{
	GtkBin *bin = GTK_BIN (widget);

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) 
		gtk_widget_size_request (bin->child, requisition);
}

static void
pe_size_allocate (GtkWidget     *widget,
		  GtkAllocation *allocation)
{
	GtkBin *bin = GTK_BIN (widget);

	widget->allocation = *allocation;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) 
		gtk_widget_size_allocate (bin->child, allocation);
}

static void
bonobo_property_editor_class_init (BonoboPropertyEditorClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;


	klass->set_value = real_property_editor_set_value;

	widget_class->size_request = pe_size_request;
	widget_class->size_allocate = pe_size_allocate;
  
	object_class->destroy = bonobo_property_editor_destroy;
}

static void
bonobo_property_editor_init (BonoboPropertyEditor *editor)
{
	GTK_WIDGET_SET_FLAGS (editor, GTK_CAN_FOCUS);
	GTK_WIDGET_SET_FLAGS (editor, GTK_NO_WINDOW);

	editor->priv = g_new0 (BonoboPropertyEditorPrivate, 1);
}

GtkType
bonobo_property_editor_get_type (void)
{
	static GtkType petype = 0;

	if (!petype) {
		static const GtkTypeInfo peinfo = {
			"BonoboPropertyEditor",
			sizeof (BonoboPropertyEditor),
			sizeof (BonoboPropertyEditorClass),
			(GtkClassInitFunc) bonobo_property_editor_class_init,
			(GtkObjectInitFunc) bonobo_property_editor_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		petype = gtk_type_unique (GTK_TYPE_BIN, &peinfo);
	}

	return petype;
}

BonoboPropertyEditor *
bonobo_property_editor_construct (GtkWidget                 *widget, 
				  BonoboPropertyEditorSetFn  set_cb,
				  CORBA_TypeCode             tc)
{
	BonoboPropertyEditor *editor;
	
	g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
	g_return_val_if_fail (set_cb != NULL, NULL);

	editor = gtk_type_new (bonobo_property_editor_get_type ());

	gtk_container_add (GTK_CONTAINER (editor), widget);

	if (tc != CORBA_OBJECT_NIL)
		editor->tc = (CORBA_TypeCode)CORBA_Object_duplicate 
			((CORBA_Object) tc, NULL);

	editor->priv->set_cb = set_cb;

	return editor;
}

void                
bonobo_property_editor_set_value (BonoboPropertyEditor *editor,
				  const BonoboArg      *value,
				  CORBA_Environment    *opt_ev)
{
	CORBA_Environment  ev, *my_ev;
	CORBA_any nv;

	bonobo_return_if_fail (editor != NULL, opt_ev);
	bonobo_return_if_fail (BONOBO_IS_PROPERTY_EDITOR(editor), opt_ev);
	bonobo_return_if_fail (value != NULL, opt_ev);

	if (!editor->priv->property)
		return;

	if (!opt_ev) {
		CORBA_exception_init (&ev);
		my_ev = &ev;
	} else
		my_ev = opt_ev;

	if (editor->tc->kind == CORBA_tk_alias && 
	    bonobo_arg_type_is_equal (value->_type, editor->tc->subtypes[0], 
				      my_ev)) {
		
		nv._type =  editor->tc;
		nv._value = value->_value;

		Bonobo_Property_setValue (editor->priv->property, &nv, my_ev);
		
	} else {
		if (!bonobo_arg_type_is_equal (value->_type, editor->tc, 
					       my_ev)) {
			bonobo_exception_set 
				(opt_ev, ex_Bonobo_Property_InvalidValue);
			g_warning ("property type change %d %d", 
				   value->_type->kind, editor->tc->kind);
			return;
		}
		Bonobo_Property_setValue (editor->priv->property, value, 
					  my_ev);
	}

	if (!opt_ev)
		CORBA_exception_free (&ev);
}

typedef struct {
	CORBA_TypeCode tc;
	GtkWidget *(*new) (CORBA_TypeCode tc);
} PEPlugin;

GtkWidget *
bonobo_property_editor_resolve (CORBA_TypeCode tc)
{
	static GHashTable *pehash = NULL;
	PEPlugin *plugin;

	if (!pehash) {

		/* fixme: we need a real plugin system */

		pehash = g_hash_table_new (NULL, NULL);

		plugin = g_new0 (PEPlugin, 1);
		plugin->tc = TC_Bonobo_Config_FileName;
		plugin->new = bonobo_property_editor_filename_new;
		g_hash_table_insert (pehash, (gpointer)plugin->tc->repo_id, 
				     plugin);

		plugin = g_new0 (PEPlugin, 1);
		plugin->tc = TC_Bonobo_Config_Color;
		plugin->new = bonobo_property_editor_color_new;
		g_hash_table_insert (pehash, (gpointer)plugin->tc->repo_id, 
				     plugin);
	}

	if ((plugin = g_hash_table_lookup (pehash, tc->repo_id)))
		return plugin->new (tc);

	switch (tc->kind) {

	case CORBA_tk_short:    
		return bonobo_property_editor_short_new ();
	case CORBA_tk_long:   
		return bonobo_property_editor_long_new ();
	case CORBA_tk_ushort: 
		return bonobo_property_editor_ushort_new ();
	case CORBA_tk_ulong: 
		return bonobo_property_editor_ulong_new ();
	case CORBA_tk_float: 
		return bonobo_property_editor_float_new ();
	case CORBA_tk_double: 
		return bonobo_property_editor_double_new ();
	case CORBA_tk_string: 
		return bonobo_property_editor_string_new ();
	case CORBA_tk_boolean: 
		return bonobo_property_editor_boolean_new (NULL);
	case CORBA_tk_enum:
		return bonobo_property_editor_enum_new ();
	case CORBA_tk_char: 
		// return bonobo_property_editor_char_new ();
	case CORBA_tk_struct:
		// return bonobo_property_editor_struct_new (tc);
	case CORBA_tk_sequence:
		// return bonobo_property_editor_list_new (tc);
	case CORBA_tk_array:  
		// return bonobo_property_editor_array_new (tc);
	default:              
		return bonobo_property_editor_default_new ();
	}
}

GtkWidget *
bonobo_property_editor_new (Bonobo_PropertyBag  pb,
			    const char         *name,
			    CORBA_TypeCode      tc, 
			    CORBA_any          *defval)
{
	GtkWidget *w;

	g_return_val_if_fail (pb != CORBA_OBJECT_NIL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (tc != CORBA_OBJECT_NIL, NULL);

	if (!(w = bonobo_property_editor_resolve (tc))) 
		return NULL;

	bonobo_property_editor_set_property (BONOBO_PROPERTY_EDITOR (w),
					     pb, name, tc, defval);

	return w;
}

static void
guard_cb (BonoboListener    *listener,
	  char              *event_name, 
	  CORBA_any         *any,
	  CORBA_Environment *ev,
	  gpointer           user_data)
{
	GtkWidget *widget = GTK_WIDGET (user_data);
	gboolean v;

	if (bonobo_arg_type_is_equal (any->_type, TC_boolean, NULL)) {

		v = BONOBO_ARG_GET_BOOLEAN (any);

		gtk_widget_set_sensitive (widget, v);
	}
}

static void
remove_listener_cb (GtkWidget *widget, Bonobo_PropertyBag bag)
{
	guint32 id;

	id = (guint32) gtk_object_get_data (GTK_OBJECT (widget), 
					    "BONOBO_LISTENER_ID");

	bonobo_event_source_client_remove_listener (bag, id, NULL);
}

void
bonobo_property_editor_set_guard (GtkWidget          *widget,
				  Bonobo_PropertyBag  bag,
				  const char         *prop_name)
{
	CORBA_Environment ev;
	char *mask;
	gboolean v;
	guint32 id;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (bag != CORBA_OBJECT_NIL);
	g_return_if_fail (prop_name != NULL);

	CORBA_exception_init (&ev);

	mask = g_strconcat ("=Bonobo/Property:change:", prop_name, NULL);

	id = bonobo_event_source_client_add_listener (bag, guard_cb, mask, 
						      NULL, widget);

	gtk_object_set_data (GTK_OBJECT (widget), "BONOBO_LISTENER_ID", 
			     (gpointer) id);

	gtk_signal_connect (GTK_OBJECT (widget), "destroy",
			    GTK_SIGNAL_FUNC (remove_listener_cb), bag);

	v = bonobo_property_bag_client_get_value_gboolean (bag, prop_name,&ev);
	
	if (!BONOBO_EX (&ev))
		gtk_widget_set_sensitive (widget, v);

	CORBA_exception_free (&ev);
}
