/**
 * bonobo-property-editor.c:
 *
 * Author:
 *   Dietmar Maurer (dietmar@ximian.com)
 *
 * Copyright 2001 Ximian, Inc.
 */

#include <bonobo/bonobo-exception.h>
#include <liboaf/liboaf.h>

#include "bonobo-property-editor.h"

#define TC_OBJECT_FROM_SERVANT(servant) \
BONOBO_PROPERTY_EDITOR (bonobo_object_from_servant (servant));

/* Parent object class in GTK hierarchy */
static GtkObjectClass *bonobo_property_editor_parent_class;

struct _BonoboPropertyEditorPrivate {
	Bonobo_Property               property;
	BonoboPropertyEditorSetFn     set_cb;
	Bonobo_EventSource_ListenerId id;
};

static void 
value_changed_cb (BonoboListener    *listener,
		  char              *event_name, 
		  CORBA_any         *any,
		  CORBA_Environment *ev,
		  gpointer           user_data)
{
	BonoboPropertyEditor *editor = BONOBO_PROPERTY_EDITOR (user_data);
	GtkWidget *widget;

	widget = bonobo_control_get_widget (BONOBO_CONTROL (editor));

	if (editor->tc)
		CORBA_Object_release ((CORBA_Object) editor->tc, ev);

	editor->tc = (CORBA_TypeCode)CORBA_Object_duplicate (
	        (CORBA_Object) any->_type, ev);

	editor->priv->set_cb (editor, any, widget);
}

static void
impl_Bonobo_PropertyEditor_setProperty (PortableServer_Servant  servant,
					const Bonobo_Property   property,
					CORBA_Environment      *ev)
{
	GtkWidget *widget;
	BonoboArg *arg;

	BonoboPropertyEditor *editor = TC_OBJECT_FROM_SERVANT(servant);

	editor->priv->property = property;

	widget = bonobo_control_get_widget (BONOBO_CONTROL (editor));
	
	arg = Bonobo_Property_getValue (property, ev);

	if (BONOBO_EX (ev) || arg == NULL)
		return;

	if (editor->tc)
		CORBA_Object_release ((CORBA_Object) editor->tc, ev);

	editor->tc = (CORBA_TypeCode)CORBA_Object_duplicate (
	        (CORBA_Object) arg->_type, ev);

	editor->priv->set_cb (editor, arg, widget);

	editor->priv->id = bonobo_event_source_client_add_listener (property, 
	        value_changed_cb, NULL, ev, editor);

	bonobo_arg_release (arg);
}

static void
bonobo_property_editor_destroy (GtkObject *object)
{
	BonoboPropertyEditor *editor = BONOBO_PROPERTY_EDITOR (object);
	CORBA_Environment ev;
	
	CORBA_exception_init (&ev);

	if (editor->priv->id && editor->priv->property)
		bonobo_event_source_client_remove_listener (
	                editor->priv->property, editor->priv->id, NULL);
		
	if (editor->tc)
		CORBA_Object_release ((CORBA_Object) editor->tc, &ev);

	CORBA_exception_free (&ev);	

	bonobo_property_editor_parent_class->destroy (object);
}

static void
bonobo_property_editor_class_init (BonoboPropertyEditorClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;
	POA_Bonobo_PropertyEditor__epv *epv = &klass->epv;

	bonobo_property_editor_parent_class = 
		gtk_type_class (bonobo_control_get_type ());

	object_class->destroy = bonobo_property_editor_destroy;

	epv->setProperty = impl_Bonobo_PropertyEditor_setProperty;
}

static void
bonobo_property_editor_init (BonoboPropertyEditor *editor)
{
	editor->priv = g_new0 (BonoboPropertyEditorPrivate, 1);
}


BONOBO_X_TYPE_FUNC_FULL (BonoboPropertyEditor, 
			 Bonobo_PropertyEditor, 
			 bonobo_control_get_type (), 
			 bonobo_property_editor);


BonoboPropertyEditor *
bonobo_property_editor_new (GtkWidget *widget, 
			    BonoboPropertyEditorSetFn set_cb)
{
	BonoboPropertyEditor *editor;
	
	g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
	g_return_val_if_fail (set_cb != NULL, NULL);

	editor = gtk_type_new (bonobo_property_editor_get_type ());

	editor->priv->set_cb = set_cb;

	bonobo_control_construct (BONOBO_CONTROL (editor), widget);
 
	return editor;
}

void                
bonobo_property_editor_set_value (BonoboPropertyEditor *editor,
				  const BonoboArg   *value,
				  CORBA_Environment *opt_ev)
{
	CORBA_Environment  ev, *my_ev;

	bonobo_return_if_fail (editor != NULL, opt_ev);
	bonobo_return_if_fail (BONOBO_IS_PROPERTY_EDITOR(editor), opt_ev);
	bonobo_return_if_fail (value != NULL, opt_ev);

	if (!editor->priv->property)
		return;

	if (!opt_ev) {
		CORBA_exception_init (&ev);
		my_ev = &ev;
	} else
		my_ev = opt_ev;

	Bonobo_Property_setValue (editor->priv->property, value, my_ev);

	if (!opt_ev)
		CORBA_exception_free (&ev);
}

static char *
tckind_to_string (CORBA_TCKind kind)
{
	switch (kind) {

	case CORBA_tk_short:    return "short";
	case CORBA_tk_long:     return "long";
	case CORBA_tk_ushort:   return "ushort";
	case CORBA_tk_ulong:    return "ulong";
	case CORBA_tk_float:    return "float";
	case CORBA_tk_double:   return "double";
	case CORBA_tk_boolean:  return "boolean";
	case CORBA_tk_char:     return "char";
	case CORBA_tk_struct:   return "struct";
	case CORBA_tk_enum:     return "enum";
	case CORBA_tk_string:   return "string";
	case CORBA_tk_sequence: return "sequence";
	case CORBA_tk_array:    return "array";
	default:                return "default";

	}
}

Bonobo_PropertyEditor
bonobo_property_editor_resolve (CORBA_TypeCode tc,  
				CORBA_Environment *ev)
{
	Bonobo_PropertyEditor editor;
	char *oafiid;

	bonobo_return_val_if_fail (tc != NULL, CORBA_OBJECT_NIL, ev);

	if (tc->repo_id) {

		oafiid = g_strconcat (BONOBO_PROPERTY_EDITOR_PREFIX, 
				      tc->repo_id, NULL);

		editor = oaf_activate_from_id (oafiid, 0, NULL, ev);

		g_free (oafiid);

		if (!BONOBO_EX (ev) && editor != CORBA_OBJECT_NIL)
			return editor;

		CORBA_exception_init (ev); /* clear all errors */
	}

	oafiid = g_strconcat (BONOBO_PROPERTY_EDITOR_PREFIX, 
			      tckind_to_string (tc->kind), NULL);

	editor = oaf_activate_from_id (oafiid, 0, NULL, ev);
	
	g_free (oafiid);

	return editor;
}
