/*
 * bonobo-moniker-config.c: Configuration moniker implementation
 *
 * Author:
 *   Dietmar Maurer (dietmar@ximian.com)
 *
 * Copyright 2000 Ximian, Inc.
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <bonobo/bonobo-moniker.h>
#include <bonobo/bonobo-moniker-util.h>
#include <bonobo/bonobo-moniker-simple.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-ui-container.h>
#include <bonobo/bonobo-widget.h>

#include <bonobo-config-bag.h>
#include <bonobo-config-property.h>

#include "bonobo-property-editor.h"
#include "gtkwtree.h"
#include "gtkwtreeitem.h"

static GConfClient       *default_client = NULL;
static BonoboTransient   *default_trans = NULL;
static BonoboEventSource *default_es = NULL;


static Bonobo_Unknown
create_bag_editor (const char        *name, 
		   CORBA_Environment *ev)
{
	GtkWidget           *tree, *item;
	BonoboControl       *control;
	BonoboConfigBag     *config_bag;
	CORBA_Object         bag;
	Bonobo_PropertyList *plist;
	int                  i;

	if (!(config_bag = bonobo_config_bag_new (default_client, name))) {
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);
		return CORBA_OBJECT_NIL;
	}
	
	bag = BONOBO_OBJREF (config_bag);

	tree = gtk_wtree_new();
	
	plist = Bonobo_PropertyBag_getProperties (bag, ev);
	
	if (BONOBO_EX (ev)) {
		bonobo_object_unref (BONOBO_OBJECT (config_bag));
		return CORBA_OBJECT_NIL;
	}

	for (i = 0; i < plist->_length; i++) {
		CORBA_TypeCode tc;
		GtkWidget *e;
		char      *pn;
		CORBA_Object editor, prop;
		BonoboUIContainer *ui_container;

		pn = Bonobo_Property_getName (plist->_buffer [i], ev);
		if (BONOBO_EX (ev))
			break;

		tc = Bonobo_Property_getType (plist->_buffer [i], ev);
		if (BONOBO_EX (ev))
			break;

		editor = bonobo_property_editor_resolve (tc, ev);

		CORBA_Object_release ((CORBA_Object)tc, ev);

		if (BONOBO_EX (ev) || editor == CORBA_OBJECT_NIL)
			break;

		ui_container = bonobo_ui_container_new ();
		
		e = bonobo_widget_new_control_from_objref (editor,
		      BONOBO_OBJREF (ui_container));
		gtk_widget_show (e);

		bonobo_object_unref (BONOBO_OBJECT (ui_container));

		prop = CORBA_Object_duplicate (plist->_buffer [i], ev);

		Bonobo_PropertyEditor_setProperty (editor, prop, ev);

		if (BONOBO_EX (ev))
			break;

    	
		item = gtk_wtree_item_new_with_widget (pn, e);
		gtk_wtree_append (GTK_WTREE (tree), item);

		CORBA_free (pn);
	}

	CORBA_free (plist);

	gtk_widget_show_all (tree);

	control = bonobo_control_new (tree);

	return CORBA_Object_duplicate (BONOBO_OBJREF (control), ev);
}

static Bonobo_Unknown
create_property_editor (const char        *name, 
			CORBA_Environment *ev)
{
	GConfValue *gv;
	BonoboArg *value;
	CORBA_Object editor, prop;

	gv =  gconf_client_get (default_client, name, NULL);

	if (!gv) {
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);
		return CORBA_OBJECT_NIL;
	}
   
	if (!(value = gconf_to_bonobo_arg (gv))) {
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);
		gconf_value_free (gv);
		return CORBA_OBJECT_NIL;
	}
	
	gconf_value_free (gv);

	editor = bonobo_property_editor_resolve (value->_type, ev);

	bonobo_arg_release (value);

	if (BONOBO_EX (ev) || editor == CORBA_OBJECT_NIL) {
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);
		return CORBA_OBJECT_NIL;
	}

	if (!CORBA_Object_is_a (editor, "IDL:Bonobo/PropertyEditor:1.0", ev)) {
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);
		return CORBA_OBJECT_NIL;
	}

	prop = bonobo_transient_create_objref (default_trans, 
					       "IDL:Bonobo/Property:1.0", 
					       name, ev);

	if (BONOBO_EX (ev) || prop == CORBA_OBJECT_NIL) {
		bonobo_object_release_unref (editor, NULL);
		return CORBA_OBJECT_NIL;
	}

	Bonobo_PropertyEditor_setProperty (editor, prop, ev);

	if (BONOBO_EX (ev) || prop == CORBA_OBJECT_NIL) {
		bonobo_object_release_unref (editor, NULL);
		return CORBA_OBJECT_NIL;
	}

	return CORBA_Object_duplicate (editor, ev);
}

static Bonobo_Unknown
config_resolve (BonoboMoniker               *moniker,
		const Bonobo_ResolveOptions *options,
		const CORBA_char            *requested_interface,
		CORBA_Environment           *ev)
{
	Bonobo_Moniker     parent;
	BonoboConfigBag   *bag;
	const gchar       *name;

	parent = bonobo_moniker_get_parent (moniker, ev);
	if (BONOBO_EX (ev))
		return CORBA_OBJECT_NIL;

	name = bonobo_moniker_get_name (moniker);

	if (parent != CORBA_OBJECT_NIL) {
		
		g_warning ("not implemented");
		
		bonobo_object_release_unref (parent, ev);

		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);

		return CORBA_OBJECT_NIL;
	}

	if (!strcmp (requested_interface, "IDL:Bonobo/PropertyBag:1.0")) {
		
		if ((bag = bonobo_config_bag_new (default_client, name)))
			return (Bonobo_Unknown) CORBA_Object_duplicate (
			      BONOBO_OBJREF (bag), ev);
		
		bonobo_exception_set (ev, ex_Bonobo_Moniker_InterfaceNotFound);

		return CORBA_OBJECT_NIL;
	}

	
 	if (!strcmp (requested_interface, "IDL:Bonobo/Property:1.0")) {

	 	return bonobo_transient_create_objref (default_trans, 
			 "IDL:Bonobo/Property:1.0", name, ev);

	} 

 	if (!strcmp (requested_interface, "IDL:Bonobo/Control:1.0")) {

		if (gconf_client_dir_exists (default_client, name, NULL)) {
			return create_bag_editor (name, ev);
		} else {
			return create_property_editor (name, ev);
		}
	} 

	return CORBA_OBJECT_NIL; /* try moniker extenders */
}


static BonoboObject *
bonobo_moniker_config_factory (BonoboGenericFactory *this, 
			       const char           *object_id,
			       void                 *closure)
{
	static gboolean initialized = FALSE;

	if (!initialized) {

		initialized = TRUE;

		default_client = gconf_client_get_default ();
		g_return_val_if_fail (default_client, CORBA_OBJECT_NIL);

		default_es = bonobo_event_source_new ();

		default_trans = 
			bonobo_config_property_transient (default_client, 
							  default_es);
		g_return_val_if_fail (default_trans, CORBA_OBJECT_NIL);
		
		bonobo_object_unref (BONOBO_OBJECT (default_es));
		
	}

	if (!strcmp (object_id, "OAFIID:Bonobo_Moniker_config")) {

		return BONOBO_OBJECT (bonobo_moniker_simple_new (
		        "config:", config_resolve));
	
	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "short")) {
		
		return bonobo_property_editor_basic_new (TC_short);
	
	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "ushort")) {
		
		return bonobo_property_editor_basic_new (TC_ushort);
	
	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "long")) {
		
		return bonobo_property_editor_basic_new (TC_long);

	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "ulong")) {
		
		return bonobo_property_editor_basic_new (TC_ulong);
	
	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "float")) {
		
		return bonobo_property_editor_basic_new (TC_float);
	
	} else if (!strcmp (object_id, 
			   BONOBO_PROPERTY_EDITOR_PREFIX "double")) {
		
		return bonobo_property_editor_basic_new (TC_double);
	
	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "string")) {
		
		return bonobo_property_editor_basic_new (TC_string);

	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "enum")) {
		
		return bonobo_property_editor_enum_new ();
		
	} else if (!strcmp (object_id, 
	 		    BONOBO_PROPERTY_EDITOR_PREFIX "boolean")) {
		
		return bonobo_property_editor_boolean_new ();
/*
	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "char")) {
		
		return bonobo_property_editor_char_new ();

	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "struct")) {
		
		return bonobo_property_editor_struct_new ();

	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "union")) {
		
		return bonobo_property_editor_union_new ();

	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "sequence")) {
		
		return bonobo_property_editor_sequence_new ();

	} else if (!strcmp (object_id, 
			    BONOBO_PROPERTY_EDITOR_PREFIX "array")) {
		
		return bonobo_property_editor_array_new ();
*/
	} else
		g_warning ("Failing to manufacture a '%s'", object_id);

	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:Bonobo_Moniker_config_Factory",
				"bonobo configuration moniker",
				bonobo_moniker_config_factory,
				NULL);
