/*
 * bonobo-config-property.c:
 *
 * Author:
 *   Dietmar Maurer (dietmar@ximian.com)
 *
 * Copyright 2000 Ximian, Inc.
 */
#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <gtk/gtk.h>
#include <libgnomeui/gnome-init.h>

#include <bonobo/bonobo-moniker.h>
#include <bonobo/bonobo-moniker-util.h>
#include <bonobo/bonobo-property.h>
#include <bonobo/bonobo-property-bag.h>
#include <bonobo/bonobo-generic-factory.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-transient.h>
#include <bonobo/bonobo-ui-node.h>
#include <bonobo/bonobo-exception.h>
#include <liboaf/liboaf.h>

#include "bonobo-config-bag.h"
#include "bonobo-config-property.h"

typedef struct {
	GConfClient       *client;
	BonoboEventSource *es;
	guint              nid;
} CPCallbackData;

static CORBA_char *
impl_Bonobo_Property_getName (PortableServer_Servant servant,
			      CORBA_Environment *ev)
{
	GConfPropertyServant *pservant = (GConfPropertyServant *) servant;

	return CORBA_string_dup (pservant->property_name);
}

static CORBA_TypeCode
impl_Bonobo_Property_getType (PortableServer_Servant servant,
			      CORBA_Environment *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	BonoboArg            *value;
	BonoboArgType         type;
	GConfValue           *gv;

	gv =  gconf_client_get (ps->client, ps->property_name, NULL);

	if (!gv) 
		return BONOBO_ARG_NULL;
   
	if (!(value = gconf_to_bonobo_arg (gv))) {
		gconf_value_free (gv);
		return BONOBO_ARG_NULL;
	}
	
	gconf_value_free (gv);

	type = (CORBA_TypeCode) CORBA_Object_duplicate 
		((CORBA_Object) value->_type, ev);

	bonobo_arg_release (value);

	return type;
}

static CORBA_any *
impl_Bonobo_Property_getValue (PortableServer_Servant servant,
			       CORBA_Environment *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	BonoboArg            *value;
	GConfValue           *gv;

	gv = gconf_client_get_without_default (ps->client, 
					       ps->property_name, NULL);
	
	value = gconf_to_bonobo_arg (gv);

	if (gv) 
		gconf_value_free (gv);
		
	return value;
}

static void
impl_Bonobo_Property_setValue (PortableServer_Servant servant,
				const CORBA_any       *any,
				CORBA_Environment     *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	GConfValue           *gv;
	GError               *err = NULL;

	gv = bonobo_arg_to_gconf (any);
	if (!gv) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
				     ex_Bonobo_Property_InvalidValue, NULL);
		return;
	}

	gconf_client_set (ps->client, ps->property_name, gv, &err);

	if (err) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
				     ex_Bonobo_Property_InvalidValue, NULL);
		return;
	}

}

static CORBA_any *
impl_Bonobo_Property_getDefault (PortableServer_Servant servant,
				 CORBA_Environment *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	BonoboArg            *value;
	GConfValue           *gv;

	gv = gconf_client_get_default_from_schema (ps->client, 
						   ps->property_name, NULL);
	
	value = gconf_to_bonobo_arg (gv);

	if (gv) 
		gconf_value_free (gv);
		
	return value;
}

static CORBA_char *
impl_Bonobo_Property_getDocString (PortableServer_Servant servant,
				   CORBA_Environment *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	GConfEntry           *ge;
	GConfSchema          *schema;
	CORBA_char           *result;

	ge = gconf_client_get_entry (ps->client, ps->property_name, 
				     NULL, FALSE, NULL);
	
	if (!ge || !ge->schema_name) 
		return CORBA_string_dup ("");
	
	schema = gconf_client_get_schema (ps->client, ge->schema_name, 
					  NULL);
	
	if (!schema || !schema->short_desc) {
		gconf_entry_free (ge);
		return CORBA_string_dup ("");
	}

	result = CORBA_string_dup (schema->short_desc);
	gconf_entry_free (ge);
	gconf_schema_free (schema);
	
	return result; 
}

static CORBA_long
impl_Bonobo_Property_getFlags (PortableServer_Servant servant,
			       CORBA_Environment *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	CORBA_long            flags;
	GError               *err = NULL;
	gboolean              writeable;

	flags = BONOBO_PROPERTY_READABLE;
	
	writeable = gconf_client_key_is_writable (ps->client, 
						  ps->property_name, &err);
	
	if (!err && writeable)
		flags |= BONOBO_PROPERTY_WRITEABLE;

	if (err) 
		g_error_free (err);
	
	return flags;
}

static Bonobo_EventSource_ListenerId
impl_Bonobo_Property_addListener (PortableServer_Servant servant,
				  const Bonobo_Listener  l,
				  CORBA_Environment     *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	Bonobo_Unknown corba_es = BONOBO_OBJREF (ps->es);
	Bonobo_EventSource_ListenerId id = 0;
	char *mask;

	mask = g_strdup_printf ("Bonobo/Property:change:%s", 
				ps->property_name);

	id = Bonobo_EventSource_addListenerWithMask (corba_es, l, mask, ev); 

	g_free (mask);

	return id;
}

static void
impl_Bonobo_Property_removeListener (PortableServer_Servant servant,
				     const Bonobo_EventSource_ListenerId id,
				     CORBA_Environment     *ev)
{
	GConfPropertyServant *ps = (GConfPropertyServant *) servant;
	Bonobo_Unknown corba_es = BONOBO_OBJREF (ps->es);

	return Bonobo_EventSource_removeListener (corba_es, id, ev); 
}

static void
impl_Bonobo_Property_ref (PortableServer_Servant servant, 
			  CORBA_Environment *ev)
{
	/* nothing to do */
}

static void
impl_Bonobo_Property_unref (PortableServer_Servant servant, 
			    CORBA_Environment *ev)
{
	/* nothing to do */
}

static CORBA_Object
impl_Bonobo_Property_queryInterface (PortableServer_Servant  servant,
				     const CORBA_char       *repoid,
				     CORBA_Environment      *ev)
{
	GConfPropertyServant *cs = (GConfPropertyServant *) servant;

	if (!strcmp (repoid, "IDL:Bonobo/Property:1.0"))
		return bonobo_transient_create_objref (cs->transient,
			"IDL:Bonobo/Property:1.0", cs->property_name, ev);
	else
		return CORBA_OBJECT_NIL;
}

static POA_Bonobo_Unknown__epv *
bonobo_config_property_get_unknown_epv (void)
{
	POA_Bonobo_Unknown__epv *epv;

	epv = g_new0 (POA_Bonobo_Unknown__epv, 1);

	epv->ref            = impl_Bonobo_Property_ref;
	epv->unref          = impl_Bonobo_Property_unref;
	epv->queryInterface = impl_Bonobo_Property_queryInterface;

	return epv;
}

static POA_Bonobo_Property__epv *
bonobo_property_get_epv (void)
{
	static POA_Bonobo_Property__epv *epv = NULL;

	if (epv != NULL)
		return epv;

	epv = g_new0 (POA_Bonobo_Property__epv, 1);

	epv->getName        = impl_Bonobo_Property_getName;
	epv->getType        = impl_Bonobo_Property_getType;
	epv->getValue       = impl_Bonobo_Property_getValue;
	epv->setValue       = impl_Bonobo_Property_setValue;
	epv->getDefault     = impl_Bonobo_Property_getDefault;
	epv->getDocString   = impl_Bonobo_Property_getDocString;
	epv->getFlags       = impl_Bonobo_Property_getFlags;
	epv->addListener    = impl_Bonobo_Property_addListener;
	epv->removeListener = impl_Bonobo_Property_removeListener;

	return epv;
}

static POA_Bonobo_Property__vepv *
bonobo_property_get_vepv (void)
{
	static POA_Bonobo_Property__vepv *vepv = NULL;

	if (vepv != NULL)
		return vepv;

	vepv = g_new0 (POA_Bonobo_Property__vepv, 1);

	vepv->Bonobo_Unknown_epv = bonobo_config_property_get_unknown_epv ();
	vepv->Bonobo_Property_epv = bonobo_property_get_epv ();

	return vepv;
}

static PortableServer_Servant
bonobo_property_create_servant (PortableServer_POA  poa,
				BonoboTransient    *bt,
				char               *name,
				void               *callback_data)
{
        GConfPropertyServant *servant;
        CORBA_Environment     ev;
	CPCallbackData       *cd = (CPCallbackData *)callback_data;

        CORBA_exception_init (&ev);

        servant = g_new0 (GConfPropertyServant, 1);
        ((POA_Bonobo_Property *)servant)->vepv = bonobo_property_get_vepv ();
        servant->property_name = g_strdup (name);
	servant->client = cd->client;
	servant->es = cd->es;
	servant->transient = bt;
        POA_Bonobo_Property__init ((PortableServer_Servant) servant, &ev);

	return servant;
}

static void
bonobo_property_destroy_servant (PortableServer_Servant servant, 
				 void *callback_data)
{
        CORBA_Environment ev;

        g_free (((GConfPropertyServant *)servant)->property_name);
        CORBA_exception_init (&ev);
        POA_Bonobo_Property__fini (servant, &ev);
        CORBA_exception_free (&ev);
	g_free (servant);
}

static void
bonobo_config_property_notify_listeners (GConfClient* client,
					 guint cnxn_id,
					 GConfEntry *entry,
					 gpointer user_data)
{
	CPCallbackData *cd = (CPCallbackData *)user_data;
	gchar          *name;
	CORBA_any      *value;		

	name = bonobo_event_make_name ("Bonobo/Property", "change", 
				       entry->key); 

	value = gconf_to_bonobo_arg (entry->value);

	bonobo_event_source_notify_listeners (cd->es, name, value, NULL);

	g_free (name);
}

static void
bonobo_config_transient_destroy (GtkObject      *object,
				 CPCallbackData *cd)
{
	bonobo_object_unref (BONOBO_OBJECT (cd->es));

	gconf_client_notify_remove (cd->client, cd->nid);
}

BonoboTransient *
bonobo_config_property_transient (GConfClient *client, BonoboEventSource *es)
{
	CPCallbackData *cd;
	BonoboTransient *bt;

	g_return_val_if_fail (client != NULL, NULL);
	g_return_val_if_fail (es != NULL, NULL);

	cd = g_new0 (CPCallbackData, 1);

	cd->client = client;
	cd->es = es;

	bonobo_object_ref (BONOBO_OBJECT (es));

	bonobo_config_init_gconf_listener (cd->client);

	cd->nid = gconf_client_notify_add (cd->client, "/", 
					   bonobo_config_property_notify_listeners,
					   cd, NULL, NULL);

	bt = bonobo_transient_new (NULL, bonobo_property_create_servant, 
				   bonobo_property_destroy_servant, cd);
	
	gtk_signal_connect (GTK_OBJECT (bt), "destroy",
			    GTK_SIGNAL_FUNC (bonobo_config_transient_destroy), cd);

	return bt;
}
