/**
 * bonobo-config-bag.c: config bag object implementation.
 *
 * Author:
 *   Dietmar Maurer (dietmar@ximian.com)
 *
 * Copyright 2000 Ximian, Inc.
 */
#include <config.h>
#include <bonobo/Bonobo.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-property.h>
#include <bonobo/bonobo-persist-stream.h>
#include <bonobo/bonobo-transient.h>
#include <bonobo/bonobo-property-bag-xml.h>
#include <string.h>

#include "bonobo-config-bag.h"
#include "bonobo-config-bag-property.h"

#define ANY_PREFIX "%CORBA:ANY%"

static GtkObjectClass        *parent_class = NULL;

#define CLASS(o) BONOBO_CONFIG_BAG_CLASS (GTK_OBJECT(o)->klass)

static char *
extract_property_name (char *key, char *path)
{
	char *rval = key;

	while (*path && *rval && *(path++) == *(rval++));

	if (*rval == '/') 
		rval++;
	
	return rval;
}

BonoboArg *
gconf_to_bonobo_arg (GConfValue *gv)
{ 
	BonoboArg         *value = NULL;
	const gchar       *str;
	BonoboUINode      *node;
        CORBA_Environment  ev;

	if (!gv)
		return bonobo_arg_new (BONOBO_ARG_NULL);

        CORBA_exception_init (&ev);

	switch (gv->type) {
	case GCONF_VALUE_INVALID:
		return NULL;
	case GCONF_VALUE_INT:
		value = bonobo_arg_new (BONOBO_ARG_INT);
		BONOBO_ARG_SET_LONG (value, gconf_value_get_int (gv));
		return value;
	case GCONF_VALUE_FLOAT:
		value = bonobo_arg_new (BONOBO_ARG_DOUBLE);
		BONOBO_ARG_SET_DOUBLE (value, gconf_value_get_float (gv));
		return value;
	case GCONF_VALUE_BOOL:
		value = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
		BONOBO_ARG_SET_BOOLEAN (value, gconf_value_get_bool (gv));
		return value;
	case GCONF_VALUE_STRING:
		str = gconf_value_get_string (gv);
		
		if (strncmp (str, ANY_PREFIX, strlen (ANY_PREFIX))) {

			value = bonobo_arg_new (BONOBO_ARG_STRING);
			BONOBO_ARG_SET_STRING (value, str);
			return value;
		}

		node = bonobo_ui_node_from_string (&str[strlen (ANY_PREFIX)]);
		if (!node)
			return NULL;

		value = bonobo_property_bag_xml_decode_any (node, &ev);
		bonobo_ui_node_free (node);

		return value;
	default: 
		return NULL;
	}

	return NULL;
} 

GConfValue *
bonobo_arg_to_gconf (const BonoboArg *ba)
{
	BonoboUINode      *node;
	gchar             *enc, *str;
        CORBA_Environment  ev;
	GConfValue        *gv;

        CORBA_exception_init (&ev);

	g_return_val_if_fail (ba != NULL, NULL);

	if (bonobo_arg_type_is_equal (ba->_type, BONOBO_ARG_STRING, NULL)) {
		gv = gconf_value_new (GCONF_VALUE_STRING);
		gconf_value_set_string (gv, BONOBO_ARG_GET_STRING (ba));
		return gv;
	}

	if (bonobo_arg_type_is_equal (ba->_type, BONOBO_ARG_INT, NULL)) {
		gv = gconf_value_new (GCONF_VALUE_INT);
		gconf_value_set_int (gv, BONOBO_ARG_GET_INT (ba));
		return gv;
	}

	if (bonobo_arg_type_is_equal (ba->_type, BONOBO_ARG_DOUBLE, NULL)) {
		gv = gconf_value_new (GCONF_VALUE_FLOAT);
		gconf_value_set_float (gv, BONOBO_ARG_GET_DOUBLE (ba));
		return gv;
	}

	if (bonobo_arg_type_is_equal (ba->_type, BONOBO_ARG_BOOLEAN, NULL)) {
		gv = gconf_value_new (GCONF_VALUE_BOOL);
		gconf_value_set_bool (gv, BONOBO_ARG_GET_BOOLEAN (ba));
		return gv;
	}

	if (!(node = bonobo_property_bag_xml_encode_any (NULL, ba, &ev)))
		return NULL;
	if (!(enc = bonobo_ui_node_to_string (node, TRUE))) {
		bonobo_ui_node_free (node);
		return NULL;
	}

	str = g_strconcat (ANY_PREFIX, enc, NULL);
	bonobo_ui_node_free_string (enc);
	bonobo_ui_node_free (node);

	gv = gconf_value_new (GCONF_VALUE_STRING);
	gconf_value_set_string (gv, str);
	g_free (str);
      
	return gv;
}

static Bonobo_PropertyList *
bonono_config_bag_get_properties (BonoboConfigBag *cb,
				  CORBA_Environment * ev)
{
	Bonobo_PropertyList *plist;
	GSList *pl, *l;
	GConfEntry *ge;
	gchar *pname;
	CORBA_Object obj;

	pl = gconf_client_all_entries (cb->client, cb->path, NULL);

	plist = Bonobo_PropertyList__alloc ();
	plist->_buffer = 
		CORBA_sequence_Bonobo_Property_allocbuf (g_slist_length (pl));
	CORBA_sequence_set_release (plist, TRUE); 
	plist->_length = 0;

	l = pl;

	while (l) {
		ge = (GConfEntry *)l->data;
		pname = extract_property_name (ge->key, cb->path);

		obj = bonobo_transient_create_objref (cb->transient, 
		        "IDL:Bonobo/Property:1.0", pname, ev);

		if (BONOBO_EX (ev))
			break;

		plist->_buffer [plist->_length++] = obj;
 
		gconf_entry_free (ge);
		l = l->next;
	}

	g_slist_free (pl);

	return plist;
}

static Bonobo_Property
bonono_config_bag_get_property_by_name (BonoboConfigBag *cb,
					const CORBA_char * name,
					CORBA_Environment * ev)
{
	/* we do not allow a slash in the name */
	if (strchr (name, '/')) {

		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
				     ex_Bonobo_PropertyBag_NotFound, NULL);
		
		return CORBA_OBJECT_NIL;
	}

	/* we always return a reference - even if the value does not exist,
	 * so that we can add values. 
	 */
       
	return bonobo_transient_create_objref (cb->transient, 
					       "IDL:Bonobo/Property:1.0",
					       name, ev); 
}

static Bonobo_PropertyNames *
bonono_config_bag_get_property_names (BonoboConfigBag   *cb,
				      CORBA_Environment *ev)
{
	Bonobo_PropertyNames *names;
	GSList *pl, *l;
	GConfEntry *ge;
	gchar *pname;

	pl = gconf_client_all_entries (cb->client, cb->path, NULL);

	names = Bonobo_PropertyNames__alloc ();
	names->_buffer = 
		CORBA_sequence_CORBA_string_allocbuf (g_slist_length (pl));
	CORBA_sequence_set_release (names, TRUE); 
	names->_length = 0;

	l = pl;

	while (l) {
		ge = (GConfEntry *)l->data;
		pname = extract_property_name (ge->key, cb->path);
		names->_buffer [names->_length++] = CORBA_string_dup (pname);
		gconf_entry_free (ge);
		l = l->next;
	}

	g_slist_free (pl);

	return names;
}

static void
bonono_config_bag_set_values (BonoboConfigBag *cb,
			      const Bonobo_PropertySet * set,
			      CORBA_Environment * ev)
{
	GConfValue           *gv;
	GError               *err = NULL;
	gchar                *path;
	gint                  i;

	for (i = 0; i < set->_length; i++) {

		path = g_strconcat (cb->path, "/", set->_buffer[i].name, NULL);
		gv = bonobo_arg_to_gconf (&set->_buffer [i].value);

		gconf_client_set (cb->client, path, gv, &err);

		gconf_value_free (gv);
		g_free (path);

		if (err) {
			g_error_free (err);
			return;
		}
	}	
}

static Bonobo_PropertySet *
bonono_config_bag_get_values (BonoboConfigBag *cb,
			      CORBA_Environment * ev)
{
	Bonobo_PropertySet *set;
	BonoboArg          *arg;
	GSList             *pl, *l;
	GConfEntry         *ge;
	gchar              *pname;

	pl = gconf_client_all_entries (cb->client, cb->path, NULL);

	set = Bonobo_PropertySet__alloc ();
	set->_buffer = 
		CORBA_sequence_Bonobo_Pair_allocbuf (g_slist_length (pl));
	CORBA_sequence_set_release (set, TRUE); 
	set->_length = 0;


	l = pl;

	while (l) {
		ge = (GConfEntry *)l->data;
		pname = extract_property_name (ge->key, cb->path);
		set->_buffer [set->_length++].name = CORBA_string_dup (pname);
		
		arg = gconf_to_bonobo_arg (ge->value);
		set->_buffer [set->_length++].value = *arg;
				   
		gconf_entry_free (ge);
		l = l->next;
	}

	g_slist_free (pl);

	return set;
}

static void
bonobo_config_bag_destroy (GtkObject *object)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (object);
	
	if (cb->transient)
		gtk_object_unref (GTK_OBJECT (cb->transient));
	
	if (cb->client) {

		gconf_client_notify_remove (cb->client, cb->nid);

		gtk_object_unref (GTK_OBJECT (cb->client));
	}

	g_free (cb->path);

	parent_class->destroy (object);
}


/* CORBA implementations */

static Bonobo_PropertyList *
impl_Bonobo_PropertyBag_getProperties (PortableServer_Servant  servant,
				       CORBA_Environment      *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));

	return CLASS (cb)->get_properties (cb, ev);
}

static Bonobo_Property
impl_Bonobo_PropertyBag_getPropertyByName (PortableServer_Servant  servant,
					   const CORBA_char       *name,
					   CORBA_Environment      *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));
       
	return CLASS (cb)->get_property_by_name (cb, name, ev);
}

static Bonobo_PropertyNames *
impl_Bonobo_PropertyBag_getPropertyNames (PortableServer_Servant  servant,
					  CORBA_Environment      *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));

	return CLASS (cb)->get_property_names (cb, ev);
}

static void                  
impl_Bonobo_PropertyBag_setValues (PortableServer_Servant servant,
				   const Bonobo_PropertySet *set,
				   CORBA_Environment *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));

	CLASS (cb)->set_values (cb, set, ev);
}

static Bonobo_PropertySet *
impl_Bonobo_PropertyBag_getValues (PortableServer_Servant servant,
				    CORBA_Environment *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));

	return CLASS (cb)->get_values (cb, ev);
}

static Bonobo_EventSource
impl_Bonobo_PropertyBag_getEventSource (PortableServer_Servant servant,
					CORBA_Environment      *ev)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (bonobo_object_from_servant (servant));

	return bonobo_object_dup_ref (BONOBO_OBJREF (cb->es), ev);
}

static void
bonobo_config_bag_notify_listeners (GConfClient* client,
				    guint cnxn_id,
				    GConfEntry *entry,
				    gpointer user_data)
{
	BonoboConfigBag *cb = BONOBO_CONFIG_BAG (user_data);
	gchar           *name;
	CORBA_any       *value;

	if (strncmp (cb->path, entry->key, strlen (cb->path)))
		return;

	if (strchr (&entry->key [strlen (cb->path) + 1], '/'))
		return;
	
	name = &entry->key [strlen (cb->path) + 1];
		
	name = bonobo_event_make_name ("Bonobo/Property", "change", name); 

	value = gconf_to_bonobo_arg (entry->value);

	bonobo_event_source_notify_listeners (cb->es, name, value, NULL);
	
	g_free (name);
}

void
bonobo_config_init_gconf_listener (GConfClient *client)
{
	static gboolean bonobo_config_gconf_init = TRUE;

	if (bonobo_config_gconf_init) {
		gconf_client_add_dir (client, "/", GCONF_CLIENT_PRELOAD_NONE,
				      NULL);
		bonobo_config_gconf_init = FALSE;
	}
}

BonoboConfigBag *
bonobo_config_bag_new (GConfClient *client,
		       const gchar *path)
{
	BonoboConfigBag *cb;

	g_return_val_if_fail (client != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);

	cb = gtk_type_new (BONOBO_CONFIG_BAG_TYPE);

	cb->client = client;
	cb->path = g_strdup (path);

	while (path [strlen (cb->path) - 1] == '/') 
		cb->path [strlen (path) - 1] = '\0';

	if (!(cb->transient = bonobo_config_bag_property_transient (cb))) {
		bonobo_object_unref (BONOBO_OBJECT (cb));
		return NULL;
	}

	if (!(cb->es = bonobo_event_source_new ())) {
		bonobo_object_unref (BONOBO_OBJECT (cb));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (cb), 
				     BONOBO_OBJECT (cb->es));

	bonobo_config_init_gconf_listener (cb->client);

	cb->nid = gconf_client_notify_add (cb->client, cb->path, 
					   bonobo_config_bag_notify_listeners,
					   cb, NULL, NULL);
	return cb;
}

static void
bonobo_config_bag_class_init (BonoboConfigBagClass *class)
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;
	BonoboConfigBagClass *cb_class = BONOBO_CONFIG_BAG_CLASS (class);
	POA_Bonobo_PropertyBag__epv *epv;
	
	parent_class = gtk_type_class (BONOBO_OBJECT_TYPE);

	cb_class->get_properties = bonono_config_bag_get_properties;
	cb_class->get_property_by_name = bonono_config_bag_get_property_by_name;
	cb_class->get_property_names = bonono_config_bag_get_property_names;
	cb_class->set_values = bonono_config_bag_set_values;
	cb_class->get_values = bonono_config_bag_get_values;

	object_class->destroy = bonobo_config_bag_destroy;

	epv = &class->epv;

	epv->getProperties        = impl_Bonobo_PropertyBag_getProperties;
	epv->getPropertyByName    = impl_Bonobo_PropertyBag_getPropertyByName;
	epv->getPropertyNames     = impl_Bonobo_PropertyBag_getPropertyNames;
	epv->getValues            = impl_Bonobo_PropertyBag_getValues;
	epv->setValues            = impl_Bonobo_PropertyBag_setValues;
	epv->getEventSource       = impl_Bonobo_PropertyBag_getEventSource;
}

static void
bonobo_config_bag_init (BonoboConfigBag *cb)
{
	/* nothing to do */
}

BONOBO_X_TYPE_FUNC_FULL (BonoboConfigBag, 
			 Bonobo_PropertyBag,
			 BONOBO_X_OBJECT_TYPE,
			 bonobo_config_bag);

