/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "App.h"
#include <algorithm>

namespace Bakery
{

//Initialize static member data:
App::type_listAppInstances App::m_listAppInstances;
bool App::m_bExiting = false;
Gnome::About* App::m_pAbout = 0;
bool App::m_bAboutShown = false;
bool App::m_bOperationCancelled = false;
std::string App::m_strCommandLine_0;

std::string App::m_strVersion;
std::string App::m_strCopyright;
std::string App::m_strDescription;
App::type_vecStrings App::m_vecAuthors;

App::App(const string& appname, const string& title)
 : Gnome::App(appname, title),
 m_Status(false, true, GNOME_PREFERENCES_NEVER)
{
  m_bInitialized = false;
  
  m_strAppName = appname;
  m_strTitle = title;

  //Connect signals:

  //GnomeClient Session management:
  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    pClient->die.connect(slot(this, &App::on_Session_die));
    pClient->save_yourself.connect(slot(this, &App::on_Session_save_yourself));
  }

  //Register an instance of this app:
  m_listAppInstances.push_back(this);
}

App::~App()
{  
  //Remove this app from the list of instances:
  remove_instance_from_list();

  //If this was the last instance:
  if(m_listAppInstances.empty())
  {
    //Delete shared static widgets if this was the last instance:
    if(m_pAbout)
    {
      delete m_pAbout;
      m_pAbout = 0;
    }

    //When the last instance goes, the application closes.
    Gnome::Main::quit();
  } 
}


void App::init()
{
  set_policy(false, true, false); //resizable
  set_default_size(640, 400); //A sensible default.
  set_wmclass(m_strAppName, m_strTitle);

  set_statusbar(m_Status);

  init_menus();
  init_toolbars();

  m_bInitialized = true;

  show_all();  

  //on_document_load(); //Show the document (even if it is empty).
}

void App::init_menus()
{
  init_menus_file();
  init_menus_edit();
  init_menus_help();

  create_menus(m_menu_UI_Infos);
  install_menu_hints();

  //Override this to add more menus.
}

void App::init_toolbars()
{
  using namespace Gnome::UI;

  m_toolbar_UI_Infos.push_back(Gnome::UI::Item(Gnome::UI::Icon(GNOME_STOCK_PIXMAP_NEW),
                         N_("New "),
                         slot(this, &App::on_menu_File_New),
                         N_("Create a new " + m_strAppName)));

  create_toolbar(m_toolbar_UI_Infos);
}


gint
App::delete_event_impl(GdkEventAny* e)
{
  //Clicking on the [x] in the title bar should be like choosing File|New
  on_menu_File_Close();

  return TRUE; //TRUE = prevent further handling.
}

void App::init_menus_file()
{
  // File menu
  type_vecGnome_UI_Info menu_file;

  //Build menu:
  menu_file.push_back(Gnome::MenuItems::New(_("New Instance"), _("Create a new instance"), slot(this, &App::on_menu_File_New)));
  menu_file.push_back(Gnome::MenuItems::Close(slot(this, &App::on_menu_File_Close)));
  menu_file.push_back(Gnome::MenuItems::Exit(slot(this, &App::on_menu_File_Exit)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::Menus::File(menu_file));
}

void App::init_menus_edit()
{
  //Edit menu
  type_vecGnome_UI_Info menu_edit;

  //Build menu:
  menu_edit.push_back(Gnome::MenuItems::Cut(slot(this, &App::on_menu_Edit_Cut)));
  menu_edit.push_back(Gnome::MenuItems::Copy(slot(this, &App::on_menu_Edit_Copy)));
  menu_edit.push_back(Gnome::MenuItems::Paste(slot(this, &App::on_menu_Edit_Paste)));
  menu_edit.push_back(Gnome::MenuItems::Clear(slot(this, &App::on_menu_Edit_Clear)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::Menus::Edit(menu_edit));
}

void App::init_menus_help()
{
  //Help menu
  type_vecGnome_UI_Info menu_help;

  //Build menu:
  menu_help.push_back(Gnome::UI::Help(m_strAppName));
  menu_help.push_back(Gnome::MenuItems::About(slot(this, &App::on_menu_Help_About)));

  //Add menu:
 
  m_menu_UI_Infos.push_back(Gnome::Menus::Help(menu_help));
}

void App::on_menu_File_New()
{
  App* pApp = new_instance();
  pApp->init(); 
}

void App::on_menu_File_Close()
{
   destroy_and_remove_from_list();
}

void App::on_menu_File_Exit()
{
  // we don't want to quit directly as we should save our work
  // therefore we need to send close to each window.

  m_bExiting = true; //One of the instances could cancel this loop.

  //Close each instance:
  type_listAppInstances::iterator i = m_listAppInstances.begin();
  while (m_bExiting && (i != m_listAppInstances.end()))
  {
    type_listAppInstances::iterator j = i;
    i++;

    App* pApp = (*j);
    if(pApp)
    {
      type_listAppInstances::size_type count = m_listAppInstances.size();
      pApp->on_menu_File_Close();

      //The iterator is invalid if an element has been removed:
      if(count != m_listAppInstances.size())
      {
        i = m_listAppInstances.begin(); //There should not be a problem with asking again.
      }
    }
  }
}

void App::on_menu_Edit_Cut()
{
  on_menu_Edit_Copy();
  on_menu_Edit_Clear();
}

void App::on_menu_Edit_Copy()
{
  //override.
}

void App::on_menu_Edit_Paste()
{
  //override.
}

void App::on_menu_Edit_Clear()
{
  //override.
}

void App::on_menu_Help_About()
{
  if(m_pAbout && m_bAboutShown) // "About" box hasn't been closed, so just raise it
  {
    m_pAbout->set_parent(*this);
    
    Gdk_Window about_win(m_pAbout->get_window());
    about_win.show();
    about_win.raise();
  }
  else
  {
    // not yet wrapped:
    gchar* pchLogo = gnome_pixmap_file("icon.png");

    string strLogo;
    if(pchLogo)
      strLogo = pchLogo;

    //Re-create About box:
    //Doesn't seem to work if you show it twice:
    if(m_pAbout)
    {
      delete m_pAbout;
      m_pAbout = 0;
    }
    m_pAbout = new Gnome::About(m_strAppName, m_strVersion,
                               m_strCopyright,
                               m_vecAuthors,
                               m_strDescription,
                               strLogo);

    m_pAbout->set_parent(*this);
    m_pAbout->close.connect(slot(this, &App::on_about_close));
    m_bAboutShown = true;
    m_pAbout->show();
  }
}

gint App::on_about_close()
{
  m_bAboutShown = false;
  return FALSE; //FALSE = continue processing this signal.
}

void App::set_about_information(const string& strVersion, const type_vecStrings& vecAuthors, const string& strCopyright, const string& strDescription)
{
  m_strVersion = strVersion;
  m_vecAuthors = vecAuthors;
  m_strCopyright = m_strCopyright;
  m_strDescription = strDescription;
}

string App::get_version() const
{
  return m_strVersion;
}


void App::on_Session_die()
{
  //"I expect you to die, Mr. Bond."

  //TODO: Questions:
  //Do we need to handle this as well as the delete event?
  //Does GnomeClient want us to close this instance or all instances?
  on_menu_File_Close();
}

gint App::on_Session_save_yourself(gint phase, GnomeSaveStyle save_style, gint is_shutdown, GnomeInteractStyle interact_style, gint is_fast)
{
  //But let the women and children save themselves first.

  //TODO: Save command line to reload the same document via the command line
  //- Need to implement command line stuff.

  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    if(m_strCommandLine_0.size()) //Need to App::call set_command_line_args().
    {
      std::vector<string> vecArgs;

      vecArgs.push_back(m_strCommandLine_0);
  
      pClient->set_clone_command (vecArgs);
      pClient->set_restart_command (vecArgs);
    }
  }

  return TRUE;
}

void App::destroy_and_remove_from_list()
{
  remove_instance_from_list();
  destroy();
}

void App::remove_instance_from_list()
{   
  //Get iterator for this instance in the list:
  type_listAppInstances::iterator iter = std::find(m_listAppInstances.begin(), m_listAppInstances.end(), this);
  if(iter != m_listAppInstances.end()) //Should never happen - it's always in the list
  {
    m_listAppInstances.erase(iter);
  }   
}

void App::set_operation_cancelled(bool bVal /* = true */)
{
  m_bOperationCancelled = bVal;
}

bool App::get_operation_cancelled()
{
  return m_bOperationCancelled;
}

void App::set_command_line_args(int argc, char **&argv)
{
  if( (argc > 0) && argv[0])
    m_strCommandLine_0 = (char*)argv[0];

  /*
  //Build C-style array of popt options:
  guint poptCount = m_vecPoptOptions.size();
  if(poptCount)
  {
    poptOption* pPoptOptions = new poptOption[poptCount];
    for(gint i = 0; i < poptCount; i++)
    {
      pPoptOptions[i] = m_vecPoptOptions[i];
    }

    //Parse
    poptContext ctx;
    gnome_init_with_popt_table(m_strAppName.c_str(), m_strVersion.c_str(), argc, argv, pPoptOptions,
				   0, &ctx);
  }
  */
}



} //namespace
