/* GConf
 * Copyright (C) 1999 Red Hat Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef GCONF_GCONF_H
#define GCONF_GCONF_H

#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include "gconf-schema.h"
#include "gconf-orbit.h"

/* Errors */

typedef enum {
  G_CONF_SUCCESS = 0,
  G_CONF_FAILED = 1,        /* Something didn't work, don't know why, probably unrecoverable
                               so there's no point having a more specific errno */

  G_CONF_NO_SERVER = 2,     /* Server can't be launched/contacted */
  G_CONF_NO_PERMISSION = 3, /* don't have permission for that */
  G_CONF_BAD_ADDRESS = 4,   /* Address couldn't be resolved */
  G_CONF_BAD_KEY = 5,       /* directory or key isn't valid (contains bad
                               characters, or malformed slash arrangement) */
  G_CONF_PARSE_ERROR = 6,   /* Syntax error when parsing */
  G_CONF_CORRUPT = 7,       /* Fatal error parsing/loading information inside the backend */
  G_CONF_TYPE_MISMATCH = 8, /* Type requested doesn't match type found */
  G_CONF_IS_DIR = 9,        /* Requested key operation on a dir */
  G_CONF_IS_KEY = 10,       /* Requested dir operation on a key */
  G_CONF_OVERRIDDEN = 11    /* Read-only source at front of path has set the value */
} GConfErrNo;

const gchar* g_conf_error          (void); /* returns strerror of current errno,
                                              combined with additional details 
                                              that may exist 
                                           */
const gchar* g_conf_strerror       (GConfErrNo en);
GConfErrNo   g_conf_errno          (void);
void         g_conf_set_error      (GConfErrNo en, const gchar* format, ...) G_GNUC_PRINTF (2, 3);
void         g_conf_clear_error    (void); /* like setting errno to 0 */


/* Meta-information about a key. Not the same as a schema; a schema
 * is normative, this is descriptive.
 */

typedef struct _GConfMetaInfo GConfMetaInfo;

struct _GConfMetaInfo {
  gchar* schema;
  gchar* mod_user; /* user owning the daemon that made the last modification */
  GTime  mod_time; /* time of the modification */
  /* anything else? */
};

#define g_conf_meta_info_schema(x)    ((x)->schema)
#define g_conf_meta_info_mod_user(x)  ((x)->mod_user)
#define g_conf_meta_info_mod_time(x)  ((x)->mod_time)

GConfMetaInfo* g_conf_meta_info_new         (void);
void           g_conf_meta_info_destroy     (GConfMetaInfo* gcmi);
void           g_conf_meta_info_set_schema  (GConfMetaInfo* gcmi,
                                             const gchar* schema_name);
void           g_conf_meta_info_set_mod_user(GConfMetaInfo* gcmi,
                                             const gchar* mod_user);
void           g_conf_meta_info_set_mod_time(GConfMetaInfo* gcmi,
                                             GTime mod_time);


/* Key-value pairs; used to list the contents of
 *  a directory
 */  

typedef struct _GConfEntry GConfEntry;

struct _GConfEntry {
  gchar* key;
  GConfValue* value;
};

/* Pair takes memory ownership of both key and value */
GConfEntry* g_conf_entry_new    (gchar* key, GConfValue* val);
void        g_conf_entry_destroy(GConfEntry* pair);

/* A configuration engine (stack of config sources); normally there's
 * just one of these on the system.  
 */
typedef struct _GConf GConf;

struct _GConf {
  gpointer dummy;
};

typedef void (*GConfNotifyFunc)(GConf* conf, guint cnxn_id, const gchar* key, GConfValue* value, gpointer user_data);

gboolean     g_conf_init            (const gchar* appname);
gboolean     g_conf_is_initialized  (void);

GConf*       g_conf_new             (void); /* Default source stack */
/* returns NULL on error; requests single specified source */
GConf*       g_conf_new_from_address(const gchar* address);
void         g_conf_unref           (GConf* conf);
void         g_conf_ref             (GConf* conf);

/* Returns ID of the notification */
guint        g_conf_notify_add(GConf* conf,
                               const gchar* namespace_section, /* dir or key to listen to */
                               GConfNotifyFunc func,
                               gpointer user_data);

void         g_conf_notify_remove  (GConf* conf,
                                    guint cnxn);


/* Low-level interfaces */
GConfValue*  g_conf_get            (GConf* conf, const gchar* key);

void         g_conf_set            (GConf* conf, const gchar* key, GConfValue* value);

void         g_conf_unset          (GConf* conf, const gchar* key);

GSList*      g_conf_all_entries    (GConf* conf, const gchar* dir);

GSList*      g_conf_all_dirs       (GConf* conf, const gchar* dir);

void         g_conf_sync           (GConf* conf);

gboolean     g_conf_dir_exists     (GConf *conf, const gchar* dir);

gboolean     g_conf_valid_key      (const gchar* key);


/* 
 * Higher-level stuff 
 */

/* 'def' (default) is used if the key is not set or if there's an error. */

gdouble      g_conf_get_float (GConf* conf, const gchar* key,
                               gdouble def);

gint         g_conf_get_int   (GConf* conf, const gchar* key,
                               gint def);

/* free the retval */
gchar*       g_conf_get_string(GConf* conf, const gchar* key,
                               const gchar* def); /* def is copied when returned, 
                                                   * and can be NULL to return 
                                                   * NULL 
                                                   */

gboolean     g_conf_get_bool  (GConf* conf, const gchar* key,
                               gboolean def);

/* this one has no default since it would be expensive and make little
   sense; it returns NULL as a default, to indicate unset or error */
/* free the retval */
/* Note that this returns the schema stored at key, NOT
   the schema that key conforms to. */
GConfSchema* g_conf_get_schema  (GConf* conf, const gchar* key);

/* No convenience functions for lists or pairs, since there are too
   many combinations of types possible
*/

/* setters return TRUE on success; note that you still have to sync */

gboolean     g_conf_set_float   (GConf* conf, const gchar* key,
                                 gdouble val);

gboolean     g_conf_set_int     (GConf* conf, const gchar* key,
                                 gint val);

gboolean     g_conf_set_string  (GConf* conf, const gchar* key,
                                 const gchar* val);

gboolean     g_conf_set_bool    (GConf* conf, const gchar* key,
                                 gboolean val);

gboolean     g_conf_set_schema  (GConf* conf, const gchar* key,
                                 GConfSchema* val);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif



