/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-paper.c: GnomePrintPaper and GnomePrintUnit
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Dirk Luetjens <dirk@luedi.oche.de>
 *    Yves Arrouye <Yves.Arrouye@marin.fdn.fr>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 1998 the Free Software Foundation and 2001-2202 Ximian, Inc.
 *
 */

#define __GNOME_PRINT_PAPER_C__

#include <config.h>
#include <math.h>
#include "gpa/gpa-node.h"
#include "gnome-print-i18n.h"
#include "gnome-print-paper.h"

/* fixme: fill in all papers and units */
/* fixme: load papers from file */
/* fixme: use some fancy unit program */

/*
 * WARNING! Do not mess up with that - we use hardcoded numbers for base units!
 */

static const GnomePrintUnit gp_units[] = {
	/* Do not insert any elements before/between first 4 */
	{0, GNOME_PRINT_UNIT_DIMENSIONLESS, 1.0, N_("Unit"), "", N_("Units"), ""},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, 1.0, N_("Point"), N_("Pt"), N_("Points"), N_("Pts")},
	{0, GNOME_PRINT_UNIT_USERSPACE, 1.0, N_("Userspace unit"), N_("User"), N_("Userspace units"), N_("User")},
	{0, GNOME_PRINT_UNIT_DEVICE, 1.0, N_("Pixel"), N_("Px"), N_("Pixels"), N_("Px")},
	/* You can add new elements from this point forward */
	{0, GNOME_PRINT_UNIT_DIMENSIONLESS, 0.01, N_("Percent"), N_("%"), N_("Percents"), N_("%")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, (72.0 / 25.4), N_("Millimeter"), N_("mm"), N_("Millimeters"), N_("mm")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, (72.0 / 2.54), N_("Centimeter"), N_("cm"), N_("Centimeters"), N_("cm")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, (72.0 / 0.0254), N_("Meter"), N_("m"), N_("meters"), N_("m")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, (72.0), N_("Inch"), N_("in"), N_("Inches"), N_("in")},
};

#define gp_num_units (sizeof (gp_units) / sizeof (gp_units[0]))

static const GnomePrintPaper gp_paper_default = {0, N_("A4"), 21.0 * 72.0 / 2.54, 29.7 * 72.0 / 2.54};

GList *gp_papers = NULL;

/* Load System paper config file */

static void
gnome_print_papers_load (void)
{
	GPANode *globals;

	globals = gpa_defaults ();

	if (globals) {
		GPANode *papers;
		papers = gpa_node_get_path_node (globals, "Globals.Media.PhysicalSizes");
		if (papers) {
			GPANode *paper;
			for (paper = gpa_node_get_child (papers, NULL); paper != NULL; paper = gpa_node_get_child (papers, paper)) {
				guchar *name;
				gdouble width, height;
				name = gpa_node_get_path_value (paper, "Name");
				gpa_node_get_length_path_value (paper, "Width", &width);
				gpa_node_get_length_path_value (paper, "Height", &height);
				if (name && (width >= 1.0) && (height >= 1.0)) {
					GnomePrintPaper *gpp;
					gpp = g_new (GnomePrintPaper, 1);
					gpp->version = 0;
					gpp->name = name;
					gpp->width = width;
					gpp->height = height;
					gp_papers = g_list_prepend (gp_papers, gpp);
				} else {
					if (name) g_free (name);
				}
				gpa_node_unref (paper);
			}
			gp_papers = g_list_reverse (gp_papers);
			gpa_node_unref (papers);
		}
		gpa_node_unref (globals);
	}

	if (!gp_papers) {
		gp_papers = g_list_prepend (NULL, (gpointer) &gp_paper_default);
	}
}

/* Returned papers are const, but lists have to be freed */

const GnomePrintPaper *
gnome_print_paper_get_default (void)
{
	if (!gp_papers) gnome_print_papers_load ();

	return gp_papers->data;
}

const GnomePrintPaper *
gnome_print_paper_get_by_name (const guchar *name)
{
	GList *l;

	g_return_val_if_fail (name != NULL, NULL);

	if (!gp_papers) gnome_print_papers_load ();

	for (l = gp_papers; l != NULL; l = l->next) {
		if (!g_ascii_strcasecmp (name, ((GnomePrintPaper *) l->data)->name)) return l->data;
	}

	return NULL;
}

#define GP_CLOSE_ENOUGH(a,b) (fabs ((a) - (b)) < 0.1)
#define GP_LESS_THAN(a,b) ((a) - (b) < 0.01)

const GnomePrintPaper *
gnome_print_paper_get_by_size (gdouble width, gdouble height)
{
	GList *l;

	/* Should we allow papers <= 1/5184 sq inch? */
	g_return_val_if_fail (width > 1.0, NULL);
	g_return_val_if_fail (height > 1.0, NULL);

	if (!gp_papers) gnome_print_papers_load ();

	for (l = gp_papers; l != NULL; l = l->next) {
		if (GP_CLOSE_ENOUGH (((GnomePrintPaper *) l->data)->width, width) &&
		    GP_CLOSE_ENOUGH (((GnomePrintPaper *) l->data)->height, height)) {
			return l->data;
		}
	}

	return NULL;
}

const GnomePrintPaper *
gnome_print_paper_get_closest_by_size (gdouble width, gdouble height, gboolean mustfit)
{
	const GnomePrintPaper *bestpaper;
	gdouble dist, best;
	GList *l;

	/* Should we allow papers <= 1/5184 sq inch? */
	g_return_val_if_fail (width > 1.0, NULL);
	g_return_val_if_fail (height > 1.0, NULL);

	if (!gp_papers) gnome_print_papers_load ();

	bestpaper = NULL;
	best = 1e18;

	for (l = gp_papers; l != NULL; l = l->next) {
		if (!mustfit ||
		    (GP_LESS_THAN (width, ((GnomePrintPaper *) l->data)->width) &&
		     GP_LESS_THAN (height, ((GnomePrintPaper *) l->data)->height))) {
			gdouble dx, dy;
			/* We fit, or it is not important */
			dx = width - ((GnomePrintPaper *) l->data)->width;
			dy = height - ((GnomePrintPaper *) l->data)->height;
			dist = sqrt (dx * dx + dy * dy);
			if (dist < best) {
				best = dist;
				bestpaper = l->data;
			}
		}
	}

	return bestpaper;
}

GList *
gnome_print_paper_get_list (void)
{
	GList *papers;

	if (!gp_papers) gnome_print_papers_load ();

	papers = g_list_copy (gp_papers);

	return papers;
}

void
gnome_print_paper_free_list (GList *papers)
{
	g_list_free (papers);
}

/* Base units are the ones used by gnome-print and paper descriptions */

const GnomePrintUnit *
gnome_print_unit_get_identity (guint base)
{
	switch (base) {
	case GNOME_PRINT_UNIT_DIMENSIONLESS:
		return &gp_units[0];
		break;
	case GNOME_PRINT_UNIT_ABSOLUTE:
		return &gp_units[1];
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		return &gp_units[2];
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		return &gp_units[3];
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit base %d", __FILE__, __LINE__, base);
		return FALSE;
		break;
	}
}

/* fixme: */
const GnomePrintUnit *
gnome_print_unit_get_default (void)
{
	return &gp_units[0];
}

const GnomePrintUnit *
gnome_print_unit_get_by_name (const guchar *name)
{
	gint i;

	g_return_val_if_fail (name != NULL, NULL);

	for (i = 0; i < gp_num_units; i++) {
		if (!g_ascii_strcasecmp (name, gp_units[i].name)) return &gp_units[i];
		if (!g_ascii_strcasecmp (name, gp_units[i].plural)) return &gp_units[i];
	}

	return NULL;
}

const GnomePrintUnit *
gnome_print_unit_get_by_abbreviation (const guchar *abbreviation)
{
	gint i;

	g_return_val_if_fail (abbreviation != NULL, NULL);

	for (i = 0; i < gp_num_units; i++) {
		if (!g_ascii_strcasecmp (abbreviation, gp_units[i].abbr)) return &gp_units[i];
		if (!g_ascii_strcasecmp (abbreviation, gp_units[i].abbr_plural)) return &gp_units[i];
	}

	return NULL;
}

GList *
gnome_print_unit_get_list (guint bases)
{
	GList *units;
	gint i;

	g_return_val_if_fail ((bases & ~GNOME_PRINT_UNITS_ALL) == 0, NULL);

	units = NULL;

	for (i = 0; i < gp_num_units; i++) {
		if (bases & gp_units[i].base) {
			units = g_list_prepend (units, (gpointer) &gp_units[i]);
		}
	}

	units = g_list_reverse (units);

	return units;
}

void
gnome_print_unit_free_list (GList *units)
{
	g_list_free (units);
}

/* These are pure utility */
/* Return TRUE if conversion is possible */
gboolean
gnome_print_convert_distance (gdouble *distance, const GnomePrintUnit *from, const GnomePrintUnit *to)
{
	g_return_val_if_fail (distance != NULL, FALSE);
	g_return_val_if_fail (from != NULL, FALSE);
	g_return_val_if_fail (to != NULL, FALSE);

	if ((from->base == GNOME_PRINT_UNIT_DIMENSIONLESS) || (to->base == GNOME_PRINT_UNIT_DIMENSIONLESS)) {
		*distance = *distance * from->unittobase / to->unittobase;
	}

	if (from->base != to->base) return FALSE;

	*distance = *distance * from->unittobase / to->unittobase;

	return TRUE;
}

/* ctm is for userspace, devicetransform is for device units */
gboolean
gnome_print_convert_distance_full (gdouble *distance, const GnomePrintUnit *from, const GnomePrintUnit *to,
				   gdouble ctmscale, gdouble devicescale)
{
	gdouble absolute;

	g_return_val_if_fail (distance != NULL, FALSE);
	g_return_val_if_fail (from != NULL, FALSE);
	g_return_val_if_fail (to != NULL, FALSE);

	if (from->base == to->base) return gnome_print_convert_distance (distance, from, to);

	if ((from->base == GNOME_PRINT_UNIT_DIMENSIONLESS) || (to->base == GNOME_PRINT_UNIT_DIMENSIONLESS)) {
		*distance = *distance * from->unittobase / to->unittobase;
	}

	switch (from->base) {
	case GNOME_PRINT_UNIT_ABSOLUTE:
		absolute = *distance * from->unittobase;
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		if (devicescale) {
			absolute = *distance * from->unittobase * devicescale;
		} else {
			return FALSE;
		}
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		if (ctmscale) {
			absolute = *distance * from->unittobase * ctmscale;
		} else {
			return FALSE;
		}
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit (base %d)", __FILE__, __LINE__, from->base);
		return FALSE;
		break;
	}

	switch (to->base) {
	case GNOME_PRINT_UNIT_DIMENSIONLESS:
	case GNOME_PRINT_UNIT_ABSOLUTE:
		*distance = absolute / to->unittobase;
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		if (devicescale) {
			*distance = absolute / (to->unittobase * devicescale);
		} else {
			return FALSE;
		}
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		if (ctmscale) {
			*distance = absolute / (to->unittobase * ctmscale);
		} else {
			return FALSE;
		}
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit (base %d)", __FILE__, __LINE__, to->base);
		return FALSE;
		break;
	}

	return TRUE;
}

