/*
 *
 * mtm-ext-selector.c :
 *
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <hestgray@ionet.net>
 *
 */

#include "mtm-view-extensions-selector.h"
#include <gtk/gtkclist.h>
#include <config.h>
#include <gnome.h>

#include "check.xpm"
#include "blank.xpm"

#define CHECK_WIDTH 20
#define FILENAME_WIDTH 100
#define TYPE_WIDTH FILENAME_WIDTH
#define URL_WIDTH FILENAME_WIDTH

static GtkScrolledWindowClass *mtm_ext_selector_parent_class;

struct _MtmExtSelectorPrivate
{
	GtkCList *clist;
	GList *exts;
	GList *unrealized;
	MtmExtSelectorOptions options;
	GdkPixmap *check;
	GdkBitmap *cmask;
	GdkPixmap *blank;
	GdkPixmap *bmask;
};

enum
{
	EXT_TOGGLED,
	LAST_SIGNAL
};

guint mtm_ext_selector_signals[LAST_SIGNAL] = { 0 };

enum
{
	ARG_0,
	ARG_OPTIONS
};

static void mtm_ext_selector_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void mtm_ext_selector_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);

static void column_clicked (GtkCList *clist, int column, MtmExtSelector *sel);
static void row_selected (GtkCList *clist, int row, int col, GdkEvent *event,
	MtmExtSelector *sel);
static void clist_toggle_state (MtmExtSelector *sel, int row, gboolean enable);
static void realize_cb (MtmExtSelector *sel, gpointer user_data);

static void
mtm_ext_selector_destroy (GtkObject *object)
{
	MtmExtSelector *sel = MTM_EXT_SELECTOR (object);
	g_list_free (sel->priv->exts);
	gdk_pixmap_unref (sel->priv->check);
	gdk_pixmap_unref (sel->priv->blank);
	gdk_bitmap_unref (sel->priv->cmask);
	gdk_bitmap_unref (sel->priv->bmask);

	g_free (sel->priv);

	if (GTK_OBJECT_CLASS(mtm_ext_selector_parent_class)->destroy)
		(*GTK_OBJECT_CLASS(mtm_ext_selector_parent_class)->destroy)(object);
}

static void
mtm_ext_selector_class_init (GtkObjectClass *object_class)
{
	mtm_ext_selector_parent_class = gtk_type_class (gtk_scrolled_window_get_type ());
	
	mtm_ext_selector_signals[EXT_TOGGLED] =
		gtk_signal_new ("ext_toggled", GTK_RUN_LAST, G_OBJECT_CLASS_TYPE (object_class),
		GTK_SIGNAL_OFFSET (MtmExtSelectorClass, ext_toggled),
		gtk_marshal_INT__POINTER, GTK_TYPE_INT, 1, GTK_TYPE_POINTER);

	object_class->set_arg = mtm_ext_selector_set_arg;
	object_class->get_arg = mtm_ext_selector_get_arg;

	gtk_object_add_arg_type ("MtmExtSelector::options", GTK_TYPE_ENUM,
		GTK_ARG_READWRITE, ARG_OPTIONS);

	object_class->destroy = mtm_ext_selector_destroy;
}

static void
mtm_ext_selector_init (GtkObject *object)
{
	MtmExtSelector *sel = MTM_EXT_SELECTOR (object);
	gchar *titles[] = { N_("Use"), N_("File"), N_("Type"), N_("URL"), NULL};
	int i;
	int widths[] = { CHECK_WIDTH, FILENAME_WIDTH, TYPE_WIDTH, URL_WIDTH, -1};

	gtk_scrolled_window_set_hadjustment (GTK_SCROLLED_WINDOW (sel),
		NULL);
	gtk_scrolled_window_set_vadjustment (GTK_SCROLLED_WINDOW (sel),
		NULL);

	sel->priv = g_new0 (MtmExtSelectorPrivate, 1);
	sel->priv->exts = NULL;
	sel->priv->options = MTM_EXT_SELECTOR_FILENAME | MTM_EXT_SELECTOR_TYPE | MTM_EXT_SELECTOR_URL;
	
	gtk_signal_connect (GTK_OBJECT (sel), "realize", realize_cb, NULL);

	for (i = 0; titles[i] != NULL; i++)
		titles[i] = gettext (titles[i]);
	
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sel),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	
	sel->priv->clist = GTK_CLIST (gtk_clist_new_with_titles (i, titles));
	gtk_container_add (GTK_CONTAINER (sel), GTK_WIDGET (sel->priv->clist));

	for (i = 0; widths[i] != -1; i++)
		gtk_clist_set_column_width (sel->priv->clist, i, widths[i]);

	gtk_clist_set_selection_mode (sel->priv->clist, GTK_SELECTION_BROWSE);

	gtk_signal_connect (GTK_OBJECT (sel->priv->clist), "select-row",
		row_selected, sel);
	gtk_signal_connect (GTK_OBJECT (sel->priv->clist), "click-column",
		column_clicked, sel);
	
	gtk_widget_show (GTK_WIDGET (sel->priv->clist));
}

GtkType
mtm_ext_selector_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"MtmExtSelector",
			sizeof (MtmExtSelector),
			sizeof (MtmExtSelectorClass),
			(GtkClassInitFunc) mtm_ext_selector_class_init,
			(GtkObjectInitFunc) mtm_ext_selector_init,
			NULL,
			NULL
		};

		type = gtk_type_unique (
			gtk_scrolled_window_get_type (), &info);
	}

	return type;
}

GtkWidget*
mtm_ext_selector_new (void)
{
	MtmExtSelector *sel = gtk_type_new (mtm_ext_selector_get_type ());

	return GTK_WIDGET (sel);
}

static void
mtm_ext_selector_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmExtSelector *sel = MTM_EXT_SELECTOR (object);

	switch (arg_id)
	{
		case ARG_OPTIONS:
			mtm_ext_selector_set_options (sel, 
				GTK_VALUE_ENUM (*arg));
			break;
	}
}

static void
mtm_ext_selector_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmExtSelector *sel = MTM_EXT_SELECTOR (object);

	switch (arg_id)
	{
		case ARG_OPTIONS:
			GTK_VALUE_ENUM (*arg) = mtm_ext_selector_get_options (sel);
	}
}

static void
column_clicked (GtkCList *clist, int column, MtmExtSelector *sel)
{
	int i;
	GList *l;

	if (column != 0)
		return;
	
	l = sel->priv->exts;
	
	for (i = 0; i < clist->rows; i++)
	{
		mtm_ext_selector_ext_toggled (sel, MTM_EXT (l->data));
		l = l->next;
	}
}

static void
row_selected (GtkCList *clist, int row, int col, GdkEvent *event,
	MtmExtSelector *sel)
{
	GdkEventType event_type;

	if (!event)
		event_type = GDK_NOTHING;
	else
		event_type = event->type;

	if (col == 0 || event_type == GDK_2BUTTON_PRESS)
		mtm_ext_selector_ext_toggled (sel,
			MTM_EXT (g_list_nth (sel->priv->exts, row)->data));
}

void
mtm_ext_selector_add_ext (MtmExtSelector *sel, MtmExt *ext, gboolean enabled)	
{
	gchar *texts[5];
	
	g_return_if_fail (sel != NULL);
	g_return_if_fail (ext != NULL);

	sel->priv->exts = g_list_append (sel->priv->exts, ext);
	
	texts[0] = NULL;
	texts[1] = ext->file;
	if (ext->handler && MTM_HANDLER (ext->handler)->desc)
		texts[2] = MTM_HANDLER (ext->handler)->desc;
	else
		texts[2] = ext->type;
	texts[3] = NULL;
	texts[4] = NULL;

	gtk_clist_append (sel->priv->clist, texts);
	
	if (sel->priv->check)
		clist_toggle_state (sel, g_list_length (sel->priv->exts) - 1,
			enabled);
	else
	{
		sel->priv->unrealized = g_list_append (sel->priv->unrealized,
			ext);
		gtk_object_set_data (GTK_OBJECT (ext),
			"sel-enable", GUINT_TO_POINTER (enabled));
	}
}

void
mtm_ext_selector_remove_ext (MtmExtSelector *sel, MtmExt *ext)	
{
	GList *l, *l2;
	int n = 0;
	
	g_return_if_fail (sel != NULL);
	g_return_if_fail (ext != NULL);

	l = sel->priv->exts;
	
	while (l)
	{
		l2 = l->next;
		if (l->data == ext)
		{
			gtk_clist_remove (sel->priv->clist, n);
			g_list_remove_link (sel->priv->exts, l);
		}
		l = l2;
		n++;
	}
}

void
mtm_ext_selector_ext_toggled (MtmExtSelector *sel, MtmExt *ext)
{
	int retval = -1;
	int n = 0;
	GList *l;
	
	g_return_if_fail (sel != NULL);
	g_return_if_fail (ext != NULL);
	
	l = g_list_find (sel->priv->exts, ext);
	g_return_if_fail (l != NULL);
	
	gtk_signal_emit (GTK_OBJECT (sel), 
		mtm_ext_selector_signals[EXT_TOGGLED], ext, &retval);
	
	if (retval == -1)
		return;
	
	l = sel->priv->exts;
	while (l)
	{
		if (l->data == ext)
			clist_toggle_state (sel, n, retval);
		n++;
		l = l->next;
	}
}	

static void
clist_toggle_state (MtmExtSelector *sel, int row, gboolean enable)
{
	if (!sel->priv->check)
		return;
	if (enable)
		gtk_clist_set_pixmap (sel->priv->clist, row, 0,
			sel->priv->check, sel->priv->cmask);
	else
		gtk_clist_set_pixmap (sel->priv->clist, row, 0,
			sel->priv->blank, sel->priv->bmask);
}
			
static void realize_cb (MtmExtSelector *sel, gpointer user_data)
{
	GList *l;
	int i = 0;
	
	sel->priv->check = gdk_pixmap_create_from_xpm_d (
		GTK_WIDGET (sel)->window, &sel->priv->cmask,
		NULL, check_xpm);
	sel->priv->blank = gdk_pixmap_create_from_xpm_d (
		GTK_WIDGET (sel)->window, &sel->priv->bmask,
		NULL, blank_xpm);

	l = sel->priv->unrealized;
	while (l)
	{
		clist_toggle_state (sel, i, GPOINTER_TO_UINT (
			gtk_object_get_data (GTK_OBJECT (l->data),
				"sel-enable")));
		gtk_object_set_data (GTK_OBJECT (l->data), "sel-enable", NULL);
		l = l->next;
		i++;
	}

	if (sel->priv->unrealized)
	{
		g_list_free (sel->priv->unrealized);
		sel->priv->unrealized = NULL;
	}
}

MtmExtSelectorOptions mtm_ext_selector_get_options (MtmExtSelector *sel)
{
	g_return_val_if_fail (sel != NULL, 0);
	
	return sel->priv->options;
}

void mtm_ext_selector_set_options (MtmExtSelector *sel, MtmExtSelectorOptions options)
{
	gboolean cols[3] = { FALSE, FALSE, FALSE };
	int i;

	g_return_if_fail (sel != NULL);

	sel->priv->options = options;
	
	if (options & MTM_EXT_SELECTOR_FILENAME)
		cols[0] = TRUE;
	if (options & MTM_EXT_SELECTOR_EXT_TYPE)
		cols[1] = TRUE;
	if (options & MTM_EXT_SELECTOR_URL)
		cols[2] = TRUE;
	
	for (i = 0; i < 3; i++)
		gtk_clist_set_column_visibility (sel->priv->clist,
			i + 1, cols[i]);
}

void
mtm_ext_selector_clear (MtmExtSelector *sel)
{
	g_return_if_fail (sel != NULL);

	gtk_clist_clear (sel->priv->clist);

	if (sel->priv->exts)
	{
		g_list_free (sel->priv->exts);
		sel->priv->exts = NULL;
	}

	if (sel->priv->unrealized)
	{
		g_list_free (sel->priv->unrealized);
		sel->priv->unrealized = NULL;
	}

}
