from utils.Observable import Observable

import gconf


#
# Configuration backend that uses GConf.
#
class GConfBackend(Observable):

    OBS_UPDATE = 0


    def __init__(self, path):

        self.__path = path
        self.__client = gconf.client_get_default()
        self.__client.add_dir(self.__path, gconf.CLIENT_PRELOAD_RECURSIVE)
        self.__client.notify_add(path, self.__on_notify)

        self.__handlers = {}



    #
    # Propagates configuration changes.
    #
    def __on_notify(self, client, cid, entry, err):

        key = entry.get_key().split("/")[-2:]
        value = entry.get_value()
        if (value):
            self.update_observer(self.OBS_UPDATE, key, value.get_string())



    #
    # Propagates changes in the list of displays.
    #
    def __on_watch(self, client, cid, entry, err):

        if (entry.get_value()):
            key = entry.get_key().split("/")[-2:]
            value = entry.get_value().get_string()
            path1 = entry.get_key()
            path2 = path1[:path1.rfind("/")]
            handler = self.__handlers.get(path1, self.__handlers.get(path2))
            if (handler): handler(key, value)



    #
    # Sets a configuration entry.
    #
    def set(self, *args):

        args = list(args)
        v = gconf.Value(gconf.VALUE_STRING)
        v.set_string(args[-1])
        path = "/".join([self.__path] + args[:-1])
        self.__client.set(path, v)



    #
    # Returns a configuration entry.
    #
    def get(self, *args):

        args = list(args)
        path = "/".join([self.__path] + args)
        v = self.__client.get(path)
        if (v):
            return v.get_string()
        else:
            return ""



    #
    # Registers callbacks for watching configuration changes.
    #
    def watch(self, *args):

        args = list(args)
        # ignore empty parts
        if ("" in args): args.remove("")
        handler = args[-1]
        path = "/".join([self.__path] + args[:-1])
        self.__client.notify_add(path, self.__on_watch)
        self.__handlers[path] = handler



    #
    # Removes the given folder from GConf.
    #
    def clear(self, folder):

        path = self.__path + "/" + folder
        entries = self.__client.all_entries(path)
        for e in entries:
            self.__client.unset(e.get_key())

        self.__client.unset(self.__path + "/" + folder)



    #
    # Returns a list of all directories at the given path.
    #
    def list(self, *args):

        args = list(args)
        path = "/".join([self.__path] + args)
        entries = self.__client.all_dirs(path)

        ret = []
        for e in entries:
            ret.append(e[len(path) + 1:])

        return ret
