#include "config.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#ifdef __FreeBSD__
#include <sys/uio.h>
#endif
#include <glib.h>
#include <signal.h>
#include <libgnome/gnome-i18n.h>

#include "cd-recorder.h"
#include "cdrecorder-marshal.h"
#include "make-iso.h"

#undef DEBUG_CDRECORD

typedef enum {
	PREPARING_TO_WRITE_CD,
	WRITING_CD,
	FIXATING_CD,
	BLANKING_CD,
} ActionType;

static char *actions [] = {
	N_("Preparing to write CD"),
	N_("Writing CD"),
	N_("Fixating CD"),
	N_("Erasing CD"),
	NULL
};

struct CDRecorderPrivate {
	GMainLoop *loop;
	int result;
	int pid;
	int cdr_stdin;
	GString *line;
	GString *cdr_stderr;
	gboolean changed_text;
	gboolean send_return;
	gboolean expect_cdrecord_to_die;
	gint music_count;
	GList *actions;

	char *last_error;
};

/* Signals */
enum {
	PROGRESS_CHANGED,
	ACTION_CHANGED,
	ANIMATION_CHANGED,
	CANCEL_FUNC_CHANGED,
	INSERT_CD_REQUEST,
	INSERT_RWCD_REQUEST,
	LAST_SIGNAL
};

static int cd_recorder_table_signals[LAST_SIGNAL] = { 0 };

static GObjectClass *parent_class = NULL;

static void cd_recorder_class_init (CDRecorderClass *class);
static void cd_recorder_init       (CDRecorder      *cdrecorder);

GType
cd_recorder_get_type (void)
{
	static GType cd_recorder_type = 0;

	if (!cd_recorder_type) {
		static const GTypeInfo cd_recorder_info = {
			sizeof (CDRecorderClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) cd_recorder_class_init,
			(GClassFinalizeFunc) NULL,
			NULL /* class_data */,
			sizeof (CDRecorder),
			0 /* n_preallocs */,
			(GInstanceInitFunc) cd_recorder_init,
		};

		cd_recorder_type = g_type_register_static (G_TYPE_OBJECT,
				"GtkPlaylist", &cd_recorder_info,
				(GTypeFlags)0);
	}

	return cd_recorder_type;
}

static gboolean
cd_needs_dvd_writer (CDDrive *recorder, const char *filename)
{
	if (!(recorder->type & CDDRIVE_TYPE_DVD_RAM_RECORDER)
			|| !(recorder->type & CDDRIVE_TYPE_DVD_RW_RECORDER)) {
		return FALSE;
	}

	/* XXX check the type of media to see if TRUE */

	return FALSE;
}

static void
cancel_cdrecord (gpointer data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;

	kill (cdrecorder->priv->pid, SIGINT);

	cdrecorder->priv->result = RESULT_CANCEL;
	g_main_loop_quit (cdrecorder->priv->loop);
}

void
cd_recorder_insert_cd_retry (CDRecorder *cdrecorder, gboolean is_reload)
{
	if (is_reload) {
		if (cdrecorder->priv->send_return) {
			write (cdrecorder->priv->cdr_stdin, "\n", 1);
		} else {
			kill (cdrecorder->priv->pid, SIGUSR1);
		}
	} else {
		cdrecorder->priv->result = RESULT_RETRY;
		g_main_loop_quit (cdrecorder->priv->loop);
	}
}

static gboolean  
cdrecord_stdout_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;
	char *line;
	char buf[1];
	unsigned int track, mb_written, mb_total;
	GIOStatus status;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	
#ifdef DEBUG_CDRECORD
	if (line) {
		g_print ("cdrecord stdout: %s", line);
	}
#endif
	if (status == G_IO_STATUS_NORMAL) {
		if (cdrecorder->priv->line) {
			g_string_append (cdrecorder->priv->line, line);
			g_free (line);
			line = g_string_free (cdrecorder->priv->line, FALSE);
			cdrecorder->priv->line = NULL;
		}
		
		if (sscanf (line, "Track %2u: %d of %d MB written",
			    &track, &mb_written, &mb_total) == 3) {
			double percent;
			if (!cdrecorder->priv->changed_text) {
				g_signal_emit (G_OBJECT (cdrecorder),
					       cd_recorder_table_signals[ACTION_CHANGED], 0,
					       WRITING_CD);
			}
			if (cdrecorder->priv->music_count == 0) {
				percent = (mb_total > 0) ? ((double)mb_written/mb_total) * 0.98 : 0.0;
			} else {			
				if (mb_total <= 0) {
					percent = 0.0;
				} else {
					percent = (double)((track-1)/(double)cdrecorder->priv->music_count)
					 +	((double)mb_written/mb_total) * 0.98 / (double)cdrecorder->priv->music_count;
				}
			}
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[PROGRESS_CHANGED], 0,
				       percent);
		} else if (g_str_has_prefix (line, "Re-load disk and hit <CR>") ||
			   g_str_has_prefix (line, "send SIGUSR1 to continue")) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[INSERT_CD_REQUEST], 0,
				       TRUE);
			cdrecorder->priv->send_return = (*line == 'R');
		} else if (g_str_has_prefix (line, "Fixating...")) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[ACTION_CHANGED], 0,
				       FIXATING_CD);
		} else if (g_str_has_prefix (line, "Fixating time:")) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[PROGRESS_CHANGED], 0,
				       1.0);
			cdrecorder->priv->result = RESULT_FINISHED;
		} else if (g_str_has_prefix (line, "Last chance to quit, ")) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[CANCEL_FUNC_CHANGED], 0,
				       cancel_cdrecord, TRUE);
		} else if (g_str_has_prefix (line, "Blanking PMA, TOC, pregap")) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[ACTION_CHANGED], 0,
				       BLANKING_CD);
		}

		g_free (line);
		
	} else if (status == G_IO_STATUS_AGAIN) {
		/* A non-terminated line was read, read the data into the buffer. */
		status = g_io_channel_read_chars (source, buf, 1, NULL, NULL);
		if (status == G_IO_STATUS_NORMAL) {
			if (cdrecorder->priv->line == NULL) {
				cdrecorder->priv->line = g_string_new (NULL);
			}
			g_string_append_c (cdrecorder->priv->line, buf[0]);
		}
	} else if (status == G_IO_STATUS_EOF) {
		return FALSE;
	}

	return TRUE;
}

static gboolean  
cdrecord_stderr_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;
	char *line;
	GIOStatus status;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
#ifdef DEBUG_CDRECORD
	if (line) {
		g_print ("cdrecord stderr: %s", line);
	}
#endif
	/* TODO: Handle errors */
	if (status == G_IO_STATUS_NORMAL && !cdrecorder->priv->expect_cdrecord_to_die) {
		g_string_prepend (cdrecorder->priv->cdr_stderr, line);
		if (strstr (line, "No disk / Wrong disk!") != NULL) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[INSERT_CD_REQUEST], 0,
				       FALSE);
			cdrecorder->priv->expect_cdrecord_to_die = TRUE;
		} else if (strstr (line, "Cannot blank disk, aborting.") != NULL) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[INSERT_RWCD_REQUEST], 0,
				       FALSE);
			cdrecorder->priv->expect_cdrecord_to_die = TRUE;
		} else if (strstr (line, "Data may not fit on current disk") != NULL) {
			cdrecorder->priv->last_error = g_strdup (_("The files selected did not fit on the CD"));
		} else if (strstr (line, "Inappropriate audio coding") != NULL) {
			cdrecorder->priv->last_error = g_strdup (_("All audio files must be stereo, 16-bit digital audio with 44100Hz samples"));
		} else if (strstr (line, "cannot write medium - incompatible format") != NULL) {
			g_signal_emit (G_OBJECT (cdrecorder),
				       cd_recorder_table_signals[INSERT_CD_REQUEST], 0,
				       FALSE);
			cdrecorder->priv->expect_cdrecord_to_die = TRUE;
		}
		
		g_free (line);
	} else if (status == G_IO_STATUS_EOF) {
		if (!cdrecorder->priv->expect_cdrecord_to_die) {
			g_main_loop_quit (cdrecorder->priv->loop);
		}
		return FALSE;
	} else {
		g_print ("cdrecord stderr read failed, status: %d\n", status);
	}

	return TRUE;
}

int
cd_recorder_write_iso (CDRecorder *cdrecorder,
		       CDDrive *recorder,
		       const char *filename,
		       int speed,
		       gboolean eject,
		       gboolean blank,
		       gboolean music,
		       gboolean dummy_write)
{
	const char *argv[200]; /* Shouldn't need more than 200 arguments */
	char *speed_str, *dev_str;
	int stdout_pipe, stderr_pipe;
	guint stdout_tag, stderr_tag;
	GIOChannel *channel;
	GError *error;
	int i, saved_i;
	gboolean needs_dvd_writer;

	i = 0;
	saved_i = 0;

	needs_dvd_writer = cd_needs_dvd_writer (recorder, filename);
	if (needs_dvd_writer) {
		argv[i++] = "dvdrecord";
	} else {
		argv[i++] = "cdrecord";
	}

	speed_str = g_strdup_printf ("speed=%d", speed);
	if (speed != 0) {
		argv[i++] = speed_str;
	}
	dev_str = g_strdup_printf ("dev=%s", recorder->cdrecord_id);
	argv[i++] = dev_str;
	if (dummy_write) {
		argv[i++] = "-dummy";
	}
	if (eject) {
		argv[i++] = "-eject";
	}
	if (blank) {
		argv[i++] = "blank=fast";
	}
	argv[i++] = "-v";
	if (music) {
		argv[i++] = "-pad";
		argv[i++] = "-audio";
		saved_i = i;
		cdrecorder->priv->music_count = build_list_of_music_arguments ("burn:///", &i, argv, 200-i-5);
	} else {
		cdrecorder->priv->music_count = 0;
		argv[i++] = "-data";
		argv[i++] = filename;
	}

	if (needs_dvd_writer) {
		argv[i++] = "-dao";
	}

	argv[i++] = NULL;

	cdrecorder->priv->cdr_stderr = NULL;
 retry:
	cdrecorder->priv->result = RESULT_ERROR;
	cdrecorder->priv->expect_cdrecord_to_die = FALSE;
	cdrecorder->priv->line = NULL;
	if (cdrecorder->priv->cdr_stderr != NULL) {
		g_string_truncate (cdrecorder->priv->cdr_stderr, 0);
	} else {
		cdrecorder->priv->cdr_stderr = g_string_new (NULL);
	}

	g_signal_emit (G_OBJECT (cdrecorder),
		       cd_recorder_table_signals[ACTION_CHANGED], 0,
		       PREPARING_TO_WRITE_CD);
	cdrecorder->priv->changed_text = FALSE;
	g_signal_emit (G_OBJECT (cdrecorder),
		       cd_recorder_table_signals[PROGRESS_CHANGED], 0,
		       0.0);
	g_signal_emit (G_OBJECT (cdrecorder),
		       cd_recorder_table_signals[ANIMATION_CHANGED], 0,
		       TRUE);
	g_signal_emit (G_OBJECT (cdrecorder),
		       cd_recorder_table_signals[CANCEL_FUNC_CHANGED], 0,
		       cancel_cdrecord, FALSE);

	error = NULL;
	if (!g_spawn_async_with_pipes  (NULL,
					(char **)argv,
					NULL,
					G_SPAWN_SEARCH_PATH,
					NULL, NULL,
					&cdrecorder->priv->pid,
					&cdrecorder->priv->cdr_stdin,
					&stdout_pipe,
					&stderr_pipe,
					&error)) {
		g_warning ("cdrecord command failed: %s\n", error->message);
		g_error_free (error);
		/* TODO: Better error handling */
	} else {
		/* Make sure we don't block on a read. */
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);
		fcntl (stderr_pipe, F_SETFL, O_NONBLOCK);

		cdrecorder->priv->loop = g_main_loop_new (NULL, FALSE);
	
		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stdout_read,
					     cdrecorder);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stderr_read,
					     cdrecorder);
		g_io_channel_unref (channel);
		
		g_signal_emit (G_OBJECT (cdrecorder),
			       cd_recorder_table_signals[CANCEL_FUNC_CHANGED], 0,
			       cancel_cdrecord, FALSE);

		g_main_loop_run (cdrecorder->priv->loop);
		g_main_loop_unref (cdrecorder->priv->loop);
		
		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);

		if (cdrecorder->priv->result == RESULT_RETRY) {
			goto retry;
		}
	}

	g_free (speed_str);
	g_free (dev_str);

	if (music) {
		int j;
		for (j=0;j<cdrecorder->priv->music_count;j++) {
			g_free ((char *)argv[saved_i+j]);
		}
	}

	g_signal_emit (G_OBJECT (cdrecorder),
		       cd_recorder_table_signals[ANIMATION_CHANGED], 0,
		       FALSE);

	return cdrecorder->priv->result;
}

GList *
cd_recorder_list_actions (CDRecorder *cdrecorder)
{
	return cdrecorder->priv->actions;
}

const char *
cd_recorder_get_error_message_details (CDRecorder *cdrecorder)
{
	g_return_val_if_fail (cdrecorder->priv->result == RESULT_ERROR, NULL);

	return (const char *)cdrecorder->priv->cdr_stderr->str;
}

const char *
cd_recorder_get_error_message (CDRecorder *cdrecorder)
{
	g_return_val_if_fail (cdrecorder->priv->result == RESULT_ERROR, NULL);

	return (const char *)cdrecorder->priv->last_error;
}

static void
cd_recorder_finalize (GObject *object)
{
	CDRecorder *cdrecorder = CD_RECORDER (object);

	g_return_if_fail (object != NULL);

	g_string_free (cdrecorder->priv->cdr_stderr, TRUE);
	//FIXME cleanup

	if (G_OBJECT_CLASS (parent_class)->finalize != NULL) {
		(* G_OBJECT_CLASS (parent_class)->finalize) (object);
	}
}

static void
cd_recorder_init (CDRecorder *cdrecorder)
{
	int i;

	cdrecorder->priv = g_new0 (CDRecorderPrivate, 1);

	for (i = 0; actions[i] != NULL; i++)
	{
		cdrecorder->priv->actions = g_list_append
			(cdrecorder->priv->actions,
			 _(actions[i]));
	}
}

CDRecorder *
cd_recorder_new (void)
{
	return g_object_new (cd_recorder_get_type (), NULL);
}

static void
cd_recorder_class_init (CDRecorderClass *klass)
{
	parent_class = g_type_class_ref (G_TYPE_OBJECT);

	G_OBJECT_CLASS (klass)->finalize = cd_recorder_finalize;

	/* Signals */
	cd_recorder_table_signals[PROGRESS_CHANGED] =
		g_signal_new ("progress-changed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					progress_changed),
				NULL, NULL,
				g_cclosure_marshal_VOID__DOUBLE,
				G_TYPE_NONE, 1, G_TYPE_DOUBLE);
	cd_recorder_table_signals[ACTION_CHANGED] =
		g_signal_new ("action-changed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					action_changed),
				NULL, NULL,
				g_cclosure_marshal_VOID__INT,
				G_TYPE_NONE, 1, G_TYPE_INT);
	cd_recorder_table_signals[ANIMATION_CHANGED] =
		g_signal_new ("animation-changed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					animation_changed),
				NULL, NULL,
				g_cclosure_marshal_VOID__BOOLEAN,
				G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
	cd_recorder_table_signals[CANCEL_FUNC_CHANGED] =
		g_signal_new ("cancel-func-changed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					cancel_func_changed),
				NULL, NULL,
				cdrecorder_marshal_VOID__POINTER_BOOLEAN,
				G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_BOOLEAN);
	cd_recorder_table_signals[INSERT_CD_REQUEST] =
		g_signal_new ("insert-cd-request",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					insert_cd_request),
				NULL, NULL,
				g_cclosure_marshal_VOID__BOOLEAN,
				G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
	cd_recorder_table_signals[INSERT_RWCD_REQUEST] =
		g_signal_new ("insert-rwcd-request",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (CDRecorderClass,
					insert_cd_rw_request),
				NULL, NULL,
				g_cclosure_marshal_VOID__BOOLEAN,
				G_TYPE_NONE, 1, G_TYPE_BOOLEAN);

}

