/* alvabrl.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "alvabrl.h"
#include "sercomm.h"


/* Definitions (!!! TBR !!! - belong to header file) */

/* Supported ALVA Devices */
typedef enum {ALVADT_GENERIC, ALVADT_380, ALVADT_570, ALVADT_544} ALVA_DEV_TYPE;

/* Device data type */
typedef struct
{
	
	unsigned char Version;	
	
	unsigned long front_keys;
	unsigned long display_keys;
	unsigned long pressed_front_keys;
	unsigned long pressed_display_keys;

	unsigned char key_codes[512];
	unsigned char sensor_codes[32];
	unsigned char LastError;
	
} ALVA_DEVICE_DATA;

/* ALVA Input Parser States */
typedef enum
{	
	AIPS_IDLE = 0, AIPS_EXP_2ND_BYTE, AIPS_EXP_CR_OR_ETX
} AIP_STATES;


/* Globals */

/*
static unsigned char Mask8[] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80 };
static unsigned short Mask16[] = {0x0001, 0x0002, 0x0004, 0x0008, 0x0010, 0x0020, 0x0040, 0x0080,
				  									0x0100, 0x0200, 0x0400, 0x0800, 0x1000, 0x2000, 0x4000, 0x8000 };
*/				  									
static unsigned long Mask32[] = {

	0x00000001,0x00000002,0x00000004,0x00000008,0x00000010,0x00000020,0x00000040,0x00000080,
	0x00000100,0x00000200,0x00000400,0x00000800,0x00001000,0x00002000,0x00004000,0x00008000,	
	0x00010000,0x00020000,0x00040000,0x00080000,0x00100000,0x00200000,0x00400000,0x00800000,
	0x01000000,0x02000000,0x04000000,0x08000000,0x10000000,0x20000000,0x40000000,0x80000000

};


static BRL_DEV_CALLBACK			ClientCallback = NULL;
static ALVA_DEV_TYPE				AlvaDevType = ALVADT_GENERIC;
static AIP_STATES						AIPState = AIPS_IDLE;
static short								AIPIndex = 0;

static ALVA_DEVICE_DATA			alvadd;

/* Functions */

/* ... */
static void alva_clear_device_data (ALVA_DEVICE_DATA *add)
{
	memset (add, 0, sizeof(ALVA_DEVICE_DATA));
}

/* ... */
static void alva_reset_aip()
{
	AIPIndex = 0;
	AIPState = AIPS_IDLE;
	alva_clear_device_data(&alvadd);
}

/* ... */
#if 0
short check_escape (unsigned char NewVal)
{
	short rv = 0;
	static short esc = 0;
	
	if (esc)
	{
		/* waiting for the second escape	 */
		if (NewVal == 0x1B)
		{
			rv = 1; /* do it */
		}
		else
		{
			reset_bip();		/* reset to idle		 */
		}
		esc = 0;
	}
	else
	{
		/* not waiting for the second escape */
		if (NewVal == 0x1B)
		{
			esc = 1;
		}
		else {
			rv = 1;	/* do it */
			esc = 0;
		}
	}

	return rv;
}

#endif


/* XXXCodes Events */

/* ... */

void alva_on_keys_changed (unsigned char key_code_1,unsigned char key_code_2)
{
	/* NOTE: called for any display, front, back, chord key changes */
	/* fires when all keys are released */
	
	int i, k;
	int kcix;
	
	/*                            0  1  2  3  4  5  6  7   8   9   10  11  12   13  14  15 */
	static int fk_5xx_remap[] = {-1, 0, 5, 2, 1, 4, 3, -1, -1, -1, 6,  7,  8,   9, -1, -1};
	
	BRAILLE_EVENT_CODE		bec;
	BRAILLE_EVENT_DATA   	bed;
		
	/* fprintf (stderr, "%02x %02x\n", key_code_1, key_code_2); */
	
	/* FILL IN THE KEYMAPS */
	
	switch (AlvaDevType)
	{
		default:
		    break;
    		case ALVADT_GENERIC:
		case ALVADT_570:			
		case ALVADT_544:
		
			switch (key_code_1)
			{
						
				case 0x71:	/* front keys and the sensors above the status field */
			
					if (key_code_2 == 0x00 || key_code_2 == 0x80)
					{
						/* this is actually a *** display key *** */
						if (key_code_2 & 0x80)
						{
							/* key release, reset the bit */
							alvadd.display_keys &= ~Mask32[key_code_2 & 0x1F];
						}
						else
						{
							/* key press, set the bit */
							alvadd.display_keys |= Mask32[key_code_2 & 0x1F];
							alvadd.pressed_display_keys |= Mask32[key_code_2 & 0x1F];
						}
					}
					else
					{
						/* this is a real front key or the 3 left sensor strip pair */
						
						/* a litle remapping... */
						k = key_code_2 & 0x7F;
						if ( k >= 0x20 && k < 0x30)
						{
							/* sensor strip above status elements, lower row */
							i = k - 0x20 + 10;
						}
						else if (k >= 0x30)
						{
							/* sensor strip above status elements, upper row */
							i = k - 0x30 + 20;
						}
						else
						{						
							/* actual front keys */
							i = fk_5xx_remap[key_code_2 & 0x0F];
						}
						
						/* set za bit */
						
						if (i >= 0 && i < 32)
						{
						
							if (key_code_2 & 0x80)
							{
								/* key release, reset the bit */
								alvadd.front_keys &= ~Mask32[i];
							}
							else
							{
								/* key press, set the bit */
								alvadd.front_keys |= Mask32[i];
								alvadd.pressed_front_keys |= Mask32[i];
							}						
						}
					}
										
				break;

				case 0x77:	/* display keys (the 2 x 6 rounded group for 544 satelite) */
							
					k = key_code_2 & 0x1F;										
					if (key_code_2 & 0x80)
					{
						/* key release, reset the bit */
						if ((key_code_2 & 0x7F) >= 0x20)
						{
							/* right block */
							alvadd.display_keys &= ~Mask32[k + 6];
						}
						else
						{
							/* left block */
							alvadd.display_keys &= ~Mask32[k];
						}
						
					}
					else
					{
						/* key press, set the bit */
						if ((key_code_2 & 0x7F) >= 0x20)
						{
							/* right block							 */
							alvadd.display_keys |= Mask32[k + 6];
							alvadd.pressed_display_keys |= Mask32[k + 6];
						}
						else
						{
							/* left block */
							alvadd.display_keys |= Mask32[k];
							alvadd.pressed_display_keys |= Mask32[k];
						}						
					}
					
				break;
		
			}
				
		break;
		
		case ALVADT_380:			
			
			switch (key_code_1)
			{
		
				case 0x71:	/* front keys and the sensors above the status field */
			
					if (key_code_2 & 0x80)
					{
						/* key release, reset the bit */
						alvadd.front_keys &= ~Mask32[key_code_2 & 0x1F];
					}
					else
					{
						/* key press, set the bit */
						alvadd.front_keys |= Mask32[key_code_2 & 0x1F];
						alvadd.pressed_front_keys |= Mask32[key_code_2 & 0x1F];
					}
						
				break;

				case 0x77:	/* display keys (the 2 x 6 rounded groub for 570 satelite) */
		
					if (key_code_2 & 0x80)
					{
						/* key release, reset the bit */
						alvadd.display_keys &= ~Mask32[key_code_2 & 0x1F];
					}
					else
					{
						/* key press, set the bit */
						alvadd.display_keys |= Mask32[key_code_2 & 0x1F];
						alvadd.pressed_display_keys |= Mask32[key_code_2 & 0x1F];
					}
					
				break;
		
			}
			
 			break;
 		
 		}
		
	/* PROCESS THE KEYMAPS */
	
	/* fprintf (stderr, "FK:%08lx DK:%08lx\n", alvadd.front_keys, alvadd.display_keys); */
	
	if (alvadd.front_keys == 0 && alvadd.display_keys == 0)
	{
		/* all (!) keys are depressed (both front and display) */
				
		kcix = 0;
		/* kcix += sprintf (&alvddd.key_codes[kcix], ""); */
				
		for (i = 0; i < 32; ++i)
		{			
			if (alvadd.pressed_display_keys & Mask32[i])
			{
				kcix += sprintf (&alvadd.key_codes[kcix], "DK%02d", i);
			}
		}
				
		for (i = 0; i < 32; ++i)
		{			
			if (alvadd.pressed_front_keys & Mask32[i])
			{
				kcix += sprintf (&alvadd.key_codes[kcix], "FK%02d", i);
			}
		}
				
		/* fire the key code event		 */
		bec = bec_key_codes;
		bed.KeyCodes = alvadd.key_codes;						
		ClientCallback (bec, &bed);
				
		/* clear presed_keys */
		alvadd.pressed_front_keys = 0;											
		alvadd.pressed_display_keys = 0;
	}
		
}

/* ... */

void alva_on_sensors_changed (unsigned char sensor_code_1,unsigned char sensor_code_2)
{

	BRAILLE_EVENT_CODE		bec;
	BRAILLE_EVENT_DATA   	bed;
		
	alvadd.sensor_codes[0] = 0;
	
	switch (sensor_code_1)
	{
		
		case 0x72:	/* lower sensors row, map as mechanical sensors			 */
			
			if ((sensor_code_2 & 0x80) == 0)
			{
				/* sensor pressed */
				sprintf (&alvadd.sensor_codes[0], "HMS%02d", sensor_code_2);
			}
			

		break;	
		
		case 0x75:	/* upper sensor row, map as optical sensors		 */
			
			if ((sensor_code_2 & 0x80) == 0)
			{
				/* sensor pressed */
				sprintf (&alvadd.sensor_codes[0], "HOS%02d", sensor_code_2);
			}
			

		break;	
		
	}
			
	
	/* fprintf (stderr, "SC: %s\n", alvadd.sensor_codes); */
	
	bec = bec_sensor;	
	bed.Sensor.SensorCodes = alvadd.sensor_codes;						
	ClientCallback (bec, &bed);
			
}


/* ... */
short alva_brl_input_parser (int NewVal)
{

	static unsigned char 		code_1;
	static unsigned char 		code_2;
	
	static BRAILLE_EVENT_CODE	bec;
	static BRAILLE_EVENT_DATA   	bed;

	
	if (!ClientCallback) return 0;	/* if we don't have a client callback doesn't make sense to go further */

	/* fprintf (stderr, "%02x ", NewVal); */

	/* raw byte callback */
	bec = bec_raw_byte;
	bed.RawByte = NewVal;
	ClientCallback (bec, &bed);

	switch (AIPState)
	{
	
		/* NOTE: All input is sent as group of two bytes. */
		/* 				Bit 7 of the 2'nd byte is 0 for the make code and 1 for the break code. */
		
		case AIPS_IDLE:
		
			switch (NewVal)
			{
				case 0x1B:	/* start up package (?) */
					AIPState = AIPS_EXP_CR_OR_ETX;
				break;
				
				case 0x71:	/* front keys and the sensos above the status field */
				case 0x72:	/* lower sennsor strip row */
				case 0x75:	/* upper sensor strip row */
				case 0x77:	/* display keys (the 2 x 6 rounded groub for 570 satelite) */
				
					code_1 = NewVal;
					AIPState = AIPS_EXP_2ND_BYTE; 				
					
				break;

			}
			
		break;
	
		case AIPS_EXP_2ND_BYTE:
			
			code_2 = NewVal;
			
			switch (code_1)
			{
				
				case 0x71:	/* front keys ans sensors above the status field */
				case 0x77:	/* display keys (2x6 rounded group for ALVA570) */
					alva_on_keys_changed (code_1, code_2);
					AIPState = AIPS_IDLE;
				break;
				
				case 0x72:	/* lower sensors row */
				case 0x75:	/* upper sensosrs row */
					alva_on_sensors_changed (code_1, code_2);
					AIPState = AIPS_IDLE;
				break;
			}
					
		break;
	
		case AIPS_EXP_CR_OR_ETX:
			
			/* for the moment just discard until end of packet... */
			if (NewVal == 0x0D || NewVal == 0x03)
			{
				AIPState = AIPS_IDLE;
			}
		
		break;
	
		default:
			/* internal error, invalid parser state */
			alva_reset_aip();
			/* assert (TRUE); */
			break;
	}

	return 0; /* 0-give next byte, 1-repeat last byte */
}

/* ... */
int alva_brl_send_dots (unsigned char *dots, short count, short blocking)
{

	int rv = 0;
		
	int realcnt;
	unsigned char sendbuff[256];
			
	realcnt = 0;
	
	sendbuff[realcnt++] = 0x1B;
	sendbuff[realcnt++] = 'B';
	sendbuff[realcnt++] = 0;
	
	switch (AlvaDevType)
	{
		case ALVADT_380:			
			sendbuff[realcnt++] = 85;	/* len */
			rv = 1;
		break;
		
		case ALVADT_570:
			sendbuff[realcnt++] = 70;		/* ??? */
			rv = 1;
		break;
		
		case ALVADT_544:
			sendbuff[realcnt++] = 44;		/* ??? */
			rv = 1;
		break;
		case ALVADT_GENERIC:
		default:
		    break;
				
	}
	
	memcpy (&sendbuff[realcnt], dots, count);	
	/* sendbuff[realcnt] = 0xFF;  TEST */
	realcnt += count;
		
	sendbuff[realcnt++] = 0x0D;
	
	rv = brl_ser_send_data (sendbuff, realcnt, blocking);
	
	return rv;
	
}

/* ... */
void alva_brl_close_device ()
{
	/* close serial communication */
	brl_ser_set_callback (NULL);				
	brl_ser_stop_timer ();	
	brl_ser_close_port ();
}

/* ... */
int alva_brl_open_device (char* DeviceName, short Port, BRL_DEV_CALLBACK DeviceCallback, BRL_DEVICE *Device)
{

	int rv = 0;
	
	alva_reset_aip();
		
	if (strcmp ("ALVA380", DeviceName) == 0)	
	{
		Device->CellCount = 85;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 10;
		
		Device->DisplayCount = 2;
		
		Device->Displays[0].StartCell = 5;
		Device->Displays[0].Width = 80;
		Device->Displays[0].Type = bdt_main;
		
		Device->Displays[1].StartCell = 0;
		Device->Displays[1].Width = 5;
		Device->Displays[1].Type = bdt_status;
		
		AlvaDevType = ALVADT_380;
		/* !!! MORE !!! */
				
		rv = 1;
	}

	else if (strcmp ("ALVA570", DeviceName) == 0) 	
	{		
		
		Device->CellCount = 70;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 22;
		
		Device->DisplayCount = 2;
		
		/* !!! TBR !!! - check this */
		Device->Displays[0].StartCell = 4;	
		Device->Displays[0].Width = 66;
		Device->Displays[0].Type = bdt_main;
		
		Device->Displays[1].StartCell = 0;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;
		
		AlvaDevType = ALVADT_570;
		
		rv = 1;
	}
	
	else if (strcmp ("ALVA544", DeviceName) == 0) 	
	{		
		
		Device->CellCount = 44;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 24;
		
		Device->DisplayCount = 2;
		
		/* !!! TBR !!! - check this */
		Device->Displays[0].StartCell = 4;	
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;
		
		Device->Displays[1].StartCell = 0;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;
		
		AlvaDevType = ALVADT_544;
		
		rv = 1;
	}


	else
	{
		/* unknown device */
		AlvaDevType = ALVADT_GENERIC;
	}

	if (rv)
	{
		
		/* fill device functions for the upper level */
		
		Device->send_dots = alva_brl_send_dots;
		Device->close_device = alva_brl_close_device;
		
		/* open serial communication */
		if (brl_ser_open_port (Port))
		{
			brl_ser_set_callback (alva_brl_input_parser);
			rv = brl_ser_set_comm_param (9600, 'N', 1, 'N');		
			/* brl_ser_start_timer (10);	 INPUT thru async callback */
			brl_ser_init_glib_poll ();		/* SYNC CALLBACK (INPUT thru GNOME loop) */
			ClientCallback = DeviceCallback;
		}	
		else
		{
			/* err, could not open ser port */
			rv = 0;
		}
	}

  return rv;
}

