/* libke.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <cspi/spi.h>
#include "libke.h"
#include "libsrconf.h"
#include "SRMessages.h"

#define XK_MISCELLANY
#include <X11/keysymdef.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>


#undef  USE_ALL_WINDOWS

#define KE_TIME_WAIT 			5000
#define NO_OF_ITEM_IN_KEYSET 		16

#define MODIFIER(K,C) (K ? C : "")
#define LINE(X)       (X == TRUE ? "-" : "")
#define STR(X)	      (X != NULL ? X : "")


enum
{
    KE_LOG_UNKNOWN	= 0,
    KE_LOG_AT_SPI 	= 1,
    KE_LOG_GNOPERNICUS  = 2  
};

static gint ke_log_flags;

typedef struct 
{
    gchar *key;
    gchar *keystring;
} KeyTransTable;

KeyTransTable key_trans_table[]=
{
    {"asciitilda",	"~"},    
    {"diez",		"#"},
    {"exclam",		"!"},
    {"caret",		"^"},
    {"dot",		"."},
    {"quotedbl",	"\""},
    {"dollar",		"$"},
    {"percent", 	"%"},
    {"ampersand", 	"&"},
    {"apostrophe",	"\'"},
    {"parenleft", 	"("},
    {"parenright", 	")"},
    {"asterisk",	"*"},
    {"plus",		"+"},
    {"comma",		","},
    {"minus",		"-"},
    {"slash",		"/"},
    {"colon",		":"},
    {"semicolon",	";"},
    {"less",		"<"},
    {"equal",		"="},
    {"greater",		">"},
    {"question",	"?"},
    {"at",		"@"},
    {"bracketleft",	"["},
    {"backslash",	"\\"},
    {"bracketright",	"]"},
    {"underscore",	"_"},
    {"braceleft",	"{"},
    {"bar",		"|"},
    {"braceright",	"}"}
};    


typedef enum
{
	KE_LAYER_STATE_IDLE,
	KE_LAYER_STATE_LEVEL
} KeyboardLayerState;

typedef struct
{
    AccessibleKeySet *keyset;
    unsigned long    modifier;
} KeySetItem;

typedef struct
{
    glong keyID;
    gint  keycode;
    gchar *keystring;   
    guint modifiers;
}KEKeyEvent;

static short *ke_get_keypad_keycodes (void);
static short *ke_get_key_keycodes (void);

static SPIBoolean 
ke_report_user_key  (const AccessibleKeystroke *stroke, gpointer user_data);
static AccessibleKeystrokeListener *ke_user_key_listener;
GSList *accessible_key_set_list = NULL;
/**
 * report_key_event:
 *
 * the key listener function
 *
**/
static SPIBoolean 
ke_report_key_event  (const AccessibleKeystroke *stroke, gpointer user_data);
static AccessibleKeystrokeListener *ke_key_listener;
static AccessibleKeySet *ke_key_keyset;
/**
 * report_layout_key:
 *
 * the key listener function for hotkeys
 *
**/
static SPIBoolean 
ke_report_layer_key  (const AccessibleKeystroke *stroke, gpointer user_data);
static AccessibleKeystrokeListener *ke_layer_listener;
KeyboardLayerState 	ke_layer_state    = KE_LAYER_STATE_IDLE;
static gint 		ke_layer_level 	  = 0;
static AccessibleKeySet *ke_layer_keyset  = NULL;
static short  		*ke_layer_keycodes = NULL;
/**
 * keyboard_echo_cb:
 *
 * The callback function used for communication with caller, is set during
 * the init process.
 *
**/
static KeyboardEchoCB ke_keyboard_echo_cb = NULL;

/**
 * KEStateEnum:
 *
 * Define the state of the keyboard echo library
 *    KE_IDLE    - library not initialized
 *    KE_RUNNING - initialized and running
**/
typedef enum 
{
    KE_IDLE,
    KE_RUNNING
} KEStateEnum;

/**
 * keyboard_echo_status
 *
 * The library status
 *
**/
static KEStateEnum ke_keyboard_echo_status = KE_IDLE;


static void
ke_log_at_spi (const AccessibleKeystroke *key)
{    
    if (!(ke_log_flags & KE_LOG_AT_SPI))
	return;
	
    fprintf (stderr, "\nAT:%xp--key event:sym %ld (%c)\tmods %x\tcode %d\ttime %ld\tkeystring \"%s\"\ttype %d (press = %d, release = %d)",
	      (unsigned int) key,
	      (glong) key->keyID, (gchar)key->keyID,
	      (guint) key->modifiers,
	      (gint) key->keycode, 
	      (long int) key->timestamp,
	      key->keystring,
	      (gint) key->type, (gint)SPI_KEY_PRESSED, (gint)SPI_KEY_RELEASED );   

}

static void
ke_log_gnopernicus (const SREvent *event)
{
    SRHotkeyData *srhotkey_data = NULL;
    
    if (!(ke_log_flags & KE_LOG_GNOPERNICUS))
	return;

    srhotkey_data = (SRHotkeyData *) event->data;
	
    fprintf (stderr, "\nGN:%xp--key event:type:%d, keyID:%ld, modifiers:%d, keystring:%s",
		     (unsigned int)event,
		     event->type,
		     srhotkey_data->keyID,
		     srhotkey_data->modifiers,
		     srhotkey_data->keystring);
}


static void
ke_print_register_return_value (SPIBoolean retval,
				const gchar *modifiers)
{
    sru_debug ("Key registry(%s): result %s", modifiers, retval ? "succeeded" : "failed");
}

/**
 * srhotkey_data_destructor:
 *
 * This function will be used to destroy data allocated in SREvent's data member.
**/
static void 
ke_srhotkey_data_destructor(gpointer data)
{
    SRHotkeyData *srhotkey_data = (SRHotkeyData *) data;
    if (srhotkey_data != NULL)
    {
	g_free (srhotkey_data->keystring);
	g_free (srhotkey_data);
    }
}

static gchar*
ke_return_keystring (gint keyID)
{
    gchar *rv = NULL;
    switch (keyID)
    {
        case XK_Alt_L:	rv = g_strdup("Alt_L");break;
        case XK_Alt_R:	rv = g_strdup("Alt_R");break;
        case XK_Shift_L:rv = g_strdup("Shift_L");break;
        case XK_Shift_R:rv = g_strdup("Shift_R");break;
        case XK_Control_L:rv = g_strdup("Control_L");break;
        case XK_Control_R:rv = g_strdup("Control_R");break;
        case XK_Caps_Lock:rv = g_strdup("Caps_Lock");break;
        case XK_Num_Lock:rv = g_strdup("Num_Lock");break;
        case XK_Home:	rv = g_strdup("Home");break;
        case XK_End:	rv = g_strdup("End");break;
        case XK_Left:	rv = g_strdup("Left");break;
        case XK_Right:	rv = g_strdup("Right");break;
        case XK_Up:	rv = g_strdup("Up");break;
        case XK_Down:	rv = g_strdup("Down");break;
        case XK_Page_Up:rv = g_strdup("Page_Up");break;
        case XK_Page_Down:rv = g_strdup("Page_Down");break;
        default:
    	    rv = NULL;
    }
/*    fprintf (stderr,"\n\n\nReturn keystring:%s",rv);*/
    return rv;
}
/**
 * call_srkey_cb
 *
 * This function is used to create the SREvent structure and to send it to
 * the callback when SR_KEY event occurs.
 *
**/
static void 
ke_call_srkey_cb (gulong keyID, 
		  guint modifiers, 
		  const gchar *keystring)
{
    SREvent *evnt = NULL;
    SRHotkeyData *srhotkey_data = NULL;
    evnt = sre_new ();
    sru_return_if_fail (evnt);
    srhotkey_data = (SRHotkeyData *) g_new0 (SRHotkeyData, 1);
	
    if (NULL != srhotkey_data )
    {
	srhotkey_data->keyID = keyID;
	srhotkey_data->modifiers = modifiers;
	if ( keyID > 255 )
	{
	    if (!keystring || strlen (keystring) == 0)
		srhotkey_data->keystring = ke_return_keystring (keyID);
	    else
		srhotkey_data->keystring = g_strdup (keystring);
	}
	else
	    srhotkey_data->keystring = g_strdup_printf ("%c", (gchar)keyID );
	    
	evnt->data = srhotkey_data;
	evnt->type = SR_EVENT_KEY;
	evnt->data_destructor = ke_srhotkey_data_destructor;
	
	ke_log_gnopernicus (evnt);
	
	if (ke_keyboard_echo_cb != NULL) 
	    ke_keyboard_echo_cb (evnt,0);
    }
    sre_release_reference (evnt);
}

/**
 * get_config_settings:
 *
 * used to get working parameters,
 *
**/
static gboolean
ke_get_config_settings (GSList **list)

{
    GSList *list_tmp = NULL;
    
    *list = NULL;            
    
    if (!srconf_get_data_with_default (SRC_USER_DEF_LIST , CFGT_LIST, 
		&list_tmp, (gpointer)NULL ,SRC_USER_DEF_SECTION)) 
	return FALSE;
    
    if (!list_tmp) 
	return FALSE;
    
    *list = list_tmp;

    return TRUE;
}

static KeySetItem*
ke_key_set_item_new (void)
{
    KeySetItem *item = NULL;
    item = (KeySetItem *) g_new0 (KeySetItem, 1);
    return item;    
}

static void
ke_key_set_item_free (KeySetItem *item)
{
    sru_return_if_fail (item);
    
    if (item->keyset)
        SPI_freeAccessibleKeySet (item->keyset);

    g_free (item);
}

static gchar*
ke_translate_key (const gchar *key)
{
    gint i;
    
    sru_return_val_if_fail (key, NULL);
    
    for (i = 0 ; i < G_N_ELEMENTS (key_trans_table) ; i++)
    {
	if (!strcmp (key, key_trans_table[i].key))
	    return g_strdup (key_trans_table[i].keystring);
    }
    
    return g_strdup (key);
}

static gchar*
ke_translate_key_revers (const gchar *key)
{
    gint i;

    sru_return_val_if_fail (key, NULL);
    
    for (i = 0 ; i < G_N_ELEMENTS (key_trans_table) ; i++)
    {
	if (!strcmp (key, key_trans_table[i].keystring))
	    return g_strdup (key_trans_table[i].key);
    }
    
    return g_strdup (key);
}

static gboolean
ke_return_modifier_mask (const gchar *data,
			unsigned long *modifiers,
			gchar  **key)
{
    gchar *delimit;
    gchar *iter;
    unsigned long rv = 0;
    
    *modifiers = SPI_KEYMASK_UNMODIFIED;
    *key = NULL;
    
    sru_return_val_if_fail (data, FALSE);
    
    delimit = g_strrstr (data, "-");

    if (!delimit)
    {
	*key = ke_translate_key (data);
	sru_debug ("No delimiter in key combination.");
	return TRUE;
    }
    
    iter = (gchar*)data;
    
    while (*iter != '-')
    {
	switch (*iter)
	{
	    case 'A':
		rv = rv | SPI_KEYMASK_ALT;
	    break;
	    case 'C':
		rv = rv | SPI_KEYMASK_CONTROL;
	    break;
	    case 'S':
		rv = rv | SPI_KEYMASK_SHIFT;
	    break;
	    case 'N':
		rv = rv | SPI_KEYMASK_MOD2;
	    break;
	    default:
		*modifiers = SPI_KEYMASK_UNMODIFIED;
		*key = ke_translate_key (data);
		return FALSE;
	    break;
	}
	iter++;
    }
    *modifiers = rv;
    *key = ke_translate_key (iter + 1);
    return TRUE;
}

static void
ke_register_key_list (GSList *list)
{
    GSList *elem = NULL;
    elem = list;
    
    while (elem)
    {
        unsigned long modifiers;
	gchar *key;
	
	if (ke_return_modifier_mask ((gchar*)elem->data, &modifiers, &key))
	{
	    KeySetItem 		*ksi = NULL;
	    AccessibleKeySet 	*keyset = NULL;
	    gchar **key_string;

	    if (!key) 
	    {
	    	elem = elem->next;
		continue;
	    }
	    
	    ksi = ke_key_set_item_new ();
	    
	    if (!ksi) 
	    {
		g_free (key);
		elem = elem->next;
		continue;
	    }
	    
	    sru_debug ("Modifier %s combination:%ld-%s",(gchar*)elem->data,modifiers,key);
	    
	    key_string = g_new (gchar*, 1);
	    
	    if (!key_string)
	    {
		ke_key_set_item_free (ksi);
		g_free (key);
		elem = elem->next;
		continue;
	    }
	    
	    key_string[0] = key;
	    keyset = SPI_createAccessibleKeySet (1, NULL, NULL, (const gchar**) key_string);
	    g_free (key);
	    
	    ksi->keyset = keyset;
	    ksi->modifier = modifiers;
	    
	    accessible_key_set_list = g_slist_append (accessible_key_set_list,
						      ksi);
	
	    if (ksi->keyset)
	    {						      
	        SPIBoolean retval = FALSE;
		retval = SPI_registerAccessibleKeystrokeListener (ke_user_key_listener,
				    			(AccessibleKeySet *) ksi->keyset,
				    			ksi->modifier,
				    			(gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				    			SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
		ke_print_register_return_value (retval, "");
		retval = SPI_registerAccessibleKeystrokeListener (ke_user_key_listener,
				    			(AccessibleKeySet *) ksi->keyset,
				    			ksi->modifier | SPI_KEYMASK_BUTTON1,
				    			(gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				    			SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
		ke_print_register_return_value (retval, "");
		retval = SPI_registerAccessibleKeystrokeListener (ke_user_key_listener,
				    			(AccessibleKeySet *) ksi->keyset,
				    			ksi->modifier | SPI_KEYMASK_BUTTON2,
				    			(gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				    			SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
		ke_print_register_return_value (retval, "");
		retval = SPI_registerAccessibleKeystrokeListener (ke_user_key_listener,
				    			(AccessibleKeySet *) ksi->keyset,
				    			ksi->modifier | SPI_KEYMASK_BUTTON3,
				    			(gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				    			SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
		ke_print_register_return_value (retval, "");
	    }	    
	}
	elem = elem->next;
    }
}

static gboolean
ke_unregister_key_list (void)
{
    GSList *elem = NULL;

    if (!accessible_key_set_list)
	return FALSE;
    	
    elem = accessible_key_set_list;
    
    while (elem)
    {
	KeySetItem *ksi = elem->data;
	
	SPI_deregisterAccessibleKeystrokeListener (ke_user_key_listener, 
						   ksi->modifier);

	SPI_deregisterAccessibleKeystrokeListener (ke_user_key_listener, 
						   ksi->modifier | 
						   SPI_KEYMASK_BUTTON1);

	SPI_deregisterAccessibleKeystrokeListener (ke_user_key_listener, 
						   ksi->modifier |
						   SPI_KEYMASK_BUTTON2);

	SPI_deregisterAccessibleKeystrokeListener (ke_user_key_listener, 
						   ksi->modifier |
						   SPI_KEYMASK_BUTTON3);
	
	ke_key_set_item_free (ksi);
	
	elem->data = NULL;
	
	elem = elem->next;
    }
    
    g_slist_free (accessible_key_set_list);
    
    accessible_key_set_list = NULL;
    
    return TRUE;
}

static gboolean
ke_user_key_verify_is_in_list (const gchar *keystring)
{
    GSList *elem = NULL;

    sru_return_val_if_fail (keystring, FALSE);
    
    if (!accessible_key_set_list)
	return FALSE;
	
    for (elem = accessible_key_set_list; elem ; elem = elem->next)
    {
	KeySetItem *ksi = (KeySetItem *)elem->data;
	if (!strcmp (((AccessibleKeySet *)(ksi->keyset))->keystrings[0], keystring))
	    return TRUE;
    }
    
    return FALSE;
}

static void
ke_user_key_list_free (GSList *list)
{
    GSList *elem = NULL;    

    sru_return_if_fail (list);

    for (elem = list ; elem ; elem = elem->next)
	g_free (elem->data);
    
    g_slist_free (list);
    
    list = NULL;
}

/**
 * ke_init:
 *
 * This function initialize the keyboard echo library, set the working
 * parameters, create and register key listener(s)
 *	kecb - the callback function used to send data to caller
**/
gboolean
ke_init(KeyboardEchoCB kecb)
{

    GSList *list       = NULL;
    SPIBoolean retval  = FALSE;
    short *ke_keycodes = NULL;
    const  gchar *log  = NULL;
  
    log = g_getenv ("GNOPERNICUS_LOG");
    if (!log)
	log = "";
    ke_log_flags = KE_LOG_UNKNOWN;
    if (strstr (log, "at-spi"))
	ke_log_flags |= KE_LOG_AT_SPI;
    if (strstr (log, "gnopernicus"))
	ke_log_flags |= KE_LOG_GNOPERNICUS;

    sru_return_val_if_fail (ke_keyboard_echo_status == KE_IDLE, FALSE);
    sru_return_val_if_fail (kecb, FALSE);
    sru_debug ("ke_init...");
    ke_keyboard_echo_cb = kecb;
  
    /* get configuration settings */
    ke_get_config_settings (&list);
    
   /* prepare the keyboard snoopers */
    ke_layer_listener 	 = SPI_createAccessibleKeystrokeListener (ke_report_layer_key, NULL);
    ke_user_key_listener = SPI_createAccessibleKeystrokeListener (ke_report_user_key, NULL);
    ke_key_listener 	 = SPI_createAccessibleKeystrokeListener (ke_report_key_event, NULL);
    
    ke_layer_keycodes = ke_get_keypad_keycodes ();
    ke_layer_keyset   = SPI_createAccessibleKeySet (NO_OF_ITEM_IN_KEYSET, NULL, ke_layer_keycodes, NULL);

    ke_keycodes   = ke_get_key_keycodes ();
    ke_key_keyset = SPI_createAccessibleKeySet (NO_OF_ITEM_IN_KEYSET, NULL, ke_keycodes, NULL);
    g_free (ke_keycodes);
    
    if (list)
    {
	ke_register_key_list   (list);
	ke_user_key_list_free  (list);
    }
    
#ifndef USE_ALL_WINDOWS
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON1 | SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B1|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON2 | SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B2|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON3 | SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
			    	      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B3|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON1 | 
				      SPI_KEYMASK_BUTTON2 |
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B1|B2|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON1 | 
				      SPI_KEYMASK_BUTTON3 |
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B1|B3|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON3 |
				      SPI_KEYMASK_BUTTON2 |
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B2|B3|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_BUTTON3 |
				      SPI_KEYMASK_BUTTON2 |
				      SPI_KEYMASK_BUTTON1 |
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
    ke_print_register_return_value (retval, "(B1|B2|B3|NUMLOCK)");
#endif
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif

    ke_print_register_return_value (retval, "(NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_SHIFTLOCK|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(SL|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_SHIFT|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(S|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_ALT|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS				
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(S|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_CONTROL|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif				      
    ke_print_register_return_value (retval, "(C|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_ALT|SPI_KEYMASK_SHIFT|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif				      
    ke_print_register_return_value (retval, "(A|S|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_CONTROL|SPI_KEYMASK_SHIFT|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(C|S|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_layer_listener,
				      (AccessibleKeySet *) ke_layer_keyset,
				      SPI_KEYMASK_CONTROL|SPI_KEYMASK_ALT|SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(C|A|NUMLOCK)");

      
  /* will listen only to unshifted key events, only press */

    retval = SPI_registerAccessibleKeystrokeListener (ke_key_listener,
				      (AccessibleKeySet *) ke_key_keyset,
				      SPI_KEYMASK_UNMODIFIED,
				      (gulong) ( SPI_KEY_PRESSED | SPI_KEY_RELEASED ),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(U)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_key_listener,
				      (AccessibleKeySet *) ke_key_keyset,
				      SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED  | SPI_KEY_RELEASED),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(U|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_key_listener,
				      (AccessibleKeySet *) ke_key_keyset,
				      SPI_KEYMASK_SHIFT,
				      (gulong) ( SPI_KEY_PRESSED  | SPI_KEY_RELEASED),
#ifndef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#else
				      SPI_KEYLISTENER_ALL_WINDOWS);
#endif
    ke_print_register_return_value (retval, "(S)");
    SPI_registerAccessibleKeystrokeListener (ke_key_listener,
				      (AccessibleKeySet *) ke_key_keyset,
				      SPI_KEYMASK_SHIFT | SPI_KEYMASK_MOD2,
				      (gulong) ( SPI_KEY_PRESSED  | SPI_KEY_RELEASED),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(S|NUMLOCK)");
    retval = SPI_registerAccessibleKeystrokeListener (ke_key_listener,
				      (AccessibleKeySet *) ke_key_keyset,
				      SPI_KEYMASK_SHIFTLOCK,
				      (gulong) ( SPI_KEY_PRESSED  | SPI_KEY_RELEASED),
#ifdef USE_ALL_WINDOWS
				      SPI_KEYLISTENER_ALL_WINDOWS);
#else
				      SPI_KEYLISTENER_NOSYNC | SPI_KEYLISTENER_CANCONSUME);
#endif
    ke_print_register_return_value (retval, "(SL)");


    ke_keyboard_echo_status = KE_RUNNING;

    sru_debug ("done.status = KE_RUNNING");

    return TRUE;
}

/**
 * ke_terminate:
 *
 * Deregister key listener(s)
 *
**/
void
ke_terminate (void)
{
    sru_return_if_fail (ke_keyboard_echo_status != KE_IDLE);
    sru_debug ("ke_terminate...");

  /* deregister keylisteners */
    SPI_deregisterAccessibleKeystrokeListener (ke_key_listener, 
					    SPI_KEYMASK_UNMODIFIED );
    SPI_deregisterAccessibleKeystrokeListener (ke_key_listener, 
					    SPI_KEYMASK_UNMODIFIED |
					    SPI_KEYMASK_MOD2 );
    SPI_deregisterAccessibleKeystrokeListener (ke_key_listener, 
					    SPI_KEYMASK_SHIFT );
    SPI_deregisterAccessibleKeystrokeListener (ke_key_listener, 
					    SPI_KEYMASK_SHIFT |
					    SPI_KEYMASK_MOD2 );
    AccessibleKeystrokeListener_unref (ke_key_listener);

#ifndef USE_ALL_WINDOWS
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON1 | 
					    SPI_KEYMASK_BUTTON3 |
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON2 | 
					    SPI_KEYMASK_BUTTON3 |
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON1 | 
					    SPI_KEYMASK_BUTTON2 |
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON1 | 
					    SPI_KEYMASK_BUTTON2 | 
					    SPI_KEYMASK_BUTTON3 |
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON1|
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON2|
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_BUTTON3|
					    SPI_KEYMASK_MOD2);
#endif
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2 | 
					    SPI_KEYMASK_SHIFTLOCK);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2 | 
					    SPI_KEYMASK_SHIFT);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2 | 
					    SPI_KEYMASK_ALT);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2 | 
					    SPI_KEYMASK_CONTROL);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2  | 
					    SPI_KEYMASK_SHIFT |
					    SPI_KEYMASK_CONTROL);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2  | 
					    SPI_KEYMASK_ALT   |
					    SPI_KEYMASK_CONTROL);
    SPI_deregisterAccessibleKeystrokeListener (ke_layer_listener, 
					    SPI_KEYMASK_MOD2  | 
					    SPI_KEYMASK_SHIFT |
					    SPI_KEYMASK_ALT);
    AccessibleKeystrokeListener_unref (ke_layer_listener);

    ke_unregister_key_list ();  

    AccessibleKeystrokeListener_unref (ke_user_key_listener);  
  
    SPI_freeAccessibleKeySet (ke_layer_keyset);
    SPI_freeAccessibleKeySet (ke_key_keyset);
    g_free (ke_layer_keycodes);
    ke_keyboard_echo_status = KE_IDLE;
    sru_debug ("done.");
}

static short*
ke_get_keypad_keycodes (void)
{
    Display *display;
    short *keycodes = (short *) g_new0 (short *, NO_OF_ITEM_IN_KEYSET);	
    gint i;

    display = GDK_DISPLAY ();
    
    keycodes [0] = XKeysymToKeycode (display, XK_KP_0);
    keycodes [1] = XKeysymToKeycode (display, XK_KP_1);
    keycodes [2] = XKeysymToKeycode (display, XK_KP_2);
    keycodes [3] = XKeysymToKeycode (display, XK_KP_3);
    keycodes [4] = XKeysymToKeycode (display, XK_KP_4);
    keycodes [5] = XKeysymToKeycode (display, XK_KP_5);
    keycodes [6] = XKeysymToKeycode (display, XK_KP_6);
    keycodes [7] = XKeysymToKeycode (display, XK_KP_7);
    keycodes [8] = XKeysymToKeycode (display, XK_KP_8);
    keycodes [9] = XKeysymToKeycode (display, XK_KP_9);
    keycodes [10] = XKeysymToKeycode (display, XK_KP_Decimal);
    keycodes [11] = XKeysymToKeycode (display, XK_KP_Enter);
    keycodes [12] = XKeysymToKeycode (display, XK_KP_Add);
    keycodes [13] = XKeysymToKeycode (display, XK_KP_Subtract);
    keycodes [14] = XKeysymToKeycode (display, XK_KP_Multiply);
    keycodes [15] = XKeysymToKeycode (display, XK_KP_Divide);
    
    for (i = 0; i < 16; ++i)
	sru_debug ("keycode[%d] = %d", i, (int) keycodes[i]);
	
    return keycodes;
}

static short*
ke_get_key_keycodes (void)
{
    Display *display;
    short *keycodes = (short *) g_new0 (short *, NO_OF_ITEM_IN_KEYSET);
/*    gint i;*/

    display = GDK_DISPLAY ();
    
    keycodes [0] = XKeysymToKeycode (display, XK_Alt_L);
    keycodes [1] = XKeysymToKeycode (display, XK_Alt_R);
    keycodes [2] = XKeysymToKeycode (display, XK_Shift_L);
    keycodes [3] = XKeysymToKeycode (display, XK_Shift_R);
    keycodes [4] = XKeysymToKeycode (display, XK_Control_L);
    keycodes [5] = XKeysymToKeycode (display, XK_Control_R);
    keycodes [6] = XKeysymToKeycode (display, XK_Caps_Lock);
    keycodes [7] = XKeysymToKeycode (display, XK_Num_Lock);
    keycodes [8] = XKeysymToKeycode (display, XK_Home);
    keycodes [9] = XKeysymToKeycode (display, XK_End);
    keycodes [10] = XKeysymToKeycode (display, XK_Left);
    keycodes [11] = XKeysymToKeycode (display, XK_Right);
    keycodes [12] = XKeysymToKeycode (display, XK_Up);
    keycodes [13] = XKeysymToKeycode (display, XK_Down);
    keycodes [14] = XKeysymToKeycode (display, XK_Page_Up);
    keycodes [15] = XKeysymToKeycode (display, XK_Page_Down);
/*
    for (i = 0; i < NO_OF_ITEM_IN_KEYSET; ++i)
	sru_debug ("keycode[%d] = %d", i, (int) keycodes[i]);
*/	
    return keycodes;
}


/**
 * report_key_event:
 *
 * the key listener function
 *
**/
static SPIBoolean
ke_report_key_event (const AccessibleKeystroke *key, gpointer user_data)
{
    static gboolean key_busy = FALSE;    
    static GQueue  *key_queue = NULL;
    KEKeyEvent     *ev;

    sru_debug ("(Key)Received key event:\n\tsym %ld (%c)\n\tmods %x\n\tcode %d\n\ttime %ld\n\tkeystring \"%s\"\n",
	     (glong) key->keyID, (gchar)key->keyID,
	     (guint) key->modifiers,
	     (gint) key->keycode,
	     (long int) key->timestamp,
	     key->keystring);   

    ke_log_at_spi (key);    
    
    if (key->type == SPI_KEY_RELEASED)
    {
	sru_debug ("keyrelease received");
	return FALSE;
    }
    
    ev = (KEKeyEvent *) g_new0 (KEKeyEvent, 1);
    
    sru_return_val_if_fail (ev, FALSE);

    if (!key_queue)
	key_queue = g_queue_new ();

    ev->keyID     = key->keyID;
    ev->modifiers = key->modifiers;
    ev->keycode	  = key->keycode;
    ev->keystring = g_strdup (key->keystring);
    
    g_queue_push_head (key_queue, ev);
    
    sru_debug ("key_busy:%d", key_busy);
    sru_return_val_if_fail (!key_busy, FALSE);

    key_busy = TRUE;
    
    while (!g_queue_is_empty (key_queue))
    {
	KEKeyEvent *key1;
	
	key1 = (KEKeyEvent *) g_queue_pop_tail (key_queue);
	
	ke_call_srkey_cb (key1->keyID, 0, key1->keystring);

	g_free (key1->keystring);
		
	g_free (key1);
    }

    g_queue_free (key_queue);
    
    key_queue = NULL;

    key_busy = FALSE;
        
    return FALSE;
}

/**
 * ke_config_changed:
 *
 * Using this function the KE can be notified for configuration changes
**/
void 
ke_config_changed (void)
{
    GSList *list = NULL;

    sru_debug ("ke_config_changed invoked.");
    
    ke_unregister_key_list ();  
    
    if (ke_get_config_settings (&list))
    {
	ke_register_key_list   (list);
	ke_user_key_list_free  (list);
    }
}

static gint
ke_return_key (glong keyID)
{
    gint layer_code = -1;
    switch (keyID)
    {
	case XK_KP_0:layer_code = 0; break;
	case XK_KP_1:layer_code = 1; break;
	case XK_KP_2:layer_code = 2; break;
	case XK_KP_3:layer_code = 3; break;
	case XK_KP_4:layer_code = 4; break;
	case XK_KP_5:layer_code = 5; break;
	case XK_KP_6:layer_code = 6; break;
	case XK_KP_7:layer_code = 7; break;
	case XK_KP_8:layer_code = 8; break;
	case XK_KP_9:layer_code = 9; break;
	case XK_KP_Decimal:layer_code = 10; break;
	case XK_KP_Enter:layer_code = 11; break;
	case XK_KP_Add:layer_code = 12; break;
	case XK_KP_Subtract:layer_code = 13; break;
	case XK_KP_Multiply:layer_code = 14; break;
	case XK_KP_Divide:layer_code = 15; break;
	default:
	break;
    }		
    return layer_code;
}

static gboolean
ke_check_is_keypad_key (glong keyID, gint keycode, gchar *keystring)
{
    switch (keyID)
    {
	case XK_KP_0:
	case XK_KP_1:
	case XK_KP_2:
	case XK_KP_3:
	case XK_KP_4:
	case XK_KP_5:
	case XK_KP_6:
	case XK_KP_7:
	case XK_KP_8:
	case XK_KP_9:
	case XK_KP_Divide:
	case XK_KP_Multiply:
	case XK_KP_Add:
	case XK_KP_Subtract:
	case XK_KP_Decimal:
	case XK_KP_Enter:
	    return TRUE;
	break;
	default:
	break;
    }
    return FALSE;
}

static gboolean
ke_is_zero (glong keyID, gint keycode, gchar *keystring)
{
    if (keyID == XK_KP_0)
	return TRUE;
    
    return FALSE;
}

static void 
ke_call_keyboard_command_layer_change_cb(const gchar *buf,gint flag)
{
    SREvent *evnt = NULL;
    
    sru_return_if_fail (buf);
    sru_return_if_fail (strlen(buf) != 0);
    evnt = sre_new ();
    sru_return_if_fail (evnt);
    
    sru_debug ("Call keyboard command_layer change callback.");
    
    evnt->data = g_strdup(buf);
    evnt->type = SR_EVENT_COMMAND_LAYER_CHANGED;
    evnt->data_destructor = g_free;

    if (evnt->data)
    {
    	ke_log_gnopernicus (evnt);
	if (ke_keyboard_echo_cb != NULL) 
	    ke_keyboard_echo_cb (evnt, flag);
    }
    
    sre_release_reference (evnt);
}

static void 
ke_call_keyboard_command_cb (const gchar *buf)
{
    SREvent *evnt = NULL;
    
    sru_return_if_fail (buf);
    sru_return_if_fail (strlen(buf) != 0);
    evnt = sre_new ();
    sru_return_if_fail (evnt);

    evnt->data = g_strdup(buf);
    evnt->type = SR_EVENT_COMMAND_LAYER;
    evnt->data_destructor = g_free;
	
    if (evnt->data)
    {
    	ke_log_gnopernicus (evnt);
        if (ke_keyboard_echo_cb != NULL) 
	    ke_keyboard_echo_cb (evnt, 0);
    }
	
    sre_release_reference (evnt);
}

static gboolean
ke_press_wait_function (gpointer data)
{
    if (ke_layer_state != KE_LAYER_STATE_IDLE)
    {
	gchar *cmd;
	
	ke_layer_state = KE_LAYER_STATE_IDLE;
				
	cmd = g_strdup_printf("L%02d", ke_layer_level);				
	
	ke_call_keyboard_command_layer_change_cb (cmd, KE_LAYER_TIME_OUT);
	
	sru_debug ("Command code %s TIMEOUT", cmd);				

	g_free (cmd);
    }
    return FALSE;
}

/**
 * report_layer_key:
 * 
 * the key listener function for hotkeys
 *
**/
static SPIBoolean 
ke_report_layer_key  (const AccessibleKeystroke *key, gpointer user_data)
{
    static gboolean layer_busy = FALSE;    
    static GQueue  *layer_queue = NULL;
    KEKeyEvent     *ev;
    
    sru_debug ("(Layer)Received key event:\n\tsym %ld (%c)\n\tmods %x\n\tcode %d\n\ttime %ld\n\tkeystring \"%s\"\n\ttype %d (press = %d, release = %d)",
	      (glong) key->keyID, (gchar)key->keyID,
	      (guint) key->modifiers,
	      (gint) key->keycode, 
	      (long int) key->timestamp,
	      key->keystring,
	      (gint) key->type, (gint)SPI_KEY_PRESSED, (gint)SPI_KEY_RELEASED );   

    ke_log_at_spi (key);	          
    
    if (key->type == SPI_KEY_RELEASED)
    {
	sru_debug ("keyrelease received");
	return FALSE;
    }

    ev = (KEKeyEvent *) g_new0 (KEKeyEvent, 1);
    
    sru_return_val_if_fail (ev, FALSE);
    
    if (!layer_queue)
	layer_queue = g_queue_new ();

    ev->keyID     = key->keyID;
    ev->modifiers = key->modifiers;
    ev->keycode	  = key->keycode;
    ev->keystring = g_strdup (key->keystring);
    
    g_queue_push_head (layer_queue, ev);

    sru_debug ("layer_busy:%d", layer_busy);
    sru_return_val_if_fail (!layer_busy, FALSE);
    
    layer_busy = TRUE;
        
    while (!g_queue_is_empty (layer_queue))
    {
	KEKeyEvent *key1;
		
	key1 = (KEKeyEvent *) g_queue_pop_tail (layer_queue);
		
	if (ke_check_is_keypad_key (key1->keyID, key1->keycode, key1->keystring))
	{
	    switch (ke_layer_state)
	    {
		case KE_LAYER_STATE_IDLE:
		{
			if (ke_is_zero (key1->keyID, key1->keycode, key1->keystring))
			{
			    sru_debug ("Layer change key pressed");
			    ke_layer_state = KE_LAYER_STATE_LEVEL;
			    g_timeout_add ( KE_TIME_WAIT , 
					    ke_press_wait_function , 
					    NULL ) ;
			}
			else
			{
			    gchar *cmd;
			    ke_layer_state = KE_LAYER_STATE_IDLE;
			    cmd = g_strdup_printf ("L%02dK%02d", 
						    ke_layer_level, 
						    ke_return_key (key1->keyID));				
			    ke_call_keyboard_command_cb (cmd);
			    sru_debug ("Command code %s",cmd);				
			    g_free (cmd);
			}
		    break;
		}
		case KE_LAYER_STATE_LEVEL:
		{
			if (ke_check_is_keypad_key (key1->keyID, key1->keycode, key1->keystring))
			{
			    gchar *cmd = NULL;
			    ke_layer_level = ke_return_key (key1->keyID);
			    cmd = g_strdup_printf ("L%02d", ke_layer_level);
			    ke_call_keyboard_command_layer_change_cb (cmd, 
								     KE_LAYER_CHANGED);
			    sru_debug ("Command:%s",cmd);
			    g_free (cmd);
			}
			    
			ke_layer_state = KE_LAYER_STATE_IDLE;
			break;
		}
		default:
		    ke_layer_state = KE_LAYER_STATE_IDLE;
		break;
	    }
	}
	else
	    ke_layer_state = KE_LAYER_STATE_IDLE;

	g_free (key1->keystring);		
	g_free (key1);
    }

    g_queue_free (layer_queue);
    layer_queue = NULL;
    layer_busy = FALSE;
  
    return TRUE;
}

static SPIBoolean 
ke_report_user_key  (const AccessibleKeystroke *key, gpointer user_data)
{
    static gboolean user_busy = FALSE;    
    static GQueue  *user_queue = NULL;
    KEKeyEvent     *ev;

    sru_debug ("(User)Received key event:\n\tsym %ld (%c)\n\tmods %x\n\tcode %d\n\ttime %ld\n\tkeystring \"%s\"\n\ttype %d (press = %d, release = %d)",
	      (glong) key->keyID, (gchar)key->keyID,
	      (guint) key->modifiers,
	      (gint) key->keycode, 
	      (long int) key->timestamp,
	      key->keystring,
	      (gint) key->type, (gint)SPI_KEY_PRESSED, (gint)SPI_KEY_RELEASED );   

    ke_log_at_spi (key);   
     
    if (key->type == SPI_KEY_RELEASED)
    {
	sru_debug ("keyrelease received");
	return FALSE;
    }
	
    sru_return_val_if_fail (ke_user_key_verify_is_in_list (key->keystring), FALSE);

    ev = (KEKeyEvent *) g_new0 (KEKeyEvent, 1);
    
    sru_return_val_if_fail (ev, FALSE);

    if (!user_queue)
	user_queue = g_queue_new ();

    ev->keyID     = key->keyID;
    ev->modifiers = key->modifiers;
    ev->keycode	  = key->keycode;
    ev->keystring = g_strdup (key->keystring);
    
    g_queue_push_head (user_queue, ev);

    sru_debug ("user_busy:%d", user_busy);
    sru_return_val_if_fail (!user_busy, FALSE);
    user_busy = TRUE;
        
    while (!g_queue_is_empty (user_queue))
    {
	KEKeyEvent *key1;
	gboolean modif = FALSE;
	gchar *cmd = NULL;
	gchar *keystr = NULL;
    
	key1 = (KEKeyEvent *) g_queue_pop_tail (user_queue);
		
	modif = ((key1->modifiers & SPI_KEYMASK_ALT)    ||
		(key1->modifiers & SPI_KEYMASK_CONTROL) ||
		(key1->modifiers & SPI_KEYMASK_MOD2)    ||
		(key1->modifiers & SPI_KEYMASK_SHIFT));

	keystr = ke_translate_key_revers (key1->keystring);
    
	cmd = g_strconcat ( MODIFIER(key1->modifiers & SPI_KEYMASK_ALT,"A"),
		    	    MODIFIER(key1->modifiers & SPI_KEYMASK_CONTROL,"C"),
		    	    MODIFIER(key1->modifiers & SPI_KEYMASK_SHIFT,"S"),
		    	    MODIFIER(key1->modifiers & SPI_KEYMASK_MOD2,"N"),
		    	    LINE(modif),
		    	    STR(keystr), NULL);
		       
	sru_debug ("(User)keyevent:%s", cmd);	  
    
	ke_call_keyboard_command_cb (cmd);
    
	g_free (keystr);
	g_free (cmd);
	g_free (key1->keystring);
	g_free (key1);
    }

    g_queue_free (user_queue);
    user_queue = NULL;
    user_busy = FALSE;
        
    return TRUE;
}
