/* gok-modifier.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-modifier.h"
#include "gok-log.h"
#include "gok-keyboard.h"
#include "main.h"

static GokModifier* m_pModifierFirst;

/**
* gok_modifier_open
*
* Initializes the modifier. Call this once at the beginning of the program.
**/
void gok_modifier_open ()
{
	m_pModifierFirst = NULL;
}

/**
* gok_modifier_close
*
* Deletes any modifiers that have been created. This must be called at the
* end of the program.
**/
void gok_modifier_close ()
{
	GokModifier* pModifier;
	GokModifier* pModifierTemp;
	
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		pModifierTemp = pModifier;
		pModifier = pModifier->pModifierNext;

		g_free (pModifierTemp->Name);
		g_free (pModifierTemp);
	}
}

/**
* gok_modifier_add
* @Name: Name of the modifier.
*
* Adds a modifier to the list of modifiers.
*
* returns: TRUE if the modifier was created, FALSE if not.
**/
gboolean gok_modifier_add (gchar* Name)
{
	GokModifier* pNewModifier;
	GokModifier* pModifierTemp;

	g_assert (Name != NULL);
	
	/* check if this modifier already exists */
	pModifierTemp = m_pModifierFirst;
	while (pModifierTemp != NULL)
	{
		if (strcmp (pModifierTemp->Name, Name) == 0)
		{
			/* the modifier already exists so don't add it again */
			return TRUE;
		}
		pModifierTemp = pModifierTemp->pModifierNext;
	}
	
	/* create a new modifier */
	pNewModifier = (GokModifier*)g_malloc (sizeof (GokModifier));
	
	/* add the name to the modifier */
	pNewModifier->Name = (gchar*)g_malloc (strlen (Name) + 1);
	strcpy (pNewModifier->Name, Name);
	
	/* initialize the modifier */
	pNewModifier->State = MODIFIER_STATE_OFF;
	pNewModifier->WrapperPre = NULL;
	pNewModifier->WrapperPost = NULL;
	pNewModifier->pModifierNext = NULL;
	pNewModifier->Type = MODIFIER_TYPE_NORMAL;
	
	/* hook the new modifier into the list of modifiers */
	if (m_pModifierFirst == NULL)
	{
		m_pModifierFirst = pNewModifier;
	}
	else
	{
		pModifierTemp = m_pModifierFirst;
		while (pModifierTemp->pModifierNext != NULL)
		{
			pModifierTemp = pModifierTemp->pModifierNext;
		}
		pModifierTemp->pModifierNext = pNewModifier;
	}
		
	return TRUE;
}

/**
* gok_modifier_set_pre
* @Name: Name of the modifier.
* @pOutput: Pointer to the output that will be set as the modifier wrapper 'pre'.
*
* Sets the wrapper 'pre' output for the given modifier.
*
* returns: TRUE if the modifier 'pre' was set, FALSE if not.
**/
gboolean gok_modifier_set_pre (gchar* Name, GokOutput* pOutput)
{
	GokModifier* pModifier;

	g_assert (Name != NULL);
	g_assert (pOutput != NULL);
	
	/* find the modifier in the list of modifiers */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (strcmp (pModifier->Name, Name) == 0)
		{
			break;
		}
		pModifier = pModifier->pModifierNext;
	}
	
	if (pModifier == NULL)
	{
		gok_log_x ("Can't add 'pre'! Modifier '%s' doesn't exist!\n", Name);
		return FALSE;
	}
	
	pModifier->WrapperPre = pOutput;
	
	return TRUE;
}

/**
* gok_modifier_set_type
* @Name: Name of the modifier.
* @Type: The type of the modifier.
*
* Sets the 'Type' attribute for the given modifier.
**/
void gok_modifier_set_type (gchar* Name, int Type)
{
	GokModifier* pModifier;
	
	/* find the modifier in the list of modifiers */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (strcmp (pModifier->Name, Name) == 0)
		{
			pModifier->Type = Type;
			break;
		}
		pModifier = pModifier->pModifierNext;
	}	
}

/**
* gok_modifier_set_post
* @Name: Name of the modifier.
* @pOutput: Pointer to the output that will be set as the modifier wrapper 'post'.
*
* Sets the 'post' output for the given modifier.
*
* returns: TRUE if the modifier 'pre' was set, FALSE if not.
**/
gboolean gok_modifier_set_post (gchar* Name, GokOutput* pOutput)
{
	GokModifier* pModifier;

	g_assert (Name != NULL);
	g_assert (pOutput != NULL);
	
	/* find the modifier in the list of modifiers */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (strcmp (pModifier->Name, Name) == 0)
		{
			break;
		}
		pModifier = pModifier->pModifierNext;
	}
	
	if (pModifier == NULL)
	{
		gok_log_x ("Can't add 'post'! Modifier '%s' doesn't exist!\n", Name);
		return FALSE;
	}
	
	pModifier->WrapperPost = pOutput;
	
	return TRUE;
}

/**
* gok_modifier_output_pre
*
* Sends all the wrapper 'pre' outputs to the system.
**/
void gok_modifier_output_pre ()
{
	GokModifier* pModifier;
	
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (pModifier->State != MODIFIER_STATE_OFF)
		{
			gok_output_send_to_system (pModifier->WrapperPre, FALSE);
		}
		pModifier = pModifier->pModifierNext;
	}
}

/**
* gok_modifier_output_post
*
* Sends all the wrapper 'post' outputs to the system.
**/
void gok_modifier_output_post ()
{
	GokModifier* pModifier;

	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (pModifier->State != MODIFIER_STATE_OFF)
		{
			gok_output_send_to_system (pModifier->WrapperPost, FALSE);
		}
		pModifier = pModifier->pModifierNext;
	}
}

/**
* gok_modifier_all_off
*
* Changes the state of modifier keys to OFF, unless they are locked on.
**/
void gok_modifier_all_off ()
{
}

/**
* gok_modifier_get_state
* @NameModifier: Name of the modifier you want the state for.
*
* returns: The state of the modifier
**/
int gok_modifier_get_state (gchar* NameModifier)
{
	GokModifier* pModifier;

	/* find the modifier in the list */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if ((pModifier->Name != NULL) &&
			(strcmp (pModifier->Name, NameModifier) == 0))
		{
			/* return its state */
			return pModifier->State;
		}
		pModifier = pModifier->pModifierNext;
	}

	/* can't find the modifier in the list so return OFF */
	gok_log_x ("Can't find modifier named %s!\n", NameModifier);
	return MODIFIER_STATE_OFF;
}

/**
* gok_modifier_get_type
* @NameModifier: Name of the modifier you want the type for.
*
* returns: The type of the modifier
**/
int gok_modifier_get_type (gchar* NameModifier)
{
	GokModifier* pModifier;

	/* find the modifier in the list */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if ((pModifier->Name != NULL) &&
			(strcmp (pModifier->Name, NameModifier) == 0))
		{
			/* return its state */
			return pModifier->Type;
		}
		pModifier = pModifier->pModifierNext;
	}

	/* can't find the modifier in the list so return MODIFIER_TYPE_NORMAL */
	gok_log_x ("Can't find modifier name!\n");
	return MODIFIER_TYPE_NORMAL;
}

/**
* gok_modifier_get_normal
*
* returns: TRUE if there are no modifiers on (or locked on). Returns
* FALSE if one or more modifiers are on (or locked on).
**/
gboolean gok_modifier_get_normal()
{
	GokModifier* pModifier;

	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (pModifier->State != MODIFIER_STATE_OFF)
		{
			return FALSE;
		}
		pModifier = pModifier->pModifierNext;
	}
	return TRUE;
}

/**
* gok_modifier_press
* @NameModifier: Name of the modifier key that has been pressed.
*
* This must be called when a modifier key has been pressed. Changes the
* state of the modifier.
**/
void gok_modifier_press (GokKey* pKey, gchar* NameModifier)
{
	GokModifier* pModifier;
	
	gok_keyboard_output_key (pKey);
}

/**
* gok_modifier_update_modifier_keys
*
* Updates the indicator on all the modifier keys for the current keyboard.
**/
void gok_modifier_update_modifier_keys (GokKeyboard* pKeyboard)
{
	GokKey* pKey;
	int state;
	gchar buffer[300];
	
	if (pKeyboard == NULL)
	{
		return;
	}

	pKey = pKeyboard->pKeyFirst;
	while (pKey != NULL)
	{
	        pKey->ComponentState.latched = 0;
	        pKey->ComponentState.locked = 0;
		if (pKey->Type == KEYTYPE_MODIFIER)
		{
			state = gok_modifier_get_state (pKey->ModifierName);
			if (state == MODIFIER_STATE_ON)
			{
#if STATUS_IN_LABEL
				strcpy (buffer, gok_key_get_label (pKey));
				if (gok_modifier_get_type (pKey->ModifierName) == MODIFIER_TYPE_NORMAL)
				{
					strcat (buffer, "+");
				}
				else
				{
					strcat (buffer, "!");
				}
				gok_key_set_button_label (pKey, buffer);
#else
				pKey->ComponentState.latched = 1;
#endif
			}
			else if (state == MODIFIER_STATE_LOCKED)
			{
#if STATUS_IN_STRING
				strcpy (buffer, gok_key_get_label (pKey));
				strcat (buffer, "*");
				gok_key_set_button_label (pKey, buffer);
#else
				pKey->ComponentState.locked = 1;
#endif
			}
			gok_key_update_toggle_state (pKey);
		}
		pKey = pKey->pKeyNext;
	}
}

/**
* gok_modifier_set_state:
* @Name: Name of the modifier.
* @State: New state of the modifier (see enum ModifierStates).
*
* Changes the state of the given modifier.
**/
void gok_modifier_set_state (gchar* Name, gint State)
{
	GokModifier* pModifier;

	/* find the modifier in the list */
	pModifier = m_pModifierFirst;
	while (pModifier != NULL)
	{
		if (strcmp (pModifier->Name, Name) == 0)
		{
			if (pModifier->State != State)
			{
				pModifier->State = State;
				
				gok_keyboard_update_labels();
				gok_modifier_update_modifier_keys (gok_main_get_current_keyboard());
			}
			return;
		}
		pModifier = pModifier->pModifierNext;
	}
	gok_log_x ("Can't find modifier: %s!\n", Name);
}

/**
 **/
gchar*
gok_modifier_first_name_from_mask (guint modmask)
{
	/* FIXME: no relation to modifiers list, unfortunately */
	if (!modmask)
		return NULL;
	if (modmask & ShiftMask)
		return "shift";
	if (modmask & LockMask)
		return "capslock";
	if (modmask & 	
	    gok_key_get_numlock_mask (gok_keyboard_get_display ()))
		return "numlock";
	if (modmask & ControlMask)
		return "ctrl";
	if (modmask & Mod1Mask)
		return "alt";
	if (modmask & Mod2Mask)
		return "mod2";
	if (modmask & Mod3Mask)
		return "mod3";
	if (modmask & Mod4Mask)
		return "mod4";
	if (modmask & Mod5Mask)
		return "mod5";
	return NULL;
}
