/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <sys/stat.h>

#include <gdk/gdkx.h>
#include <gnome.h>
#include <glade/glade.h>

#include <libxklavier/xklavier.h>
#include <libxklavier/xklavier_config.h>

#include "gswitchit_capplet.h"

static GtkWidget *grabDialog;
static GtkWidget *userDefinedMenu;
static GtkWidget *switchcutsOMenu;

static void CappletSetStateToChanged( GtkWidget * w,
                                      GSwitchItCapplet * gswic )
{
  gswic->stateChanged = TRUE;
  gtk_widget_set_sensitive( CappletGetGladeWidget( gswic, "btnOk" ), TRUE );
}

static void CappletConfigUpdate( GSwitchItCapplet * gswic )
{
  GnomeIconEntry *ie[XkbNumKbdGroups];
  gint i;

  for( i = XklGetNumGroups(  ); --i >= 0; )
  {
    char sz[30];
    struct stat buf;
    char *fn;

    g_snprintf( sz, sizeof( sz ), "iconentryFlag.%d", i );
    ie[i] =
      GNOME_ICON_ENTRY( gtk_object_get_data
                        ( GTK_OBJECT( gswic->capplet ), sz ) );

    fn = gnome_icon_entry_get_filename( ie[i] );
    g_free( gswic->config.imageFiles[i] );
    if( stat( fn, &buf ) == -1 )
    {
      gswic->config.imageFiles[i] = NULL;
      continue;
    }
    gswic->config.imageFiles[i] = g_strdup( fn );
  }

  gswic->config.doBeep =
    gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON
                                  ( CappletGetGladeWidget( gswic,
                                                           "doBeep" ) ) );

  gswic->config.handleIndicators =
    gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON
                                  ( CappletGetGladeWidget( gswic,
                                                           "handleIndicators" ) ) );
}

static void CappletGroupPerWindowChanged( GtkWidget * w,
                                          GSwitchItCapplet * gswic )
{
  GtkWidget *handleIndicators;

  CappletSetStateToChanged( w, gswic );

  gswic->config.groupPerApp =
    gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( w ) );

  handleIndicators = CappletGetGladeWidget( gswic, "handleIndicators" );
  gtk_widget_set_sensitive( handleIndicators, gswic->config.groupPerApp );
}

static void CappletSecChanged( GtkWidget * w, GSwitchItCapplet * gswic )
{
  int idx, mask;

  CappletSetStateToChanged( w, gswic );

  idx = GPOINTER_TO_INT( gtk_object_get_data( GTK_OBJECT( w ), "idx" ) );
  mask = 1 << idx;
  if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( w ) ) )
    gswic->config.secondaryGroupsMask |= mask;
  else
    gswic->config.secondaryGroupsMask &= ~mask;

  if( ( gswic->config.secondaryGroupsMask + 1 ) == ( 1 << XklGetNumGroups(  ) ) )       // all secondaries?
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( w ), FALSE );
}

static void CappletDefaultGroupChanged( GtkWidget * menuItem,
                                        GSwitchItCapplet * gswic )
{
  CappletSetStateToChanged( menuItem, gswic );

  gswic->config.defaultGroup =
    GPOINTER_TO_INT( gtk_object_get_data( GTK_OBJECT( menuItem ), "group" ) );
}

static void CappletOk( GtkDialog * capplet, GSwitchItCapplet * gswic )
{
  CappletConfigUpdate( gswic );

  GSwitchItConfigSaveAppletParams( &gswic->config );

  gtk_main_quit(  );
}

static void CappletCancel( GtkDialog * capplet, GSwitchItCapplet * si )
{
  gtk_main_quit(  );
}

static void CappletHelp( GtkWidget * w, GSwitchItCapplet * si )
{
  GSwitchItHelp( GTK_WINDOW( si->capplet ), "gswitchitPropsCapplet" );
}

static void CappletSetup( GSwitchItCapplet * gswic )
{
  GladeXML *data;
  GtkWidget *widget, *capplet;

  GtkWidget *secondary;
  GtkWidget *frameFlag;
  GtkWidget *iconentryFlag;
  GtkWidget *doBeep;
  GtkWidget *groupPerApp;
  GtkWidget *handleIndicators;
  GtkWidget *menuItem;
  GtkWidget *hboxFlags;
  GtkWidget *defaultGroupsOMenu;
  GtkWidget *defaultGroupsMenu;
  int i;
  GtkTooltips *tooltips;
  const char *groupName;
  GroupDescriptionsBuffer groupNames;

  const char *secondaryTooltip =
    _( "Make the layout accessible from the applet popup menu ONLY.\n"
       "No way to switch to this layout using the keyboard." );

  glade_gnome_init(  );

  data = glade_xml_new( GLADE_DIR "/gswitchit-properties.glade", "gswitchit_capplet", NULL );   // default domain!
  XklDebug( 125, "data: %p\n", data );

  gswic->capplet = capplet =
    glade_xml_get_widget( data, "gswitchit_capplet" );

  gtk_object_set_data( GTK_OBJECT( capplet ), "gladeData", data );

  g_signal_connect_swapped( GTK_OBJECT( capplet ), "destroy",
                            G_CALLBACK( g_object_unref ), data );

  g_signal_connect( GTK_OBJECT( capplet ), "close",
                    G_CALLBACK( gtk_main_quit ), data );

  doBeep = glade_xml_get_widget( data, "doBeep" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( doBeep ),
                                gswic->config.doBeep );

  tooltips = gtk_tooltips_data_get( doBeep )->tooltips;

  g_signal_connect( GTK_OBJECT( doBeep ),
                    "toggled",
                    G_CALLBACK( CappletSetStateToChanged ), gswic );

  groupPerApp = glade_xml_get_widget( data, "groupPerApp" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( groupPerApp ),
                                gswic->config.groupPerApp );

  g_signal_connect( G_OBJECT( groupPerApp ),
                    "toggled",
                    G_CALLBACK( CappletGroupPerWindowChanged ), gswic );

  handleIndicators = glade_xml_get_widget( data, "handleIndicators" );

  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( handleIndicators ),
                                gswic->config.handleIndicators );
  gtk_widget_set_sensitive( handleIndicators, gswic->config.groupPerApp );

  g_signal_connect( GTK_OBJECT( handleIndicators ),
                    "toggled",
                    G_CALLBACK( CappletSetStateToChanged ), gswic );

  hboxFlags = glade_xml_get_widget( data, "hboxFlags" );

  GSwitchItConfigLoadGroupDescriptions( &gswic->config, groupNames );
  groupName = ( const char * ) groupNames;
  for( i = 0; i < XklGetNumGroups(  );
       i++, groupName += sizeof( groupNames[0] ) )
  {
    gchar sz[30];
    GtkWidget *vboxFlag;
    GtkWidget *vboxFlagImg;
    g_snprintf( sz, sizeof( sz ), "frameFlag.%d", i );

    frameFlag = gtk_frame_new( groupName );
    gtk_widget_set_name( frameFlag, sz );
    gtk_object_set_data( GTK_OBJECT( capplet ), sz, frameFlag );
    gtk_widget_show( frameFlag );

    vboxFlag = gtk_vbox_new( FALSE, 0 );
    g_snprintf( sz, sizeof( sz ), "vboxFlag.%d", i );
    gtk_widget_set_name( vboxFlag, sz );
    gtk_object_set_data( GTK_OBJECT( capplet ), sz, vboxFlag );

    gtk_box_pack_start( GTK_BOX( vboxFlag ), gtk_label_new( "" ), TRUE,
                        TRUE, 0 );
    gtk_box_pack_start( GTK_BOX( vboxFlag ), frameFlag, FALSE, FALSE, 0 );
    gtk_box_pack_start( GTK_BOX( vboxFlag ), gtk_label_new( "" ), TRUE,
                        TRUE, 0 );

    gtk_box_pack_start( GTK_BOX( hboxFlags ), vboxFlag, TRUE, TRUE, 0 );

    gtk_frame_set_shadow_type( GTK_FRAME( frameFlag ),
                               GTK_SHADOW_ETCHED_OUT );

    iconentryFlag = gnome_icon_entry_new( NULL, NULL );
    g_snprintf( sz, sizeof( sz ), "iconentryFlag.%d", i );
    gtk_widget_set_name( iconentryFlag, sz );
    gtk_object_set_data( GTK_OBJECT( capplet ), sz, iconentryFlag );
    gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( iconentryFlag ),
                                   gswic->config.imageFiles[i] );

    gtk_widget_show( iconentryFlag );

    gtk_tooltips_set_tip( tooltips, iconentryFlag, groupName, groupName );

    vboxFlagImg = gtk_vbox_new( FALSE, 0 );
    g_snprintf( sz, sizeof( sz ), "vboxFlagImg.%d", i );
    gtk_widget_set_name( vboxFlagImg, sz );
    gtk_object_set_data( GTK_OBJECT( capplet ), sz, vboxFlagImg );

    secondary = gtk_check_button_new_with_label( _( "Secondary" ) );
    g_snprintf( sz, sizeof( sz ), "secondary.%d", i );
    gtk_widget_set_name( secondary, sz );
    gtk_object_set_data( GTK_OBJECT( capplet ), sz, secondary );

    gtk_object_set_data( GTK_OBJECT( secondary ),
                         "idx", GINT_TO_POINTER( i ) );

    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( secondary ),
                                  ( gswic->config.
                                    secondaryGroupsMask & ( 1 << i ) ) != 0 );
    gtk_tooltips_set_tip( tooltips, secondary,
                          secondaryTooltip, secondaryTooltip );

    gtk_box_pack_start( GTK_BOX( vboxFlagImg ), secondary, FALSE, FALSE, 0 );
    gtk_box_pack_start( GTK_BOX( vboxFlagImg ), iconentryFlag, FALSE, FALSE,
                        0 );

    gtk_container_add( GTK_CONTAINER( frameFlag ), vboxFlagImg );
    g_signal_connect( GTK_OBJECT( iconentryFlag ),
                      "changed",
                      G_CALLBACK( CappletSetStateToChanged ), gswic );
    g_signal_connect( GTK_OBJECT( secondary ), "toggled",
                      G_CALLBACK( CappletSecChanged ), gswic );
  }

  defaultGroupsMenu = gtk_menu_new(  );

  gtk_object_set_data( GTK_OBJECT( capplet ), "defaultGroupsMenu",
                       defaultGroupsMenu );

  menuItem = gtk_menu_item_new_with_label( _( "not used" ) );
  gtk_object_set_data( GTK_OBJECT( menuItem ), "group",
                       GINT_TO_POINTER( -1 ) );
  g_signal_connect( GTK_OBJECT( menuItem ), "activate",
                    G_CALLBACK( CappletDefaultGroupChanged ), gswic );

  gtk_menu_shell_append( GTK_MENU_SHELL( defaultGroupsMenu ), menuItem );
  gtk_widget_show( menuItem );
  groupName = ( const char * ) groupNames;
  for( i = 0; i < XklGetNumGroups(  );
       i++, groupName += sizeof( groupNames[0] ) )
  {
    menuItem = gtk_menu_item_new_with_label( groupName );
    gtk_object_set_data( GTK_OBJECT( menuItem ), "group",
                         GINT_TO_POINTER( i ) );
    g_signal_connect( GTK_OBJECT( menuItem ), "activate",
                      G_CALLBACK( CappletDefaultGroupChanged ), gswic );
    gtk_menu_shell_append( GTK_MENU_SHELL( defaultGroupsMenu ), menuItem );
    gtk_widget_show( menuItem );
  }

  defaultGroupsOMenu = glade_xml_get_widget( data, "defaultGroupsOMenu" );
  gtk_option_menu_set_menu( GTK_OPTION_MENU( defaultGroupsOMenu ),
                            defaultGroupsMenu );
  // initial value - ( group no + 1 )
  gtk_option_menu_set_history( GTK_OPTION_MENU( defaultGroupsOMenu ),
                               ( gswic->config.defaultGroup <
                                 XklGetNumGroups(  ) )? gswic->config.
                               defaultGroup + 1 : 0 );
  glade_xml_signal_connect_data( data, "on_btnHelp_clicked",
                                 GTK_SIGNAL_FUNC( CappletHelp ), gswic );
  glade_xml_signal_connect_data( data, "on_btnOk_clicked",
                                 GTK_SIGNAL_FUNC( CappletOk ), gswic );
  glade_xml_signal_connect_data( data, "on_btnCancel_clicked",
                                 GTK_SIGNAL_FUNC( CappletCancel ), gswic );
  glade_xml_signal_connect_data( data, "on_gswitchit_capplet_unrealize",
                                 GTK_SIGNAL_FUNC( CappletCancel ), gswic );

  gtk_widget_set_sensitive( glade_xml_get_widget( data, "btnOk" ), FALSE );

  gtk_widget_show_all( capplet );
}

int main( int argc, char **argv )
{
  GSwitchItCapplet gswic;
  GError *gconf_error = NULL;
  bindtextdomain( PACKAGE, GNOMELOCALEDIR );
  bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
  textdomain( PACKAGE );
  memset( &gswic, 0, sizeof( gswic ) );

  gnome_program_init( "gswitchit-properties", VERSION, LIBGNOMEUI_MODULE,
                      argc, argv, GNOME_PARAM_NONE );

  if( !gconf_init( argc, argv, &gconf_error ) )
  {
    g_warning( _( "Failed to init GConf: %s\n" ), gconf_error->message );
    g_error_free( gconf_error );
    return 1;
  }
  gconf_error = NULL;

  //GSwitchItInstallGlibLogAppender(  );

  XklInit( GDK_DISPLAY(  ) );
  XklConfigInit(  );
  XklConfigLoadRegistry(  );
  GSwitchItConfigInit( &gswic.config );
  GSwitchItConfigLoadAppletParams( &gswic.config );
  CappletSetup( &gswic );
  gtk_main(  );
  GSwitchItConfigTerm( &gswic.config );
  XklConfigFreeRegistry(  );
  XklConfigTerm(  );
  XklTerm(  );

  return 0;
}
