/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*  PDF view widget
 *
 *  Copyright (C) 2002 Martin Kretzschmar
 *
 *  Author:
 *    Martin Kretzschmar <Martin.Kretzschmar@inf.tu-dresden.de>
 *
 * GPdf is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GPdf is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <aconf.h>
#include <math.h>
#include "gpdf-view.h"
#include "gpdf-util.h"
#include "gpdf-marshal.c"
#include "gpdf-g-switch.h"
#  include <gdk/gdkkeysyms.h>
#  include <libgnome/gnome-macros.h>
#  include <libgnomeprint/gnome-print.h>
#  include <libgnomeprintui/gnome-print-preview.h>
#  include "gtkgesture.h"
#include "gpdf-g-switch.h"
#include "GPOutputDev.h"
#include "PDFDoc.h"

BEGIN_EXTERN_C

struct _GPdfViewPrivate {
	GnomePrintContext *print_context;
	GPOutputDev *output_dev;
	PDFDoc *pdf_doc;

	GnomeCanvasItem *page_background;
	GnomeCanvasItem *page_shadow;

	GtkGestureHandler *gesture_handler;

	gint requested_page;
	gint current_page;
	gdouble zoom;
	guint dragging : 1;
	gint drag_anchor_x, drag_anchor_y;
	gint drag_offset_x, drag_offset_y;
};

#define PARENT_TYPE GNOME_TYPE_CANVAS
GPDF_CLASS_BOILERPLATE (GPdfView, gpdf_view, GnomeCanvas, PARENT_TYPE);

#define GPDF_IS_NON_NULL_VIEW(obj) (((obj) != NULL) && (GPDF_IS_VIEW ((obj))))
#define GPDF_IS_VIEW_WITH_DOC(obj) \
(GPDF_IS_NON_NULL_VIEW (obj) && (obj)->priv->pdf_doc)

enum {
	ZOOM_CHANGED_SIGNAL,
	PAGE_CHANGED_SIGNAL,
	LAST_SIGNAL
};

static guint gpdf_view_signals [LAST_SIGNAL];


#define PAGE_PAD 3

static void
gpdf_view_page_background_show (GPdfView *gpdf_view)
{
	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	gnome_canvas_item_show (gpdf_view->priv->page_background);
	gnome_canvas_item_show (gpdf_view->priv->page_shadow);
}

static void
gpdf_view_page_background_hide (GPdfView *gpdf_view)
{
	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	gnome_canvas_item_hide (gpdf_view->priv->page_background);
	gnome_canvas_item_hide (gpdf_view->priv->page_shadow);
}

static void
gpdf_view_page_background_resize (GPdfView *gpdf_view, 
				  gdouble width, gdouble height)
{
	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	gnome_canvas_item_set (gpdf_view->priv->page_background,
			       "x2", width, "y2", height, NULL);
	gnome_canvas_item_set (gpdf_view->priv->page_shadow,
			       "x2", width + PAGE_PAD - 1.0,
			       "y2", height + PAGE_PAD - 1.0,
			       NULL);
}

static void
setup_upside_down_transform (gdouble *transform, gdouble height)
{
	transform [0] = 1.0;
	transform [1] = 0.0;
	transform [2] = 0.0;
	transform [3] = -1.0;
	transform [4] = 0.0;
	transform [5] = height;
}	

static gboolean
gpdf_view_render_page (GPdfView *gpdf_view)
{
	GPdfViewPrivate *priv;
	GnomePrintConfig *config;
	PDFDoc *pdf_doc;
	gint page;
	gint page_rotate;
	ArtDRect region;
	gdouble page_transform [6];

	g_return_val_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view), FALSE);

	priv = gpdf_view->priv;

	if (priv->requested_page == priv->current_page)
		return FALSE;

	page = priv->requested_page;

	if (priv->print_context) {
		g_object_unref (priv->print_context);
		priv->print_context = NULL;
	}

	pdf_doc = priv->pdf_doc;

	config = gnome_print_config_default ();

	gnome_print_config_set_length (
		config,
		(const guchar *)GNOME_PRINT_KEY_LAYOUT_WIDTH,
		pdf_doc->getPageWidth (page),
		GNOME_PRINT_PS_UNIT);

	gnome_print_config_set_length (
		config,
		(const guchar *)GNOME_PRINT_KEY_LAYOUT_HEIGHT,
		pdf_doc->getPageHeight (page),
		GNOME_PRINT_PS_UNIT);

	setup_upside_down_transform (page_transform,
				     pdf_doc->getPageHeight (page));

	/* Set scrolling region */
	region.x0 = region.y0 = 0.0 - PAGE_PAD;
	region.x1 = pdf_doc->getPageWidth (page) + PAGE_PAD;
	region.y1 = pdf_doc->getPageHeight (page) + PAGE_PAD;

	gpdf_view_page_background_resize (gpdf_view, 
					  pdf_doc->getPageWidth (page),
					  pdf_doc->getPageHeight (page));

	priv->print_context = gnome_print_preview_new_full (
		config, GNOME_CANVAS (gpdf_view), page_transform, &region);

	gnome_print_config_unref (config);

	priv->output_dev->setPrintContext (priv->print_context);
	priv->pdf_doc->displayPage (priv->output_dev,
				    page,
				    72,
				    0,
				    gFalse);

	priv->current_page = page;

	return FALSE;
}

static void
gpdf_view_emit_page_changed (GPdfView *gpdf_view, gint page)
{
	g_signal_emit (G_OBJECT (gpdf_view),
		       gpdf_view_signals [PAGE_CHANGED_SIGNAL],
		       0, page);
}

void
gpdf_view_goto_page (GPdfView *gpdf_view, gint page)
{
	GPdfViewPrivate *priv;
	gint last_page;

	priv = gpdf_view->priv;
	last_page = priv->pdf_doc->getNumPages ();

	priv->requested_page = CLAMP (page, 1, last_page);
	gpdf_view_emit_page_changed (gpdf_view, priv->requested_page);
	g_idle_add ((GSourceFunc)gpdf_view_render_page, gpdf_view);
}

void
gpdf_view_page_prev (GPdfView *gpdf_view)
{
	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	gpdf_view_goto_page (gpdf_view, gpdf_view->priv->requested_page - 1);
}

void
gpdf_view_page_next (GPdfView *gpdf_view)
{
	g_return_if_fail(GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	gpdf_view_goto_page (gpdf_view, gpdf_view->priv->requested_page + 1);
}

void
gpdf_view_page_first (GPdfView *gpdf_view)
{
	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	gpdf_view_goto_page (gpdf_view, 1);
}

void
gpdf_view_page_last (GPdfView *gpdf_view)
{
	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	gpdf_view_goto_page (gpdf_view,
			     gpdf_view->priv->pdf_doc->getNumPages ());
}

gboolean
gpdf_view_is_first_page (GPdfView *gpdf_view)
{
	g_return_val_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view), FALSE);
	
	return gpdf_view->priv->requested_page == 1;
}

gboolean
gpdf_view_is_last_page (GPdfView *gpdf_view)
{
	gint last_page;

	g_return_val_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view), FALSE);

	last_page = gpdf_view->priv->pdf_doc->getNumPages ();

	return gpdf_view->priv->requested_page == last_page;
}


#define MAGSTEP  1.2
#define IMAGSTEP 0.8333333333333

#define MIN_ZOOM 0.05409
#define MAX_ZOOM 18.4884

#define ZOOM_IN_FACTOR  MAGSTEP
#define ZOOM_OUT_FACTOR IMAGSTEP

static void
gpdf_view_emit_zoom_changed (GPdfView *gpdf_view, gdouble zoom)
{
	g_signal_emit (G_OBJECT (gpdf_view),
		       gpdf_view_signals [ZOOM_CHANGED_SIGNAL],
		       0, zoom);
}

void
gpdf_view_zoom (GPdfView *gpdf_view, gdouble factor, gboolean relative)
{
	GPdfViewPrivate *priv;
	gdouble zoom;

	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	priv = gpdf_view->priv;

	if (relative) {
		zoom = priv->zoom * factor;
	} else {
		zoom = factor;
	}

	priv->zoom = CLAMP (zoom, MIN_ZOOM, MAX_ZOOM);
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (gpdf_view),
					  priv->zoom);
	gpdf_view_emit_zoom_changed (gpdf_view, priv->zoom);
}

void
gpdf_view_zoom_default (GPdfView *gpdf_view)
{
	gpdf_view_zoom (gpdf_view, 1.0, FALSE);
}

void
gpdf_view_zoom_in (GPdfView *gpdf_view)
{
	gpdf_view_zoom (gpdf_view, ZOOM_IN_FACTOR, TRUE);
}

void
gpdf_view_zoom_out (GPdfView *gpdf_view)
{
	gpdf_view_zoom (gpdf_view, ZOOM_OUT_FACTOR, TRUE);
}

#define DOUBLE_EQUAL(a,b) (fabs (a - b) < 1e-6)
void
gpdf_view_zoom_fit (GPdfView *gpdf_view, gint width, gint height)
{
	GPdfViewPrivate *priv;
	gdouble x1, y1, x2, y2;
	gdouble page_w, page_h, zoom_w, zoom_h;

	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	priv = gpdf_view->priv;

	gnome_canvas_get_scroll_region (GNOME_CANVAS (gpdf_view),
					&x1, &y1, &x2, &y2);

	page_w = x2 - x1;
	page_h = y2 - y1;

	if (DOUBLE_EQUAL (page_w, 0.0) || DOUBLE_EQUAL (page_h, 0.0))
		return;

	zoom_w = width / page_w;
	zoom_h = height / page_h;

	gpdf_view_zoom (gpdf_view, (zoom_w < zoom_h) ? zoom_w : zoom_h, FALSE);
}

void
gpdf_view_zoom_fit_width (GPdfView *gpdf_view, gint width)
{
	GPdfViewPrivate *priv;
	gdouble x1, y1, x2, y2;
	gdouble page_w;

	g_return_if_fail (GPDF_IS_VIEW_WITH_DOC (gpdf_view));

	priv = gpdf_view->priv;

	gnome_canvas_get_scroll_region (GNOME_CANVAS (gpdf_view),
					&x1, &y1, &x2, &y2);
	page_w = x2 - x1 + 0.5;
	if (DOUBLE_EQUAL (page_w, 0.0))
		return;

	gpdf_view_zoom (gpdf_view, width / page_w, FALSE);
}


static gboolean
gpdf_view_key_press_event (GtkWidget *widget, GdkEventKey *event)
{
	GPdfView *gpdf_view;
	gboolean big_step;
	gboolean do_scroll;
	gint height, width;
	gint x, y;

	g_return_val_if_fail (GPDF_IS_NON_NULL_VIEW (widget), FALSE);

	gpdf_view = GPDF_VIEW (widget);
	gnome_canvas_get_scroll_offsets (GNOME_CANVAS (gpdf_view), &x, &y);
	height = widget->allocation.height;
	width = widget->allocation.width;
	big_step = event->state & GDK_SHIFT_MASK;

	switch (event->keyval) {
	case GDK_plus:
	case GDK_KP_Add:
	case GDK_equal:
		gpdf_view_zoom_in (gpdf_view);
		break;
	case GDK_minus:
	case GDK_KP_Subtract:
	case GDK_underscore:
		gpdf_view_zoom_out (gpdf_view);
		break;
	case GDK_KP_Right:
	case GDK_Right:
		do_scroll = TRUE;
		if (big_step)
			x += width;
		else
			x += 10;
		break;
	case GDK_KP_Left:
	case GDK_Left:
		do_scroll = TRUE;
		if (big_step)
			x -= width;
		else
			x -= 10;
		break;
	case GDK_KP_Up:
	case GDK_Up:
		if (big_step) {
			goto page_up;
		} else {
			do_scroll = TRUE;
			y -= 10;
		}
		break;
	case GDK_KP_Down:
	case GDK_Down:
		if (big_step) {
			goto page_down;
		} else {
			do_scroll = TRUE;
			y += 10;
		}
		break;
	case GDK_KP_Page_Up:
	case GDK_Page_Up:
	case GDK_Delete:
	case GDK_KP_Delete:
	case GDK_BackSpace:
	page_up:
		do_scroll = TRUE;
		if (y <= 0) {
			if (gpdf_view->priv->current_page > 1) {
				gpdf_view_page_prev (gpdf_view);
				/* 
				 * FIXME height might be outdated for
				 * new page
				 */
				y = GTK_LAYOUT (widget)->height - height;
			}
		} else {
			y -= height;
		}
		break;
	case GDK_KP_Page_Down:
	case GDK_Page_Down:
	case GDK_space:
	page_down:
		if (y >= GTK_LAYOUT (widget)->height - height) {
			if (gpdf_view->priv->current_page < 
			    gpdf_view->priv->pdf_doc->getNumPages ()) {
				gpdf_view_page_next (gpdf_view);
				y = 0;
			}
		} else {
			y += height;
		}
		break;
	default:
		return FALSE;
	}

	if (do_scroll)
		gnome_canvas_scroll_to (GNOME_CANVAS (widget), x, y);

	return TRUE;
}

static gboolean
gpdf_view_button_press_event (GtkWidget *widget, GdkEventButton *event)
{
	GPdfViewPrivate *priv;

	g_return_val_if_fail (GPDF_IS_NON_NULL_VIEW (widget), FALSE);

	priv = GPDF_VIEW (widget)->priv;

	if (!GTK_WIDGET_HAS_FOCUS (widget))
		gtk_widget_grab_focus (widget);

	if (priv->dragging)
		return FALSE;

	switch (event->button) {
	case 1:
		GdkCursor *cursor;

		priv->dragging = TRUE;

		gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget),
						 &priv->drag_offset_x,
						 &priv->drag_offset_y);

		priv->drag_anchor_x = (gint)(event->x - priv->drag_offset_x);
		priv->drag_anchor_y = (gint)(event->y - priv->drag_offset_y);

		cursor = gdk_cursor_new (GDK_FLEUR);
		gdk_pointer_grab (widget->window, FALSE,
				  (enum GdkEventMask)
				  (GDK_POINTER_MOTION_MASK | 
				   GDK_POINTER_MOTION_HINT_MASK |
				   GDK_BUTTON_RELEASE_MASK),
				  NULL, cursor, event->time);
		gdk_cursor_unref (cursor);
		return TRUE;
	default:
		break;
	}

	return FALSE;
}

static gboolean
gpdf_view_button_release_event (GtkWidget *widget, GdkEventButton *event)
{
	GPdfViewPrivate *priv;

	g_return_val_if_fail (GPDF_IS_NON_NULL_VIEW (widget), FALSE);

	priv = GPDF_VIEW (widget)->priv;
	
	if (!priv->dragging || event->button != 1)
		return FALSE;

	priv->dragging = FALSE;
	gdk_pointer_ungrab (event->time);
	
	return TRUE;
}

static gboolean
gpdf_view_motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
	GPdfViewPrivate *priv;
	gint x, y, dx, dy;
	GdkModifierType mod;

	g_return_val_if_fail (GPDF_IS_NON_NULL_VIEW (widget), FALSE);
	
	priv = GPDF_VIEW (widget)->priv;

	if (!priv->dragging)
		return FALSE;

	if (event->is_hint) {
		gdk_window_get_pointer (widget->window, &x, &y, &mod);
	} else {
		x = (gint)event->x;
		y = (gint)event->y;
	}

	dx = priv->drag_anchor_x - x;
	dy = priv->drag_anchor_y - y;

	gnome_canvas_scroll_to (GNOME_CANVAS (widget), 
				priv->drag_offset_x + dx,
				priv->drag_offset_y + dy);

	priv->drag_anchor_x = (gint)event->x;
	priv->drag_anchor_y = (gint)event->y;
	gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget),
					 &priv->drag_offset_x,
					 &priv->drag_offset_y);

	return TRUE;
}

static void
gesture_page_next_event_cb (GtkGestureHandler *gh,
			    GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_page_next (gpdf_view);
}

static void
gesture_page_prev_event_cb (GtkGestureHandler *gh,
			    GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_page_prev (gpdf_view);
}

static void
gesture_page_first_event_cb (GtkGestureHandler *gh,
			     GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_page_first (gpdf_view);
}

static void
gesture_page_last_event_cb (GtkGestureHandler *gh,
			    GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_page_last (gpdf_view);
}

static void
gesture_zoom_in_event_cb (GtkGestureHandler *gh,
			  GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_zoom_in (gpdf_view);
}

static void
gesture_zoom_out_event_cb (GtkGestureHandler *gh,
			   GPdfView *gpdf_view, gpointer p)
{
	gpdf_view_zoom_out (gpdf_view);
}


void
gpdf_view_set_pdf_doc (GPdfView *gpdf_view, PDFDoc *pdf_doc)
{
	GPdfViewPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	priv = gpdf_view->priv;

	/*
	 * All PDFDocs in GPdf code are owned by a GPdfPersistStream,
	 * so don't delete our old pdf_doc
	 */
	priv->pdf_doc = pdf_doc;

	priv->current_page = 0;
	priv->output_dev->startDoc (pdf_doc->getXRef());
	gpdf_view_page_background_show (gpdf_view);
	gpdf_view_goto_page (gpdf_view, 1);
}


static void
gpdf_view_dispose (GObject *object)
{
	GPdfViewPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (object));

	priv = GPDF_VIEW (object)->priv;
	
	if (priv->output_dev) {
		delete priv->output_dev;
		priv->output_dev = NULL;
	}
	
	if (priv->print_context) {
		g_object_unref (priv->print_context);
		priv->print_context = NULL;
	}

	if (priv->gesture_handler) {
		gtk_gesture_handler_destroy (priv->gesture_handler);
		priv->gesture_handler = NULL;
	}

	GNOME_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
gpdf_view_finalize (GObject *object)
{
	GPdfView *gpdf_view;

	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (object));
	
	gpdf_view = GPDF_VIEW (object);

	if (gpdf_view->priv) {
		g_free (gpdf_view->priv);
		gpdf_view->priv = NULL;
	}

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_view_class_init (GPdfViewClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = gpdf_view_dispose;
	object_class->finalize = gpdf_view_finalize;

	widget_class->button_press_event = gpdf_view_button_press_event;
	widget_class->button_release_event = gpdf_view_button_release_event;
	widget_class->motion_notify_event = gpdf_view_motion_notify_event;
	widget_class->key_press_event = gpdf_view_key_press_event;

	gpdf_view_signals [ZOOM_CHANGED_SIGNAL] = g_signal_new (
		"zoom_changed",
		G_TYPE_FROM_CLASS (object_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (GPdfViewClass, zoom_changed),
		NULL, NULL,
		gpdf_marshal_VOID__DOUBLE,
		G_TYPE_NONE, 1, G_TYPE_DOUBLE);

	gpdf_view_signals [PAGE_CHANGED_SIGNAL] = g_signal_new (
		"page_changed",
		G_TYPE_FROM_CLASS (object_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (GPdfViewClass, page_changed),
		NULL, NULL,
		gpdf_marshal_VOID__INT,
		G_TYPE_NONE, 1, G_TYPE_INT);
}

static void
gpdf_view_instance_init (GPdfView *gpdf_view)
{
	gpdf_view->priv = g_new0 (GPdfViewPrivate, 1);
}

static void
gpdf_view_setup_gesture_handler (GPdfView *gpdf_view)
{
	GtkGestureHandler *gh;

	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	gh = gtk_gesture_handler_new (GTK_WIDGET (gpdf_view));
	gpdf_view->priv->gesture_handler = gh;

	gtk_gesture_add_callback (
		gh, "012",
		GTK_GESTURE_FUNC (gesture_page_next_event_cb),
		gpdf_view, NULL);
	gtk_gesture_add_callback (
		gh, "210",
		GTK_GESTURE_FUNC (gesture_page_prev_event_cb),
		gpdf_view, NULL);
	gtk_gesture_add_callback (
		gh, "630",
		GTK_GESTURE_FUNC (gesture_page_first_event_cb),
		gpdf_view, NULL);
	gtk_gesture_add_callback (
		gh, "036",
		GTK_GESTURE_FUNC (gesture_page_last_event_cb),
		gpdf_view, NULL);

	gtk_gesture_add_callback (
		gh, "048",
		GTK_GESTURE_FUNC (gesture_zoom_in_event_cb),
		gpdf_view, NULL);
	gtk_gesture_add_callback (
		gh, "840",
		GTK_GESTURE_FUNC (gesture_zoom_out_event_cb),
		gpdf_view, NULL);
}

static void
gpdf_view_setup_page_background (GPdfView *gpdf_view)
{
	GPdfViewPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view));

	priv = gpdf_view->priv;

	priv->page_background = gnome_canvas_item_new (
		gnome_canvas_root (GNOME_CANVAS (gpdf_view)),
		GNOME_TYPE_CANVAS_RECT,
		"x1", 0.0, "y1", 0.0, "x2", 72.0, "y2", 72.0,
		"fill_color", "white",
		"outline_color", "black",
		"width_pixels", 1, NULL);

	gnome_canvas_item_lower_to_bottom (priv->page_background);

	priv->page_shadow = gnome_canvas_item_new (
		gnome_canvas_root (GNOME_CANVAS (gpdf_view)),
		GNOME_TYPE_CANVAS_RECT,
		"x1", PAGE_PAD - 1.0, "y1", (gdouble)PAGE_PAD - 1.0,
		"x2", 72.0 + PAGE_PAD - 1.0, "y2", 72.0 + PAGE_PAD - 1.0,
		"fill_color", "black", NULL);

	gnome_canvas_item_lower_to_bottom (priv->page_shadow);

	gpdf_view_page_background_hide (gpdf_view);
}

GPdfView *
gpdf_view_construct (GPdfView *gpdf_view)
{
	GPdfViewPrivate *priv;
	GnomePrintConfig *config;
	GnomePrintContext *pc;

	g_return_val_if_fail (GPDF_IS_NON_NULL_VIEW (gpdf_view), NULL);

	gpdf_view_setup_page_background (gpdf_view);
	gpdf_view_setup_gesture_handler (gpdf_view);

	priv = gpdf_view->priv;
	priv->output_dev = new GPOutputDev ();
	priv->zoom = 1.0;

	return gpdf_view;
}

GtkWidget *
gpdf_view_new (void)
{
	GPdfView *gpdf_view;

	gpdf_view = GPDF_VIEW (g_object_new (GPDF_TYPE_VIEW,
					     "aa", TRUE,
					     NULL));

	return GTK_WIDGET (gpdf_view_construct (gpdf_view));
}

END_EXTERN_C
