/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *            (C) 2001, 2002 Jorn Baayen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)
#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);

#include "toolbar.h"
#include "galeon-spinner.h"
#include "galeon-window.h"
#include "gul-bonobo-extensions.h"
#include "gul-string.h"
#include "gul-gui.h"
#include "location-entry.h"
#include "galeon-shell.h"
#include "galeon-embed-favicon.h"
#include "galeon-dnd.h"
#include "gul-toolbar-bonobo-view.h"
#include "prefs-strings.h"
#include "eel-gconf-extensions.h"
#include "gul-toolbar-item-factory.h"
#include "gul-tbi-zoom.h"
#include "galeon-navigation-button.h"
#include "galeon-reload-button.h"
#include "galeon-go-button.h"
#include "galeon-home-button.h"
#include "galeon-new-button.h"
#include "galeon-stop-button.h"

#include <string.h>
#include <bonobo/bonobo-i18n.h>
#include <bonobo/bonobo-window.h>
#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-ui-toolbar-button-item.h>
#include <bonobo/bonobo-property-bag.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkmenu.h>

#define DEFAULT_TOOLBAR_SETUP \
	"back_menu=navigation_button(direction=back,arrow=TRUE);" \
	"up_menu=navigation_button(direction=up,arrow=TRUE);" \
	"forward_menu=navigation_button(direction=forward,arrow=TRUE);" \
	"stop=stop_button;" \
	"reload=reload_button;" \
	"home=home_button;" \
	"favicon=favicon;" \
	"location=location;" \
	"zoom=zoom;" \
	"spinner=spinner;"

#define ZOOM_DELAY 50

#define GALEON_HOMEPAGE "http://galeon.sourceforge.net/"

static void toolbar_class_init (ToolbarClass *klass);
static void toolbar_init (Toolbar *t);
static void toolbar_finalize (GObject *object);
static void toolbar_set_window (Toolbar *t, GaleonWindow *window);
static void toolbar_get_widgets (Toolbar *t);
static void toolbar_changed_cb (GulToolbar *gt, Toolbar *t);
static void
toolbar_set_property (GObject *object,
                      guint prop_id,
                      const GValue *value,
                      GParamSpec *pspec);
static void
toolbar_get_property (GObject *object,
                      guint prop_id,
                      GValue *value,
                      GParamSpec *pspec);


enum
{
	PROP_0,
	PROP_GALEON_WINDOW
};

enum 
{
        TOOLBAR_ITEM_STYLE_PROP,
        TOOLBAR_ITEM_ORIENTATION_PROP,
	TOOLBAR_ITEM_PRIORITY_PROP
};

static GObjectClass *parent_class = NULL;

struct ToolbarPrivate
{
	GaleonWindow *window;
	BonoboUIComponent *ui_component;
	GulTbBonoboView *bview;

	GtkWidget *spinner;
	gboolean visibility;

	GSList *navigation_buttons;
	GaleonReloadButton *reload_button;
	GaleonGoButton *go_button;
	GaleonHomeButton *home_button;
	GaleonStopButton *stop_button;
	GaleonNewButton *new_button;
	GtkWidget *location_entry;
	GtkTooltips *tooltips;
	GtkWidget *favicon;
	GtkWidget *favicon_ebox;
	GtkWidget *zoom_spinbutton;
	guint zoom_timeout_id;
	gboolean zoom_lock;
};

GType 
toolbar_get_type (void)
{
        static GType toolbar_type = 0;

        if (toolbar_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (ToolbarClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) toolbar_class_init,
                        NULL,
                        NULL, /* class_data */
                        sizeof (Toolbar),
                        0, /* n_preallocs */
                        (GInstanceInitFunc) toolbar_init
                };

                toolbar_type = g_type_register_static (GUL_TYPE_TOOLBAR,
						       "Toolbar",
						       &our_info, 0);
        }

        return toolbar_type;

}

static void
toolbar_class_init (ToolbarClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        parent_class = g_type_class_peek_parent (klass);

        object_class->finalize = toolbar_finalize;

	object_class->set_property = toolbar_set_property;
	object_class->get_property = toolbar_get_property;

	g_object_class_install_property (object_class,
                                         PROP_GALEON_WINDOW,
                                         g_param_spec_object ("GaleonWindow",
                                                              "GaleonWindow",
                                                              "Parent window",
                                                              GALEON_WINDOW_TYPE,
                                                              G_PARAM_READWRITE));

	gul_toolbar_item_register_type ("navigation_button", (GulTbItemConstructor) galeon_navigation_button_new);
	gul_toolbar_item_register_type ("reload_button", (GulTbItemConstructor) galeon_reload_button_new);
	gul_toolbar_item_register_type ("go_button", (GulTbItemConstructor) galeon_go_button_new);
	gul_toolbar_item_register_type ("home_button", (GulTbItemConstructor) galeon_home_button_new);
	gul_toolbar_item_register_type ("stop_button", (GulTbItemConstructor) galeon_stop_button_new);
	gul_toolbar_item_register_type ("new_button", (GulTbItemConstructor) galeon_new_button_new);
}

static void
toolbar_set_property (GObject *object,
                      guint prop_id,
                      const GValue *value,
                      GParamSpec *pspec)
{
        Toolbar *t = TOOLBAR (object);

        switch (prop_id)
        {
	case PROP_GALEON_WINDOW: 
		toolbar_set_window (t, g_value_get_object (value));
		break;
	}
}

static void
toolbar_get_property (GObject *object,
                      guint prop_id,
                      GValue *value,
                      GParamSpec *pspec)
{
        Toolbar *t = TOOLBAR (object);

        switch (prop_id)
        {
	case PROP_GALEON_WINDOW:
		g_value_set_object (value, t->priv->window);
		break;
        }
}

static void
toolbar_location_url_activate_cb (GaleonLocationEntry *entry, const gchar *url, GaleonWindow *window)
{
	galeon_window_load_url (window, url);
}

static void
each_url_get_data_binder (GaleonDragEachSelectedItemDataGet iteratee, 
        		  gpointer iterator_context, gpointer data)
{
	const char *location;
	GaleonTab *tab;
	GaleonWindow *window = GALEON_WINDOW(iterator_context);

	tab = galeon_window_get_active_tab (window);
	location = galeon_tab_get_location (tab);
	
	iteratee (location, -1, -1, -1, -1, data);
}

static void
favicon_drag_data_get_cb (GtkWidget *widget,
                          GdkDragContext *context,
                          GtkSelectionData *selection_data,
                          guint info,
                          guint32 time,
                          GaleonWindow *window)
{
        g_assert (widget != NULL);
        g_return_if_fail (context != NULL);

        galeon_dnd_drag_data_get (widget, context, selection_data,
                info, time, window, each_url_get_data_binder);
}

static void
toolbar_setup_favicon_ebox (Toolbar *t, GtkWidget *w)
{
	ToolbarPrivate *p = t->priv;
	
	g_return_if_fail (w == p->favicon_ebox);

	p->favicon = g_object_ref (galeon_embed_favicon_new
				   (galeon_window_get_active_embed (p->window)));
	gtk_container_add (GTK_CONTAINER (p->favicon_ebox), p->favicon);
	gtk_container_set_border_width (GTK_CONTAINER (p->favicon_ebox), 2);

	galeon_dnd_url_drag_source_set (p->favicon_ebox);

	g_signal_connect (G_OBJECT (p->favicon_ebox),
			  "drag_data_get",
			  G_CALLBACK (favicon_drag_data_get_cb),
			  p->window);
	gtk_widget_show_all (p->favicon_ebox);
}

static gboolean
toolbar_zoom_timeout_cb (gpointer data)
{
	Toolbar *t = data;
	gint zoom = toolbar_get_zoom (t);
	
	g_return_val_if_fail (IS_GALEON_WINDOW (t->priv->window), FALSE);

	galeon_window_set_zoom (t->priv->window, zoom);

	return FALSE;
}

static void
toolbar_zoom_spinbutton_value_changed_cb (GtkSpinButton *sb, Toolbar *t)
{
	ToolbarPrivate *p = t->priv;
	
	if (p->zoom_timeout_id != 0)
	{
		g_source_remove (p->zoom_timeout_id);
	}
	if (!p->zoom_lock)
	{
		p->zoom_timeout_id = g_timeout_add (ZOOM_DELAY, toolbar_zoom_timeout_cb, t);
	}
}

static void
toolbar_setup_zoom_spinbutton (Toolbar *t, GtkWidget *w)
{
	g_signal_connect (w, "value_changed", 
			  G_CALLBACK (toolbar_zoom_spinbutton_value_changed_cb), t);
}

static void
toolbar_setup_location_entry (Toolbar *t, GtkWidget *w)
{
	GaleonAutocompletion *ac = galeon_shell_get_autocompletion (galeon_shell);
	GaleonLocationEntry *e;

	g_return_if_fail (w == t->priv->location_entry);
	g_return_if_fail (GALEON_IS_LOCATION_ENTRY (w));

	e = GALEON_LOCATION_ENTRY (w);
	galeon_location_entry_set_autocompletion (e, ac);

	g_signal_connect (e, "url-activated",
			  G_CALLBACK (toolbar_location_url_activate_cb),
			  t->priv->window);
}
	
static gboolean
toolbar_spinner_button_release_event_cb (GtkWidget *w, GdkEventButton *ev, Toolbar *t)
{
	if (ev->button >= 0 && ev->button <= 3)
	{
		ToolbarPrivate *p = t->priv;
		galeon_window_load_url (p->window, GALEON_HOMEPAGE);
	}
	return FALSE;
}

static void
toolbar_setup_spinner (Toolbar *t, GtkWidget *w)
{
	ToolbarPrivate *p = t->priv;
        GtkWidget *spinner;
	
	g_return_if_fail (w == p->spinner);

        /* build the spinner and insert it into the box */
        spinner = galeon_spinner_new ();
	galeon_spinner_set_small_mode (GALEON_SPINNER (spinner), TRUE);
	gtk_container_add (GTK_CONTAINER (p->spinner), spinner);
	gtk_widget_show (spinner);

	/* don't care about the box anymore */
	g_object_unref (p->spinner);
	p->spinner = g_object_ref (spinner);

	g_signal_connect (p->spinner, "button-release-event",
			  G_CALLBACK (toolbar_spinner_button_release_event_cb), t);
}

static void
toolbar_setup_style (Toolbar *t)
{
	static const char *styles[] = { "both_horiz", "both", "text", "icon" };
	const char *style;
	GConfValue *value;

	g_return_if_fail (t->priv->ui_component != NULL);

	style = "";
	value = eel_gconf_get_value (CONF_TOOLBAR_STYLE);
	if (value != NULL)
	{
		guint i = gconf_value_get_int (value);
		if (i < G_N_ELEMENTS(styles))
		{
			style = styles[i];
		}
		gconf_value_free (value);
	}

	gul_bonobo_set_look (t->priv->ui_component, "/Toolbar", style);
}

static void
toolbar_style_changed_cb (GConfClient *client,
		          guint cnxn_id,
			  GConfEntry *entry,
			  gpointer t)
{
	toolbar_setup_style (TOOLBAR (t));
}

static void
toolbar_set_window (Toolbar *t, GaleonWindow *window)
{
	g_return_if_fail (t->priv->window == NULL);
	
	t->priv->window = window;
	t->priv->ui_component = g_object_ref (t->priv->window->ui_component);

	gul_tb_bonobo_view_set_path (t->priv->bview, t->priv->ui_component, "/Toolbar");

	eel_gconf_notification_add (CONF_TOOLBAR_STYLE,
			            toolbar_style_changed_cb, t);

	toolbar_setup_style (t);
	toolbar_get_widgets (t);
}

static void
toolbar_get_widgets (Toolbar *t)
{
	ToolbarPrivate *p;
	GulToolbar *gt;
	GulTbItem *it;
	GSList *li;
	const gchar *nav_buttons_ids[] = {"back", "back_menu", "up", "up_menu", "forward", "forward_menu" };
	guint i;

	DEBUG_MSG (("in toolbar_get_widgets\n"));

	g_return_if_fail (IS_TOOLBAR (t));
	p = t->priv;
	g_return_if_fail (IS_GALEON_WINDOW (p->window));
	g_return_if_fail (BONOBO_IS_UI_COMPONENT (p->ui_component));

	/* release all the widgets */

	for (li = p->navigation_buttons; li; li = li->next)
	{
		g_object_unref (li->data);
	}
	g_slist_free (p->navigation_buttons);
	p->navigation_buttons = NULL;

	if (p->reload_button)
	{
		g_object_unref (p->reload_button);
	}
	p->reload_button = NULL;

	if (p->go_button)
	{
		g_object_unref (p->go_button);
	}
	p->go_button = NULL;

	if (p->home_button)
	{
		g_object_unref (p->home_button);
	}
	p->home_button = NULL;

	if (p->stop_button)
	{
		g_object_unref (p->stop_button);
	}
	p->stop_button = NULL;

	if (p->new_button)
	{
		g_object_unref (p->new_button);
	}
	p->new_button = NULL;

	if (p->favicon_ebox)
	{
		g_object_unref (p->favicon_ebox);
		p->favicon_ebox = NULL;
	}
	
	if (p->favicon)
	{
		g_object_unref (p->favicon);
		p->favicon = NULL;
	}
	
	if (p->location_entry)
	{
		g_object_unref (p->location_entry);
		p->location_entry = NULL;
	}
	
	if (p->spinner)
	{
		g_object_unref (p->spinner);
		p->spinner = NULL;
	}

	if (p->zoom_spinbutton)
	{
		g_object_unref (p->zoom_spinbutton);
		p->zoom_spinbutton = NULL;
	}
	
	gt = GUL_TOOLBAR (t);

	for (i = 0; i < G_N_ELEMENTS (nav_buttons_ids); ++i)
	{
		it = gul_toolbar_get_item_by_id (gt, nav_buttons_ids[i]);
		if (it)
		{
			if (GALEON_IS_NAVIGATION_BUTTON (it))
			{	
				DEBUG_MSG (("    got a navigation button\n"));
				p->navigation_buttons = g_slist_prepend (p->navigation_buttons, g_object_ref (it));
				if (p->window)
				{
					galeon_tbi_set_window (GALEON_TBI (it), p->window);
				}
			}
			else
			{
				g_warning ("An unexpected button has been found in your toolbar. "
					   "Maybe your setup is too old.");
			}
		}
	}
	
	it = gul_toolbar_get_item_by_id (gt, "reload");
	if (it)
	{
		if (GALEON_IS_RELOAD_BUTTON (it))
		{	
			DEBUG_MSG (("    got a reload button\n"));
			p->reload_button = g_object_ref (it);
			if (p->window)
			{
				galeon_tbi_set_window (GALEON_TBI (it), p->window);
			}
		}
		else
		{
			g_warning ("An unexpected button has been found in your toolbar. "
				   "Maybe your setup is too old.");
		}
	}

	it = gul_toolbar_get_item_by_id (gt, "go");
	if (it)
	{
		if (GALEON_IS_GO_BUTTON (it))
		{	
			DEBUG_MSG (("    got a go button\n"));
			p->go_button = g_object_ref (it);
			if (p->window)
			{
				galeon_tbi_set_window (GALEON_TBI (it), p->window);
			}
		}
		else
		{
			g_warning ("An unexpected button has been found in your toolbar. "
				   "Maybe your setup is too old.");
		}
	}

	it = gul_toolbar_get_item_by_id (gt, "home");
	if (it)
	{
		if (GALEON_IS_HOME_BUTTON (it))
		{	
			DEBUG_MSG (("    got a home button\n"));
			p->home_button = g_object_ref (it);
			if (p->window)
			{
				galeon_tbi_set_window (GALEON_TBI (it), p->window);
			}
		}
		else
		{
			g_warning ("An unexpected button has been found in your toolbar. "
				   "Maybe your setup is too old.");
		}
	}

	it = gul_toolbar_get_item_by_id (gt, "stop");
	if (it)
	{
		if (GALEON_IS_STOP_BUTTON (it))
		{	
			DEBUG_MSG (("    got a stop button\n"));
			p->stop_button = g_object_ref (it);
			if (p->window)
			{
				galeon_tbi_set_window (GALEON_TBI (it), p->window);
			}
		}
		else
		{
			g_warning ("An unexpected button has been found in your toolbar. "
				   "Maybe your setup is too old.");
		}
	}

	it = gul_toolbar_get_item_by_id (gt, "new");
	if (it)
	{
		if (GALEON_IS_NEW_BUTTON (it))
		{	
			DEBUG_MSG (("    got a new button\n"));
			p->new_button = g_object_ref (it);
			if (p->window)
			{
				galeon_tbi_set_window (GALEON_TBI (it), p->window);
			}
		}
		else
		{
			g_warning ("An unexpected button has been found in your toolbar. "
				   "Maybe your setup is too old.");
		}
	}

	it = gul_toolbar_get_item_by_id (gt, "location");
	if (it)
	{
		p->location_entry = gul_tb_item_get_widget (it);
		g_object_ref (p->location_entry);
		toolbar_setup_location_entry (t, p->location_entry);

		DEBUG_MSG (("    got a location entry\n"));
	}
	
	it = gul_toolbar_get_item_by_id (gt, "favicon");
	if (it)
	{
		p->favicon_ebox = gul_tb_item_get_widget (it);
		g_object_ref (p->favicon_ebox);
		toolbar_setup_favicon_ebox (t, p->favicon_ebox);

		DEBUG_MSG (("    got a favicon ebox\n"));
	}

	it = gul_toolbar_get_item_by_id (gt, "spinner");
	if (it)
	{
		p->spinner = gul_tb_item_get_widget (it);
		g_object_ref (p->spinner);
		toolbar_setup_spinner (t, p->spinner);

		DEBUG_MSG (("    got a spinner\n"));
	}

	it = gul_toolbar_get_item_by_id (gt, "zoom");
	if (it)
	{
		p->zoom_spinbutton = gul_tbi_zoom_get_spinner (GUL_TBI_ZOOM (it));
		g_object_ref (p->zoom_spinbutton);
		toolbar_setup_zoom_spinbutton (t, p->zoom_spinbutton);

		DEBUG_MSG (("    got a zoom control\n"));
	}

	/* update the controls */
	galeon_window_update_all_controls (p->window);
}

static void
toolbar_init (Toolbar *t)
{
        t->priv = g_new0 (ToolbarPrivate, 1);

	t->priv->window = NULL;
	t->priv->ui_component = NULL;
	t->priv->visibility = TRUE;
	t->priv->navigation_buttons = NULL;
	t->priv->tooltips = gtk_tooltips_new ();
	g_object_ref (t->priv->tooltips);
	gtk_object_sink (GTK_OBJECT (t->priv->tooltips));

	if (!gul_toolbar_listen_to_gconf (GUL_TOOLBAR (t), CONF_TOOLBAR_SETUP))
	{
		/* FIXME: make this a dialog? */
		g_warning ("An incorrect toolbar configuration has been found, resetting to the default");

		/* this is to make sure we get a toolbar, even if the
		   setup is wrong or there is no schema */
		eel_gconf_set_string (CONF_TOOLBAR_SETUP, DEFAULT_TOOLBAR_SETUP);
	}

	g_signal_connect (t, "changed", G_CALLBACK (toolbar_changed_cb), t);

	t->priv->bview = gul_tb_bonobo_view_new ();
	gul_tb_bonobo_view_set_toolbar (t->priv->bview, GUL_TOOLBAR (t));
}

static void
toolbar_changed_cb (GulToolbar *gt, Toolbar *t)
{
	g_return_if_fail (gt == GUL_TOOLBAR (t));

	if (t->priv->window)
	{
		toolbar_get_widgets (t);
	}
}

static void
toolbar_finalize (GObject *object)
{
	Toolbar *t;
	ToolbarPrivate *p;
	GSList *li;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_TOOLBAR (object));

	t = TOOLBAR (object);
	p = t->priv;

        g_return_if_fail (p != NULL);

	if (p->location_entry) g_object_unref (p->location_entry);
	if (p->favicon_ebox) g_object_unref (p->favicon_ebox);
	if (p->favicon) g_object_unref (p->favicon);
	if (p->spinner) g_object_unref (p->spinner);
	if (p->tooltips) g_object_unref (p->tooltips);
	if (p->zoom_spinbutton) g_object_unref (p->zoom_spinbutton);
	if (p->zoom_timeout_id != 0)
	{
		g_source_remove (p->zoom_timeout_id);
	}

	g_object_unref (t->priv->bview);
	
	for (li = t->priv->navigation_buttons; li; li = li->next)
	{
		g_object_unref (li->data);
	}
	g_slist_free (t->priv->navigation_buttons);

	if (t->priv->reload_button)
	{
		g_object_unref (p->reload_button);
	}

	if (t->priv->go_button)
	{
		g_object_unref (p->go_button);
	}

	if (t->priv->home_button)
	{
		g_object_unref (p->home_button);
	}

	if (t->priv->stop_button)
	{
		g_object_unref (p->stop_button);
	}

	if (t->priv->new_button)
	{
		g_object_unref (p->new_button);
	}

        g_free (t->priv);

	DEBUG_MSG (("Toolbar finalized\n"));

        G_OBJECT_CLASS (parent_class)->finalize (object);
}

Toolbar *
toolbar_new (GaleonWindow *window)
{
	Toolbar *t;

	t = TOOLBAR (g_object_new (TOOLBAR_TYPE, 
				   "GaleonWindow", window,
				   NULL));

	g_return_val_if_fail (t->priv != NULL, NULL);

	return t;
}

void
toolbar_set_visibility (Toolbar *t, gboolean visibility)
{
	if (visibility == t->priv->visibility) return;

	t->priv->visibility = visibility;

	gul_bonobo_set_hidden (BONOBO_UI_COMPONENT(t->priv->ui_component),
                               "/Toolbar",
                               !visibility);
}

void 
toolbar_activate_location (Toolbar *t)
{
	if (t->priv->location_entry)
	{
		galeon_location_entry_activate 
			(GALEON_LOCATION_ENTRY(t->priv->location_entry));
	}
}

void 
toolbar_spinner_start (Toolbar *t)
{
	if (t->priv->spinner)
	{
		galeon_spinner_start (GALEON_SPINNER(t->priv->spinner));
	}
}

void 
toolbar_spinner_stop (Toolbar *t)
{
	if (t->priv->spinner)
	{
		galeon_spinner_stop (GALEON_SPINNER(t->priv->spinner));
	}
}

static void
toolbar_navigation_button_set_sensitive (Toolbar *t, GaleonNavigationDirection d, gboolean sensitivity)
{
	GSList *li;
	ToolbarPrivate *p = t->priv;

	for (li = p->navigation_buttons; li; li = li->next)
	{
		GaleonNavigationButton *b = GALEON_NAVIGATION_BUTTON (li->data);
		if (galeon_navigation_button_get_direction (b) == d)
		{
			galeon_navigation_button_set_sensitive (b, sensitivity);
		}
	}
}

void 
toolbar_button_set_sensitive (Toolbar *t,
			      ToolbarButtonID id,
			      gboolean sensitivity)
{
	ToolbarPrivate *p = t->priv;
	switch (id)
	{
	case TOOLBAR_BACK_BUTTON:
		toolbar_navigation_button_set_sensitive (t, GALEON_NAVIGATION_DIRECTION_BACK, sensitivity);
		break;
	case TOOLBAR_FORWARD_BUTTON:
		toolbar_navigation_button_set_sensitive (t, GALEON_NAVIGATION_DIRECTION_FORWARD, sensitivity);
		break;
	case TOOLBAR_UP_BUTTON:
		toolbar_navigation_button_set_sensitive (t, GALEON_NAVIGATION_DIRECTION_UP, sensitivity);
		break;
	case TOOLBAR_STOP_BUTTON:
		if (p->stop_button)
		{
			galeon_stop_button_set_sensitive (p->stop_button, sensitivity);
		}
		break;
	}
}

void
toolbar_set_location (Toolbar *t,
		      const char *location)
{
	g_return_if_fail (location != NULL);
	
	if (t->priv->location_entry)
	{
		galeon_location_entry_set_location 
			(GALEON_LOCATION_ENTRY (t->priv->location_entry), location);
	}
}

void
toolbar_update_favicon (Toolbar *t)
{
	if (t->priv->favicon)
	{
		galeon_embed_favicon_set_embed (GALEON_EMBED_FAVICON (t->priv->favicon),
						galeon_window_get_active_embed (t->priv->window));
	}
}

char *
toolbar_get_location (Toolbar *t)
{
	gchar *location;
	if (t->priv->location_entry)
	{
		location = galeon_location_entry_get_location
			(GALEON_LOCATION_ENTRY (t->priv->location_entry));
	}
	else
	{
		location = g_strdup ("");
	}
	return location;
}

gint
toolbar_get_zoom (Toolbar *t)
{
	gint zoom;
	if (t->priv->zoom_spinbutton)
	{
		zoom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (t->priv->zoom_spinbutton));
	}
	else
	{
		zoom = 100;
	}
	return zoom;
}

void
toolbar_set_zoom (Toolbar *t, gint zoom)
{
	ToolbarPrivate *p = t->priv;
	if (p->zoom_spinbutton)
	{
		p->zoom_lock = TRUE;
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (p->zoom_spinbutton), zoom);
		p->zoom_lock = FALSE;
	}
}

