/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 *
 * Copyright (C) 2006 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <glib.h>

#include "nautilus-burn-init.h"
#include "nautilus-burn-drive-monitor.h"
#include "nautilus-burn-drive-monitor-private.h"

static gboolean burn_already_initialized = FALSE;
G_LOCK_DEFINE_STATIC (burn_already_initialized);

/**
 * nautilus_burn_init:
 *
 * If nautilus-burn is not already initialized, initialize it. This must be
 * called prior to performing any other nautilus-burn operations, and may
 * be called multiple times without error.
 *
 * Return value: %TRUE if nautilus-burn is successfully initialized (or was
 * already initialized).
 */
gboolean
nautilus_burn_init (void)
{
 	if (! g_thread_supported ()) {
 		g_thread_init (NULL);
	}

	G_LOCK (burn_already_initialized);

	if (! burn_already_initialized) {
		nautilus_burn_get_drive_monitor ();
	}

	burn_already_initialized = TRUE;
	G_UNLOCK (burn_already_initialized);

	return TRUE;
}

/**
 * nautilus_burn_initialized:
 *
 * Detects if nautilus-burn has already been initialized (nautilus-burn must be
 * initialized prior to using any methods or operations).
 *
 * Return value: %TRUE if nautilus-burn has already been initialized.
 */
gboolean
nautilus_burn_initialized (void)
{
	gboolean out;

	G_LOCK (burn_already_initialized);
	out = burn_already_initialized;
	G_UNLOCK (burn_already_initialized);
	return out;
}

/**
 * nautilus_burn_shutdown:
 *
 * Cease all active nautilus-burn operations.
 *
 */
void
nautilus_burn_shutdown (void)
{
	_nautilus_burn_drive_monitor_shutdown ();
}
