#include <gnome.h>
#include <glade/glade.h>

#include "gstcontrol.h"

struct _GstControlPrivate {
	GladeXML            *xml;
	GObject             *adjustment_volume;
	GObject             *adjustment_seek;
	gboolean             seek_in_progress;
	gboolean             allow_fullscreen;
	GstMediaPlayMode     display_mode;
};

enum {
	BUTTON_PLAY,
	BUTTON_STOP,
	BUTTON_NEXT,
	BUTTON_PREV,
	BUTTON_LIST,
	BUTTON_MUTE,
	VOLUME_CHANGE,
	SEEK_CHANGE,
	DISPLAY_MODE_CHANGE,
	LAST_SIGNAL
};

static gint gst_control_signals [LAST_SIGNAL] = { 0 };

static void  gst_control_destroy   (GtkObject  *object);
static void  gst_control_finalize  (GObject    *object); 
static gint  gst_control_expose    (GtkWidget  *widget, GdkEventExpose *event);
static void  gst_control_event_check (GtkWidget* widget, GstControl *control);

GNOME_CLASS_BOILERPLATE (GstControl, gst_control, 
			 GtkVBox, GTK_TYPE_VBOX)



static void
volume_changed (GtkAdjustment *adjustment, GstControl *control)
{
	g_return_if_fail(GST_IS_CONTROL(control));
	g_signal_emit (control, gst_control_signals [VOLUME_CHANGE], 0,
	               gtk_adjustment_get_value(GTK_ADJUSTMENT(control->_priv->adjustment_volume)));
}

static gboolean
seek_changed (GtkWidget *widget, GdkEventButton *event, GstControl *control)
{
	g_return_if_fail(GST_IS_CONTROL(control));
	g_signal_emit (control, gst_control_signals [SEEK_CHANGE], 0, 
	               gtk_adjustment_get_value(GTK_ADJUSTMENT(control->_priv->adjustment_seek)));
	control->_priv->seek_in_progress = FALSE;
	return FALSE;
}

static gboolean
seek_started (GtkWidget *widget, GdkEventButton *event, GstControl *control)
{
	g_return_val_if_fail (GST_IS_CONTROL (control), FALSE);
	control->_priv->seek_in_progress = TRUE;
	return FALSE;
}

static void
gst_control_class_init (GstControlClass *class)
{
	GtkWidgetClass  *widget_class = (GtkWidgetClass *) class;
	GtkObjectClass  *object_class = (GtkObjectClass *) class;
	GObjectClass    *gobject_class = (GObjectClass *) class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = gst_control_destroy;

	gobject_class->finalize = gst_control_finalize;

	widget_class->expose_event = gst_control_expose;

	gst_control_signals [BUTTON_PLAY] =
		g_signal_new ("button_play",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_play),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	gst_control_signals [BUTTON_STOP] =
		g_signal_new ("button_stop",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_stop),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	
	gst_control_signals [BUTTON_NEXT] =
		g_signal_new ("button_next",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_next),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	gst_control_signals [BUTTON_PREV] =
		g_signal_new ("button_prev",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_prev),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	gst_control_signals [BUTTON_LIST] =
		g_signal_new ("button_list",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_list),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);	

	gst_control_signals [BUTTON_MUTE] =
		g_signal_new ("button_mute",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, button_mute),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);	

	gst_control_signals [VOLUME_CHANGE] =
		g_signal_new ("volume_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, volume_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE, 1,
			      G_TYPE_DOUBLE);

	gst_control_signals [SEEK_CHANGE] =
		g_signal_new ("seek_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, seek_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE, 1,
			      G_TYPE_DOUBLE);

	gst_control_signals [DISPLAY_MODE_CHANGE] =
		g_signal_new ("display_mode_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, display_mode_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT);

	class->button_play = NULL;
	class->button_stop = NULL;
	class->button_next = NULL;
	class->button_prev = NULL;
	class->button_list = NULL;
	class->volume_change = NULL;
}


static void
gst_control_instance_init (GstControl *control)
{
	GstControlPrivate  *priv;

	control->_priv = g_new0 (GstControlPrivate, 1);
	priv = control->_priv;

	/* seek slider */
	priv->adjustment_seek = G_OBJECT (gtk_adjustment_new (0.0, 0.0, 1.0, 0.1, 0.1, 0.01));
	priv->seek_in_progress = FALSE;

	/* volume slider */
	priv->adjustment_volume = G_OBJECT (gtk_adjustment_new (0.9, 0.0, 1.0, 0.1, 0.10, 0.10));
	priv->allow_fullscreen = FALSE;
}

GtkWidget *
gst_control_new (GladeXML *xml)
{
	GstControl *control;
	GstControlPrivate  *priv;

	control = g_object_new (GST_TYPE_CONTROL, NULL);

	priv = control->_priv;

	priv->xml = xml;
	glade_xml_signal_connect_data(priv->xml, "clicked_handler", G_CALLBACK (gst_control_event_check), control);

	gtk_box_pack_start (GTK_BOX (control), glade_xml_get_widget(priv->xml, "vbox_media_control"), TRUE, TRUE, 0);

	/* seek slider */
	gtk_range_set_adjustment(GTK_RANGE(glade_xml_get_widget(priv->xml, "hscale_seek")), 
	                         GTK_ADJUSTMENT(priv->adjustment_seek));
	glade_xml_signal_connect_data(priv->xml, "seek_started_handler", G_CALLBACK (seek_started), control);
	glade_xml_signal_connect_data(priv->xml, "seek_changed_handler", G_CALLBACK (seek_changed), control);

	/* volume slider */
	gtk_range_set_adjustment(GTK_RANGE(glade_xml_get_widget(priv->xml, "hscale_volume")), 
	                         GTK_ADJUSTMENT(priv->adjustment_volume));
	glade_xml_signal_connect_data(priv->xml, "volume_changed_handler", G_CALLBACK (volume_changed), control);

	gst_control_set_display_mode(control, GST_MEDIA_PLAY_NORMAL);
	return GTK_WIDGET (control);
}

static void
gst_control_destroy (GtkObject *object)
{
	GstControl *control;

	g_return_if_fail (GST_IS_CONTROL (object));

	control = GST_CONTROL (object);

	GNOME_CALL_PARENT (GTK_OBJECT_CLASS, destroy, (object));
}

static void
gst_control_finalize (GObject *object)
{
	GstControl *control;

	g_return_if_fail (GST_IS_CONTROL (object));

	control = GST_CONTROL (object);

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static gint
gst_control_expose (GtkWidget  *widget, GdkEventExpose *event)
{
	GstControl *control;
	g_return_val_if_fail (GST_IS_CONTROL (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	control = GST_CONTROL(widget);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		return GNOME_CALL_PARENT_WITH_DEFAULT
			(GTK_WIDGET_CLASS, expose_event, (widget, event), FALSE);
	}

	return FALSE;
}


void
gst_control_set_display_mode  (GstControl *control, GstMediaPlayMode display_mode)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));

	priv = control->_priv;

	if (display_mode == GST_MEDIA_PLAY_NORMAL){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "hbox_mediacontrol"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_mini"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_fullscreen"));
		gtk_widget_set_sensitive(glade_xml_get_widget(priv->xml, "button_fullscreen"), priv->allow_fullscreen);

		gtk_widget_hide(glade_xml_get_widget(priv->xml, "button_play_top"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "button_fs_to_normal"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "button_mini_to_normal"));
	}
	else if (display_mode == GST_MEDIA_PLAY_FULLSCREEN){
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "hbox_mediacontrol"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "button_mini_to_normal"));

		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_play_top"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_fs_to_normal"));
	}
	else {
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "hbox_mediacontrol"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "button_fs_to_normal"));

		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_play_top"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "button_mini_to_normal"));
	}

	priv->display_mode = display_mode;
}

void
gst_control_set_state  (GstControl *control, GstElementState state)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));

	priv = control->_priv;

	if (state == GST_STATE_PLAYING){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_paused_bot"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_paused_top"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_playing_bot"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_playing_top"));
	}
	else {
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_paused_bot"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_paused_top"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_playing_bot"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_playing_top"));
	}
}

GtkAdjustment*
gst_control_get_volume_adjustment (GstControl *control)
{
	g_return_if_fail (GST_IS_CONTROL (control));
	return GTK_ADJUSTMENT(control->_priv->adjustment_volume);

}

void
gst_control_set_mute  (GstControl *control, gboolean mute)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	if (mute){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume_mute"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume"));
	}
	else {
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume_mute"));
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume"));
	}
}

void
gst_control_set_enable_seek (GstControl* control, gboolean enable)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	gtk_widget_set_sensitive(glade_xml_get_widget(priv->xml, "hscale_seek"), enable);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(priv->adjustment_seek), 0.0);
}

void
gst_control_allow_fullscreen (GstControl* control, gboolean allow)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	priv->allow_fullscreen = allow;
	gtk_widget_set_sensitive(glade_xml_get_widget(priv->xml, "button_fullscreen"), allow);
}

void
gst_control_playlist_is_open (GstControl* control, gboolean is_open)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(priv->xml, "button_playlist")), is_open);
}

void
gst_control_set_seek_pos (GstControl* control, gdouble seek_pos)
{
	g_return_if_fail (GST_IS_CONTROL (control));

	if (control->_priv->seek_in_progress == FALSE){
		gtk_adjustment_set_value(GTK_ADJUSTMENT(control->_priv->adjustment_seek), seek_pos);
	}
}

static void
gst_control_event_check (GtkWidget* widget, GstControl *control)
{
	GstControlPrivate *priv;

	g_return_if_fail(GST_IS_CONTROL(control));

	priv = control->_priv;

	if (widget == glade_xml_get_widget(priv->xml, "button_play_bot") ||
	    widget == glade_xml_get_widget(priv->xml, "button_play_top")){
		g_signal_emit (control, gst_control_signals [BUTTON_PLAY], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_stop")){
		g_signal_emit (control, gst_control_signals [BUTTON_STOP], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_next")){
		g_signal_emit (control, gst_control_signals [BUTTON_NEXT], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_prev")){
		g_signal_emit (control, gst_control_signals [BUTTON_PREV], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_playlist")){
		g_signal_emit (control, gst_control_signals [BUTTON_LIST], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_mute")){
		g_signal_emit (control, gst_control_signals [BUTTON_MUTE], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_fullscreen")){
		g_signal_emit (control, gst_control_signals [DISPLAY_MODE_CHANGE], 0, GST_MEDIA_PLAY_FULLSCREEN);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_mini_to_normal")){
		g_signal_emit (control, gst_control_signals [DISPLAY_MODE_CHANGE], 0, GST_MEDIA_PLAY_NORMAL);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_mini")){
		g_signal_emit (control, gst_control_signals [DISPLAY_MODE_CHANGE], 0, GST_MEDIA_PLAY_MINI);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_fs_to_normal")){
		g_signal_emit (control, gst_control_signals [DISPLAY_MODE_CHANGE], 0, GST_MEDIA_PLAY_NORMAL);
		return;
	}
}
