/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include <gtk/gtkimagemenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkimage.h>
#include "bookmarks-gtk-menu.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-string.h"
#include "bookmarks-context-menu.h"

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GbGtkMenuPrivate {
	GbFolder *root;
	GtkMenuShell *ms;
	guint rebuild_timeout;

	GHashTable *bookmark_to_git;

	GbLocationSource *location_source;
};

typedef struct _GmItem GmItem;
struct _GmItem {
	GbGtkMenu *gm;
	GbBookmark *bm;
	GtkWidget *widget;
	gboolean built : 1;
	gboolean built_childs : 1;
	gboolean ignore_next_activation : 1; /* little hack */
};

#define REBUILD_TIMEOUT 500

/**
 * Private functions, only availble from this file
 */
static void		gb_gtk_menu_class_init			(GbGtkMenuClass *klass);
static void		gb_gtk_menu_init			(GbGtkMenu *e);
static void		gb_gtk_menu_finalize_impl		(GObject *o);
static void		gb_gtk_menu_build_submenu		(GbGtkMenu *gm, GbFolder *item,
								 GtkMenuShell *ms);
static void		gb_gtk_menu_build_folder		(GbGtkMenu *gm, GbFolder *item, 
								 GtkMenuShell *ms, guint index);
static void		gb_gtk_menu_build_site			(GbGtkMenu *gm, GbSite *item,
							 	GtkMenuShell *ms, guint index);
static void		gb_gtk_menu_build_childs		(GbGtkMenu *gm, GbFolder *folder);
static GtkWidget *	gb_gtk_menu_create_menuitem		(GbBookmark *bm);
static void		gb_gtk_menu_cb				(GtkWidget *w, GmItem *git);

static GmItem *		gm_item_new				(GbGtkMenu *gm, GbBookmark *b);
static void		gm_item_destroy_from_hash		(gpointer key, gpointer data, 
								 gpointer user_data);
static gboolean		gb_gtk_menu_menuitem_button_press_cb	(GtkWidget *wid, GdkEventButton *event, 
								 GmItem *bm);
static void		gb_gtk_menu_event_after_cb	 	(GtkWidget *wid, GdkEventAny *event, 
								 GmItem *bm);
static void		gb_gtk_menu_popup_deactivated_cb	(GbContextMenu *cm, GmItem *git);
static void		gb_gtk_menu_descendant_modied_cb	(GbFolder *f, GbBookmark *bm, 
								 GbGtkMenu *gm);
static void		gb_gtk_menu_descendant_added_cb		(GbFolder *f, GbFolder *p, GbBookmark *c, 
								 gint position, GbGtkMenu *gm);
static void		gb_gtk_menu_descendant_removed_cb 	(GbFolder *f, GbFolder *p, GbBookmark *c, 
								 gint position, GbGtkMenu *gm);
static void		gb_gtk_menu_rebuild			(GbGtkMenu *gm);
static void 		gb_gtk_menu_rebuild_with_timeout	(GbGtkMenu *gm);
static gboolean		gb_gtk_menu_rebuild_timeout_cb		(gpointer data);


static gpointer g_object_class;

enum GbGtkMenuSignalsEnum {
	GB_GTK_MENU_BOOKMARK_ACTIVATED,
	GB_GTK_MENU_LAST_SIGNAL
};
static gint GbGtkMenuSignals[GB_GTK_MENU_LAST_SIGNAL];

/**
 * GtkMenu object
 */

MAKE_GET_TYPE (gb_gtk_menu, "GbGtkMenu", GbGtkMenu, gb_gtk_menu_class_init,
	       gb_gtk_menu_init, G_TYPE_OBJECT);

static void
gb_gtk_menu_class_init (GbGtkMenuClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_gtk_menu_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	GbGtkMenuSignals[GB_GTK_MENU_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbGtkMenuClass, gb_gtk_menu_bookmark_activated), 
		NULL, NULL, 
		galeon_marshal_VOID__OBJECT_STRING_INT,
		G_TYPE_NONE, 3, GB_TYPE_BOOKMARK, G_TYPE_STRING, G_TYPE_UINT);
}

static void 
gb_gtk_menu_init (GbGtkMenu *m)
{
	GbGtkMenuPrivate *p = g_new0 (GbGtkMenuPrivate, 1);

	m->priv = p;
	
}

static void
gb_gtk_menu_finalize_impl (GObject *o)
{
	GbGtkMenu *gm = GB_GTK_MENU (o);
	GbGtkMenuPrivate *p = gm->priv;

	DEBUG_MSG (("Finalizing GbGtkMenu\n"));

	gb_gtk_menu_set_location_source (gm, NULL);

	g_signal_handlers_disconnect_matched (p->root, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, gm);

	if (p->rebuild_timeout)
	{
		g_source_remove (p->rebuild_timeout);
	}

	g_hash_table_foreach (p->bookmark_to_git, gm_item_destroy_from_hash, NULL);
	g_hash_table_destroy (p->bookmark_to_git);
	g_object_unref (G_OBJECT (p->root));
	g_object_unref (G_OBJECT (p->ms));
	g_free (p);

	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GbGtkMenu *
gb_gtk_menu_new (GbFolder *root, GtkMenuShell *ms)
{
	GbGtkMenu *ret = g_object_new (GB_TYPE_GTK_MENU, NULL);
	GbGtkMenuPrivate *p = ret->priv;

	p->root = root;
	g_object_ref (root);

	g_signal_connect (root, "descendant-modified", 
			  G_CALLBACK (gb_gtk_menu_descendant_modied_cb), ret);
	g_signal_connect (root, "descendant-added", 
			  G_CALLBACK (gb_gtk_menu_descendant_added_cb), ret);
	g_signal_connect (root, "descendant-removed", 
			  G_CALLBACK (gb_gtk_menu_descendant_removed_cb), ret);

	p->ms = ms;
	g_object_ref (G_OBJECT (ms));

	gb_gtk_menu_rebuild (ret);

	return ret;
}

static void
gb_gtk_menu_build_submenu (GbGtkMenu *gm, GbFolder *item, GtkMenuShell *ms)
{
	guint index = 0;
	GbBookmark *i;

	for (i = item->child; i != NULL; i = i->next)
	{
		if (GB_IS_SITE (i))
		{
			gb_gtk_menu_build_site (gm, GB_SITE (i), ms, index);
		}
		else if (GB_IS_FOLDER (i))
		{
			gb_gtk_menu_build_folder (gm, GB_FOLDER (i), ms, index);
		}
		else if (GB_IS_SEPARATOR (i))
		{
			GtkWidget *sep = gtk_menu_item_new ();
			gtk_container_add (GTK_CONTAINER (ms), sep);
			gtk_widget_show (sep);
		}
		else
		{
			g_warning ("something skipped when building bookmarks menu");
		}
		index++;
	}
}

static GtkWidget *
gb_gtk_menu_create_menuitem (GbBookmark *bm)
{
	gchar *name = gul_string_shorten (GB_BOOKMARK (bm)->name, GB_GTK_MENU_NAME_MAX_LENGTH);
	GtkWidget *ret = gtk_image_menu_item_new_with_label (name);
	GdkPixbuf *pixbuf = gb_bookmark_get_icon (bm);
	g_free (name);
	
	if (GB_IS_SITE (bm))
	{
		// gchar *tip = misc_string_shorten (GB_SITE (bm)->url, GB_GTK_MENU_TIP_MAX_LENGTH);
		// TODO: do we really want the tooltip?
		// g_free (tip);
	}
	
	if (pixbuf)
	{
		GtkWidget *image = gtk_image_new_from_pixbuf (pixbuf);
		g_object_unref (G_OBJECT (pixbuf));
		gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (ret), image);
	}

	gtk_widget_show_all (ret);
	return ret;
}

static void
gb_gtk_menu_build_folder (GbGtkMenu *gm, GbFolder *item, GtkMenuShell *ms, guint index)
{
	GmItem *git;
		
	GtkWidget *menuitem = gb_gtk_menu_create_menuitem (GB_BOOKMARK (item));
	GtkWidget *menu = gtk_menu_new ();

	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);

	gtk_container_add (GTK_CONTAINER (ms), menuitem);
	git = gm_item_new (gm, GB_BOOKMARK (item));
	git->widget = menuitem;
	git->built = FALSE;
	
	g_signal_connect (menuitem, "map", G_CALLBACK (gb_gtk_menu_cb), git);

 	g_signal_connect (menu, "event-after",
			  G_CALLBACK (gb_gtk_menu_event_after_cb), git);

}

static void
gb_gtk_menu_build_site (GbGtkMenu *gm, GbSite *item, GtkMenuShell *ms, guint index)
{
	GmItem *git;

	GtkWidget *menuitem = gb_gtk_menu_create_menuitem (GB_BOOKMARK (item));

	
	gtk_container_add (GTK_CONTAINER (ms), menuitem);
	git = gm_item_new (gm, GB_BOOKMARK (item));
	git->widget = menuitem;
	git->built = FALSE;
				
 	g_signal_connect (menuitem, "button_press_event",
			  G_CALLBACK (gb_gtk_menu_menuitem_button_press_cb), git);

	g_signal_connect (menuitem, "activate", G_CALLBACK (gb_gtk_menu_cb), git);
}

static void
gm_item_destroy_from_hash (gpointer key, gpointer data, gpointer user_data)
{
	/* data is a GmItem that is being removed from the hashtable */
	g_free (data);
}

static GmItem *
gm_item_new (GbGtkMenu *gm, GbBookmark *b)
{
	GmItem *ret = g_new0 (GmItem, 1);
	ret->gm = gm;
	ret->bm = b;
	ret->widget = NULL;
	g_hash_table_insert (gm->priv->bookmark_to_git, b, ret);
	return ret;
}

static void
gb_gtk_menu_cb (GtkWidget *w, GmItem *git)
{
	if (git->ignore_next_activation)
	{
		git->ignore_next_activation = FALSE;
		return;
	}
	if (GB_IS_FOLDER (git->bm))
	{
		if (!git->built)
		{
			GtkMenuShell *ms = GTK_MENU_SHELL (gtk_menu_item_get_submenu 
							   (GTK_MENU_ITEM (git->widget)));
			gb_gtk_menu_build_submenu (git->gm, GB_FOLDER (git->bm), ms);
			git->built = TRUE;
		}
		if (!git->built_childs)
		{
			gb_gtk_menu_build_childs (git->gm, GB_FOLDER (git->bm));
			git->built_childs = TRUE;
		}
	}
	else if (GB_IS_SITE (git->bm))
	{
		g_signal_emit (git->gm, GbGtkMenuSignals[GB_GTK_MENU_BOOKMARK_ACTIVATED], 0, 
			       git->bm, GB_SITE (git->bm)->url, GB_BAF_DEFAULT);
	}
	else
	{
		g_warning ("unexpected bookmark type");
	}
}

static void
gb_gtk_menu_build_childs (GbGtkMenu *gm, GbFolder *folder)
{
	GbBookmark *i;
	for (i = folder->child; i != NULL; i = i->next)
	{
		if (GB_IS_FOLDER (i))
		{
			GmItem *git = g_hash_table_lookup (gm->priv->bookmark_to_git, i);
			if (git)
			{
				GtkMenuShell *ms = GTK_MENU_SHELL (gtk_menu_item_get_submenu 
								   (GTK_MENU_ITEM (git->widget)));
				gb_gtk_menu_build_submenu (git->gm, GB_FOLDER (git->bm), ms);
				git->built = TRUE;
			}
		}
	}
}

static gboolean
gb_gtk_menu_menuitem_button_press_cb (GtkWidget *wid, GdkEventButton *event, 
				      GmItem *git)
{
	if (event->button == 3)
	{
		GbContextMenu *cm = gb_context_menu_new_quick (git->bm, git->gm->priv->location_source, 
							       G_OBJECT (git->gm));
		g_signal_connect (cm, "deactivated", G_CALLBACK (gb_gtk_menu_popup_deactivated_cb), git);
		gb_context_menu_popup (cm, event);
		g_object_unref (cm);
		return TRUE;
	}
	else if (event->button == 2)
	{
		const gchar *url = GB_IS_SITE (git->bm) ? GB_SITE (git->bm)->url : NULL;
		g_signal_emit_by_name (git->gm, "bookmark-activated", git->bm, url, 
				       GB_BAF_NEW_TAB_OR_WINDOW);

		/* avoid activating this bookmark */
		git->ignore_next_activation = TRUE;

		return TRUE;
	}
	return FALSE;
}

static void
gb_gtk_menu_event_after_cb (GtkWidget *wid, GdkEventAny *event, 
			    GmItem *git)
{
	if (event->type == GDK_BUTTON_PRESS
	    && (((GdkEventButton *) event)->button == 3)
	    /* this is a bit hackish, avoids popping more than one menu */
	    && !GTK_MENU_SHELL (wid)->active_menu_item) 
	{
		GbContextMenu *cm = gb_context_menu_new_quick (git->bm, git->gm->priv->location_source, 
							       G_OBJECT (git->gm));
		g_signal_connect (cm, "deactivated", G_CALLBACK (gb_gtk_menu_popup_deactivated_cb), git);
		gb_context_menu_popup (cm, (GdkEventButton *) event);
		g_object_unref (cm);
	}
	else if (event->type == GDK_BUTTON_PRESS
		 && (((GdkEventButton *) event)->button == 2))
	{
		const gchar *url = GB_IS_SITE (git->bm) ? GB_SITE (git->bm)->url : NULL;
		g_signal_emit_by_name (git->gm, "bookmark-activated", git->bm, url, 
				       GB_BAF_NEW_TAB_OR_WINDOW);
	}
}

static void
gb_gtk_menu_popup_deactivated_cb (GbContextMenu *cm, GmItem *git)
{
	gtk_menu_shell_deactivate (git->gm->priv->ms);
}

void
gb_gtk_menu_set_location_source (GbGtkMenu *gm, GbLocationSource *src)
{
	GbGtkMenuPrivate *p = gm->priv;

	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}

	p->location_source = src;

	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}
}

static gboolean
gb_gtk_menu_rebuild_timeout_cb (gpointer data)
{
	GbGtkMenu *gm = data;
	GbGtkMenuPrivate *p = gm->priv;
	DEBUG_MSG (("GbGtkMenu rebuild timeout\n"));

	p->rebuild_timeout = 0;

	gb_gtk_menu_rebuild (data);
	return FALSE;
}

static void 
gb_gtk_menu_rebuild_with_timeout (GbGtkMenu *gm) 
{
	GbGtkMenuPrivate *p = gm->priv;

	if (p->rebuild_timeout)
	{
		g_source_remove (p->rebuild_timeout);
	}
	
	p->rebuild_timeout = g_timeout_add (REBUILD_TIMEOUT, 
					    gb_gtk_menu_rebuild_timeout_cb, gm);
}

static void
gb_gtk_menu_rebuild (GbGtkMenu *gm)
{
	GbGtkMenuPrivate *p = gm->priv;
	GList *children;
	GList *li;

	DEBUG_MSG (("gb_gtk_menu_rebuild called\n"));
	
	/* this could be more fine grained. For now, just destroy everything and 
	   build it again */
	
	if (p->bookmark_to_git)
	{
		DEBUG_MSG (("Rebuilding GbGtkMenu\n"));
		g_hash_table_foreach (p->bookmark_to_git, gm_item_destroy_from_hash, NULL);
		g_hash_table_destroy (p->bookmark_to_git);
	}

	p->bookmark_to_git = g_hash_table_new (NULL, NULL);
	
	children = gtk_container_get_children (GTK_CONTAINER (p->ms));
	for (li = children; li; li = li->next)
	{
		gtk_container_remove (GTK_CONTAINER (p->ms), li->data);
	}
	g_list_free (children);

	gb_gtk_menu_build_submenu (gm, p->root, p->ms);
	gb_gtk_menu_build_childs (gm, p->root);
	
}

static void
gb_gtk_menu_descendant_modied_cb (GbFolder *f, GbBookmark *bm, GbGtkMenu *gm)
{
	gb_gtk_menu_rebuild_with_timeout (gm);
}

static void
gb_gtk_menu_descendant_added_cb (GbFolder *f, GbFolder *p, GbBookmark *c, 
				 gint position, GbGtkMenu *gm)
{
	gb_gtk_menu_rebuild_with_timeout (gm);
}

static void
gb_gtk_menu_descendant_removed_cb (GbFolder *f, GbFolder *p, GbBookmark *c, 
				   gint position, GbGtkMenu *gm)
{
	gb_gtk_menu_rebuild_with_timeout (gm);
}

