#include <config.h>

#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <asm/page.h>
#include <linux/tasks.h>
#include <time.h>
#include <unistd.h>
#include <signal.h>

#include <gnome.h>

#include "procview.h"
#include "fromtop.h"
#include "procbar.h"
#include "proc.h"
#include "details.h"
#include "global.h"

#include "asc.xpm"
#include "dsc.xpm"

#include "cpu.xpm"
#include "mem.xpm"
#include "swap.xpm"

#define CF ((proc_field *)c->data)
#define CG (*((gint *)g->data))

static GtkWidget *nLabel;
static GtkWidget *clist;
static GtkWidget *clist_menu;
static GtkWidget *vbox;
static GtkWidget *handle_box;
static GtkWidget *summary;

static gint cfg_summary;
static ProcInfo summary_info;

gint cfg_show_summary;
gint cfg_show_summary_cpu;
gint cfg_show_summary_mem;
gint cfg_show_summary_swap;
gint cfg_has_swap;

GtkWidget *summary_cpu;
GtkWidget *summary_mem;
GtkWidget *summary_swap = NULL;

static GdkColor bar_cpu_colors [PROC_CPU_SIZE-1] = {
	{0, 0xffff, 0xffff, 0x4fff},
	{0, 0xdfff, 0xdfff, 0xdfff},
	{0, 0xafff, 0xafff, 0xafff},
	{0, 0, 0, 0},
};

static GdkColor bar_mem_colors [PROC_MEM_SIZE-1] = {
	{0, 0xbfff, 0xbfff, 0x4fff},
	{0, 0xefff, 0xefff, 0x4fff},
	{0, 0xafff, 0xafff, 0xafff},
	{0, 0, 0x8fff, 0},
};

static GdkColor bar_swap_colors [PROC_SWAP_SIZE-1] = {
	{0, 0xcfff, 0x5fff, 0x5fff},
	{0, 0, 0x8fff, 0},
};

static GList *field_list = NULL;
static GList *geometry_list = NULL;
static GtkWidget *sort_asc, *sort_dsc;
static gint field_list_length;
static gint select_pid = -1;

static gint read_proc_mask = PROC_FILLUSR | PROC_FILLMEM;
static uid_t user_uid;

/* procview idle run or frozen */
static gint cfg_run;
static gint run_tag = -1;
static gint cfg_sf, cfg_order;

static GtkWidget *xpm_run;
static GtkWidget *xpm_stop;

static GtkWidget *xpm_cpu;
static GtkWidget *xpm_mem;
static GtkWidget *xpm_swap;

static gint p_current_time;
static time_t p_time;
static float p_elapsed_time;
static gint p_total_time;

/* cumulative mode */
static gint p_cl_sum = 0;

static unsigned p_gl_main_mem;

static proc_t **proc_tab = NULL;
static int prev_count = 0;

ProcProcData **proc_data = NULL;

static gint first = 1;

static gint procview_update ();
static void procview_init (gchar *, gchar *);
static void procview_open_details ();
static GtkWidget * procview_clist_prepare ();

static GtkWidget     * renice_dialog = NULL;
static GtkAdjustment * renice_aj;
static GtkWidget     * renice_msg;

static void procview_cfg_read ();

#define CMP_DINT(NAME) \
\
static int cmp_ ## NAME (const ProcProcData **P, const ProcProcData **Q) \
{ \
	  return sort_order * ((*P)-> ## NAME - (*Q)-> ## NAME); \
}

#define CMP_INT(NAME) \
\
static int cmp_ ## NAME (const ProcProcData **P, const ProcProcData **Q) \
{ \
	  return sort_order * ((*P)->p-> ## NAME - (*Q)->p-> ## NAME); \
}

#define CMP_INT2(NAME,V1,V2) \
\
static int cmp_ ## NAME (const ProcProcData **P, const ProcProcData **Q) \
{ \
	  return sort_order * ((*P)->p-> ## V1 + (*P)->p-> ## V2 - (\
                               (*Q)->p-> ## V1 + (*Q)->p-> ## V2)); \
}

#define CMP_STR(NAME) \
\
static int cmp_ ## NAME (const ProcProcData **P, const ProcProcData **Q) \
{ \
	  return sort_order * strcasecmp ((*P)->p-> ## NAME, (*Q)->p-> ## NAME); \
}

#define SORT_ASC 1
#define SORT_DSC -1

static int sort_field = -1;
static int sort_order = SORT_ASC;

CMP_INT (stime)
CMP_INT (utime)
CMP_INT2 (time, stime, utime)
CMP_INT (pid)
CMP_DINT (pcpu)
CMP_DINT (pmem)
CMP_INT (size)
CMP_INT (rss)
CMP_INT (resident)
CMP_INT (nice)
CMP_INT (priority)
CMP_INT (share)

CMP_STR (user)
CMP_STR (cmd)

proc_field p_fields [] = {

	{ N_("PID"), N_("Process ID"),
	  P_PID, cmp_pid, SORT_ASC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("User"), N_("User name"),
	  P_USER, cmp_user, SORT_ASC, PROC_FILLUSR, GTK_JUSTIFY_LEFT, NULL},
	{ N_("Pri"), N_("Priority"),
	  P_PRIORITY, cmp_priority, SORT_ASC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Ni"), N_("Nice"),
	  P_NICE, cmp_nice, SORT_ASC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Size"), N_("Used memory size"),
	  P_SIZE, cmp_size, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("RSS"), N_("Resident memory set size"),
	  P_RSS, cmp_rss, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Share"), N_("Shared memory size"),
	  P_SHARE, cmp_share, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Stat"), N_("Process state"),
	  P_STATE, NULL, SORT_ASC, 0, GTK_JUSTIFY_LEFT, NULL},
	{ N_("CPU"), N_("Percent CPU usage"),
	  P_PCPU, cmp_pcpu, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("MEM"), N_("Percent memory usage"),
	  P_PMEM, cmp_pmem, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Time"), N_("Time consumed"),
	  P_TIME, cmp_time, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("UTime"), N_("User time consumed"),
	  P_UTIME, cmp_utime, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("STime"), N_("System time consumed"),
	  P_STIME, cmp_stime, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ N_("Cmd"), N_("Command"),
	  P_CMD, cmp_cmd, SORT_ASC, PROC_FILLCMD, GTK_JUSTIFY_LEFT, NULL},
	{ N_("Resident"), N_("Resident memory size"),
	  P_RESIDENT, cmp_resident, SORT_DSC, 0, GTK_JUSTIFY_RIGHT, NULL},
	{ NULL, NULL, 0, NULL, 0, 0, 0, NULL},
};

struct _gtop_signal {
	gchar *name;
	gint sig_num;
} gtop_signals [] = {
	{"SIGHUP", SIGHUP},
	{"SIGINT", SIGINT},
	{"SIGQUIT", SIGQUIT},
	{"SIGILL", SIGILL},
	{"SIGTRAP", SIGTRAP},
	{"SIGABRT", SIGABRT},
	{"SIGIOT", SIGIOT},
	{"SIGBUS", SIGBUS},
	{"SIGFPE", SIGFPE},
	{"SIGKILL", SIGKILL},
	{"SIGUSR1", SIGUSR1},
	{"SIGSEGV", SIGSEGV},
	{"SIGUSR2", SIGUSR2},
	{"SIGPIPE", SIGPIPE},
	{"SIGALRM", SIGALRM},
	{"SIGTERM", SIGTERM},
#ifdef SIGSTKFLT
	{"SIGSTKFLT", SIGSTKFLT},
#endif
	{"SIGCHLD", SIGCHLD},
	{"SIGCONT", SIGCONT},
	{"SIGSTOP", SIGSTOP},
	{"SIGTSTP", SIGTSTP},
	{"SIGTTIN", SIGTTIN},
	{"SIGTTOU", SIGTTOU},
	{"SIGURG", SIGURG},
	{"SIGXCPU", SIGXCPU},
	{"SIGXFSZ", SIGXFSZ},
	{"SIGVTALRM", SIGVTALRM},
	{"SIGPROF", SIGPROF},
	{"SIGWINCH", SIGWINCH},
	{"SIGPOLL", SIGPOLL},
	{"SIGIO", SIGIO},
	{"SIGPWR", SIGPWR},
#ifdef SIGUNUSED
	{"SIGUNUSED", SIGUNUSED},
#endif
	{NULL, 0},
};

static void procview_click_column (GtkCList *cl, gint n);
void procview_handler (GtkNotebook *, GtkNotebookPage *, GTopPageSignal);

void
procview_all_procs ()
{
	read_proc_mask &= ~(PROC_ANYTTY | PROC_UID);

	procview_update ();
}

void
procview_tty_procs ()
{
	read_proc_mask |= PROC_ANYTTY;

	procview_update ();
}

void
procview_user_procs ()
{
	read_proc_mask |= PROC_UID;

	procview_update ();
}

void
procview_start_stop (gint start)
{
	/* char **pd = (cfg_run) ? tb_timer_stopped_xpm : tb_timer_xpm; */

	if (!start) {
		if (run_tag != -1) {
			gtk_timeout_remove (run_tag);
			run_tag = -1;
		}

	} else {

		if (run_tag != -1)
			return;

		procview_update ();
		run_tag = gtk_timeout_add (3000, procview_update, NULL);
	}

	gnome_stock_pixmap_widget_set_icon ((GnomeStock *) (TBC (RUN)->icon),
					    (start) ? GNOME_STOCK_PIXMAP_TIMER :
					    GNOME_STOCK_PIXMAP_TIMER_STOP);

	gtk_widget_queue_draw (GTK_WIDGET (TBC (RUN)->widget));
}

static void
procview_tbar_time (GtkWidget *w, gpointer gp)
{
	cfg_run = !cfg_run;
	procview_start_stop (cfg_run);
}

GnomeUIInfo procViewMenu [] = {
  {GNOME_APP_UI_ITEM, N_("All processes"), NULL, procview_all_procs, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("User processes"), NULL, procview_user_procs, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("TTY processes"), NULL, procview_tty_procs, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  GNOMEUIINFO_END
};

gint
procview_cpu ()
{
	proc_read_cpu (&summary_info);
	gnome_proc_bar_set_values (GNOME_PROC_BAR (summary_cpu), summary_info.cpu);

	return TRUE;
}

gint
procview_mem ()
{
	proc_read_mem (&summary_info);
	gnome_proc_bar_set_values (GNOME_PROC_BAR (summary_mem), summary_info.mem);
	if (summary_swap)
		gnome_proc_bar_set_values (GNOME_PROC_BAR (summary_swap), summary_info.swap);

	return TRUE;
}

static GtkWidget *
procview_summary ()
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (TRUE, GNOME_PAD_SMALL);

	proc_read_mem (&summary_info);
	summary_cpu = gnome_proc_bar_new (xpm_cpu, PROC_CPU_SIZE-1, bar_cpu_colors,
				   procview_cpu);
	summary_mem = gnome_proc_bar_new (xpm_mem, PROC_MEM_SIZE-2, bar_mem_colors,
				   procview_mem);
	cfg_has_swap = (summary_info.swap [PROC_SWAP_TOTAL]);

	summary_swap = gnome_proc_bar_new (xpm_swap, PROC_SWAP_SIZE-1, bar_swap_colors,
					   NULL);

	gtk_box_pack_start_defaults (GTK_BOX (hbox), summary_cpu);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), summary_mem);
	if (cfg_has_swap)
		gtk_box_pack_start_defaults (GTK_BOX (hbox), summary_swap);

	if (cfg_show_summary_cpu)
		gtk_widget_show (summary_cpu);

	if (cfg_has_swap && cfg_show_summary_mem)
		gtk_widget_show (summary_mem);

	if (cfg_show_summary_swap && cfg_has_swap)
		gtk_widget_show (summary_swap);

	return hbox;
}

#define PIX(x) widget = gnome_pixmap_new_from_xpm_d (x ## _xpm); \
                        gtk_widget_ref (widget);
               

void *
addProcessesView ()
{
	GtkWidget *widget;

	procview_cfg_read ();

	sort_asc = PIX (asc);
	sort_dsc = PIX (dsc);

	xpm_run = PIX (tb_timer);
	xpm_stop = PIX (tb_timer_stopped);

	xpm_cpu = PIX (cpu);
	xpm_mem = PIX (mem);
	xpm_swap = PIX (swap);

	procview_init (gnome_config_get_string
		       ("/gtop/procview/fields=0,1,2,3,4,5,6,7,8,9,10,13"),
		       gnome_config_get_string
		       ("/gtop/procview/geometry"
			"=50,80,30,30,50,50,50,50,60,60,60,200"));	

	nLabel = gtk_label_new (_("Processes"));
	clist = procview_clist_prepare ();

	procview_click_column (GTK_CLIST (clist), cfg_sf);

	summary = procview_summary ();
	handle_box = gtk_handle_box_new ();
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	/* GTK_HANDLE_BOX (handle_box)->shrink_on_detach = 0; */
#if 0
	GTK_HANDLE_BOX (handle_box)->shadow_type = GTK_SHADOW_NONE;
#endif
	gtk_container_border_width (GTK_CONTAINER (vbox), GNOME_PAD_SMALL);

	gtk_container_add (GTK_CONTAINER (handle_box), summary);
	gtk_box_pack_start (GTK_BOX (vbox), handle_box, FALSE, TRUE, 0);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), clist);

	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, nLabel);

	if (cfg_show_summary)
		gtk_widget_show (summary);

	gtk_widget_show (nLabel);
	gtk_widget_show (clist);
	gtk_widget_show (handle_box);
	gtk_widget_show (vbox);

	/* pt = openproc (PROC_FILLUSR); */
	/* when I don't call meminfo, nice & priority values are malformed */
	/* don't know why ?!? */
	meminfo ();
	/* procview_update (); */

	return procview_handler;
}

static void
procview_cfg_read ()
{
	printf ("proview_cfg_read\n");

	cfg_run = gnome_config_get_int ("/gtop/procview/run=1");

	cfg_show_summary = gnome_config_get_int ("/gtop/procview/show_summary=1");
	cfg_show_summary_cpu  = gnome_config_get_int ("/gtop/procview/show_summary_cpu=1");
	cfg_show_summary_mem  = gnome_config_get_int ("/gtop/procview/show_summary_mem=1");
	cfg_show_summary_swap = gnome_config_get_int ("/gtop/procview/show_summary_swap=1");

	cfg_sf    = gnome_config_get_int ("/gtop/procview/sort_field=8");
	cfg_order = gnome_config_get_int ("/gtop/procview/sort_order=1");

	p_fields [cfg_sf].order = (cfg_order) ? SORT_DSC : SORT_ASC;
}

static void
procview_cfg_save ()
{
	/* printf ("proview_cfg_save\n"); */

	gnome_config_set_int ("/gtop/procview/run", cfg_run);
	gnome_config_set_int ("/gtop/procview/show_summary", cfg_show_summary);
	gnome_config_set_int ("/gtop/procview/show_summary_cpu", cfg_show_summary_cpu);
	gnome_config_set_int ("/gtop/procview/show_summary_mem", cfg_show_summary_mem);
	gnome_config_set_int ("/gtop/procview/show_summary_swap", cfg_show_summary_swap);

	gnome_config_sync ();
}

void
procview_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s)
{
	/*printf ("procview handler %x %x\n", page->child, vbox);*/

	/* check if it is for me */
	if (page->child == vbox) {

		switch (s) {

		case GTOP_PAGE_CFG_READ:

			procview_cfg_read ();
			break;

		case GTOP_PAGE_CFG_SAVE:

			procview_cfg_save ();
			break;

		case GTOP_PAGE_ENTER:

			/*printf ("procview page entered\n");*/
			/* cfg_run = !cfg_run_save; */

			gtop_time_cb = procview_tbar_time;
			procview_start_stop (cfg_run);

			if (!cfg_summary) {
				gnome_proc_bar_start (GNOME_PROC_BAR (summary_cpu), 200);
				gnome_proc_bar_start (GNOME_PROC_BAR (summary_mem), 200);
			}

			gnome_app_menu_show (GNOME_APP(window),
					     GTOP_MENU_VIEW, -1);

			break;

		case GTOP_PAGE_LEAVE:

			/*printf ("procview page leaved\n");*/
			gtop_time_cb = NULL;
			procview_start_stop (0);

			if (!cfg_summary) {
				gnome_proc_bar_stop (GNOME_PROC_BAR (summary_cpu));
				gnome_proc_bar_stop (GNOME_PROC_BAR (summary_mem));
			}

			gnome_app_menu_hide (GNOME_APP(window),
					     GTOP_MENU_VIEW, -1);
			break;
		}
	}
}

static void
procview_click_column (GtkCList *cl, gint n)
{
	GtkWidget *arrow;
	GList *c = g_list_nth (field_list, n);
	gint nsf = CF - p_fields;

	/* has compare function ??*/
	if (!CF->compare)
		return;

	if (sort_field >= 0) {

		arrow = (p_fields [sort_field].order == SORT_ASC) ? sort_asc : sort_dsc;

		gtk_widget_hide (arrow);
		gtk_widget_unrealize (arrow);
		gtk_widget_hide (p_fields [sort_field].hb);
		gtk_container_remove (GTK_CONTAINER (p_fields [sort_field].hb),
				      arrow);
		gtk_widget_show (p_fields [sort_field].hb);
	}

	if (nsf == sort_field)

		CF->order = (CF->order == SORT_ASC) ?
			SORT_DSC : SORT_ASC;
	
	cfg_sf = n;
	cfg_order = CF->order;
	sort_field = nsf;
	arrow = (CF->order == SORT_ASC) ? sort_asc : sort_dsc;
	gtk_box_pack_start (GTK_BOX (CF->hb), arrow,
			    FALSE, FALSE, 2);
	gtk_widget_show (arrow);
	procview_update ();
}

static void
procview_un_select_row (GtkCList *cl, gint row, gint col, GdkEventButton * e, gpointer gp)
{
	gint i = (gint)gp;

	/* printf ("select_row: %d col: %d pid: %d\n", row, col, proc_data [row]->p->pid); */
	if (row < prev_count)
		if (!i)
			select_pid = proc_data [row]->p->pid;
		else
			select_pid = -1;

	if (e)
		if (e->button == 1 && (dwin || e->type == GDK_2BUTTON_PRESS))
			procview_open_details ();
}

static void
procview_clist_button_press (GtkCList *cl, GdkEventButton *e)
{
	/*printf ("clist button press\n");*/
	/*printf ("type: %u button: %u\n", e->type, e->button);*/

	if (e->button != 1)
		gtk_signal_emit_stop_by_name (GTK_OBJECT (cl), "button_press_event");

	if (select_pid != -1 && e->button == 3)
		gtk_menu_popup(GTK_MENU (clist_menu), NULL, NULL, NULL,
			       NULL, e->button, time(NULL));
}

static void
procview_send_signal (GtkWidget *w, gint s)
{
	if (select_pid != -1)
		kill (select_pid, s);
}

static void
procview_open_details ()
{
	if (select_pid != -1)
		procview_details (select_pid);
}

static void
renice_destroy ()
{
	renice_dialog = NULL;
}

static void
renice_cb (GtkWidget *w, gint n)
{
	if (n!=2) {
		gint  rv, nv = renice_aj->value;
		gchar *msg, tmp [256];

		/* printf ("set priotiry of process %d to %d\n", select_pid, nv); */

		rv = setpriority (PRIO_PROCESS, select_pid, nv);

		msg = NULL; /* keep gcc happy */

		if (rv == -1) {
			switch (errno) {
			case ESRCH:
				g_snprintf (tmp, 256, _("Cannot locate process %d"), select_pid);
				msg = tmp;
				break;
			case EACCES:
				msg = _("Only root can set negative priority");
				break;
			case EPERM:
				msg = _("Permission denied");
				break;
			}
			n = 1;
		} else {
			procview_update ();
			msg = "";
		}

		/* printf ("renice msg: %s\n", msg); */

		gtk_label_set (GTK_LABEL (renice_msg), msg);
	}

	if (n==2 || !n)
		gtk_widget_destroy (renice_dialog);
}

static void
procview_renice ()
{
	if (!renice_dialog) {

		GtkWidget *hs;
		GtkWidget *lb;

		renice_dialog = gnome_dialog_new (_("Renice"),
						  GNOME_STOCK_BUTTON_OK,
						  GNOME_STOCK_BUTTON_APPLY,
						  GNOME_STOCK_BUTTON_CANCEL,
						  NULL);
		renice_aj = (GtkAdjustment *) gtk_adjustment_new (getpriority (PRIO_PROCESS,
									       select_pid),
						-20, 20, 1, 1, 0);
		hs = gtk_hscale_new (renice_aj);
		gtk_scale_set_digits (GTK_SCALE (hs), 0);
		lb = gtk_label_new (_("Set priority"));
		renice_msg = gtk_label_new ("");
		gtk_misc_set_alignment (GTK_MISC (lb), 0, 1);
		gtk_misc_set_alignment (GTK_MISC (renice_msg), 0, 1);

		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (renice_dialog)->vbox),
					     lb, TRUE, TRUE, 0);
		gtk_box_pack_start_defaults (GTK_BOX (GNOME_DIALOG (renice_dialog)->vbox),
					     hs);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (renice_dialog)->vbox),
					     renice_msg, TRUE, TRUE, 0);

		gnome_dialog_button_connect (GNOME_DIALOG (renice_dialog),
					     0, (GtkSignalFunc) renice_cb, (gpointer) 0);
		gnome_dialog_button_connect (GNOME_DIALOG (renice_dialog),
					     1, (GtkSignalFunc) renice_cb, (gpointer) 1);
		gnome_dialog_button_connect (GNOME_DIALOG (renice_dialog),
					     2, (GtkSignalFunc) renice_cb, (gpointer) 2);
		gtk_signal_connect (GTK_OBJECT (renice_dialog), "destroy",
				    (GtkSignalFunc) renice_destroy, NULL);

		gtk_widget_show (lb);
		gtk_widget_show (hs);
		gtk_widget_show (renice_msg);
		gtk_widget_show (renice_dialog);
	}
}

static GtkWidget *
procview_clist_menu_prepare ()
{
	GtkWidget *menu;
	GtkWidget *smenu;
	GtkWidget *mi;
	gint i;

	menu = gtk_menu_new ();
	mi = gtk_menu_item_new_with_label (_("Send"));
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	smenu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (mi), smenu);

	for (i=0; gtop_signals [i].name; i++) {
		mi = gtk_menu_item_new_with_label (_(gtop_signals [i].name));
		gtk_signal_connect (GTK_OBJECT (mi), "activate",
				    GTK_SIGNAL_FUNC (procview_send_signal),
				    (gpointer) gtop_signals [i].sig_num);
		gtk_menu_append (GTK_MENU (smenu), mi);
		gtk_widget_show (mi);

		if (!((i+1) % 17)) {
			/* separator */
			mi = gtk_menu_item_new ();
			gtk_menu_append (GTK_MENU (smenu), mi);
			gtk_widget_show (mi);

			mi = gtk_menu_item_new_with_label (_("More ..."));
			gtk_menu_append (GTK_MENU (smenu), mi);
			gtk_widget_show (mi);

			smenu = gtk_menu_new ();
			gtk_menu_item_set_submenu (GTK_MENU_ITEM (mi), smenu);
		}
	}

	/* separator */
	mi = gtk_menu_item_new ();
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	mi = gtk_menu_item_new_with_label (_("SIGTERM"));
	gtk_signal_connect (GTK_OBJECT (mi), "activate",
			    GTK_SIGNAL_FUNC (procview_send_signal),
			    (gpointer) SIGTERM);
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	mi = gtk_menu_item_new_with_label (_("SIGKILL"));
	gtk_signal_connect (GTK_OBJECT (mi), "activate",
			    GTK_SIGNAL_FUNC (procview_send_signal),
			    (gpointer) SIGKILL);
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	/* separator */
	mi = gtk_menu_item_new ();
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	mi = gtk_menu_item_new_with_label (_("Renice ..."));
	gtk_signal_connect (GTK_OBJECT (mi), "activate",
			    GTK_SIGNAL_FUNC (procview_renice),
			    NULL);
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	mi = gtk_menu_item_new_with_label (_("Details ..."));
	gtk_signal_connect (GTK_OBJECT (mi), "activate",
			    GTK_SIGNAL_FUNC (procview_open_details),
			    NULL);
	gtk_menu_append (GTK_MENU (menu), mi);
	gtk_widget_show (mi);

	return menu;
}

static GtkWidget *
procview_clist_prepare ()
{
	GtkWidget *cl;
	gint i=0;
	GList *c = field_list;
	GList *g = geometry_list;
	gchar **text = g_new (gchar *, field_list_length);
	GtkWidget *l;
	GtkTooltips *tips;

	cl = gtk_clist_new (field_list_length);
	gtk_clist_column_titles_show (GTK_CLIST (cl));
	gtk_widget_set_name (cl, "ProcessList");

	tips = gtk_tooltips_new ();
	gtk_object_set_data (GTK_OBJECT (window), "tooltips", tips);

	while (c) {
		l = gtk_label_new (_(CF->label));
		gtk_label_set_justify (GTK_LABEL (l), CF->justification);
		gtk_widget_show (l);
		gtk_tooltips_set_tip (tips, l,
				      CF->long_info, "tip");
		CF->hb = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start_defaults (GTK_BOX (CF->hb), l);
		gtk_widget_show (CF->hb);
		gtk_clist_set_column_widget (GTK_CLIST (cl), i, CF->hb);

		i++;
		c = c->next;
	}

	clist_menu = procview_clist_menu_prepare ();

	gtk_clist_set_policy (GTK_CLIST (cl),
			      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_signal_connect (GTK_OBJECT (cl), "click_column",
			    GTK_SIGNAL_FUNC (procview_click_column), NULL);
	gtk_signal_connect (GTK_OBJECT (cl), "select_row",
			    GTK_SIGNAL_FUNC (procview_un_select_row), (gpointer) 0);
	gtk_signal_connect (GTK_OBJECT (cl), "unselect_row",
			    GTK_SIGNAL_FUNC (procview_un_select_row), (gpointer) 1);
	gtk_signal_connect (GTK_OBJECT (cl), "button_press_event",
			    GTK_SIGNAL_FUNC (procview_clist_button_press), NULL);
        

	g_free (text);

	c = field_list;
	i = 0;
	while (c) {
		gtk_clist_set_column_justification (GTK_CLIST (cl), i,
						    CF->justification);
		gtk_clist_set_column_width (GTK_CLIST (cl), i++, CG);
		c = c->next;
		g = g->next;
	}

	gtk_tooltips_set_tip (tips, cl,
			      "tip", "tip");

	return cl;
}

void
procview_init (gchar *s, gchar *g)
{
	gchar *c = g_strdup (s);
	gchar *t;
	gchar *gc = g_strdup (g);
	gchar *gt;
	gchar *cc=c;
	gchar *gcc=gc;
	gint ndx;
	gint *ip;

	/*printf ("procview init: %s\n", s);*/

	user_uid = getuid ();

	field_list_length = 0;
	do {
		t=strchr (c, ',');
		if (c && *c) {
			if (t)
				*t=0;
			ndx = atoi (c);
			/*printf ("%s %d: %s\n", s, ndx, p_fields [ndx].label);*/
			field_list = g_list_append (field_list, p_fields+ndx);
			field_list_length++;
			if (t)
				c=t+1;
		}	     

		gt=strchr (gc, ',');
		if (gc && *gc) {
			if (gt)
				*gt=0;
			ip = g_new (gint, 1);
			*ip = atoi (gc);
			/*printf ("%s %d\n", g, *ip);*/
			geometry_list = g_list_append (geometry_list, ip);
			if (gt)
				gc=gt+1;
		}	     
	} while (t && gt);

	g_free (cc);
	g_free (gcc);
}

gchar *
sprint_fmt (ProcProcData *d, p_fmt fmt)
{
	proc_t *t = d->p;
	gchar tmp[256];
	gchar *rv;

	switch (fmt) {
	case P_TIME:
	case P_UTIME:
	case P_STIME:
	case P_PID:
	case P_PRIORITY:
	case P_NICE:
	case P_RSS:
	case P_SIZE:
	case P_SHARE:
	case P_PCPU:
	case P_PMEM:
	case P_RESIDENT:
		switch (fmt) {
		case P_TIME:
			sprint_time (tmp, t->utime + t->stime);
			break;
		case P_UTIME:
			sprint_time (tmp, t->utime);
			break;
		case P_STIME:
			sprint_time (tmp, t->stime);
			break;
		case P_PID:
			sprintf (tmp, "%d", t->pid);
			break;
		case P_PRIORITY:
			sprintf (tmp, "%d", t->priority);
			break;
		case P_NICE:
			sprintf (tmp, "%d", t->nice);
			break;
		case P_RSS:
			sprintf (tmp, "%ld", t->rss << (PAGE_SHIFT - 10));
			break;
		case P_SIZE:
			sprintf (tmp, "%ld", t->size << (PAGE_SHIFT - 10));
			break;
		case P_SHARE:
			sprintf (tmp, "%ld", t->share << (PAGE_SHIFT - 10));
			break;
		case P_PCPU:
			sprintf (tmp, "%2d.%1d", d->pcpu / 10, d->pcpu % 10);
			break;
		case P_PMEM:
			sprintf (tmp, "%2d.%1d", d->pmem / 10, d->pmem % 10);
			break;
		case P_RESIDENT:
			sprintf (tmp, "%ld", t->resident << (PAGE_SHIFT - 10));
			break;
		default:
			break;
		}
		rv = tmp;
		break;
	case P_STATE:
		rv = status (t);
		break;
	case P_USER:
		rv = t->user;
		break;
	case P_CMD:
		rv = t->cmd;
		break;
	default:
		rv = "?";
	}

	return g_strdup (rv);
}

int
procview_clist_thaw (gpointer p)
{
	gtk_clist_thaw (GTK_CLIST (clist));

	return FALSE;
}

void
procview_update_clist (ProcProcData **pd)
{
	gint i=0;
	gint j, k;
	gchar **text = g_new (gchar *, field_list_length);
	ProcProcData *d;
	GList *c;
	float old_adj_hval = GTK_SCROLLBAR (GTK_CLIST (clist)->hscrollbar)
		->range.adjustment->value;
	float old_adj_vval = GTK_SCROLLBAR (GTK_CLIST (clist)->vscrollbar)
		->range.adjustment->value;
	gint old_hoffset = GTK_CLIST (clist)->hoffset;
	gint old_voffset = GTK_CLIST (clist)->voffset;
	gint row = -1;
	
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_clear (GTK_CLIST (clist));

	while ((d = pd [i])) {
		i++;
		c = field_list;
		j = 0;

		while (c) {
			text [j] = sprint_fmt (d, CF->fmt);
			j++;
			c = c->next;
		}
		gtk_clist_append (GTK_CLIST (clist), text);

		for (k=0; k<j; k++)
			g_free (text [k]);
	}

	/* keep old view position - somewhat ughly, but works now */
	GTK_SCROLLBAR (GTK_CLIST (clist)->hscrollbar)
		->range.adjustment->value = old_adj_hval;
	GTK_SCROLLBAR (GTK_CLIST (clist)->vscrollbar)
		->range.adjustment->value = old_adj_vval;

	GTK_CLIST (clist)->hoffset = old_hoffset;
	GTK_CLIST (clist)->voffset = old_voffset;

	for (i=0; i < prev_count; i++)
		if (pd [i]->p->pid == select_pid)
			row = i;
	if (row >= 0)
		gtk_clist_select_row (GTK_CLIST (clist), row, -1);

	gtk_clist_thaw (GTK_CLIST (clist));

	g_free (text);
}

gint
procview_update ()
{
	/* printf ("procview update\n");*/

	static struct save_hist save_history [NR_TASKS];
	struct save_hist new_save_hist [NR_TASKS];
	int stime, utime;

	ProcProcData *d;
	proc_t *p;
	gint i, j, n=0;

	if (proc_tab)
		freeproctab (proc_tab);
	proc_tab = readproctab (read_proc_mask, &user_uid, 1);

	p_current_time = uptime(0, 0);
	p_time = time (NULL);
	p_elapsed_time = get_elapsed_time ();
	p_gl_main_mem = read_total_main ();

	/* count number of processes */
	while (proc_tab [n])
		n++;

	if (proc_data) {
		for (i = 0; i < prev_count; i++)
			g_free (proc_data [i]);
		g_free (proc_data);
	}
	proc_data = (ProcProcData **) g_new (ProcProcData *, n+1);
	proc_data [n] = NULL;

	for (i = 0; i < n; i++) {
		/* allocate proc_data */
		d = proc_data [i] = (ProcProcData *) g_new (ProcProcData, 1);
		p = d->p = proc_tab [i];

		/* compute some own fields for proc_data */
		/* PCPU */
		p_total_time = (p->utime + p->stime +
				(p_cl_sum ? p->cutime + p->cstime : 0));

		/* history */
		new_save_hist[i].ticks = p_total_time;
		new_save_hist[i].pid = p->pid;
		stime = p->stime;
		utime = p->utime;
		new_save_hist[i].stime = stime;
		new_save_hist[i].utime = utime;

		/* find matching entry from previous pass */
		j = 0;
		while (!first && j < MIN (prev_count, NR_TASKS)) {
			if (save_history[j].pid == p->pid) {
				p_total_time -= save_history[j].ticks;
				stime -= save_history[j].stime;
				utime -= save_history[j].utime;

				j = NR_TASKS;
			}
			j++;
		}

		if (first) {
			p_elapsed_time = (p_current_time*100 -
					  p->start_time)/HZ;
			/*printf ("%d %d %f\n", HZ, p_current_time, p_elapsed_time);*/
		}

		d->pcpu = (p_elapsed_time > 0.001) ? (unsigned int)
			(((p_total_time * 10 * 100)/HZ) / p_elapsed_time) : 0;
		if (d->pcpu > 999)
			d->pcpu = 999;
		/* PMEM */
		d->pmem = p->rss * 1000 /
			(p_gl_main_mem >> PAGE_SHIFT);

	}

	/* copy the relevant info for the next pass */
	for (i = 0; i < n; i++) {
		save_history[i].pid = new_save_hist[i].pid;
		save_history[i].ticks = new_save_hist[i].ticks;
		save_history[i].stime = new_save_hist[i].stime;
		save_history[i].utime = new_save_hist[i].utime;
	}

	prev_count = n;

	if (p_fields [sort_field].compare) {
		sort_order = p_fields [sort_field].order;
		qsort (proc_data, n, sizeof (ProcProcData *),
		       (int (*)(const void *, const void *))
		       p_fields [sort_field].compare);
	}

	/* update screen*/
	procview_update_clist (proc_data);
	first = 0;

	if (dwin)
		procview_open_details ();

	return TRUE;
}
