/*
 * $Id: g_session.c,v 1.9 1998/07/30 21:04:49 jochen Exp $
 * GXSNMP -- An snmp management application
 * Copyright (C) 1998 Gregory McLean & Jochen Friedrich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Snmp session support
 */
#ifndef lint
static char const copyright[] =
"@(#) Copyright (c) 1998 Gregory McLean & Jochen Friedrich";
#endif
static char const rcsid[] =
"$Id: g_session.c,v 1.9 1998/07/30 21:04:49 jochen Exp $";

#include <g_snmp.h>
#include <sys/socket.h>
#include <sys/time.h>

extern int     errno;

static GSList  *rq_list     = NULL;   /* track the active requests */
static gint     id          = 1;      /* SNMP request id */

/*
 * lookup address of node and set port number or default.
 * Maybe this one should be nuked completely. gnome has a nice async
 * dns helper and the collector might use cached IP addresses for speedup
 * anyways.
 *
 */

gboolean
g_setup_address (host_snmp *host)
{
  if (!g_lookup_address(host->domain, host->name, &host->address))
    return FALSE;
 
  return TRUE;
}

/* 
 * query/set one mib from a snmp host
 *
 * host    -- Host info in question
 * callback-- Pointer to function that will handle the reply 
 *
 */

/* Asynchronous SNMP functions */

gpointer
g_async_send (host_snmp *host, guint req, GSList *objs, guint arg1, guint arg2)
{
  snmp_request * request;
  time_t         now;
  guint          len;

  now = time(NULL);

  request = g_malloc(sizeof(snmp_request));

  if (host->done_callback)
    request->callback = host->done_callback;
  else
    request->callback = NULL;

  if (host->time_callback)
    request->timeout  = host->time_callback;
  else
    request->timeout  = NULL;

  if (!g_setup_address(host))
    {
      g_slist_free(objs);
      g_free(request);
      return NULL;
    }

  request->pdu.request.id           = id++;
  request->pdu.request.type         = req;
  request->pdu.request.error_status = arg1;
  request->pdu.request.error_index  = arg2;
  request->pdu.request.variables    = objs;

  if (req == SNMP_PDU_SET)
    {
      request->auth=g_string_new(host->wcomm);
    }
  else
    {
      request->auth=g_string_new(host->rcomm);
    }

  request->retries                  = host->retries;
  request->timeoutval               = host->timeoutval;
  request->magic                    = host->magic;
  request->version                  = host->version;
  request->domain                   = host->domain;
  request->address                  = host->address;
  request->host                     = host;
  request->time                     = now + request->timeoutval;
  
  sendPdu(request->domain, request->address, PMODEL_SNMPV1, SMODEL_SNMPV1, 
          request->auth, SLEVEL_NANP, NULL, NULL, PDUV1, &request->pdu, TRUE);

  rq_list = g_slist_append(rq_list, request);

  return request;
}

gpointer
g_async_get (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_GET, pdu, 0, 0);
}

gpointer
g_async_getnext (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_NEXT, pdu, 0, 0);
}

gpointer
g_async_bulk (host_snmp *host, GSList *pdu, guint nonrep, guint maxiter)
{
  return g_async_send(host, SNMP_PDU_BULK, pdu, nonrep, maxiter);
}

gpointer
g_async_set (host_snmp *host, GSList *pdu)
{
  return g_async_send(host, SNMP_PDU_SET, pdu, 0, 0);
}

/* Synchronous SNMP functions */

struct inputcb {
  int sock_nr;
  void (*receiveMessage)();
};

static GSList *sockets;
static GSList *result;
static int end;

static void
cb_time(host_snmp *host, void *magic)
{
  end = 1;
  result = NULL;
}

static void
cb_done (host_snmp *host, void *magic, SNMP_PDU *spdu, GSList *objs)
{
  end = 1;
  result = objs;
}

static void
cb_register(guint socket, void (*receiveMessage)())
{
  int *socknr;
  struct inputcb *icb;
  
  icb = g_malloc(sizeof(struct inputcb));
  icb->sock_nr = socket;
  icb->receiveMessage = receiveMessage;
  sockets = g_slist_append(sockets, icb);
}

GSList *
g_sync_send (host_snmp *host, guint req, GSList *objs, guint arg1, guint arg2)
{
  struct timeval to;
  int numfds;
  fd_set fdset;
  int count;
  struct inputcb *icb;
  GSList *sck = NULL;

  sockets = NULL;
  g_snmp_reinit(cb_register);

  host->done_callback = cb_done;
  host->time_callback = cb_time;

  if (!g_async_send (host, req, objs, arg1, arg2))
    {
      g_slist_free(sockets);
      sockets = NULL;
      return NULL;
    }

  end = 0;
  while (!end) 
    {
      to.tv_sec = 1;
      to.tv_usec = 0;

      sck = sockets;
      numfds = 0;
      FD_ZERO(&fdset);

      while(sck)
        {
          icb = sck->data;
          if (numfds <= icb->sock_nr) numfds = icb->sock_nr + 1;
          FD_SET(icb->sock_nr, &fdset);
          sck = sck->next;
        }
      count = select(numfds, &fdset, 0, 0, &to);
      if (count > 0)
        {
          sck = sockets;
          while(sck)
            {
              icb = sck->data;
              if (FD_ISSET(icb->sock_nr, &fdset))
                icb->receiveMessage();
              sck = sck->next;
            }
        }
      else
        g_snmp_timeout_cb(NULL);
    }
  g_slist_free(sockets);
  sockets = NULL;
  return result;
}

GSList *
g_sync_get (host_snmp *host, GSList *pdu)
{
  return g_sync_send(host, SNMP_PDU_GET, pdu, 0, 0);
}

GSList *
g_sync_getnext (host_snmp *host, GSList *pdu)
{
  return g_sync_send(host, SNMP_PDU_NEXT, pdu, 0, 0);
}

GSList *
g_sync_bulk (host_snmp *host, GSList *pdu, guint nonrep, guint maxiter)
{
  return g_sync_send(host, SNMP_PDU_BULK, pdu, nonrep, maxiter);
}

GSList *
g_sync_set (host_snmp *host, GSList *pdu)
{
  return g_sync_send(host, SNMP_PDU_SET, pdu, 0, 0);
}

gboolean
g_pdu_add_name(GSList **pdu, guchar *name, guchar type, gpointer value)
{
  guint  id_len;
  gulong id[SNMP_SIZE_OBJECTID];
  
  id_len  = SNMP_SIZE_OBJECTID;

  if (!read_objid (name, id, &id_len))
    return FALSE;
  return g_pdu_add_oid(pdu, id, id_len, type, value);
}

gboolean
g_pdu_add_oid(GSList **pdu, gulong *myoid, guint mylength, guchar type, 
              gpointer value)
{
  struct _SNMP_OBJECT *obj;

  obj = g_malloc(sizeof(struct _SNMP_OBJECT));
  obj->id = g_malloc(mylength * sizeof(gulong));

  g_memmove(obj->id, myoid, mylength * sizeof(gulong));
  obj->request = 0;
  obj->id_len  = mylength;
  obj->type    = type;

  switch(type) 
    {
      case SNMP_INTEGER:
        obj->syntax.lng     = *((glong *) value);
        obj->syntax_len     = sizeof(glong);
        break;
      case SNMP_COUNTER:
      case SNMP_GAUGE:
      case SNMP_TIMETICKS:
        obj->syntax.ulng    = *((gulong *) value);
        obj->syntax_len     = sizeof(gulong);
        break;
      case SNMP_OCTETSTR:
      case SNMP_OPAQUE:
        obj->syntax_len    = strlen((guchar *) value);
        obj->syntax.string = strdup((guchar *) value);
        break;
      case SNMP_NULL:
        break;
      default:
        g_free(obj);
        return FALSE;
    }

  *pdu = g_slist_append(*pdu, obj);
  return TRUE;
}

/* This should be nuked once the new parser and mib module are available.
   For now, either use this or the print function in struct tree          */

void 
g_snmp_printf(char *buf, int buflen, struct _SNMP_OBJECT *obj)
{
  int timeticks, seconds, minutes, hours, days;

  /*
   * Changed all the sprintf's to snprintf, paranoid I know but
   * I'd rather not get caught with any buffer overflows..
   */
  switch(obj->type)
    {
      case SNMP_INTEGER:
        snprintf(buf, buflen, "%d", obj->syntax.lng);
        break;
      case SNMP_COUNTER:
      case SNMP_GAUGE:
        snprintf(buf, buflen, "%u", obj->syntax.ulng);
        break;
      case SNMP_TIMETICKS:
	/* replaced this duplicated code with a call to existing code */
	timetick_string (obj->syntax.ulng, buf);
        break;
      case SNMP_OCTETSTR:
      case SNMP_OPAQUE:
	/*
         * Changed to strncpy to prevent potential buffer overruns
         * as that appears to be all the rage to attempt to crash/hack/
         * crack systems. Am I being paranoid?
         */
        strncpy(buf, obj->syntax.string,
                obj->syntax_len > buflen ? buflen: obj->syntax_len);
	/*
         * BUG FIX: the returned object is not _always_ terminated
         *          with a null. This ensures that it in fact is.
         */
	buf[obj->syntax_len > buflen ? buflen: obj->syntax_len] = '\0';
        g_free(obj->syntax.string);
        break;
      case SNMP_IPADDR:
        if (obj->syntax_len == 4) /* IPv4 */
          snprintf(buf, buflen, "%d.%d.%d.%d", obj->syntax.string[0],
                                               obj->syntax.string[1],
                                               obj->syntax.string[2],
                                               obj->syntax.string[3]);
        if (obj->syntax_len == 16) /* IPv6 */
          snprintf(buf, buflen, "%x:%x:%x:%x:%x:%x:%x:%x", 
                                               256*obj->syntax.string[0]+
                                               obj->syntax.string[1],
                                               256*obj->syntax.string[2]+
                                               obj->syntax.string[3],
                                               256*obj->syntax.string[4]+
                                               obj->syntax.string[5],
                                               256*obj->syntax.string[6]+
                                               obj->syntax.string[7],
                                               256*obj->syntax.string[8]+
                                               obj->syntax.string[9],
                                               256*obj->syntax.string[10]+
                                               obj->syntax.string[11],
                                               256*obj->syntax.string[12]+
                                               obj->syntax.string[13],
                                               256*obj->syntax.string[14]+
                                               obj->syntax.string[15]);
        g_free(obj->syntax.string);
    }
}

/*
 * The request queue functions
 */

snmp_request *
g_find_request (guint reqid)
{
  GSList       * list;
  snmp_request * retval;

  list = rq_list;
  while(list)
    {
      retval = (snmp_request *) list->data;
      if(retval->pdu.request.id == reqid)
	  return retval;
      list = list->next;
    }
  return NULL;
}

gboolean
g_remove_request (snmp_request *request)
{
  rq_list = g_slist_remove(rq_list, request);
  g_free(request);
  return TRUE;
}

/*
 * The low level callbacks
 */

int
g_snmp_timeout_cb (gpointer data)
{
  GSList       *mylist;
  time_t        now;
  snmp_request *request;
  char          buf[256];

  now = time(NULL);
  mylist = rq_list;

  while(mylist)
    {
      request = (snmp_request *) mylist->data;
      mylist = mylist->next;
      if (request->time <= now)
        {
          if (request->retries)
            {
              request->retries--;
              request->time = now + request->timeoutval;
	      /* 
               * Again what happens on a -1 return to sendto
	       */
              sendPdu(request->domain, request->address, PMODEL_SNMPV1, 
                SMODEL_SNMPV1, request->auth, SLEVEL_NANP, NULL, NULL, PDUV1, 
                &request->pdu, TRUE);
            }
          else
            {
              if (request->timeout)
                {
                   request->timeout(request->host, request->magic);
                }
              g_remove_request(request);
            }
        }
    }
  return TRUE;
}

void
g_session_response_pdu (guint messageProcessingModel,
  guint securityModel, GString *securityName, guint securityLevel, 
  GString *contextEngineID, GString *contextName, guint pduVersion,
  SNMP_PDU *PDU)
{
  GSList             *objs;
  snmp_request       *request;

  if (PDU->type == SNMP_PDU_TRAP1)
    objs = PDU->trap.variables;
  else
    objs = PDU->request.variables;
  if (request = g_find_request(PDU->request.id))
    {
      if (memcmp(securityName->str, request->auth->str, securityName->len))
        {
          g_slist_free(objs);
          return;
        }
      request->host->status = PDU->request.error_status;
      if (request->callback)
        {
          request->callback(request->host, request->magic, PDU, objs);
        }
      g_remove_request(request);
    }
  g_slist_free(objs); 
}

/* EOF */

