#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <asm/page.h>
#include <string.h>
#include <proc/readproc.h>
#include <gnome.h>

#include "memusage.h"
#include "graph.h"
#include "proc.h"
#include "global.h"

#define PROC_CMD_LEN 40

struct _MemProcInfo {
	char          *cmd;
	gint           n;
	unsigned long  value;
};

typedef struct _MemProcInfo MemProcInfo;

enum _FieldType {
	RESIDENT,
	SHARED,
	SIZE
};

typedef enum _FieldType FieldType;

static void memusage_type_set (GtkWidget *, gpointer);

GnomeUIInfo memUsageMenu [] = {
  {GNOME_APP_UI_ITEM, N_("Resident"), NULL, memusage_type_set, (gpointer)RESIDENT, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Shared"), NULL, memusage_type_set, (gpointer)SHARED, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Size"), NULL, memusage_type_set, (gpointer)SIZE, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  GNOMEUIINFO_END
};

static GtkWidget *sw;
static Graph     *graph;
static GtkWidget *f;

static proc_t **proc_tab = NULL;
static MemProcInfo *memusage_data = NULL;
static unsigned long value_threshold = 1024;
static unsigned long value_total;
static unsigned long mem_total;
static unsigned long mem_kernel;
static gint cfg_run;
static gint run_tag = -1;
static FieldType ftype = RESIDENT;
static gchar *graph_head;
static gchar *graph_tail;

static void     memusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s);
static gpointer memusage_data_fn (GraphCmd cmd, gpointer data);
static gint     memusage_update ();
static void     memusage_init ();
static void     memusage_start_stop (gint start);

void *
addMemUsageView ()
{
	GtkWidget *gw;

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_border_width (GTK_CONTAINER (sw), GNOME_PAD_SMALL);
	gtk_widget_set_name (sw, "MemUsageGraph");

	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (f), GTK_SHADOW_NONE);
	gtk_container_border_width (GTK_CONTAINER (f), GNOME_PAD_SMALL << 1);
 
	graph = graph_new (memusage_data_fn);
	gw = graph_widget (graph);
	graph_colors_set (graph, graph_default_colors, GRAPH_DEFAULT_COLORS);

	gtk_container_add (GTK_CONTAINER (f), gw);
	gtk_container_add (GTK_CONTAINER (sw), f);

	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), sw,
				  gtk_label_new (_("Memory usage")));

	gtk_widget_show (gw);
	gtk_widget_show (f);
	gtk_widget_show (sw);

	memusage_init ();

	return memusage_handler;
}

static void
memusage_cfg_save ()
{
	/* printf ("proview_cfg_save\n"); */

	//gnome_config_set_int ("/gtop/procview/run", cfg_run);
	//gnome_config_set_int ("/gtop/procview/summary", !cfg_summary);
	/*gnome_config_set_int ("/gtop/procview/sort_field", cfg_sf);*/
	/*gnome_config_set_int ("/gtop/procview/sort_order", !cfg_order);*/

	gnome_config_set_int ("/gtop/memusage/field_type", ftype);
	gnome_config_set_int ("/gtop/memusage/run", cfg_run);

	gnome_config_sync ();
}

static void
memusage_time_cb (GtkWidget *w, gpointer gp)
{
	cfg_run = !cfg_run;
	memusage_start_stop (cfg_run);
}

static void
memusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s)
{
	/*printf ("procview handler %x %x\n", page->child, vbox);*/

	/* check if it is for me */
	if (page->child == sw) {

		switch (s) {

		case GTOP_PAGE_CFG_SAVE:

			memusage_cfg_save ();
			break;

		case GTOP_PAGE_ENTER:

			/*printf ("procview page entered\n");*/

			gtop_time_cb = memusage_time_cb;
			memusage_update ();
			memusage_start_stop (cfg_run);

			gnome_app_menu_show (GNOME_APP(window),
					     GTOP_MENU_MEMUSAGE, -1);

			break;

		case GTOP_PAGE_LEAVE:

			/*printf ("procview page leaved\n");*/

			gtop_time_cb = NULL;
			memusage_start_stop (0);

			gnome_app_menu_hide (GNOME_APP(window),
					     GTOP_MENU_MEMUSAGE, -1);
			break;

		default:
			break;
		}
	}
}

static void
memusage_type_set (GtkWidget *w, gpointer p)
{
	ftype = (FieldType)p;

	switch (ftype) {
	case RESIDENT:
		graph_head = _("Resident Sizes of Processes");
		graph_tail = _("Sum of Resident Sizes: %ldk");
		break;
	case SHARED:
		graph_head = _("Shared Sizes of Processes");
		graph_tail = _("Sum of Shared Sizes: %ldk");
		break;
	case SIZE:
		graph_head = _("Total Sizes of Processes");
		graph_tail = _("Sum of Total Sizes: %ldk");
		break;
	}

	memusage_update ();
}

static gpointer
memusage_data_fn (GraphCmd cmd, gpointer data)
{
	MemProcInfo *info = data;
	static gchar buf [256];
	static gchar tail [256];

	switch (cmd) {

	case GRAPH_FIRST:

		return (memusage_data) ? (memusage_data [0].cmd) ? memusage_data : NULL : NULL;

	case GRAPH_NEXT:

		return (info [1].cmd) ? info+1 : NULL;

	case GRAPH_VALUE:

		return (gpointer)(info->value);

	case GRAPH_LABEL:

		if (info->n > 1)
			sprintf (buf, "%20s (%2d): %8ldk", info->cmd, info->n, info->value);
		else
			sprintf (buf, "%20s     : %8ldk", info->cmd, info->value);

		return buf;

	case GRAPH_HEAD:

		return graph_head;

	case GRAPH_TAIL:

		sprintf (tail, graph_tail,
			 value_total >> 10);

		return tail;

	default:

		return NULL;
	}
}

static int
memusage_cmd_cmp (const void *i1,
		  const void *i2)
{
	return strcmp (((MemProcInfo *)i1)->cmd, ((MemProcInfo *)i2)->cmd);
}

static gint
memusage_update ()
{
	MemProcInfo *ti;
	ProcInfo info;
	gint n = 0, i, j, k = 0, l;
	unsigned long value;
	char *cmd;
	static gchar bts [256];

	/* printf ("memusage_update\n"); */

	/* firts: read proctab */
	if (proc_tab)
		freeproctab (proc_tab);
	proc_tab = readproctab (PROC_FILLMEM | PROC_FILLCMD);
	proc_read_mem (&info);

	/* count number of processes */
	while (proc_tab [n]) {
		n++;
	}

	/* temporary info */

	ti = g_new (MemProcInfo, n);

	value = 0; /* keep gcc happy */
	value_total = 0;

	for (i=0; i<n; i++) {
		ti [i].cmd      = proc_tab [i]->cmd;
		switch (ftype) {
		case RESIDENT:
			value = proc_tab [i]->rss;
			break;
		case SHARED:
			value = proc_tab [i]->share;
			break;
		case SIZE:
			value = proc_tab [i]->size;
			break;
		}
		ti [i].value = value << (PAGE_SHIFT - 10);
		value_total += value << PAGE_SHIFT;
	}

	/*
	 * sort info by cmd
	 * we need same processes cmd grouped
	 *
	 */

	qsort (ti, n, sizeof (MemProcInfo), memusage_cmd_cmp);

	/*
	 * compute # of processes > threshold to k
	 *
	 */

	for (i=0, k=0; i<n;) {

		value = 0;
		cmd = ti [i].cmd;

		do {
			value += ti [i].value;
			i++;

		} while (i<n && !strcmp (cmd, ti [i].cmd));

		if (value >= value_threshold)
			k++;
	}

	if (memusage_data)
		g_free (memusage_data);

	memusage_data = g_new (MemProcInfo, k+2);

	/* mem_used += mem_kernel;
	mem_used += info.mem [PROC_MEM_BUF];
	mem_free  = mem_total - mem_used; */

	/* mem_used = info.mem [PROC_MEM_USED] + mem_kernel;
	mem_free = info.mem [PROC_MEM_FREE];

	memusage_data [0].cmd = _("Free");
	memusage_data [0].value = mem_free >> 10;
	memusage_data [0].n = 1;

	sprintf (bts, "< %dk", value_threshold);
	memusage_data [k+1].cmd = bts;
	memusage_data [k+1].value = 0;
	memusage_data [k+1].n = 0;

	memusage_data [k+2].cmd = _("Buffers");
	memusage_data [k+2].value = info.mem [PROC_MEM_BUF] >> 10;
	memusage_data [k+2].n = 1;

	memusage_data [k+3].cmd = _("Kernel");
	memusage_data [k+3].value = mem_kernel >> 10;
	memusage_data [k+3].n = 1; */

	sprintf (bts, "< %ldk", value_threshold);
	memusage_data [k].cmd = bts;
	memusage_data [k].value = 0;
	memusage_data [k].n = 0;

	memusage_data [k+1].cmd = NULL;

	/* memusage_data ++; */

	/*
	 * second pass
	 *
	 *   store data and fill Below field
	 *
	 */
	
	for (i=0, j=0; i<n;) {

		value = 0;
		l   = 0;
		cmd = ti [i].cmd;

		do {
			value += ti [i].value;

			i++;
			l++;

		} while (i<n && !strcmp (cmd, ti [i].cmd));

		if (value >= value_threshold) {

			memusage_data [j].cmd = cmd;
			memusage_data [j].n = l;
			memusage_data [j].value = value;
			j++;

		} else {
			memusage_data [k].value += value;
			memusage_data [k].n ++;
		}
	}

	g_free (ti);
	
	/* memusage_data --; */

	graph_minimal_height_set (graph, sw->allocation.height - 8*GNOME_PAD_SMALL);
	graph_update (graph);

	return TRUE;
}

static void
memusage_start_stop (gint start)
{
	/* printf ("memusage_start_stop %d\n", start); */

	if (!start) {
		if (run_tag != -1) {
			gtk_timeout_remove (run_tag);
			run_tag = -1;
		}

	} else {

		if (run_tag != -1)
			return;

		memusage_update ();
		run_tag = gtk_timeout_add (3000, memusage_update, NULL);
	}

	gnome_stock_pixmap_widget_set_icon ((GnomeStockPixmapWidget *) (TBC (RUN)->icon),
					    (start) ? GNOME_STOCK_PIXMAP_TIMER :
					    GNOME_STOCK_PIXMAP_TIMER_STOP);

	gtk_widget_queue_draw (GTK_WIDGET (TBC (RUN)->widget));
}

static void
memusage_init ()
{
	struct stat st;
	ProcInfo info;
	gboolean def;

	stat ("/proc/kcore", &st);

	mem_total = st.st_size;

	proc_read_mem (&info);
	mem_kernel = mem_total - info.mem [PROC_MEM_TOTAL];

	ftype   = gnome_config_get_int_with_default ("/gtop/memusage/field_type=0", &def);
	cfg_run = gnome_config_get_int ("/gtop/memusage/run=1");

	if (def)
		ftype = RESIDENT;

	memusage_type_set (NULL, (gpointer)ftype);

	/* printf ("mem_kernel %ld\n", mem_kernel); */
}
