
/*  ----------------------------------------------------------------------

    Copyright (C) 1998  Cesar Miquel  (miquel@df.uba.ar)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    ---------------------------------------------------------------------- */


#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#include "gtk/gtk.h"
#include "logview.h"
#include "close.xpm"

#define INFO_WIDTH            350
#define INFO_HEIGHT           140

/*
 *    -------------------
 *    Function Prototypes
 *    -------------------
 */

void repaint (GtkWidget * canvas, GdkRectangle * area);
void CreateMainWin ();
void CreateMenus (GtkWidget * window, GtkWidget * box);
void log_repaint (GtkWidget * canvas, GdkRectangle * area);
void PointerMoved (GtkWidget * canvas, GdkEventMotion * event);
void HandleLogKeyboard (GtkWidget * win, GdkEventKey * event_key);
void ExitProg (GtkWidget * widget, gpointer user_data);
void LoadLogMenu (GtkWidget * widget, gpointer user_data);
void CloseLogMenu (GtkWidget * widget, gpointer user_data);
void CalendarMenu (GtkWidget * widget, gpointer user_data);
void MonitorMenu (GtkWidget* widget, gpointer user_data); 
void AboutShowWindow (GtkWidget* widget, gpointer user_data);
void AboutHideWindow ();
void LogInfo (GtkWidget * widget, gpointer user_data);
void CloseApp ();
void CloseLog (Log *);
void ShowErrMessage (char *);
void CloseLogInfo ();
GtkWidget *new_pixmap_from_data (char  **, GdkWindow *, GdkColor *);
int InitApp ();
int InitPages ();
int RepaintLogInfo (GtkWidget * widget, GdkEventExpose * event);
GtkWidget *create_menu (char *item[], int n);

/*
 *    ,-------.
 *    | Menus |
 *    `-------'
 */

MenuItem file_menu[] =
{
   {"Open log..             ", 0, LoadLogMenu, NULL},
   {"Export log..           ", 0, StubCall, NULL},
   {"Close log              ", 0, CloseLogMenu, NULL},
   {"Switch log             ", 0, StubCall, NULL},
   {"Monitor..              ", 0, MonitorMenu, NULL},
   {"Exit                   ", 0, ExitProg, NULL},
   {NULL, 0, NULL, NULL},
};

MenuItem view_menu[] =
{
   {"Calendar              ", 0, CalendarMenu, NULL},
   {"Log stats             ", 0, LogInfo, NULL},
   {"Zoom                  ", 0, StubCall, NULL},
   {NULL, 0, NULL, NULL}
};

MenuItem filter_menu[] =
{
   {"Select..                     ", 0, StubCall, NULL},
   {"Find..                       ", 0, StubCall, NULL},
   {NULL, 0, NULL, NULL},
};

MenuItem help_menu[] =
{
   {"About                  ", 0, AboutShowWindow, NULL},
   {NULL, 0, NULL, NULL},
};

/*
 *       ----------------
 *       Global variables
 *       ----------------
 */

ConfigData *cfg;
int loginfovisible;
GtkWidget *InfoDialog;

extern GdkGC *gc;
extern Log *curlog, *loglist[];
extern int numlogs, curlognum;


static gint init_timer = -1;

void
destroy (void)
{
   gtk_exit (0);
}

int
main (int argc, char *argv[])
{
   /*  Initialize gtk */
   gtk_init (&argc, &argv);

   /*  Load graphics config */
   cfg = CreateConfig();
  
   /*  Show about window */
   AboutShowWindow (NULL, NULL);

   /*  Add a timer that will start initialization after */
   /*  after the title window appears. */
   init_timer = gtk_timeout_add (1000, (GtkFunction) InitApp, NULL);

   /*  Loop application */
   gtk_main ();

   return 0;
}

/* ----------------------------------------------------------------------
   NAME:        InitApp
   DESCRIPTION: Main initialization routine.
   ---------------------------------------------------------------------- */

int
InitApp ()
{
  /*  Initialize variables */
  loginfovisible = FALSE;

  /*  Read files and init data. */
  if (InitPages () < 0)
    ShowErrMessage ("No log files to open");

  /*  Remove about window */
  AboutHideWindow ();

  /*  Display main window */
  CreateMainWin ();

  /*  Remove timer by returning FALSE */
  return FALSE;

}

/* ----------------------------------------------------------------------
   NAME:        CreateMainWin
   DESCRIPTION: Creates the main window.
   ---------------------------------------------------------------------- */

void
CreateMainWin ()
{
   GtkWidget *window;
   GtkWidget *canvas;
   GtkWidget *box;
   GtkWidget *frame;

   window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title (GTK_WINDOW (window), "logview");
   gtk_signal_connect (GTK_OBJECT (window), "destroy",
		       GTK_SIGNAL_FUNC (destroy), NULL);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (destroy), NULL);
   gtk_container_border_width (GTK_CONTAINER (window), 0);
   gtk_widget_set_style (window, cfg->main_style);


   box = gtk_vbox_new (FALSE, 0);
   gtk_container_add (GTK_CONTAINER (window), box);
   gtk_widget_show (box);

   CreateMenus (window, box);

   frame = gtk_frame_new (NULL);
   gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
   gtk_container_border_width (GTK_CONTAINER (frame), 3);
   gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 0);
   gtk_widget_set_style (frame, cfg->main_style);
   gtk_widget_show (frame);


   canvas = gtk_drawing_area_new ();
   gtk_drawing_area_size (GTK_DRAWING_AREA (canvas), LOG_CANVAS_W, LOG_CANVAS_H);
   gtk_container_add (GTK_CONTAINER (frame), canvas);

   /*  Install event handlers */
   gtk_signal_connect (GTK_OBJECT (canvas), "expose_event",
		       GTK_SIGNAL_FUNC (log_repaint), NULL);
   gtk_signal_connect (GTK_OBJECT (canvas), "motion_notify_event",
		       GTK_SIGNAL_FUNC (PointerMoved), NULL);
   gtk_signal_connect (GTK_OBJECT (canvas), "key_press_event",
		       GTK_SIGNAL_FUNC (HandleLogKeyboard), NULL);
   gtk_widget_set_events (canvas, GDK_EXPOSURE_MASK |
			  GDK_POINTER_MOTION_MASK |
			  GDK_KEY_PRESS_MASK);

   gtk_widget_set_style (canvas, cfg->white_bg_style);
   gtk_widget_show (canvas);

   gtk_widget_show (window);

}

/* ----------------------------------------------------------------------
   NAME:        CreateMenus
   DESCRIPTION: Make main menus.
   ---------------------------------------------------------------------- */

void
CreateMenus (GtkWidget * window, GtkWidget * box)
{
   MenuItem main_menu[] =
   {
      {"File", 0, NULL, file_menu},
      {"View", 0, NULL, view_menu},
      {"Filter", 0, NULL, filter_menu},
      {"Help", 0, NULL, help_menu},
      {NULL, 0, NULL, NULL}};
   GtkWidget *menu;
   GtkWidget *menubar;
   GtkWidget *menuitem;
   int i;

   menubar = gtk_menu_bar_new ();
   gtk_box_pack_start (GTK_BOX (box), menubar, FALSE, TRUE, 0);
   gtk_widget_set_style (menubar, cfg->main_style);
   gtk_widget_show (menubar);

   i = 0;
   while (main_menu[i].submenu != NULL)
   {
      menu = AddMenu (main_menu[i].submenu);
      menuitem = gtk_menu_item_new_with_label (main_menu[i].name);
      if (strcmp("Help", main_menu[i].name) == 0)
	gtk_menu_item_right_justify(GTK_MENU_ITEM(menuitem));
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);
      gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
      gtk_widget_set_style (menuitem, cfg->main_style);
      gtk_widget_show (menuitem);
      i++;
   }

   

}

/* ----------------------------------------------------------------------
   NAME:          LogInfo
   DESCRIPTION:   Display the statistics of the log.
   ---------------------------------------------------------------------- */

void
LogInfo (GtkWidget * widget, gpointer user_data)
{
   GtkWidget *infoarea;
   GtkWidget *frame;
   GtkWidget *vbox;
   GtkWidget *toolbar;
   GtkWidget *icon;
   int h1, h2;

   if (curlog == NULL || loginfovisible)
      return;

   InfoDialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_signal_connect (GTK_OBJECT (InfoDialog), "destroy",
		       (GtkSignalFunc) CloseLogInfo,
		       &InfoDialog);
   gtk_signal_connect (GTK_OBJECT (InfoDialog), "delete_event",
		       (GtkSignalFunc) CloseLogInfo,
		       &InfoDialog);
   gtk_window_set_title (GTK_WINDOW (InfoDialog), "Log stats");
   gtk_container_border_width (GTK_CONTAINER (InfoDialog), 0);
   gtk_widget_set_style (InfoDialog, cfg->main_style);
   gtk_widget_realize (InfoDialog);

   vbox = gtk_vbox_new (FALSE, 2);
   gtk_container_border_width (GTK_CONTAINER (vbox), 4);
   gtk_container_add (GTK_CONTAINER (InfoDialog), vbox);
   gtk_widget_show (vbox);

   /* Create toolbar */
   toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
			      GTK_TOOLBAR_ICONS);
   icon = new_pixmap_from_data (closexpm, InfoDialog->window, 
				&InfoDialog->style->bg[GTK_STATE_NORMAL]);
   gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
			    "close", "Hide log stats", NULL,
			    icon,
			    (GtkSignalFunc) CloseLogInfo, 
			    toolbar);

   gtk_box_pack_start (GTK_BOX (vbox), toolbar, TRUE, TRUE, 0);
   gtk_widget_show (toolbar);

   frame = gtk_frame_new (NULL);
   gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
   gtk_container_border_width (GTK_CONTAINER (frame), 3);
   gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
   gtk_widget_set_style (frame, cfg->main_style);
   gtk_widget_show (frame);

   infoarea = gtk_drawing_area_new ();
   gtk_signal_connect (GTK_OBJECT (infoarea), "expose_event",
		       (GtkSignalFunc) RepaintLogInfo, NULL);

   h1 = cfg->headingb->descent + cfg->headingb->ascent;
   h2 = cfg->fixedb->descent + cfg->fixedb->ascent+2;
   gtk_drawing_area_size (GTK_DRAWING_AREA (infoarea), INFO_WIDTH, h1*2+8*h2);
   gtk_widget_set_events (infoarea, GDK_EXPOSURE_MASK);
   gtk_widget_set_style (infoarea, cfg->white_bg_style);
   gtk_container_add (GTK_CONTAINER (frame), infoarea);
   gtk_widget_show (infoarea);
   gtk_widget_realize (infoarea);

   gtk_widget_show (InfoDialog);

   loginfovisible = TRUE;
}

int
RepaintLogInfo (GtkWidget * widget, GdkEventExpose * event)
{
   static GdkDrawable *canvas;
   char buffer[255];
   int x, y, h, w;

   if (widget != NULL)
      canvas = widget->window;

   /* Erase page */
   gdk_window_clear (canvas);

   /* Draw title */
   h = cfg->headingb->descent + cfg->headingb->ascent;
   w = gdk_string_measure (cfg->fixedb, "X");
   x = 2*w;
   y = h*3/2;
   gdk_gc_set_foreground (gc, &cfg->green);
   gdk_draw_string (canvas, cfg->headingb, gc, x, y, "Log Information");
   gdk_draw_line (canvas, gc, x, y + 3, INFO_WIDTH - x, y + 3);
   y += 2 * h;


   /* Draw Info */
   gdk_gc_set_foreground (gc, &cfg->black);

   h = cfg->fixedb->descent + cfg->fixedb->ascent+2;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Log:");
   y += h;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Size:");
   y += h;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Modified:");
   y += h;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Start date:");
   y += h;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Last date:");
   y += h;
   gdk_draw_string (canvas, cfg->fixedb, gc, x, y, "Num. lines:");

   gdk_gc_set_font (gc, cfg->fixed);

   /* Check that there is at least one log */
   if (curlog == NULL)
      return -1;

   h = cfg->headingb->descent + cfg->headingb->ascent;
   y = h*3/2;
   y += 2*h;
   x = 15*w;
   sprintf (buffer, "%s", curlog->name);
   h = cfg->fixedb->descent + cfg->fixedb->ascent+2;
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);
   y += h;
   sprintf (buffer, "%ld bytes", (long) curlog->lstats.size);
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);
   y += h;
   sprintf (buffer, "%s ", ctime (&curlog->lstats.mtime));
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);
   y += h;
   sprintf (buffer, "%s ", ctime (&curlog->lstats.startdate));
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);
   y += h;
   sprintf (buffer, "%s ", ctime (&curlog->lstats.enddate));
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);
   y += h;
   sprintf (buffer, "%ld ", curlog->lstats.numlines);
   gdk_draw_string (canvas, cfg->fixed, gc, x, y, buffer);

   return TRUE;
}

/* ----------------------------------------------------------------------
   NAME:          CloseLogInfo
   DESCRIPTION:   Callback called when the log info window is closed.
   ---------------------------------------------------------------------- */

void
CloseLogInfo (GtkWidget * widget, GtkWindow ** window)
{
   if (loginfovisible)
      gtk_widget_hide (InfoDialog);
   InfoDialog = NULL;
   loginfovisible = FALSE;
}



/* ----------------------------------------------------------------------
   NAME:          CloseLogMenu
   DESCRIPTION:   Close the current log.
   ---------------------------------------------------------------------- */

void
CloseLogMenu (GtkWidget * widget, gpointer user_data)
{
   int i;

   if (numlogs == 0)
      return;

   CloseLog (curlog);
   numlogs--;
   if (numlogs == 0)
   {
      curlog = NULL;
      loglist[0] = NULL;
      curlognum = 0;
      log_repaint (NULL, NULL);
      if (loginfovisible)
	 RepaintLogInfo (NULL, NULL);
      return;
   }
   for (i = curlognum; i < numlogs; i++)
      loglist[i] = loglist[i + 1];
   loglist[i] = NULL;

   if (curlognum > 0)
      curlognum--;
   curlog = loglist[curlognum];
   log_repaint (NULL, NULL);

   if (loginfovisible)
      RepaintLogInfo (NULL, NULL);
}


/* ----------------------------------------------------------------------
   NAME:          ExitProg
   DESCRIPTION:   Callback to call when program exits.
   ---------------------------------------------------------------------- */

void 
ExitProg (GtkWidget * widget, gpointer user_data)
{
   CloseApp ();
}

/* ----------------------------------------------------------------------
   NAME:          CloseApp
   DESCRIPTION:   Close everything and exit.
   ---------------------------------------------------------------------- */

void 
CloseApp ()
{
   int i;

   for (i = 0; i < numlogs; i++)
      CloseLog (loglist[i]);

   gtk_exit (0);
}
