//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.typechecker.annotations.builtin;

import static org.eclipse.escet.common.java.Strings.fmt;

import org.eclipse.escet.cif.common.CifAnnotationUtils;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProblemReporter;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProvider;
import org.eclipse.escet.common.typechecker.SemanticProblemSeverity;

/**
 * Annotation provider for "partial" annotations.
 *
 * <p>
 * A "partial" annotation can be added to a specification to indicate that it is a partial specification, such as a
 * library, meant to be included in other specifications. The annotation is used by the CIF type checker to disable
 * certain types of warnings that may give a lot of false positives for partial specifications.
 * </p>
 *
 * <p>
 * Since the "partial" annotation is used by the CIF type checker, even before annotations have been type checked, it is
 * treated as a special case by the CIF type checker. The behavior of the "partial" annotation also differs from other
 * annotations in relation to imports. A "partial" annotation from an imported specification is not merged into the
 * importing specifications as with other annotations; that is, the "partial" specification applies only to the specific
 * file in which it is specified.
 * </p>
 */
public class PartialSpecAnnotationProvider extends AnnotationProvider {
    /** The name of the annotations handled by this annotation provider. */
    private final String annotationName;

    /**
     * Constructor for the {@link ControllerPropertiesAnnotationProvider} class.
     *
     * @param annotationName The name of the annotations handled by this annotation provider.
     */
    public PartialSpecAnnotationProvider(String annotationName) {
        this.annotationName = annotationName;
    }

    @Override
    public final void checkAnnotation(AnnotatedObject annotatedObject, Annotation annotation,
            AnnotationProblemReporter reporter)
    {
        // May only be used on specifications.
        if (!(annotatedObject instanceof Specification)) {
            reporter.reportProblem(annotation, "annotation may only be used on specifications.",
                    annotation.getPosition(), SemanticProblemSeverity.ERROR);
            // Non-fatal problem.
        }

        // Must not have any arguments.
        if (!annotation.getArguments().isEmpty()) {
            reporter.reportProblem(annotation, "annotation has an argument.", annotation.getPosition(),
                    SemanticProblemSeverity.ERROR);
            // Non-fatal problem.
        }
    }

    @Override
    public final void checkGlobal(Specification spec, AnnotationProblemReporter reporter) {
        // Warn about duplicates.
        long count = CifAnnotationUtils.getAnnotations(spec, annotationName).count();
        if (count > 1) {
            reporter.reportProblem(annotationName,
                    fmt("the specification has more than one \"%s\" annotation, namely %d.", annotationName, count),
                    spec.getPosition(), SemanticProblemSeverity.WARNING);
            // Non-fatal problem.
        }
    }
}
