/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* lib_sox.c: Interface between spettro and libsox */

#include "spettro.h"
#include "lib_sox.h"

#if HAVE_LIBSOX

#include <string.h>	/* for strerror() */
#include <errno.h>
#include <sox.h>

/* Open a file, setting af->{sample_rate,channels,frames}
 * On failure, returns FALSE.
 */
bool
libsox_open(audio_file_t *af, char *filename)
{
    static bool initialized = FALSE;

    if (!initialized) {
	if (sox_format_init() != 0) return FALSE;
	initialized = TRUE;
    }

    af->sox_format = sox_open_read(filename, NULL, NULL, NULL);
    if (af->sox_format == NULL) goto fail;

    af->filename = filename;
    af->channels = af->sox_format->signal.channels;
    af->sample_rate = af->sox_format->signal.rate;
    switch (af->sox_format->signal.length) {
	/* "0 if unknown, -1 if unspecified" */
    case 0:
	/* Report a 0-length file */
	break;
    case -1:
	fprintf(stdout, "libsox says the length of %s is unspecified.\n",
		filename);
	goto fail;
    default:
	if (af->sox_format->signal.length < 0) {
	    fprintf(stdout, "libsox says the length of %s is negative.\n",
		    filename);
	    goto fail;
	}
	break;
    }
    af->frames = af->sox_format->signal.length;

    return TRUE;
fail:
    sox_format_quit();
    initialized = FALSE;
    return FALSE;
}

/* Seek to the "start"th sample from the MP3 file.
 * Returns TRUE on success, FALSE on failure.
 */
bool
libsox_seek(audio_file_t *af, int start)
{
    int errno;

    errno = sox_seek(af->sox_format, (sox_uint64_t)start, SOX_SEEK_SET);
    if (errno != SOX_SUCCESS) {
	fprintf(stdout, "Failed to seek in audio file: %s\n", af->sox_format->sox_errstr);
	return FALSE;
    }

    return TRUE;
}

/*
 * Read samples from the audio file and convert them to the desired format
 * into the buffer "write_to".
 *
 * Returns the number of frames written, or a negative value on errors.
 */
int
libsox_read_frames(audio_file_t *af,
		   void *write_to,
		   frames_t frames_to_read,
		   af_format_t format)
{
    sox_uint64_t samples_to_read;	/* Number of samples to read, not frames */
    sox_uint64_t samples_read;	/* Number of samples returned from the file */
    int frames_read;		/* Number of frames returned from the file */
    sox_sample_t *soxbuf;	/* A buffer for sox's 32-bit signed samples */

    samples_to_read = frames_to_read * af->channels;

    soxbuf = Calloc(samples_to_read, sizeof(sox_sample_t));

    samples_read = sox_read(af->sox_format, soxbuf, samples_to_read);

    /* sox_read does not distinguish between EOF and errors.
     * We shouldn't ask it to read past the end of the file
     * so treat both as errors. */
    if (samples_read == 0) {
        fprintf(stdout, "sox_read returned 0; sox_errno=%d\n", af->sox_format->sox_errno);
	goto fail;
    }

    frames_read = samples_read / af->channels;

    switch (format) {
	short *sp;
	float *fp;
	sox_sample_t *soxp;
	sox_uint64_t i;
    case af_signed:
	soxp = soxbuf; sp = (short *)write_to;
	for (i=samples_read; i != 0; i--) *sp++ = (*soxp++) >> 16;
	break;
    case af_float:  
	soxp = soxbuf; fp = (float *)write_to;
	if (af->channels == 1) {
	    for (i=samples_read; i != 0; i--) {
		*fp++ = (*soxp++ / (float)(1<<31));
	    }
	} else {
	    /* Convert the frames to mono */
	    for (i=samples_read; i != 0; i--) {
		float total = 0.0;
		int chan;
		for (chan=0; chan < af->channels; chan++) {
		    total += (float) (*soxp++) / (float)(1<<31);
		}
		*fp++ = total / af->channels;
	    }
	}
	break;
    }

    free(soxbuf);
    return frames_read;

fail:
    free(soxbuf);
    return -1;
}

void
libsox_close(audio_file_t *af)
{
    sox_close(af->sox_format);
}
#endif /* HAVE_LIBSOX */
