/*
 * SPDX-License-Identifier: MIT
 * Copyright (C) 2004 - 2022 AJA Video Systems, Inc.
 */
///////////////////////////////////////////////////////////////
//
// HEVC Linux Device Driver for AJA HEVC boards.
//
////////////////////////////////////////////////////////////
//
// Filename: hevcconstants.h
// Purpose:	 Constant definition for MB86M31.
// Notes:	 
//
///////////////////////////////////////////////////////////////

#ifndef HEVCCONSTANTS_H
#define HEVCCONSTANTS_H

// constants from example dirver

#define SIZEOF_32BIT							4

#define FMB_VERSION_SIZE_MCPU_FIRM				4							// dwords
#define FMB_VERSION_SIZE_HEVC_ENCODER_FIRM		(15 * SIZEOF_32BIT)			// bytes
#define FMB_VERSION_SIZE_SYSTEM_FIRM			( 5 * SIZEOF_32BIT)			// bytes

#define FMB_PCI_VENDOR_ID 			0x10CF
#define FMB_PCI_DEVICE_ID 			0x2049

#define FMB_PCI_BAR0_ADDR	0xF1E00000U
#define FMB_PCI_BAR2_ADDR	0x08000000U
#define FMB_PCI_BAR4_ADDR	0x20000000U
#define FMB_PCI_BAR5_ADDR	0xF0000000U

#define FMB_PCI_BAR0_SIZE	0x00010000U
#define FMB_PCI_BAR2_SIZE	0x00008000U
#define FMB_PCI_BAR4_SIZE	0x00200000U
#define FMB_PCI_BAR5_SIZE	0x00020000U

#define FMB_PCI_BAR0_OFFSET_ADDR_MASK	0x0000FFFFU		//  64KB (16bit)
#define FMB_PCI_BAR2_OFFSET_ADDR_MASK	0x00007FFFU		//  32KB (15bit)
#define FMB_PCI_BAR4_OFFSET_ADDR_MASK	0x001FFFFFU		//   2MB (21bit)
#define FMB_PCI_BAR5_OFFSET_ADDR_MASK	0x0001FFFFU		// 128KB (17bit)

#define FMB_PCI_REG_AXI_MASTER_REMAP_ADDR_OFFSET	0x000008B4U

#define FMB_PCI_POLL_COUNT_REG_WRITE_COMPLETE	1000

#define FMB_CODEC_STATE_RESET_POLLING_TIME		1000  	// 1 ms
#define FMB_CODEC_STATE_RESET_POLLING_COUNT		3000	// 3000 times

// bar type
enum fmb_pci_bar_type 
{
	FMB_PCI_BAR0 = 0,
	FMB_PCI_BAR2 = 2,
	FMB_PCI_BAR4 = 4,
	FMB_PCI_BAR5 = 5
};

// continuity counter type
enum fmb_pci_continuity_cnt_type 
{
	FMB_PCI_CONT_COMMAND,
	FMB_PCI_CONT_DMA_VEI,
	FMB_PCI_CONT_DMA_SEO,
	FMB_PCI_CONT_TYPE_MAX
};

// interrupt factor id
enum fmb_factor_id 
{
	FMB_FACT_ID_CMD_ACK = 0,
	FMB_FACT_ID_CMD_RESULT = 1,

	FMB_FACT_ID_MESSAGE = 2,

	FMB_FACT_ID_DMA_VEI_ACK = 16,
	FMB_FACT_ID_DMA_VEI_COMPLETE = 17,

	FMB_FACT_ID_DMA_SEO_ACK = 18,
	FMB_FACT_ID_DMA_SEO_COMPLETE = 19,

	FMB_FACT_ID_FATAL_ERROR = 31,

	FMB_FACT_IRQ_MAX = 32
};

// codec stream types
enum fmb_stream_type
{
	FMB_STREAM_TYPE_VEI,
	FMB_STREAM_TYPE_SEO,
	FMB_STREAM_TYPE_MAX
};

#define FMB_SET_ID_BIT(id_bits, id)	\
( (id_bits) = ( (id_bits) | (0x00000001U << (id)) ) )

// codec registers

#define FMB_REG_FATAL_ERROR_BASE				0x08000100U
#define FMB_REG_FATAL_ERROR_NUM					64

#define FMB_REG_MAIN_STATE						0x08000200U
#define FMB_REG_VIN_STATE_BASE					0x08000220U
#define FMB_REG_EH_STATE_BASE					0x080002A0U

#define FMB_REG_VIN_STATE( n )	\
		(FMB_REG_VIN_STATE_BASE + ( 4 * n ) )
#define FMB_REG_EH_STATE( n )	\
		(FMB_REG_EH_STATE_BASE + ( 4 * n ) )

#define FMB_REG_VERSION_BASE						0x08000000U
#define FMB_REG_VERSION_MCPU						(FMB_REG_VERSION_BASE + 0x00000000U)
#define FMB_REG_VERSION_HEVC_ENCODER_STD_SINGLE		(FMB_REG_VERSION_BASE + 0x00000010U)
#define FMB_REG_VERSION_HEVC_ENCODER_STD_MULTI		(FMB_REG_VERSION_BASE + 0x0000004CU)
#define FMB_REG_VERSION_SYSTEM						(FMB_REG_VERSION_BASE + 0x000000ECU)

#define FMB_REG_USER_FIRM_VERSION_BASE				0x08006D00U
#define FMB_REG_VERSION_HEVC_ENCODER_USER_SINGLE	(FMB_REG_USER_FIRM_VERSION_BASE + 0x00000000U)
#define FMB_REG_VERSION_HEVC_ENCODER_USER_MULTI		(FMB_REG_USER_FIRM_VERSION_BASE + 0x0000003CU)

#define FMB_REG_COMMAND_BASE					0x08000400U
#define FMB_REG_COMMAND_CONTINUITY_CNT			(FMB_REG_COMMAND_BASE + 0x00000000U)
#define FMB_REG_COMMAND_TARGET					(FMB_REG_COMMAND_BASE + 0x00000004U)
#define FMB_REG_COMMAND_ID						(FMB_REG_COMMAND_BASE + 0x00000008U)
#define FMB_REG_COMMAND_PARAM_BASE				(FMB_REG_COMMAND_BASE + 0x0000000CU)
#define FMB_COMMAND_PARAM_MAX					17

#define FMB_REG_COMMAND_ACK_BASE				0x08000800U
#define FMB_REG_COMMAND_ACK_CONTINUITY_CNT		(FMB_REG_COMMAND_ACK_BASE + 0x00000000U)
#define FMB_REG_COMMAND_ACK_TARGET				(FMB_REG_COMMAND_ACK_BASE + 0x00000004U)
#define FMB_REG_COMMAND_ACK_ID					(FMB_REG_COMMAND_ACK_BASE + 0x00000008U)
#define FMB_REG_COMMAND_ACK_RESULT				(FMB_REG_COMMAND_ACK_BASE + 0x0000000CU)

#define FMB_REG_COMMAND_RESULT_BASE				0x08000820U
#define FMB_REG_COMMAND_RESULT_CONTINUITY_CNT	(FMB_REG_COMMAND_RESULT_BASE + 0x00000000U)
#define FMB_REG_COMMAND_RESULT_TARGET			(FMB_REG_COMMAND_RESULT_BASE + 0x00000004U)
#define FMB_REG_COMMAND_RESULT_ID				(FMB_REG_COMMAND_RESULT_BASE + 0x00000008U)
#define FMB_REG_COMMAND_RESULT_PARAM_BASE		(FMB_REG_COMMAND_RESULT_BASE + 0x0000000CU)
#define FMB_REG_COMMAND_RESULT_PARAM_MAX		18

#define FMB_REG_MESSAGE_RESULT_BASE				0x08000C20U
#define FMB_REG_MESSAGE_RESULT_TARGET			(FMB_REG_MESSAGE_RESULT_BASE + 0x00000000U)
#define FMB_REG_MESSAGE_RESULT_ID				(FMB_REG_MESSAGE_RESULT_BASE + 0x00000004U)
#define FMB_REG_MESSAGE_RESULT_PARAM_BASE		(FMB_REG_MESSAGE_RESULT_BASE + 0x00000008U)
#define FMB_REG_MESSAGE_RESULT_PARAM_MAX		((0x0000010CU / SIZEOF_32BIT) - 2)

#define FMB_REG_DMA_VEI_BASE					0x08001200U
#define FMB_REG_DMA_VEI_CONTINUITY_CNT			(FMB_REG_DMA_VEI_BASE + 0x00000000U)
#define FMB_REG_DMA_VEI_DATA_ID					(FMB_REG_DMA_VEI_BASE + 0x00000004U)
#define FMB_REG_DMA_VEI_NUMBER_OF_DESCRIPTORS	(FMB_REG_DMA_VEI_BASE + 0x00000008U)
#define FMB_REG_DMA_VEI_TOTAL_SIZE				(FMB_REG_DMA_VEI_BASE + 0x0000000CU)
#define FMB_REG_DMA_VEI_LAST_FRAME_MARKER		(FMB_REG_DMA_VEI_BASE + 0x00000010U)
#define FMB_REG_DMA_VEI_PICTURE_INFO_ADDRESS_L	(FMB_REG_DMA_VEI_BASE + 0x00000014U)
#define FMB_REG_DMA_VEI_PICTURE_INFO_ADDRESS_H	(FMB_REG_DMA_VEI_BASE + 0x00000018U)
#define FMB_REG_DMA_VEI_PICTURE_INFO_SIZE		(FMB_REG_DMA_VEI_BASE + 0x0000001CU)

#define FMB_REG_DMA_VEI_NOT_LAST_FRAME			0x00000000U
#define FMB_REG_DMA_VEI_LAST_FRAME				0xFFFFFFFFU

#define FMB_REG_DMA_VEI_DESCRIPTOR_BASE			0x08001360U
#define FMB_REG_DMA_VEI_DESCRIPTOR_NUM			512
#define FMB_REG_DMA_VEI_DESCRIPTOR_SIZE			0x0000000CU /* byte */

#define FMB_REG_DMA_VEI_DESCRIPTOR_ADDRESS_L( n )	\
	(FMB_REG_DMA_VEI_DESCRIPTOR_BASE + ( FMB_REG_DMA_VEI_DESCRIPTOR_SIZE * (n) ) + 0x00000000U )
#define FMB_REG_DMA_VEI_DESCRIPTOR_ADDRESS_H( n )	\
	(FMB_REG_DMA_VEI_DESCRIPTOR_BASE + ( FMB_REG_DMA_VEI_DESCRIPTOR_SIZE * (n) ) + 0x00000004U )
#define FMB_REG_DMA_VEI_DESCRIPTOR_DMA_SIZE( n )	\
	(FMB_REG_DMA_VEI_DESCRIPTOR_BASE + ( FMB_REG_DMA_VEI_DESCRIPTOR_SIZE * (n) ) + 0x00000008U )

#define FMB_REG_DMA_VEI_ACK_BASE				0x08001220U
#define FMB_REG_DMA_VEI_ACK_CONTINUITY_CNT		(FMB_REG_DMA_VEI_ACK_BASE + 0x00000000U)
#define FMB_REG_DMA_VEI_ACK_DATA_ID				(FMB_REG_DMA_VEI_ACK_BASE + 0x00000004U)
#define FMB_REG_DMA_VEI_ACK_RESULT				(FMB_REG_DMA_VEI_ACK_BASE + 0x00000008U)

#define FMB_REG_DMA_VEI_COMPLETE_BASE				0x08001240U
#define FMB_REG_DMA_VEI_COMPLETE_CONTINUITY_CNT		(FMB_REG_DMA_VEI_COMPLETE_BASE + 0x00000000U)
#define FMB_REG_DMA_VEI_COMPLETE_DATA_ID			(FMB_REG_DMA_VEI_COMPLETE_BASE + 0x00000004U)
#define FMB_REG_DMA_VEI_COMPLETE_RESULT				(FMB_REG_DMA_VEI_COMPLETE_BASE + 0x00000008U)
#define FMB_REG_DMA_VEI_COMPLETE_TOTAL_SIZE			(FMB_REG_DMA_VEI_COMPLETE_BASE + 0x0000000CU)

#define FMB_REG_DMA_SEO_BASE					0x08004200U
#define FMB_REG_DMA_SEO_CONTINUITY_CNT			(FMB_REG_DMA_SEO_BASE + 0x00000000U)
#define FMB_REG_DMA_SEO_DATA_ID					(FMB_REG_DMA_SEO_BASE + 0x00000004U)
#define FMB_REG_DMA_SEO_NUMBER_OF_DESCRIPTORS	(FMB_REG_DMA_SEO_BASE + 0x00000008U)
#define FMB_REG_DMA_SEO_TOTAL_SIZE				(FMB_REG_DMA_SEO_BASE + 0x0000000CU)
#define FMB_REG_DMA_SEO_ES_INFO_ADDRESS_L		(FMB_REG_DMA_SEO_BASE + 0x00000010U)
#define FMB_REG_DMA_SEO_ES_INFO_ADDRESS_H		(FMB_REG_DMA_SEO_BASE + 0x00000014U)
#define FMB_REG_DMA_SEO_ES_INFO_SIZE			(FMB_REG_DMA_SEO_BASE + 0x00000018U)

#define FMB_REG_DMA_SEO_DESCRIPTOR_BASE			0x08004360U
#define FMB_REG_DMA_SEO_DESCRIPTOR_NUM			256
#define FMB_REG_DMA_SEO_DESCRIPTOR_SIZE			0x0000000CU 	// bytes

#define FMB_REG_DMA_SEO_DESCRIPTOR_ADDRESS_L( n )	\
	(FMB_REG_DMA_SEO_DESCRIPTOR_BASE + ( FMB_REG_DMA_SEO_DESCRIPTOR_SIZE * (n) ) + 0x00000000U )
#define FMB_REG_DMA_SEO_DESCRIPTOR_ADDRESS_H( n )	\
	(FMB_REG_DMA_SEO_DESCRIPTOR_BASE + ( FMB_REG_DMA_SEO_DESCRIPTOR_SIZE * (n) ) + 0x00000004U )
#define FMB_REG_DMA_SEO_DESCRIPTOR_DMA_SIZE( n )	\
	(FMB_REG_DMA_SEO_DESCRIPTOR_BASE + ( FMB_REG_DMA_SEO_DESCRIPTOR_SIZE * (n) ) + 0x00000008U )

#define FMB_REG_DMA_SEO_ACK_BASE				0x08004220U
#define FMB_REG_DMA_SEO_ACK_CONTINUITY_CNT		(FMB_REG_DMA_SEO_ACK_BASE + 0x00000000U)
#define FMB_REG_DMA_SEO_ACK_DATA_ID				(FMB_REG_DMA_SEO_ACK_BASE + 0x00000004U)
#define FMB_REG_DMA_SEO_ACK_RESULT				(FMB_REG_DMA_SEO_ACK_BASE + 0x00000008U)

#define FMB_REG_DMA_SEO_COMPLETE_BASE				0x08004240U
#define FMB_REG_DMA_SEO_COMPLETE_CONTINUITY_CNT		(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x00000000U)
#define FMB_REG_DMA_SEO_COMPLETE_DATA_ID			(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x00000004U)
#define FMB_REG_DMA_SEO_COMPLETE_RESULT				(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x00000008U)
#define FMB_REG_DMA_SEO_COMPLETE_TOTAL_SIZE			(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x0000000CU)
#define FMB_REG_DMA_SEO_COMPLETE_LAST_MARKER		(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x00000010U)
#define FMB_REG_DMA_SEO_COMPLETE_ES_INFO_TOTAL_SIZE	(FMB_REG_DMA_SEO_COMPLETE_BASE + 0x00000014U)

#define FMB_REG_DMA_SEO_LAST_ES						0xFFFFFFFFU


#define FMB_REG_HSIO_BASE							0xF1E00000U
#define FMB_REG_HSIO_PCIE0_INT_REG_SET				(FMB_REG_HSIO_BASE + 0x00000090U)

#define FMB_REG_HSIO_PCIE0_INT_REG_CMD				    ( 1U << 0 )
#define FMB_REG_HSIO_PCIE0_INT_REG_CMD_RESULT_ACK		( 1U << 1 )

#define FMB_REG_HSIO_PCIE0_INT_REG_MESSAGE_ACK			( 1U << 2 )

#define FMB_REG_HSIO_PCIE0_INT_REG_DMA_VEI_REQ			( 1U << 16 )
#define FMB_REG_HSIO_PCIE0_INT_REG_DMA_VEI_COMPLETE_ACK	( 1U << 17 )

#define FMB_REG_HSIO_PCIE0_INT_REG_DMA_SEO_REQ			( 1U << 18 )
#define FMB_REG_HSIO_PCIE0_INT_REG_DMA_SEO_COMPLETE_ACK	( 1U << 19 )

#define FMB_REG_HSIO_PCIE1_INT_REG_CLR				(FMB_REG_HSIO_BASE + 0x00000194U)
#define FMB_REG_HSIO_PCIE1_INT_REG_STATUS			(FMB_REG_HSIO_BASE + 0x00000198U)

#define FMB_REG_MCPU_FIRMWARE_RESTART_REQUEST		0x01003E00U
#define FMB_MCPU_FIRMWARE_RESTART					0x00000001U

#define FMB_REG_ENCODE_WORK_BASE					0x14000000U
#define FMB_REG_ENCODE_WORK_END						(FMB_REG_ENCODE_WORK_BASE + 0x01800000U - 1U)

// streams
#define FMB_STREAM_ID_MAX	 	    				4
#define FMB_CONTROL_ID_MAX	 	    				FMB_STREAM_ID_MAX

// Main state information
#define FMB_MAIN_STATE_BOOT							0x00000001U
#define FMB_MAIN_STATE_INIT							0x00000002U
#define FMB_MAIN_STATE_ENCODE						0x00000003U
#define FMB_MAIN_STATE_ERROR						0xFFFFFFFFU

// VIN_ID state information
#define FMB_VIN_ID_STATE_STOP						0x00000001U
#define FMB_VIN_ID_STATE_START						0x00000002U

// EH_ID state information
#define FMB_EH_ID_STATE_STOP						0x00000001U
#define FMB_EH_ID_STATE_START						0x00000002U
#define FMB_EH_ID_STATE_READY_TO_STOP				0x00000003U

// Main state mode of encode
#define FMB_ENCODE_MODE_SINGLE_CH					0x00000001U
#define FMB_ENCODE_MODE_MULTI_CH					0x00000002U

/* Type of HEVC encoder firmware */
#define FMB_HEVC_ENCODER_FIRM_STD					0x00000001U
#define FMB_HEVC_ENCODER_FIRM_USER					0x00000002U

// commands
#define FMB_COMMAND_TARGET_MAIN						0x00000001U
#define FMB_COMMAND_TARGET_VIN_ID					0x00000002U
#define FMB_COMMAND_TARGET_EH_ID					0x00000003U
#define FMB_COMMAND_TARGET_GPIO						0x00000004U
#define FMB_COMMAND_TARGET_SPI						0x00000005U

#define FMB_COMMAND_ID_STATE_CHANGE					0x00000002U

#define FMB_COMMAND_ID_EH_ID_ENCODE_PARAM_CHANGE			0x00000003U
#define FMB_COMMAND_ID_EH_ID_ENCODE_PICTURE_TYPE_CHANGE		0x00000004U
#define FMB_COMMAND_ID_EH_ID_ENCODE_ADDITIONAL_INFO			0x00000005U

#define FMB_COMMAND_ID_GPIO_FUNCTION				0x00000001U
#define FMB_COMMAND_ID_GPIO_DIRECTION				0x00000002U
#define FMB_COMMAND_ID_GPIO_SET_VALUE				0x00000003U
#define FMB_COMMAND_ID_GPIO_GET_VALUE				0x00000004U

/* encoder param target */
#define FMB_CHANGE_PARAM_VBR                        0x00000001U
#define FMB_CHANGE_PARAM_CBR                        0x00000002U
#define FMB_CHANGE_PARAM_RESOLUTION					0x00000004U
#define FMB_CHANGE_PARAM_FRAME_RATE					0x00000008U

/* change the sequence or not */
#define FMB_CHANGE_PARAM_CONTINUOUS_SEQUENCE        0x00000001U
#define FMB_CHANGE_PARAM_DISCONTINUOUS_SEQUENCE     0x00000002U

/* picture type */
#define FMB_CHANGE_PICTURE_TYPE_IDR					0x00000001U

#define FMB_COMMAND_PARAM_CHANGE_ENCODE_PARAM_MAX	17
//#define FMB_COMMAND_PARAM_MAX						FMB_COMMAND_PARAM_CHANGE_ENCODE_PARAM_MAX
#define FMB_COMMAND_PARAM_SIZE						4

/* GPIO port */
#define FMB_GPIO_PORT_MAX							63

/* GPIO function information */
#define FMB_GPIO_FUNCTION_GPIO						0x00000001U
#define FMB_GPIO_FUNCTION_PERIPHERAL				0x00000002U

/* GPIO direction information */
#define FMB_GPIO_DIRECTION_IN						0x00000001U
#define FMB_GPIO_DIRECTION_OUT						0x00000002U

/* GPIO value information */
#define FMB_GPIO_VALUE_LOW							0x00000001U
#define FMB_GPIO_VALUE_HIGH							0x00000002U

#define FMB_COMMAND_ID_SPI_CONFIG					0x00000001U
#define FMB_COMMAND_ID_SPI_SEND_RECEIVE				0x00000002U

#define FMB_COMMAND_PARAM0_SPI_CONFIG_FORMAT		0x00000001U
#define FMB_COMMAND_PARAM0_SPI_CONFIG_SCR			0x00000002U
#define FMB_COMMAND_PARAM0_SPI_CONFIG_CPSDVSR		0x00000003U
#define FMB_COMMAND_PARAM0_SPI_CONFIG_SPO			0x00000004U
#define FMB_COMMAND_PARAM0_SPI_CONFIG_SPH			0x00000005U
#define FMB_COMMAND_PARAM0_SPI_SEND_DATA			0x00000001U
#define FMB_COMMAND_PARAM0_SPI_RECEIVE_DATA			0x00000002U

#define FMB_COMMAND_PARAM1_SPI_BIT_WIDTH_POSITION	16

#define FMB_SPI_BIT_WIDTH_TO_PARAM1(w)				((w) -1)

#define FMB_SPI_RESULT2_TO_BIT_WIDTH(w)				((w) +1)

#define FMB_COMMAND_RESULT2_SPI_RECEIVE_DATA_MASK	0x0000FFFFU
#define FMB_COMMAND_RESULT2_SPI_BIT_WIDTH_POSITION	16

#define FMB_SET_PARAM_BITS(target, name, value)	\
{ \
	(target) &= (~((FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_##name##_BIT_MASK) << \
				   (FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_##name##_POSITION))); \
	(target) |= (((value) & (FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_##name##_BIT_MASK)) << \
				 (FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_##name##_POSITION)); \
}

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_HSIZE_EH_POSITION							16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_HSIZE_EH_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_HSIZE_EH_BIT_MASK							( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_HSIZE_EH_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VSIZE_EH_POSITION							0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VSIZE_EH_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VSIZE_EH_BIT_MASK							( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VSIZE_EH_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_LEFT_POSITION						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_LEFT_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_LEFT_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_LEFT_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_RIGHT_POSITION						0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_RIGHT_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_RIGHT_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_RIGHT_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_TOP_POSITION							16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_TOP_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_TOP_BIT_MASK							( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_TOP_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_BOTTOM_POSITION						0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_BOTTOM_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_BOTTOM_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_CROP_BOTTOM_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_LEFT_POSITION				16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_LEFT_BIT_WIDTH				16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_LEFT_BIT_MASK				( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_LEFT_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_RIGHT_POSITION				0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_RIGHT_BIT_WIDTH				16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_RIGHT_BIT_MASK				( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_RIGHT_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_TOP_POSITION				16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_TOP_BIT_WIDTH				16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_TOP_BIT_MASK				( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_TOP_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_BOTTOM_POSITION				0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_BOTTOM_BIT_WIDTH			16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_BOTTOM_BIT_MASK				( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_PAN_SCAN_RECT_BOTTOM_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_SIGNAL_TYPE_POSITION				28
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_SIGNAL_TYPE_BIT_WIDTH				4
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_SIGNAL_TYPE_BIT_MASK				( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_SIGNAL_TYPE_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FORMAT_POSITION						24
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FORMAT_BIT_WIDTH					4
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FORMAT_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FORMAT_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FULL_RANGE_FLAG_POSITION			20
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FULL_RANGE_FLAG_BIT_WIDTH			4
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FULL_RANGE_FLAG_BIT_MASK			( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_VIDEO_FULL_RANGE_FLAG_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_DESCRIPTION_PRESENT_FLAG_POSITION	16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_DESCRIPTION_PRESENT_FLAG_BIT_WIDTH	4
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_DESCRIPTION_PRESENT_FLAG_BIT_MASK	( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_DESCRIPTION_PRESENT_FLAG_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_PRIMARIES_POSITION					8
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_PRIMARIES_BIT_WIDTH				8
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_PRIMARIES_BIT_MASK					( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_COLOUR_PRIMARIES_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_TRANSFER_CHARACTERISTICS_POSITION			0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_TRANSFER_CHARACTERISTICS_BIT_WIDTH		8
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_TRANSFER_CHARACTERISTICS_BIT_MASK			( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_TRANSFER_CHARACTERISTICS_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_MATRIX_COEFFS_POSITION					24
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_MATRIX_COEFFS_BIT_WIDTH					8
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_MATRIX_COEFFS_BIT_MASK					( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_MATRIX_COEFFS_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_ASPECT_RATIO_IDC_POSITION					24
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_ASPECT_RATIO_IDC_BIT_WIDTH				8
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_ASPECT_RATIO_IDC_BIT_MASK					( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_ASPECT_RATIO_IDC_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_WIDTH_POSITION						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_WIDTH_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_WIDTH_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_WIDTH_BIT_WIDTH ) - 1 )

#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_HEIGHT_POSITION						0
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_HEIGHT_BIT_WIDTH						16
#define FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_HEIGHT_BIT_MASK						( ( 0x00000001U << FMB_COMMAND_PARAM_ENCODE_PARAM_CHANGE_SAR_HEIGHT_BIT_WIDTH ) - 1 )

// command ACK
#define FMB_COMMAND_ACK_RESULT_OK					0x00000001U
#define FMB_COMMAND_ACK_RESULT_INPROGRESS			0x00000006U

// command result
#define FMB_COMMAND_RESULT_0_OK						0x00000001U

// message target
#define FMB_MESSAGE_TARGET_VALUE_VI					0x00000002U
#define FMB_MESSAGE_TARGET_VALUE_EH					0x00000006U

// vi message id
#define FMB_MESSAGE_ID_CAPTURE_START				0x00000001U
#define FMB_MESSAGE_ID_VSYNC						0x00000002U

// eh message id
#define FMB_MESSAGE_ID_ENCODE_ADDITIONAL_INFO		0x00000001U

// dma result
#define FMB_DMA_RESULT_OK							0x00000001U

// memory area allocate param
#define FMB_VRAW_Y_DATA_SIZE_CH0				(((4096 * 2304) * 10) / 8)     /* bytes */
#define FMB_VRAW_C_DATA_SIZE_CH0				(FMB_VRAW_Y_DATA_SIZE_CH0 / 2) /* bytes */
#define FMB_VRAW_Y_DATA_SIZE					(FMB_VRAW_Y_DATA_SIZE_CH0 / 4) /* bytes */
#define FMB_VRAW_C_DATA_SIZE					(FMB_VRAW_Y_DATA_SIZE / 2)     /* bytes */

#define FMB_TRANSFER_VEI_BUFFER_SIZE_CH0		(FMB_VRAW_Y_DATA_SIZE_CH0 + FMB_VRAW_C_DATA_SIZE_CH0 + FMB_VRAW_C_DATA_SIZE_CH0) /* bytes */
#define FMB_TRANSFER_VEI_BUFFER_SIZE			(FMB_VRAW_Y_DATA_SIZE + FMB_VRAW_C_DATA_SIZE + FMB_VRAW_C_DATA_SIZE) /* bytes */
#define FMB_TRANSFER_VEI_PIC_INFO_MIN_SIZE		(0x14) /* bytes (0x14 = Size from picture_serial_number to picture_number_of additional_datas) */
#define FMB_TRANSFER_VEI_PIC_INFO_MAX_SIZE		(FMB_TRANSFER_VEI_PIC_INFO_MIN_SIZE + ((4 + 4 + 256) * 16)) /* bytes */
#define FMB_TRANSFER_SEO_BUFFER_SIZE			(6 * 1024 * 1024) /* bytes */
#define FMB_TRANSFER_SEO_ES_INFO_SIZE			(0x7C + ((4 + 4 + 256) * 16)) /* bytes (0x7C = Size from es_serial_number to es_number_of additional_datas) */

#define FMB_TRANSFER_ALIGNMENT_SIZE 			(4 * 1024)

#define FMB_TRANSFER_VEI_SIZE_PER_DESCRIPTOR	(FMB_TRANSFER_VEI_BUFFER_SIZE_CH0 / FMB_REG_DMA_VEI_DESCRIPTOR_NUM)

#define FMB_TRANSFER_VEI_PHY_BUFFER_SIZE		(((FMB_TRANSFER_VEI_SIZE_PER_DESCRIPTOR + (FMB_TRANSFER_ALIGNMENT_SIZE - 1)) / FMB_TRANSFER_ALIGNMENT_SIZE) * FMB_TRANSFER_ALIGNMENT_SIZE)
#define FMB_TRANSFER_VEI_PHY_BUFFER_GROUP_NUM	2

#define FMB_TRANSFER_VEI_PHY_BUFFER_NUM_CH0  	(FMB_TRANSFER_VEI_BUFFER_SIZE_CH0 / FMB_TRANSFER_VEI_PHY_BUFFER_SIZE)
#define FMB_TRANSFER_VEI_PHY_BUFFER_NUM  		(FMB_TRANSFER_VEI_BUFFER_SIZE / FMB_TRANSFER_VEI_PHY_BUFFER_SIZE)

#define FMB_TRANSFER_SEO_PHY_BUFFER_NUM  		FMB_REG_DMA_SEO_DESCRIPTOR_NUM
#define FMB_TRANSFER_SEO_PHY_BUFFER_SIZE	  	(FMB_TRANSFER_SEO_BUFFER_SIZE / FMB_TRANSFER_SEO_PHY_BUFFER_NUM)
#define FMB_TRANSFER_SEO_PHY_BUFFER_GROUP_NUM	2

#endif
