﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class DeleteContainerImageRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API DeleteContainerImageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteContainerImage"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the container service for which to delete a registered container
   * image.</p>
   */
  inline const Aws::String& GetServiceName() const { return m_serviceName; }
  inline bool ServiceNameHasBeenSet() const { return m_serviceNameHasBeenSet; }
  template <typename ServiceNameT = Aws::String>
  void SetServiceName(ServiceNameT&& value) {
    m_serviceNameHasBeenSet = true;
    m_serviceName = std::forward<ServiceNameT>(value);
  }
  template <typename ServiceNameT = Aws::String>
  DeleteContainerImageRequest& WithServiceName(ServiceNameT&& value) {
    SetServiceName(std::forward<ServiceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the container image to delete from the container service.</p>
   * <p>Use the <code>GetContainerImages</code> action to get the name of the
   * container images that are registered to a container service.</p>
   * <p>Container images sourced from your Lightsail container service, that are
   * registered and stored on your service, start with a colon (<code>:</code>). For
   * example, <code>:container-service-1.mystaticwebsite.1</code>. Container images
   * sourced from a public registry like Docker Hub don't start with a colon. For
   * example, <code>nginx:latest</code> or <code>nginx</code>.</p>
   */
  inline const Aws::String& GetImage() const { return m_image; }
  inline bool ImageHasBeenSet() const { return m_imageHasBeenSet; }
  template <typename ImageT = Aws::String>
  void SetImage(ImageT&& value) {
    m_imageHasBeenSet = true;
    m_image = std::forward<ImageT>(value);
  }
  template <typename ImageT = Aws::String>
  DeleteContainerImageRequest& WithImage(ImageT&& value) {
    SetImage(std::forward<ImageT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_serviceName;

  Aws::String m_image;
  bool m_serviceNameHasBeenSet = false;
  bool m_imageHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
