﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codepipeline/CodePipelineRequest.h>
#include <aws/codepipeline/CodePipeline_EXPORTS.h>
#include <aws/codepipeline/model/ActionCategory.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodePipeline {
namespace Model {

/**
 */
class GetActionTypeRequest : public CodePipelineRequest {
 public:
  AWS_CODEPIPELINE_API GetActionTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetActionType"; }

  AWS_CODEPIPELINE_API Aws::String SerializePayload() const override;

  AWS_CODEPIPELINE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Defines what kind of action can be taken in the stage. The following are the
   * valid values:</p> <ul> <li> <p> <code>Source</code> </p> </li> <li> <p>
   * <code>Build</code> </p> </li> <li> <p> <code>Test</code> </p> </li> <li> <p>
   * <code>Deploy</code> </p> </li> <li> <p> <code>Approval</code> </p> </li> <li>
   * <p> <code>Invoke</code> </p> </li> <li> <p> <code>Compute</code> </p> </li>
   * </ul>
   */
  inline ActionCategory GetCategory() const { return m_category; }
  inline bool CategoryHasBeenSet() const { return m_categoryHasBeenSet; }
  inline void SetCategory(ActionCategory value) {
    m_categoryHasBeenSet = true;
    m_category = value;
  }
  inline GetActionTypeRequest& WithCategory(ActionCategory value) {
    SetCategory(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The creator of an action type that was created with any supported integration
   * model. There are two valid values: <code>AWS</code> and
   * <code>ThirdParty</code>.</p>
   */
  inline const Aws::String& GetOwner() const { return m_owner; }
  inline bool OwnerHasBeenSet() const { return m_ownerHasBeenSet; }
  template <typename OwnerT = Aws::String>
  void SetOwner(OwnerT&& value) {
    m_ownerHasBeenSet = true;
    m_owner = std::forward<OwnerT>(value);
  }
  template <typename OwnerT = Aws::String>
  GetActionTypeRequest& WithOwner(OwnerT&& value) {
    SetOwner(std::forward<OwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The provider of the action type being called. The provider name is specified
   * when the action type is created.</p>
   */
  inline const Aws::String& GetProvider() const { return m_provider; }
  inline bool ProviderHasBeenSet() const { return m_providerHasBeenSet; }
  template <typename ProviderT = Aws::String>
  void SetProvider(ProviderT&& value) {
    m_providerHasBeenSet = true;
    m_provider = std::forward<ProviderT>(value);
  }
  template <typename ProviderT = Aws::String>
  GetActionTypeRequest& WithProvider(ProviderT&& value) {
    SetProvider(std::forward<ProviderT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string that describes the action type version.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  GetActionTypeRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  ActionCategory m_category{ActionCategory::NOT_SET};

  Aws::String m_owner;

  Aws::String m_provider;

  Aws::String m_version;
  bool m_categoryHasBeenSet = false;
  bool m_ownerHasBeenSet = false;
  bool m_providerHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodePipeline
}  // namespace Aws
