﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/ModelCardExportOutputConfig.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateModelCardExportJobRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateModelCardExportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateModelCardExportJob"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name or Amazon Resource Name (ARN) of the model card to export.</p>
   */
  inline const Aws::String& GetModelCardName() const { return m_modelCardName; }
  inline bool ModelCardNameHasBeenSet() const { return m_modelCardNameHasBeenSet; }
  template <typename ModelCardNameT = Aws::String>
  void SetModelCardName(ModelCardNameT&& value) {
    m_modelCardNameHasBeenSet = true;
    m_modelCardName = std::forward<ModelCardNameT>(value);
  }
  template <typename ModelCardNameT = Aws::String>
  CreateModelCardExportJobRequest& WithModelCardName(ModelCardNameT&& value) {
    SetModelCardName(std::forward<ModelCardNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the model card to export. If a version is not provided, then
   * the latest version of the model card is exported.</p>
   */
  inline int GetModelCardVersion() const { return m_modelCardVersion; }
  inline bool ModelCardVersionHasBeenSet() const { return m_modelCardVersionHasBeenSet; }
  inline void SetModelCardVersion(int value) {
    m_modelCardVersionHasBeenSet = true;
    m_modelCardVersion = value;
  }
  inline CreateModelCardExportJobRequest& WithModelCardVersion(int value) {
    SetModelCardVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the model card export job.</p>
   */
  inline const Aws::String& GetModelCardExportJobName() const { return m_modelCardExportJobName; }
  inline bool ModelCardExportJobNameHasBeenSet() const { return m_modelCardExportJobNameHasBeenSet; }
  template <typename ModelCardExportJobNameT = Aws::String>
  void SetModelCardExportJobName(ModelCardExportJobNameT&& value) {
    m_modelCardExportJobNameHasBeenSet = true;
    m_modelCardExportJobName = std::forward<ModelCardExportJobNameT>(value);
  }
  template <typename ModelCardExportJobNameT = Aws::String>
  CreateModelCardExportJobRequest& WithModelCardExportJobName(ModelCardExportJobNameT&& value) {
    SetModelCardExportJobName(std::forward<ModelCardExportJobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The model card output configuration that specifies the Amazon S3 path for
   * exporting.</p>
   */
  inline const ModelCardExportOutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = ModelCardExportOutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = ModelCardExportOutputConfig>
  CreateModelCardExportJobRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_modelCardName;

  int m_modelCardVersion{0};

  Aws::String m_modelCardExportJobName;

  ModelCardExportOutputConfig m_outputConfig;
  bool m_modelCardNameHasBeenSet = false;
  bool m_modelCardVersionHasBeenSet = false;
  bool m_modelCardExportJobNameHasBeenSet = false;
  bool m_outputConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
