﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appflow/AppflowRequest.h>
#include <aws/appflow/Appflow_EXPORTS.h>
#include <aws/appflow/model/ConnectorType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Appflow {
namespace Model {

/**
 */
class ListConnectorEntitiesRequest : public AppflowRequest {
 public:
  AWS_APPFLOW_API ListConnectorEntitiesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListConnectorEntities"; }

  AWS_APPFLOW_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The name of the connector profile. The name is unique for each
   * <code>ConnectorProfile</code> in the Amazon Web Services account, and is used to
   * query the downstream connector. </p>
   */
  inline const Aws::String& GetConnectorProfileName() const { return m_connectorProfileName; }
  inline bool ConnectorProfileNameHasBeenSet() const { return m_connectorProfileNameHasBeenSet; }
  template <typename ConnectorProfileNameT = Aws::String>
  void SetConnectorProfileName(ConnectorProfileNameT&& value) {
    m_connectorProfileNameHasBeenSet = true;
    m_connectorProfileName = std::forward<ConnectorProfileNameT>(value);
  }
  template <typename ConnectorProfileNameT = Aws::String>
  ListConnectorEntitiesRequest& WithConnectorProfileName(ConnectorProfileNameT&& value) {
    SetConnectorProfileName(std::forward<ConnectorProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The type of connector, such as Salesforce, Amplitude, and so on. </p>
   */
  inline ConnectorType GetConnectorType() const { return m_connectorType; }
  inline bool ConnectorTypeHasBeenSet() const { return m_connectorTypeHasBeenSet; }
  inline void SetConnectorType(ConnectorType value) {
    m_connectorTypeHasBeenSet = true;
    m_connectorType = value;
  }
  inline ListConnectorEntitiesRequest& WithConnectorType(ConnectorType value) {
    SetConnectorType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> This optional parameter is specific to connector implementation. Some
   * connectors support multiple levels or categories of entities. You can find out
   * the list of roots for such providers by sending a request without the
   * <code>entitiesPath</code> parameter. If the connector supports entities at
   * different roots, this initial request returns the list of roots. Otherwise, this
   * request returns all entities supported by the provider. </p>
   */
  inline const Aws::String& GetEntitiesPath() const { return m_entitiesPath; }
  inline bool EntitiesPathHasBeenSet() const { return m_entitiesPathHasBeenSet; }
  template <typename EntitiesPathT = Aws::String>
  void SetEntitiesPath(EntitiesPathT&& value) {
    m_entitiesPathHasBeenSet = true;
    m_entitiesPath = std::forward<EntitiesPathT>(value);
  }
  template <typename EntitiesPathT = Aws::String>
  ListConnectorEntitiesRequest& WithEntitiesPath(EntitiesPathT&& value) {
    SetEntitiesPath(std::forward<EntitiesPathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the API that's used by the connector.</p>
   */
  inline const Aws::String& GetApiVersion() const { return m_apiVersion; }
  inline bool ApiVersionHasBeenSet() const { return m_apiVersionHasBeenSet; }
  template <typename ApiVersionT = Aws::String>
  void SetApiVersion(ApiVersionT&& value) {
    m_apiVersionHasBeenSet = true;
    m_apiVersion = std::forward<ApiVersionT>(value);
  }
  template <typename ApiVersionT = Aws::String>
  ListConnectorEntitiesRequest& WithApiVersion(ApiVersionT&& value) {
    SetApiVersion(std::forward<ApiVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items that the operation returns in the response.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListConnectorEntitiesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that was provided by your prior <code>ListConnectorEntities</code>
   * operation if the response was too big for the page size. You specify this token
   * to get the next page of results in paginated response.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListConnectorEntitiesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectorProfileName;

  ConnectorType m_connectorType{ConnectorType::NOT_SET};

  Aws::String m_entitiesPath;

  Aws::String m_apiVersion;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_connectorProfileNameHasBeenSet = false;
  bool m_connectorTypeHasBeenSet = false;
  bool m_entitiesPathHasBeenSet = false;
  bool m_apiVersionHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Appflow
}  // namespace Aws
